/*
  Copyright (C) 2004-2005 Tommi Tervonen, Petteri Klemola, Pasi Orovuo

  This file is part of Kajaani Kombat.

  Kajaani Kombat is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  Kajaani Kombat is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with Kajaani Kombat; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "game_server.h"

game_server::~game_server()
{
  shutdown_server();
  SDLNet_TCP_Close(ssocket);
  if (terra)
    delete terra;
  while (!output_queue.empty())
    delete output_queue.pop();
  SDL_DestroySemaphore(pl_arr_smp);
  SDL_DestroySemaphore(d_smp);
}

game_server::game_server(IPaddress _ip, int _num_castles, int _wins) throw(string &)
{
  con_th = 0;
  ip = _ip;
  character_selected=0;
  castles_placed = 0;
  mode = ID_MODE_NOTACTIVE;
  num_castles = _num_castles;
  wins = _wins;
  game_on = false;
  terra = NULL;
  round = 1;

  /* init semaphoress */
  pl_arr_smp = SDL_CreateSemaphore(1); 
  d_smp = SDL_CreateSemaphore(1); 


  // init net on init_net!
}

void game_server::init_net()
{
  ssocket = SDLNet_TCP_Open (&ip);

  if (ssocket == NULL)
    {
      throw string ("[SRV] Can't bind to port: ")+string(SDLNet_GetError());
    }
  
  // OK, init server thread to listen to clients
  con_th = SDL_CreateThread (&game_server::connection_accepter, (void *) this);
  if (con_th < 0)
    {
      fprintf (stderr, "[SRV] Error starting thread: %s\n", SDL_GetError());
      exit(2);
    }
}

void game_server::shutdown_server()
{
  if (con_th)
    {
      mode = ID_MODE_SHUTTING_DOWN;
      SDL_WaitThread(con_th, NULL);
      con_th = 0;
    }
  if (SDL_SemWait(d_smp) == -1) {
    fprintf (stderr, "[SRV] Error locking semaphore: %s\n", SDL_GetError());
    exit(1);
  }
  while (conns.size() > 0)
    {

      remove_connect(conns[0]);

    }
  if (SDL_SemPost (d_smp) == -1)
    {
      fprintf (stderr, "[SRV] Fatal error (in semaphore unlocking): %s .. server will shut down.\n", SDL_GetError());
      exit (1);
    }

}

void game_server::remove_connect(connects *c)
{

  vector<connects *>::iterator it = find(conns.begin(), conns.end(), c);
  assert (it != conns.end());
  const vector<int> &v = c->get_player_indices();
  for (unsigned int i=0;i<v.size();i++)
    remove_player(v[i]);

  conns.erase(it);

  // if it's not a local connect, delete it (local connects are deleted outside the server obj)
  if (typeid (*c) != typeid(local_connects))
    delete c;

#ifdef DEBUG
  printf ("[SRV] Removed connect, %d connects active\n", (int) conns.size());
#endif

  game_on = conns.size() > 0;

}

void game_server::start_game()
{
  assert(con_th);

  output_queue.push(new msg_start_game(wins));

  round = 1;
  do{
    if (terra) 
      {
	delete terra;
	terra = 0;
      }
    
    data_transfer();
    //      printf ("Creating terrain:\n");
    //printf ("- Placing %d castles ...\n", num_castles);
    terra = new terrain (num_castles);

#ifdef DEBUG
    printf ("[SRV] - Assigning castles to %d players ...\n", no_players());
#endif
    assign_default_castles();

#ifdef DEBUG
    printf ("\tdone.\n");

    printf( "[SRV] - Exchanging data with clients...\n" );
#endif

    if (round == 1)
      dispatch_playerinfo();
#ifdef DEBUG
    printf( "\tdone.\n" );
#endif
    
    const vector<coord> v = terra->get_castle_centers();
    vector<coord>::const_iterator it = v.begin();
    while (it != v.end())
      {
	coord c = *it;
	tile &ot = terra->get_tile(c);
#ifdef DEBUG
	printf ("[SRV] Sending castle at (%d,%d)\n", c.getTX(), c.getTY());
#endif
	output_queue.push (new msg_castle_location(c, ot.owner));
	it++;
      }
    
    
    //      printf ("Waiting for playfield synchronization to complete ...");
    fflush (stdout);
    
    data_transfer();
    
    //      printf (" done.\n");
    
    main_gameloop();
    round++;
    if (terra) 
      {
	delete terra;
	terra = 0;
      }
  }while(game_on && no_alive_players() > 0);//game is on if no player has enough wins
}

void game_server::main_gameloop()
{
  game_won = false;
  game_on = true;

  //  printf ("Entering CASTLE SELECT mode\n");
  mode_character_select();
  mode_castle_select();
  if (no_alive_players() < 2) return;

  while ( !game_won )
    {
      //      printf ("Entering PLACE CANNONS mode.\n");
      mode_place_cannons();
      if (no_alive_players() < 2) return;
      
      //      printf ("Entering SHOOTING mode.\n");
      mode_shooting();
      if (no_alive_players() < 2) return;
      
      //      printf ("Entering REPAIR mode.\n");
      mode_repair();
      if (no_alive_players() < 2) return;
      
      //      printf ("Entering ENDROUND mode.\n");
      mode_end_round();
      
      if ( game_won == true ) {
		mode_end_game();
		return;
      }
      if (no_alive_players() < 2) return;
    }
}


void game_server::mode_place_cannons()
{
  mode = ID_MODE_PLACEC;
  clear_connection_readys();

  terra->check_conquered();

  // fill amount of cannons / player
  for (int i=0;i<MAX_PLAYERS;i++)
	{
	  if (players[i].is_alive()) continue;
	  vector<const tile *> cas = terra->conquered_tiles(tile::TYPE_CASTLE_C, i);
	  cannons_to_place[i] = cas.size()*2;
	  big_cannons_to_place[i] = cas.size()*1;
	  //	  printf ("%d cannons for player %d\n", cas.size()*2, i);
	}

  output_queue.push (new msg_start_placecan());

  while (no_ready_connections() < no_connections()) {
    data_transfer();
  }
}

int game_server::no_ready_connections() const
{
  if (SDL_SemWait(d_smp) == -1){
    fprintf (stderr, "[SRV] Error locking semaphore: %s\n", SDL_GetError());
    exit(1);
  }
  int c = 0;
  for (unsigned int i=0;i<conns.size();i++)
    if (conns[i]->is_ready()) c++;
  

  if (SDL_SemPost (d_smp) == -1)
    {
      fprintf (stderr, "[SRV] Fatal error (in semaphore unlocking): %s .. server will shut down.\n", SDL_GetError());
      exit (1);
    }

  return c;
}

int game_server::no_connections() const
{
  return conns.size();
}

void game_server::mode_shooting()
{
  mode = ID_MODE_SHOOT;
  output_queue.push (new msg_start_shoot());

  clear_connection_readys();
  // osion odotus, odotetaan kunnes kaikki pelaajat haluavat lopettaa.
  while (no_ready_connections() < no_connections())
    data_transfer();
}

void game_server::mode_repair()
{
  mode = ID_MODE_REPAIR;
  clear_connection_readys();

  terra->check_conquered();

  for (int i=0;i<MAX_PLAYERS;i++)
    {
      block_placers[i].set_type(block::TYPE_L);
      block_placers[i].set_orientation(0);
      if ( players[i].is_alive())
	{
	  coord c = coord::tc(-1,-1);
	  block_placers[i].set_center(c);
	  continue;
	}
    }
  
  output_queue.push (new msg_start_repair());
  data_transfer();
  
  const vector<coord> cent = terra->get_castle_centers();
  vector<coord>::const_iterator it = cent.begin();
  while (it != cent.end()) {
    const coord c = *it++;
    tile t = terra->get_tile (c);
    if (t.owner == OWNER_NOONE)
      continue;
    
    block_placers[t.owner].set_center(c);
    block_placers[t.owner].set_type(block::TYPE_L);
  }
  
  // osion odotus, odotetaan kunnes kaikki pelaajat haluavat lopettaa.
  while (no_ready_connections() < no_connections())
    data_transfer();
}

void game_server::mode_end_round()
{
  int count = 0;
  bool some_dead = false;
  /*Muita paikkoja pitisi varmaan kyll kans muokata niin ett tippuneilta pelaajilta
    tulevat messut ignorataan tai jotain...
  */
  
  data_transfer();
  output_queue.push (new msg_start_endround());
  clear_connection_readys();
  mode = ID_MODE_ENDR;
  
  // Toi vitun MSG_ENDR_START ei ehdi clienteille ajoissa (tai ne ei ehdi vaihtaa
  // modea) ennen kuin aletaan lhetell MSG_DEAD_PLAYER
  // --- eikhn ehi, tai jos ei niin korjataan muulla tavalla. -tommi
  //SDL_Delay(100);

  terra->check_conquered();

  for ( int i = 0; i < MAX_PLAYERS; i++ ) 
    {
      if ( players[i].is_connected()) 
	{
	  vector<const tile *>	castles = terra->conquered_tiles( tile::TYPE_CASTLE_C, i );
	  
	  if ( castles.size() == 0) 
	    {
#ifdef DEBUG
	      printf( "[SRV] Player %d is dead\n", i );
#endif
	      players[i].kill();
	      some_dead = true;
	      
	      if (terra)
		terra->dead_player_remove(i);
	    }
	  else 
	    {
#ifdef DEBUG
	      printf( "[SRV] Got %d conquered castles for player %d.\n", (int) castles.size(), i );
#endif
	      count++;
	    }
	} // end if
    } // end for

  data_transfer();

  if ( count < 2 && some_dead) {
    // Eli meilll on yksi tai vhemmn pelaajia, ja joku on kuollut
    game_won = true;
    if (count == 1)
      for (int i=0;i<MAX_PLAYERS;i++)
	if (players[i].is_alive())
	  {
	    players[i].win();
	    break;
	  }
    
  }

  // osion odotus, odotetaan kunnes kaikki pelaajat haluavat lopettaa.
  while (no_ready_connections() < no_connections())
    data_transfer();
}

void game_server::mode_end_game() {
  mode = ID_MODE_ENDGAME;
  output_queue.push( new msg_start_endgame());
  clear_connection_readys();

  data_transfer();
  /*
    Tss pitisi lhett clienteille tiedot voittajasta ja pisteist jne???
  */
  //  printf( "ROUND OVER\n" );

  /* seuraavassa tyhjennetn kaikki tiedot seuraavaa pelikierrosta varten */
  castles_placed=0;
  
  for (int i=0;i<MAX_PLAYERS;i++)
    if(players[i].is_connected()) players[i].rebirth();
  
  //vaihettiin kohtaa
  while (no_ready_connections() < no_connections())
    data_transfer();


  /* lets calculate if game is over */
  int mostwins=0;

  for (int i=0; i < MAX_PLAYERS; i++)
    {
      if(players[i].get_wins() > mostwins)
	mostwins = players[i].get_wins();
    }
  if(mostwins == wins)
    {
      output_queue.push (new msg_game_over());
      game_on=false;
    }
  else
    output_queue.push (new msg_start_game());

  data_transfer();

  //0.5 versioo varten nopee korjaus
  clear_connection_readys();
  while (no_ready_connections() < no_connections())
    data_transfer();
}

void game_server::mode_character_select()
{
  mode = ID_MODE_CHSELECT;

  output_queue.push (new msg_start_castleselect());
  
  /* delay until all players have placed castles */
  while (character_selected < no_alive_players())
    data_transfer();
  
  //  printf ("All players have chosen starting castles.\n");

  //build_default_walls(); // build and send default walls.

  // checking conquered
  //terra->check_conquered();
}


void game_server::mode_castle_select()
{
  mode = ID_MODE_CSELECT;

  output_queue.push (new msg_start_castleselect());
  
  /* delay until all players have placed castles */
  while (castles_placed < no_alive_players())
    data_transfer();
  
  //  printf ("All players have chosen starting castles.\n");

  build_default_walls(); // build and send default walls.

  // checking conquered
  terra->check_conquered();
}


void game_server::build_default_walls()
{
  clear_connection_readys();

  data_transfer();

  // Odotellessa rakennetaan omat muurit
  const vector <coord> vec = terra->get_castle_centers();
  vector<coord>::const_iterator it = vec.begin();

  int x, y;

  while (it != vec.end())
    {
      const coord coo = *it++;
      tile &t = terra->get_tile(coo);
      if (t.owner == OWNER_NOONE)
	continue;
      
      // wall north of castle
      y = coo.getTY() - CASTLE_WALL_RADIUS;
      for (x=coo.getTX()-CASTLE_WALL_RADIUS;x<=coo.getTX()+CASTLE_WALL_RADIUS;x++)
	{
	  coord c = coord::tc(x,y);
	  tile t (tile::TYPE_WALL, t.owner);
	  terra->set_tile (c, t);
	}
      // west wall
      x = coo.getTX() - CASTLE_WALL_RADIUS;
      for (y=coo.getTY()-CASTLE_WALL_RADIUS+1;y<=coo.getTY()+CASTLE_WALL_RADIUS;y++)
	{
	  coord c = coord::tc(x,y);
	  tile t (tile::TYPE_WALL, t.owner);
	  terra->set_tile (c, t);
	}
      // south wall
      y = coo.getTY() + CASTLE_WALL_RADIUS;
      for (x=coo.getTX()-CASTLE_WALL_RADIUS+1;x<=coo.getTX()+CASTLE_WALL_RADIUS;x++)
	{
	  coord c = coord::tc(x,y);
	  tile t (tile::TYPE_WALL, t.owner);
	  terra->set_tile (c, t);
	}
      // east wall
      x = coo.getTX() + CASTLE_WALL_RADIUS;
      for (y=coo.getTY()-CASTLE_WALL_RADIUS+1;y<=coo.getTY()+CASTLE_WALL_RADIUS-1;y++)
	{
	  coord c = coord::tc(x,y);
	  tile t (tile::TYPE_WALL, t.owner);
	  terra->set_tile (c, t);
	}
      // walls built for the castle
    } // end while  
}


void game_server::data_transfer()
{

  if (SDL_SemWait(d_smp) == -1) {
    exit(1);
  }

  msg *m = 0;

  for (unsigned int i=0;i<conns.size();i++)
    {
      while (conns[i]->has_msgs())
	{
	  try {
	    m = conns[i]->receive();
	  }
	  catch (string &s) 
	    {
	      remove_connect(conns[i]);
	      if (SDL_SemPost (d_smp) == -1)
		{
		  fprintf (stderr, "[SRV] Fatal error (in semaphore unlocking): %s .. server will shut down.\n", SDL_GetError());
		  exit (1);
		}

	      data_transfer();
	      return;
	    }

	  if (m->get_id() == msg::DISCONNECT)
	    {
	      remove_connect(conns[i]);
	      if (SDL_SemPost (d_smp) == -1)
		{
		  fprintf (stderr, "[SRV] Fatal error (in semaphore unlocking): %s .. server will shut down.\n", SDL_GetError());
		  exit (1);
		}
	      data_transfer();
	      return;
	    }

	  switch (mode)
	    {
	    case ID_MODE_NOTACTIVE:
	      if ( m->get_id() == msg::TALKMESSAGE)
		output_queue.push (new msg_talkmessage(*(msg_talkmessage *)m));
	      else
		assert(0); // no msgs should come in this mode.
	      break;
	    case ID_MODE_PLACEC:
	      if (m->get_id() == msg::END)
		conns[i]->set_ready(true);
	      else if ( m->get_id() == msg::TALKMESSAGE)
		output_queue.push (new msg_talkmessage(*(msg_talkmessage *)m));
	      else if (m->get_id() == msg::MOVE_CANNONPLACER)
		output_queue.push (new msg_move_cannonplacer(*(msg_move_cannonplacer *)m));
	      else if (m->get_id() == msg::PLACE_CANNON)
		{
		  msg_place_cannon *mpc = dynamic_cast<msg_place_cannon *>(m);
		  process_place_cannon (mpc->get_owner(), mpc->get_position());
		}
	      else if (m->get_id() == msg::PLACE_BIG_CANNON)
		{
		  msg_place_big_cannon *mpc = dynamic_cast<msg_place_big_cannon *>(m);
		  process_place_big_cannon (mpc->get_owner(), mpc->get_position());
		}
	      else if (m->get_id() == msg::CHANGE_CANNON)
		{
		  msg_change_cannon *mpc = dynamic_cast<msg_change_cannon *>(m);
		  int ow = mpc->get_owner();
		  process_change_cannon(ow); 
		  //process_place_big_cannon (mpc->get_owner(), mpc->get_position());
		}
	      else
		fprintf (stderr, "[SRV] Received unknown message: %d\n", m->get_id());
	      break;
	    case ID_MODE_SHOOT:
	      if (m->get_id() == msg::END)
		conns[i]->set_ready(true);
	      else if (m->get_id() == msg::MOVE_CURSOR)
		output_queue.push(new msg_move_cursor(*(msg_move_cursor*)m));
	      else if ( m->get_id() == msg::TALKMESSAGE)
		output_queue.push (new msg_talkmessage(*(msg_talkmessage *)m));
	      else if (m->get_id() == msg::SHOOT)
		{
		  msg_shoot *ms = dynamic_cast<msg_shoot *>(m);
		  process_shoot(ms->get_to(), ms->get_shooter(), ms->get_ammosize());
		  output_queue.push(new msg_shoot(*ms));
		}
	      else 
		fprintf (stderr, "[SRV] Received unknown message: %d\n", m->get_id());
	      break;
	    case ID_MODE_REPAIR:
	      if (m->get_id() == msg::END)
		conns[i]->set_ready(true);
	      else if ( m->get_id() == msg::TALKMESSAGE)
		output_queue.push (new msg_talkmessage(*(msg_talkmessage *)m));
	      else if (m->get_id() == msg::MOVE_BLOCKPLACER)
		{
		  msg_move_blockplacer *mvb = dynamic_cast<msg_move_blockplacer *>(m);
		  block_placers[mvb->get_owner()].set_center(mvb->get_position());
		  output_queue.push(new msg_move_blockplacer(*mvb));
		}
	      else if (m->get_id() == msg::PLACE_BLOCK)
		{
		  msg_place_block *mpb = dynamic_cast<msg_place_block *>(m);
		  process_place_block (mpb->get_owner(), mpb->get_next_type(), mpb->get_next_orientation());
		  output_queue.push(new msg_place_block(*mpb));
		}
	      else if (m->get_id() == msg::ROTATE_BL)
		{
		  msg_rotate_block *mr = dynamic_cast<msg_rotate_block *>(m);
		  block_placers[mr->get_owner()].rotate();
		  output_queue.push(new msg_rotate_block(*mr));
		}
	      else
		fprintf (stderr, "[SRV] Received unknown message: %d.\n", m->get_id()); // unknown msg
	      break;
	    case ID_MODE_ENDR:
	      if (m->get_id() == msg::END)
		conns[i]->set_ready(true);
	      else if ( m->get_id() == msg::TALKMESSAGE)
		output_queue.push (new msg_talkmessage(*(msg_talkmessage *)m));
	      else
		fprintf (stderr, "[SRV] Received unknown message: %d.\n", m->get_id()); // unknown msg
	      break;
	    case ID_MODE_ENDGAME:
	      if(m->get_id()==msg::END)
		conns[i]->set_ready(true);
	      else if ( m->get_id() == msg::TALKMESSAGE)
		output_queue.push (new msg_talkmessage(*(msg_talkmessage *)m));
	      else
		fprintf (stderr, "[SRV] Received unknown message: %d.\n", m->get_id()); // unknown msg
	      break;
	    case ID_MODE_CHSELECT:
	      if (m->get_id() == msg::CHOOSE_CHARACTER)
		{
		  msg_choose_character *mpc = dynamic_cast<msg_choose_character *>(m);
		  process_choose_character (mpc->get_owner(), mpc->get_position(), mpc->get_character());
		}
	      else if (m->get_id() == msg::MOVE_CHARACTERCHOOSER)
		output_queue.push (new msg_move_characterchooser(*(msg_move_characterchooser *)m));
	      else if ( m->get_id() == msg::END) 
		conns[i]->set_ready(true);
	      else if ( m->get_id() == msg::TALKMESSAGE)
		output_queue.push (new msg_talkmessage(*(msg_talkmessage *)m));
	      else
		fprintf (stderr, "[SRV] Received unknown message: %d.\n", m->get_id()); // unknown msg
	      break;
	    case ID_MODE_CSELECT:
	      if (m->get_id() == msg::MOVE_CHOOSER_REQ)
		{
		  msg_move_chooser_req *mm = dynamic_cast<msg_move_chooser_req *>(m);
		  process_chooser_req(mm->get_owner(), mm->is_down());
		}
	      else if (m->get_id() == msg::CHOOSER_LOCK)
		{
		  msg_chooser_lock *mm = dynamic_cast<msg_chooser_lock *>(m);
		  process_chooser_lock(mm->get_owner());
		}
	      else if ( m->get_id() == msg::TALKMESSAGE)
		output_queue.push (new msg_talkmessage(*(msg_talkmessage *)m));
	      else if ( m->get_id() == msg::END) 
		conns[i]->set_ready(true);
	      else
		fprintf (stderr, "[SRV] Received unknown message: %d.\n", m->get_id()); // unknown msg

	      break;
	    case ID_MODE_SHUTTING_DOWN:
	      break;
	    default:
	      /* should never get here */
	      assert(0);
	      break;
	    } // end switch
	  delete m;
	}
    }

  // msgs received, send + filter pending msgs 

  while (!output_queue.empty())
    {
      msg *sm = output_queue.pop();
      for (unsigned int i=0;i<conns.size();i++)
	{
  
	  if (typeid(*sm) == typeid(msg_move_cannonplacer))
	    {
	      msg_move_cannonplacer *mc = dynamic_cast<msg_move_cannonplacer *>(sm);
	      if (conns[i]->includes_player(mc->get_owner()))
		continue;
	    }
	  else if (typeid(*sm) == typeid(msg_move_characterchooser))
	    {
	      msg_move_characterchooser *mc = dynamic_cast<msg_move_characterchooser *>(sm);
	      if (conns[i]->includes_player(mc->get_owner()))
		continue;
	    }
	  else if (typeid(*sm) == typeid(msg_talkmessage))
	    {
	      msg_talkmessage *mc = dynamic_cast<msg_talkmessage *>(sm);
	      if (conns[i]->includes_player(mc->get_owner()))
		continue;
	    }
	  else if (typeid(*sm) == typeid(msg_move_blockplacer))
	    {
	      msg_move_blockplacer *mc = dynamic_cast<msg_move_blockplacer *>(sm);
	      if (conns[i]->includes_player(mc->get_owner()))
		continue;
	    }
	  else if (typeid(*sm) == typeid(msg_place_cannon))
	    {
	      msg_place_cannon *mc = dynamic_cast<msg_place_cannon *>(sm);
	      if (conns[i]->includes_player(mc->get_owner()))
		continue;
	    }
	  else if (typeid(*sm) == typeid(msg_place_big_cannon))
	    {
	      msg_place_big_cannon *mc = dynamic_cast<msg_place_big_cannon *>(sm);
	      if (conns[i]->includes_player(mc->get_owner()))
		continue;
	    }
	  else if (typeid(*sm) == typeid(msg_change_cannon))
	    {
	      msg_change_cannon *mc = dynamic_cast<msg_change_cannon *>(sm);
	      if (conns[i]->includes_player(mc->get_owner()))
		continue;
	    }
	  else if (typeid(*sm) == typeid(msg_chooser_lock))
	    {
	      // send always
	    }
	  else if (typeid(*sm) == typeid(msg_rotate_block))
	    {
	      msg_rotate_block *mc = dynamic_cast<msg_rotate_block *>(sm);
	      if (conns[i]->includes_player(mc->get_owner()))
		continue;
	    }
	  else if (typeid(*sm) == typeid(msg_place_block))
	    {
	      msg_place_block *mc = dynamic_cast<msg_place_block *>(sm);
	      if (conns[i]->includes_player(mc->get_owner()))
		continue;
	    }
	  else if (typeid(*sm) == typeid(msg_shoot))
	    {
	      msg_shoot *mc = dynamic_cast<msg_shoot *>(sm);
	      if (conns[i]->includes_player(mc->get_shooter()))
		continue;
	    }
	  else if (typeid(*sm) == typeid(msg_move_cursor))
	    {
	      msg_move_cursor *mc = dynamic_cast<msg_move_cursor *>(sm);

	      if (conns[i]->includes_player(mc->get_owner()))
		continue;
	    }
	  try {
	    conns[i]->send(sm);
	  }
	  catch (string &e)
	    {
	      if (SDL_SemWait(d_smp) == -1) {
		exit(1);
	      }      
	      remove_connect(conns[i]);
	      if (SDL_SemPost (d_smp) == -1)
		{
		  fprintf (stderr, "[SRV] Fatal error (in semaphore unlocking): %s .. server will shut down.\n", SDL_GetError());
		  exit (1);
		}
	    }
	}
      delete sm;
    }	  
  if (SDL_SemPost (d_smp) == -1)
    {
      fprintf (stderr, "[SRV] Fatal error (in semaphore unlocking): %s .. server will shut down.\n", SDL_GetError());
      exit (1);
    }
  SDL_Delay(1);
}

int game_server::connection_accepter (void *data)
{
  game_server *gs = (game_server *) data;

  TCPsocket s;
  int pl_ind;
  msg *m = 0;
  connects *c = 0;

  while (gs->mode == ID_MODE_NOTACTIVE)
    {
      s = SDLNet_TCP_Accept(gs->ssocket);
      if (!s)
	{
	  SDL_Delay(ACCEPT_YIELDWAIT);
	}
      else
	{
	  pl_ind = -1;
	  try {
	    m = msg::recv(s);
	    msg_clienthello *mch = dynamic_cast<msg_clienthello *>(m);
	    if (mch == 0)
	      throw string ("Error: received message type not clienthello.");
	    
	    int nump = mch->get_num_players();
	    if (gs->no_players() + nump > MAX_PLAYERS) // should inform client
	      throw string ("Error: too many players trying to connect.");

	    c = new tcp_connects(s);
	    
	    for (int i=0;i<nump;i++)
	      {
		pl_ind = gs->add_player (mch->get_name(i), c);
		c->add_player(pl_ind);
		if (pl_ind < 0)
		  throw string ("Error adding player.\n");
#ifdef DEBUG
		printf ("[SRV] Player (%s) added to index %d\n", gs->players[pl_ind].get_name().c_str(), pl_ind);
#endif
	      }
	    msg_serverhello reply (c->get_player_indices());
	    reply.send(s);
	    gs->conns.push_back(c);
#ifdef DEBUG
	    printf ("[SRV] Added connection with %d players.\n", c->num_players());
#endif
	    gs->game_server::dispatch_playerinfo();
	  }
	  catch (string &st) {
#ifdef DEBUG
	    printf ("[SRV] Error in handshake: %s\n", st.c_str());
#endif
	    if (SDL_SemWait(gs->d_smp) == -1)
	      exit(1);
	    if (c)
	      gs->remove_connect(c);
	    if (SDL_SemPost (gs->d_smp) == -1)
	      {
		fprintf (stderr, "[SRV] Fatal error (in semaphore unlocking): %s .. server will shut down.\n", SDL_GetError());
		exit (1);
	      }


	  }	  
	  if (m) {delete m; m=0; }
	  c = 0;
	} // end else
    }
  return 0;
}

void game_server::process_chooser_lock (int player_index)
{
  // tt pit parantaa tekemn tarkistus onko pelaaja jo lukinnut
  // linnanvalitsimensa
  output_queue.push (new msg_chooser_lock(player_index));
  castles_placed++;
}
int game_server::no_players() const
{
  int c = 0;
  for (int i=0;i<MAX_PLAYERS;i++)
    if (players[i].is_connected())
      c++;
  
  return c;
}

int game_server::no_alive_players() const
{
  int c = 0;
  for (int i=0;i<MAX_PLAYERS;i++)
    if (players[i].is_alive())
      c++;
  return c;
}

int game_server::add_player (const string &name, connects *c)
{
  int i;
  int ret = 0;

  if (SDL_SemWait (pl_arr_smp) == -1)
    {
      fprintf (stderr, "[SRV] Error locking semaphore: %s\n", SDL_GetError());
      exit(2);
    }
  
  if (no_players() == MAX_PLAYERS)
    {
      fprintf (stderr, "[SRV] Maximum number of players joined, join refused!\n");
      return -1;
    }
  else
    {
      for (i=0;i<MAX_PLAYERS;i++)
	if (!players[i].is_connected())
	  break;
      ret = i;
      players[i] = player(c, name);
    }
  
  if (SDL_SemPost (pl_arr_smp) == -1)
	{
	  fprintf (stderr, "[SRV] Fatal error (in semaphore unlocking): %s .. server will shut down.\n", SDL_GetError());
	  exit (1);
	}
  return ret;
}

bool game_server::remove_player(int index)
{
  assert (index < MAX_PLAYERS);

  if (SDL_SemWait (pl_arr_smp) == -1)
    {
      fprintf (stderr, "[SRV] Fatal error on locking semaphore: %s. Server will shut down.\n", SDL_GetError());
      exit(1);
    }

  if (!players[index].is_connected())
    {
      fprintf (stderr, "[SRV] Error removing player - not connected\n");
      if (SDL_SemPost (pl_arr_smp) == -1)
	{
	  fprintf (stderr, "[SRV] Fatal error (in semaphore unlocking): %s .. server will shut down.\n", SDL_GetError());
	  exit(1);
	}
      return false;
    }

  players[index] = player();

  if (mode != ID_MODE_NOTACTIVE)
    {
      cannons_to_place[index] = 0;
      home_castles[index] = coord::tc(-1,-1);
      if (terra)
	terra->dead_player_remove(index);
    }

  if (SDL_SemPost (pl_arr_smp) == -1)
	{
	  fprintf (stderr, "[SRV] Fatal error (in semaphore unlocking): %s .. server will shut down.\n", SDL_GetError());
	  exit(1);
	}
  output_queue.push(new msg_player_disconnected(index));

#ifdef DEBUG
  printf ("[SRV] Removed player with index %d\n", index);
#endif
  return true;
}

void game_server::process_chooser_req (int owner, bool down)
{
  const vector<coord> v = terra->get_castle_centers();

  vector<coord>::const_iterator it = v.begin();

  vector<coord>::const_iterator vanha;
  vector<coord>::const_iterator uusi;

  coord uu_coo;

  //#ifdef DEBUG
  //printf ("process req pl %d\n", owner);
  //#endif

  while (it != v.end())
	{
	  const coord ic = *it;
	  tile t = terra->get_tile(ic);
	  vanha = it;
	  it++;
	  if ( t.owner != owner) continue;

	  /* found index !*/
	  if (!down)
		{
		  uusi = it - 1;
		  while (1)
			{
			  if (uusi < v.begin()) // jos alle ekan indeksin
				uusi = v.end()-1; // hyptn viimeiseen

			  uu_coo = *uusi;
			  tile ti = terra->get_tile(uu_coo);
			  if (ti.owner == OWNER_NOONE)
			    break;
			  
			  uusi--;
			}
		  
		}
	  else 
	    {
	      uusi = it;
	      while (true)
		{
		  if (uusi == v.end())
		    uusi = v.begin();
		  
		  uu_coo = *uusi;
		  tile ti = terra->get_tile(uu_coo);
		  
		  if (ti.owner == OWNER_NOONE)
		    break;
		  
		  uusi++;
		}
	    }
	  
	  const coord va_coo = *vanha;
	  
	  tile uutile (tile::TYPE_CASTLE_C, owner);
	  tile vatile (tile::TYPE_CASTLE_C, OWNER_NOONE);

	  terra->set_tile (uu_coo, uutile);
	  terra->set_tile (va_coo, vatile);

	  output_queue.push (new msg_move_chooser(owner, uu_coo));
	  return;
	}
  assert(0);
}

void game_server::assign_default_castles()
{
  int i;
  int assigned = 0;
  const vector<coord> vec = terra->get_castle_centers();
  vector<coord>::const_iterator it = vec.begin();
  for (i=0;i<MAX_PLAYERS;i++)
    {
      if (!players[i].is_connected()) continue;
      if (it == vec.end())
	assert (0); // should never happen! there should always be enough castles to accommodate the players.
      const coord c = *it;
      it++;
      // assign tile to the player
      terra->set_tile (c, tile::TYPE_CASTLE_C, i);
      assigned++;
    }
  // now we should have assigned a castle to all the players
  assert (assigned == no_players());
}

void game_server::dispatch_playerinfo() {
  
  for ( int src = 0; src < MAX_PLAYERS; src++ ) {
    
    if (!players[src].is_connected())
      continue;
    
    output_queue.push (new msg_playerinfo(src, players[src].get_name()));
  }
  data_transfer();
}

void game_server::process_choose_character(int owner, const coord &place, int ch)
{

  /*  terra->set_tile (place, tile::TYPE_CANNON_C, owner);
  terra->set_tile (place+coord::tc(0,1), tile::TYPE_CANNON_A, owner);
  terra->set_tile (place+coord::tc(1,1), tile::TYPE_CANNON_A, owner);
  terra->set_tile (place+coord::tc(1,0), tile::TYPE_CANNON_A, owner);

  cannons_to_place[owner]--;
  */
  for (int i=0;i<MAX_PLAYERS;i++)
    {
      if ( players[i].is_alive() && players[i].get_character()==ch && owner!=i)
	{
	  ch=NOONE;
	  break;
	}
    }
  
  players[owner].put_character(ch);

  if(ch!=NOONE)
    character_selected++;
  
  output_queue.push (new msg_choose_character(owner, place, ch));
}

void game_server::process_place_cannon(int owner, const coord &place)
{
  terra->set_tile (place, tile::TYPE_CANNON_C, owner);
  terra->set_tile (place+coord::tc(0,1), tile::TYPE_CANNON_A, owner);
  terra->set_tile (place+coord::tc(1,1), tile::TYPE_CANNON_A, owner);
  terra->set_tile (place+coord::tc(1,0), tile::TYPE_CANNON_A, owner);

  cannons_to_place[owner]--;

  output_queue.push (new msg_place_cannon(owner, place));
}

void game_server::process_place_big_cannon(int owner, const coord &place)
{
  //tm pit viel teh
  
  terra->set_tile (place, tile::TYPE_CANNON_C, owner);
  terra->set_tile (place+coord::tc(0,1), tile::TYPE_BIG_CANNON_A, owner);
  terra->set_tile (place+coord::tc(1,1), tile::TYPE_BIG_CANNON_A, owner);
  terra->set_tile (place+coord::tc(1,0), tile::TYPE_BIG_CANNON_A, owner);
  terra->set_tile (place+coord::tc(2,0), tile::TYPE_BIG_CANNON_A, owner);
  terra->set_tile (place+coord::tc(0,2), tile::TYPE_BIG_CANNON_A, owner);
  terra->set_tile (place+coord::tc(2,1), tile::TYPE_BIG_CANNON_A, owner);
  terra->set_tile (place+coord::tc(1,2), tile::TYPE_BIG_CANNON_A, owner);
  terra->set_tile (place+coord::tc(2,2), tile::TYPE_BIG_CANNON_A, owner);


  big_cannons_to_place[owner]--;

  output_queue.push (new msg_place_big_cannon(owner, place));
}

void game_server::process_change_cannon(int owner)
{
  output_queue.push (new msg_change_cannon(owner));
}

void game_server::process_shoot (const coord &c, int owner, int ammosize)
{
  tile t = terra->get_tile(c);

  if (t.type == tile::TYPE_WALL)
    {
      t.type = tile::TYPE_EMPTY;
      t.owner = OWNER_NOONE;
      t.conquered = false;
      terra->set_tile( c, t );
    }
  else if (t.type == tile::TYPE_CANNON_C || t.type == tile::TYPE_CANNON_A)
    {
      coord cc = terra->get_center_tile(c);
      terra->hit(cc, ammo(coord::pc(0,0), c, owner, ammosize));
    }
}

void game_server::process_place_block (int owner, int next_block, int next_ori)
{
  //    printf ("placing block of owner %d type %d x %d y %d next block %d orientation %d\n", 
  //	    owner, block_placers[owner].get_type(), cen.x, cen.y, next_block, orientation);

  const vector<coord> v = block_placers[owner].get_tiles();
  vector<coord>::const_iterator it = v.begin();
  
  coord cen = block_placers[owner].get_center();
  while (it != v.end())
    {
      const coord &c = *it++;
      terra->set_tile (c+cen, tile::TYPE_WALL, owner);
      //	  printf ("tile hit to %d %d", c.x+cen.x, c.y+cen.y);
    }
  
  block_placers[owner].set_type(next_block);
  block_placers[owner].set_orientation(next_ori);
}

void game_server::clear_connection_readys()
{
  for (unsigned int i=0;i<conns.size();i++)
    conns[i]->set_ready(false);
}

void game_server::probe_disconnected_players()
{
  if (mode != ID_MODE_NOTACTIVE) return;
  data_transfer();
}

void game_server::add_local_connects(vector<string> & pl_names, monitor_queue<msg *> *in, monitor_queue<msg *> *out)
{
  assert(!game_on);
  assert(con_th == 0);

  local_connects *c = new local_connects(in, out);

  for (unsigned int i=0;i<pl_names.size();i++)
    {
      int pind = add_player(pl_names[i], c);
      c->add_player(pind);

#ifdef DEBUG
      printf ("[SRV] Added connection with player %s\n", pl_names[i].c_str());
#endif
      
      assert(pind == (int) i);
    }
  conns.push_back(c);

}

int game_server::get_mode() const
{
  return mode;
}
