/*
  Copyright (C) 2004-2005 Tommi Tervonen, Petteri Klemola, Pasi Orovuo

  This file is part of Kajaani Kombat.

  Kajaani Kombat is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  Kajaani Kombat is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with Kajaani Kombat; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "gui_screens.h"

typedef struct server_numpl {
  game_server *server;
  int num_players;
};

const char * main_screen::button_files[4][3] = {
  { "main_butstart.png", "main_butstart_focus.png", "main_butstart_act.png" },
  { "main_buthost.png", "main_buthost_focus.png", "main_buthost_act.png" },
  { "main_butopt.png", "main_butopt_focus.png", "main_butopt_act.png" },
  { "main_butquit.png", "main_butquit_focus.png", "main_butquit_act.png" }
};

const char * numpl_screen::button_files[4][3] = {
  { "numpl_butone.png", "numpl_butone_focus.png", "numpl_butone_act.png" },
  { "numpl_buttwo.png", "numpl_buttwo_focus.png", "numpl_buttwo_act.png" }
};

const char *numpl_screen::imgtext_files[1] = {
  "numpl_imgtext.png"
};

const char *connect_screen::imgtext_files[4] = {
  "connect_imgname.png", "connect_imgname2.png", "connect_imgserv.png", "connect_imgport.png"
};

const char *host_screen::imgtext_files[6] = {
  "host_imgname.png", "host_imgname2.png", "host_imgport.png", "host_imgnumpl.png", "host_imgnumcastles.png",
  "host_imgnumwins.png"
};

const char *host_screen::button_files[][3] = {
  { "host_butstart.png", "host_butstart_focus.png", "host_butstart_act.png" }
};

const char *connect_screen::button_files[][3] = {
  { "connect_butconn.png", "connect_butconn_focus.png", "connect_butconn_act.png" }
};

/*
const char *options_screen::imgtext_files[8] = {
  "options_player1.png", "options_player2.png",
  "options_keys_up.png", "options_keys_down.png", "options_keys_left.png", "options_keys_right.png",
  "options_keys_action1.png", "options_keys_action2.png"
};
*/

const char *options_screen::button_files[][3] = {
  { "options_apply.png", "options_apply_focus.png", "options_apply_act.png" },
  { "options_cancel.png", "options_cancel_focus.png", "options_cancel_act.png" }
};


  main_screen::main_screen(SDL_Surface *scr, TTF_Font *font, gfx *graphix, starfield_efect *se, audio *au, config *cfg)
  : gui_screen (scr, font, graphix, se, au, cfg)
{
  for(int p=0;p<4;p++)
    {
      for (int i=0;i<3;i++)
	{
	  //	  printf ("Loading button image %s\n", button_files[p][i]);
	  button_gfx[p][i] = img_load (button_files[p][i]);
	  if (button_gfx[p][i] == NULL)
	    {
	      fprintf (stderr, "Error loading image %s: %s\n", button_files[p][i], IMG_GetError());
	      exit (1);
	    }
	}
    }
  logogfx = img_load("logo.png");
  if (logogfx == NULL)
    {
      fprintf (stderr, "Error loading image %s: %s\n", "logo.png", IMG_GetError());
      exit (1);
    }

  //  printf ("Main screen images loaded\n");
  // create sub-screens.
  ns =  new numpl_screen(screen, font, graphix, starfield, aud, cfg);
  opts = new options_screen( screen, font, graphix, starfield, aud, cfg );
  qs = new confirm_quit_screen( screen, font, graphix, starfield, aud, cfg );

  // init buttons.
  coord p0 = coord::pc (center_xcoord(logogfx), 50);
  gui_image *logoimg = new gui_image(p0, logogfx);
  coord p1 = coord::pc (center_xcoord(button_gfx[0][0]), p0.getPY() + logoimg->get_size().getPY()+10);
  gui_button *b1 = new gui_button (p1, button_gfx[0][0], button_gfx[0][1], button_gfx[0][2]);
  coord p2 = coord::pc(center_xcoord(button_gfx[1][0]), p1.getPY() + b1->get_size().getPY()+10);
  gui_button *b2 = new gui_button (p2, button_gfx[1][0], button_gfx[1][1], button_gfx[1][2]);
  coord p3 = coord::pc(center_xcoord(button_gfx[2][0]), p2.getPY() + b2->get_size().getPY()+10);
  gui_button *b3 = new gui_button (p3, button_gfx[2][0], button_gfx[2][1], button_gfx[2][2]);
  // exit from screen-button
  coord p4 = coord::pc(center_xcoord(button_gfx[3][0]), p3.getPY() + b3->get_size().getPY()+10);
  qbtn = new gui_button( p4, button_gfx[3][0], button_gfx[3][1], button_gfx[3][2] );


  // set event handlers
  b1->set_act_screen(ns, numpl_screen::next_connect);
  b2->set_act_screen(ns, numpl_screen::next_host);
  b3->set_act_screen( opts, 0 );
  qbtn->set_act_screen( qs, 0 );

  // add image
  add_component(logoimg);
  // add buttons
  add_component(b1);
  add_component(b2);
  add_component(b3);
  add_component( qbtn );

  focus_first_component();
}

main_screen::~main_screen()
{
  for (int i=0;i<3;i++)
    for (int j=0;j<3;j++)
      SDL_FreeSurface (button_gfx[i][j]);

  delete logogfx;
  delete ns;
}

void main_screen::activate(int signum)
{

  bool enterpushed=false;

  re_init(false);
  draw();

  aud->play_intro_music();

  SDL_Event ev;
  while (true)
    {
      draw();
      try {
	while (SDL_PollEvent(&ev))
	  {
	    if (ev.type == SDL_KEYDOWN)
	      {
		if(ev.key.keysym.sym == SDLK_DOWN)
		  focus_down();
		else if(ev.key.keysym.sym == SDLK_UP)
		  focus_up();
		else if (ev.key.keysym.sym == SDLK_ESCAPE)
		  qbtn->action( SDLK_RETURN, ev.key.keysym.mod ); //back_one_screen();
		else
		  {
		    gui_focus_component *t = dynamic_cast<gui_focus_component *>(comps[cur_focus]);
		    if (t && (ev.key.keysym.sym == SDLK_RETURN || 
			      dynamic_cast<gui_button *>(t) == 0))
		      {
			click(t);
			enterpushed=true;
		      }
		    
		    if (t)
		      {
			t->action(ev.key.keysym.sym, ev.key.keysym.mod);
			if(enterpushed==true)
			  {
			    this->re_init(true);
			    enterpushed=false;
			  }
		      }
		  }
	      }
	  }
      } // end try
      catch (int i)
	{
	  if (i == 1) return;
	}
    }
};

void host_screen::activate(int signum)
{

  bool enterpushed=false;

  numlocal_pl = signum;
  namebox->set_text( conf->get_name( config::PLAYER_1 ) );

  if (numlocal_pl == 1) // 1 player
    {
      pl2img->hide();
      namebox2->hide();
    }
  else if (numlocal_pl == 2) // 2 players
    {
      namebox2->set_text( conf->get_name( config::PLAYER_2 ) );
      pl2img->unhide();
      namebox2->unhide();
    }
  else assert(0); // shouldn't get here

  portbox->set_text( conf->get_port() );
  
  re_init(false);
  draw();

  SDL_Event ev;
  while (true)
    {
      draw();
      try {
	while (SDL_PollEvent(&ev))
	  {
	    if (ev.type == SDL_KEYDOWN)
	      {
		if(ev.key.keysym.sym == SDLK_DOWN)
		  focus_down();
		else if(ev.key.keysym.sym == SDLK_UP)
		  focus_up();
		else if (ev.key.keysym.sym == SDLK_ESCAPE)
		  back_one_screen();
		else
		  {
		    gui_focus_component *t = dynamic_cast<gui_focus_component *>(comps[cur_focus]);
		    if (t && (ev.key.keysym.sym == SDLK_RETURN &&
			      dynamic_cast<gui_button *>(t) != 0))
		      {
			click(t);
			enterpushed=true;
		      }
		    if (t)
		      {
			t->action(ev.key.keysym.sym, ev.key.keysym.mod);
			if(enterpushed==true)
			  {
			    //this->re_init(true);
			    enterpushed=false;
			  }
			
		      }
		  }
	      }
	  }
      } // end try
      catch (int i)
	{
	  if (i == 1) return;
	}
    }
};

connect_screen::connect_screen(SDL_Surface *scr, TTF_Font *font, gfx *graphix, starfield_efect *se, audio *au, config *cfg)
  : gui_screen (scr, font, graphix, se, au, cfg)
{
  //  printf ("Loading connect_screen images:\n");
  for (int i=0;i<4;i++)
    {
      //      printf ("Loading image 'text' %s\n", imgtext_files[i]);
      SDL_Surface *s = img_load (imgtext_files[i]);
      if (s == NULL)
	{
	  fprintf (stderr, "Error loading image %s: %s\n", imgtext_files[i], IMG_GetError());
	  exit (1);
	}
      imgtext_gfx[i] = s;
    }

  for(int p=0;p<1;p++)
    {
      for (int i=0;i<3;i++)
	{
	  //	  printf ("Loading button image %s\n", button_files[p][i]);
	  SDL_Surface *s = img_load (button_files[p][i]);
	  if (s == NULL)
	    {
	      fprintf (stderr, "Error loading image %s: %s\n", button_files[p][i], IMG_GetError());
	      exit (1);
	    }
	  button_gfx[p][i] = s;
	}
    }

  coord c1 = coord::pc(SCR_WIDTH/2-imgtext_gfx[0]->w, 100);
  gui_image *t1 = new gui_image (c1, imgtext_gfx[0]);
  coord c1_2 = coord::pc (SCR_WIDTH/2-imgtext_gfx[1]->w, c1.getPY() + 30);
  pl2img = new gui_image(c1_2, imgtext_gfx[1]);
  coord c2 = coord::pc(SCR_WIDTH/2-imgtext_gfx[2]->w, c1_2.getPY() + 30);
  gui_image *t2 = new gui_image (c2, imgtext_gfx[2]);
  coord c3 = coord::pc(SCR_WIDTH/2-imgtext_gfx[3]->w, c2.getPY() + 30);
  gui_image *t3 = new gui_image (c3, imgtext_gfx[3]);

  // editbox's
  coord c4 = t1->get_position() + t1->get_size() + coord::pc(10,0);
  namebox = new gui_editbox(c4, font);
  coord c4_2 = pl2img->get_position() + pl2img->get_size() + coord::pc(10,0);
  namebox2 = new gui_editbox(c4_2, font);
  coord c5 = t2->get_position() + t2->get_size() + coord::pc(10,0);
  serverbox = new gui_editbox(c5, font);
  coord c6 = t3->get_position() + t3->get_size() + coord::pc(10,0);
  portbox = new gui_editbox(c6, font);

  // connect button
  coord c7 = coord::pc (center_xcoord(button_gfx[0][0]), c3.getPY() + t3->get_size().getPY() + 50);
  gui_button *b1 = new gui_button (c7, button_gfx[0][0], button_gfx[0][1], button_gfx[0][2]);
  b1->set_act_screen(this);
  b1->set_act_callback(start_game);

  add_component(b1);
  add_component(t1);
  add_component(pl2img);
  add_component(t2);
  add_component(t3);
  add_component(namebox);
  add_component(namebox2);
  add_component(serverbox);
  add_component(portbox);

  focus_first_component();
}

connect_screen::~connect_screen()
{
  for (int i=0;i<1;i++)
    for (int j=0;j<3;j++)
      SDL_FreeSurface (button_gfx[i][j]);

  for (int i=0;i<4;i++)
    delete imgtext_gfx[i];
}

void connect_screen::activate(int signum)
{
  numlocal_pl = signum;

  namebox->set_text( conf->get_name( config::PLAYER_1 ) );

  if (numlocal_pl == 1) // 1 player
  {
    pl2img->hide();
    namebox2->hide();
  }
  else if (numlocal_pl == 2) // 2 players
  {
    namebox2->set_text( conf->get_name( config::PLAYER_2 ) );
    pl2img->unhide();
    namebox2->unhide();
  }
  else assert(0); // shouldn't get here

  serverbox->set_text( conf->get_server() );
  portbox->set_text( conf->get_port() );

  re_init(false);

  SDL_Event ev;
  while (true)
    {
      draw();
      try {
	while (SDL_PollEvent(&ev))
	  {
	    if (ev.type == SDL_KEYDOWN)
	      {
		if(ev.key.keysym.sym == SDLK_DOWN)
		  focus_down();
		else if(ev.key.keysym.sym == SDLK_UP)
		  focus_up();
		else if (ev.key.keysym.sym == SDLK_ESCAPE)
		  back_one_screen();
		else
		  {
		    gui_focus_component *t = dynamic_cast<gui_focus_component *>(comps[cur_focus]);
		    if (t && (ev.key.keysym.sym == SDLK_RETURN && 
			      dynamic_cast<gui_button *>(t) != 0))
		      click(t);
		    if (t)
		      {
			t->action(ev.key.keysym.sym, ev.key.keysym.mod);
			//this-re_init(true);
		      }
		  }
	      }
	  }
      } // end try
      catch (int i)
	{
	  if (i == 1) return;
	}
    }
}

void connect_screen::start_game(gui_screen *t)
{
  IPaddress server_ip;

  connect_screen *ls = dynamic_cast<connect_screen *>(t);
  string name = ls->namebox->get_text();
  string name2 = ls->namebox2->get_text();

  if ( !name.length() ) {
    ls->objs.push_back( new text_object( 10, "Name for player 1 required" ) );
    return;
  }
  else if ( ls->numlocal_pl == 2 && !name2.length() ) {
    ls->objs.push_back( new text_object( 10, "Name for player 2 required" ) );
    return;
  }

  string names[] = {name, name2};

  string server = ls->serverbox->get_text();
  string port = ls->portbox->get_text();

  ls->conf->set_name( config::PLAYER_1, name );
  ls->conf->set_name( config::PLAYER_2, name2 );
  ls->conf->set_server( server );
  ls->conf->set_port( port );

  int iport = atoi(port.c_str());

  
  //  printf ("Connecting to %s:%i as %s\n", server.c_str(),iport, name.c_str());
  // check that host exists!
  if (SDLNet_ResolveHost (&server_ip, (char *) server.c_str(), iport) < 0) {
    string errs ("Error resolving server host ");
    errs += server;
    errs += ":";
    errs += port;
    errs += ".";
    ls->objs.push_back(new text_object(-1, errs));
    return;
  }

  // init client, connect to server
  game_client *client = new game_client (names, &server_ip, ls->screen, ls->font, ls->numlocal_pl, ls->graphix, ls->starfield, ls->aud, ls->conf);
  //  printf ("Client object created\n");

  try {
    client->connect();
  }
  catch (string &s) {
    ls->objs.push_back(new text_object(-1, s));
    return;
  }
  // start client activity
  client->activate(0);

  ls->remove_text_objects();
  ls->reset_last_drawn();
  vector<string> er = client->get_errors();

  for (unsigned int i=0;i<er.size();i++)
    ls->objs.push_back(new text_object (-1, er[i]));

  // back to intro music
  ls->aud->play_gui_select_sound();
  ls->aud->play_intro_music();
  
  //  printf ("back from activate\n");
  delete client;
}

host_screen::host_screen(SDL_Surface *scr, TTF_Font *font, gfx *graphix, starfield_efect *se, audio *au, config *cfg)
  : gui_screen (scr, font, graphix, se, au, cfg)
{
  //  printf ("Loading host_screen images:\n");
  for (int i=0;i<6;i++)
    {
      //      printf ("Loading image 'text' %s\n", imgtext_files[i]);
      SDL_Surface *s = img_load (imgtext_files[i]);
      if (s == NULL)
	{
	  fprintf (stderr, "Error loading image %s: %s\n", imgtext_files[i], IMG_GetError());
	  exit (1);
	}
      imgtext_gfx[i] = s;
    }

  for(int p=0;p<1;p++)
    {
      for (int i=0;i<3;i++)
	{
	  //	  printf ("Loading button image %s\n", button_files[p][i]);
	  SDL_Surface *s = img_load (button_files[p][i]);
	  if (s == NULL)
	    {
	      fprintf (stderr, "Error loading image %s: %s\n", button_files[p][i], IMG_GetError());
	      exit (1);
	    }
	  button_gfx[p][i] = s;
	}
    }

  coord c1 = coord::pc(SCR_WIDTH/2-imgtext_gfx[0]->w, 50);
  gui_image *t1 = new gui_image (c1, imgtext_gfx[0]);
  coord c1_2 = coord::pc(SCR_WIDTH/2-imgtext_gfx[1]->w, c1.getPY() + 30);
  pl2img = new gui_image (c1_2, imgtext_gfx[1]);
  coord c2 = coord::pc(SCR_WIDTH/2-imgtext_gfx[2]->w, c1_2.getPY() + 30);
  gui_image *t2 = new gui_image (c2, imgtext_gfx[2]);
  coord c3 = coord::pc(SCR_WIDTH/2-imgtext_gfx[3]->w, c2.getPY() + 30);
  gui_image *t3 = new gui_image (c3, imgtext_gfx[3]);
  coord c4 = coord::pc(SCR_WIDTH/2-imgtext_gfx[4]->w, c3.getPY() + 30);
  gui_image *t4 = new gui_image (c4, imgtext_gfx[4]);
  coord c5 = coord::pc(SCR_WIDTH/2-imgtext_gfx[5]->w, c4.getPY() + 30);
  gui_image *t5 = new gui_image (c5, imgtext_gfx[5]);

  // editbox's
  coord c6 = t1->get_position() + t1->get_size() + coord::pc(10,0);
  namebox = new gui_editbox(c6, font);
  coord c6_2 = pl2img->get_position() + pl2img->get_size() + coord::pc(10,0);
  namebox2 = new gui_editbox(c6_2, font);
  coord c7 = t2->get_position() + t2->get_size() + coord::pc(10,0);
  portbox = new gui_editbox(c7, font);
  coord c8 = t3->get_position() + t3->get_size() + coord::pc(10,0);
  // number choosers
  num_plchooser = new gui_numchooser(c8, font, 3, 2);
  coord c9 = t4->get_position() + t4->get_size() + coord::pc(10,0);
  num_castchooser = new gui_numchooser(c9, font, 11, 4);
  coord c10 = t5->get_position() + t5->get_size() + coord::pc(10,0);
  num_winschooser = new gui_numchooser(c10, font, 5, 1);

  // connect button
  coord c11 = coord::pc(center_xcoord(button_gfx[0][0]), num_winschooser->get_position().getPY() + num_winschooser->get_size().getPY() + 50);
  gui_button *b1 = new gui_button (c11, button_gfx[0][0], button_gfx[0][1], button_gfx[0][2]);
  b1->set_act_screen(this);
  b1->set_act_callback(start_game);

  num_plchooser->set_choice(DEFAULT_NUM_PLAYERS);
  num_castchooser->set_choice(DEFAULT_NUM_CASTLES);
  num_winschooser->set_choice(DEFAULT_NUM_WINS);

  add_component(b1);
  add_component(t1);
  add_component(pl2img);
  add_component(t2);
  add_component(t3);
  add_component(t4);
  add_component(t5);
  add_component(namebox);
  add_component(namebox2);
  add_component(portbox);
  add_component(num_plchooser);
  add_component(num_castchooser);
  add_component(num_winschooser);

  focus_first_component();
}

host_screen::~host_screen()
{
  for (int i=0;i<1;i++)
    for (int j=0;j<3;j++)
      SDL_FreeSurface (button_gfx[i][j]);

  for (int i=0;i<6;i++)
    delete imgtext_gfx[i];
}

void host_screen::start_game(gui_screen *t)
{
  IPaddress ip;
  IPaddress server_ip;
  game_server *serv = 0;
  game_client *client = 0;

  host_screen *ls = dynamic_cast<host_screen *>(t);
  string name = ls->namebox->get_text();
  string name2 = ls->namebox2->get_text();
  string server ("localhost");
  string port = ls->portbox->get_text();

 string names[] = {name, name2};
  int num_pl = ls->num_plchooser->get_choice();
  int num_castles = ls->num_castchooser->get_choice();
  int wins = ls->num_winschooser->get_choice();

  int iport = atoi(port.c_str());

  if ( !name.length() ) {
    ls->objs.push_back( new text_object( 10, "Name for player 1 required" ) );
    return;
  }
  else if ( ls->numlocal_pl == 2 && !name2.length() ) {
    ls->objs.push_back( new text_object( 10, "Name for player 2 required" ) );
    return;
  }
      
  ls->conf->set_name( config::PLAYER_1, name );
  if ( num_pl == 2 )
    ls->conf->set_name( config::PLAYER_2, name2 );
  ls->conf->set_port( port );

   //  printf ("Starting server..\n");

  if (SDLNet_ResolveHost (&ip, "localhost", iport) == -1)
    {
      string errs("Error resolving host: ");
      errs += SDLNet_GetError();
      //      fprintf (stderr, "Error resolving host: %s\n", SDLNet_GetError());
      ls->objs.push_back (new text_object (-1, errs));
      return;
    }

  if (SDLNet_ResolveHost (&server_ip, NULL, iport) == -1)
    {
      string errs("Error resolving host: ");
      errs += SDLNet_GetError();
      ls->objs.push_back (new text_object (-1, errs));
      return;
    }

  // try max 10 times
  for (int i=0;i<10;i++)
    {
      ls->draw();
      try {
	if (serv) delete serv;
	serv = new game_server(server_ip, num_castles, wins);
	
	// init client, connect to server
	if (client) delete client;
	client = NULL;
	client = new game_client (names, &ip, ls->screen, ls->font, ls->numlocal_pl, ls->graphix, ls->starfield, ls->aud, ls->conf);
	//	printf ("Client object created\n");

	monitor_queue<msg *> *in = 0, *out = 0;
	client->connect_local( &in, &out);

	vector<string> nv (ls->numlocal_pl);
	for (int kk=0;kk<ls->numlocal_pl;kk++)
	  nv[kk] = names[kk];
	
	serv->add_local_connects(nv, out, in);

	// init server net
	serv->init_net();

	// start client activity
	server_numpl sn;
	sn.server = serv;
	sn.num_players = num_pl;

	SDL_Thread *th = SDL_CreateThread (serv_starter, &sn);
	if (th == 0)
	  {
	    printf ("Error starting thread!\n");
	    exit(1);
	  }
	client->activate(0);
	ls->remove_text_objects();
	ls->reset_last_drawn();
	vector<string> es = client->get_errors();
	for (unsigned int k=0;k<es.size();k++)
	  ls->objs.push_back(new text_object(-1, es[k]));
	serv->shutdown_server();

	SDL_WaitThread(th, NULL);
	th = 0;
	i=10;
	break;
      }
      catch (string &s)
	{
	  if (client) {delete client; client = 0;}
	  //	  printf ("Error starting server: %s .. delaying and trying again.\n", ce);
	  s += " .. delaying and trying again.";
	  ls->objs.push_back (new text_object (-1, s));
	  ls->draw();
	  SDL_Delay(1000);
	}

    }

  // shutdown server
  if (serv)
    {
      //      printf ("Shutting down server..\n");      
      delete serv;
      serv = 0;
    }
  if (client)
    {
      delete client;
      client = 0;
    }

  ls->aud->play_gui_select_sound();
  ls->aud->play_intro_music();
}

int serv_starter (void *data)
{
  server_numpl *sn = static_cast<server_numpl *>(data);
  
  while (sn->server->get_mode() != game_server::ID_MODE_SHUTTING_DOWN) 
    {
      if (sn->server->no_players() == sn->num_players)
	{
	  sn->server->start_game();
	  break;
	}
      else
	SDL_Delay(100);
      // probe for disconnected players.
      sn->server->probe_disconnected_players();
    }
  return 0;
}

numpl_screen::numpl_screen(SDL_Surface *scr, TTF_Font *font, gfx *graphix, starfield_efect *se, audio *au, config *cfg)
  : gui_screen (scr, font, graphix, se, au, cfg)
{
  for (int i=0;i<1;i++)
    {
      //      printf ("Loading image 'text' %s\n", imgtext_files[i]);
      SDL_Surface *s = img_load (imgtext_files[i]);
      if (s == NULL)
	{
	  fprintf (stderr, "Error loading image %s: %s\n", imgtext_files[i], IMG_GetError());
	  exit (1);
	}
      imgtext_gfx[i] = s;
    }

  for(int p=0;p<2;p++)
    {
      for (int i=0;i<3;i++)
	{
	  //	  printf ("Loading button image %s\n", button_files[p][i]);
	  button_gfx[p][i] = img_load (button_files[p][i]);
	  if (button_gfx[p][i] == NULL)
	    {
	      fprintf (stderr, "Error loading image %s: %s\n", button_files[p][i], IMG_GetError());
	      exit (1);
	    }
	}
    }
  //  printf ("Number of players screen images loaded\n");
  // create sub-screens.
  ls = new connect_screen(screen, font, graphix, starfield, aud, cfg);
  hs = new host_screen(screen, font, graphix, starfield, aud, cfg);

  // init text
  coord c1 = coord::pc(center_xcoord(imgtext_gfx[0]), 100);
  gui_image *t1 = new gui_image (c1, imgtext_gfx[0]);

  // init buttons.
  coord c2 = coord::pc (center_xcoord(button_gfx[0][0]), c1.getPY() + t1->get_size().getPY() + 50);
  b1 = new gui_button (c2, button_gfx[0][0], button_gfx[0][1], button_gfx[0][2]);
  coord c3 = coord::pc (center_xcoord(button_gfx[1][0]), c2.getPY() + b1->get_size().getPY() + 20);
  b2 = new gui_button (c3, button_gfx[1][0], button_gfx[1][1], button_gfx[1][2]);

  // set event handlers - set in activate
  //  b1->set_act_screen(ls);
  //b2->set_act_screen(ls);

  // add buttons
  add_component(t1);
  add_component(b1);
  add_component(b2);

  focus_first_component();
}

numpl_screen::~numpl_screen()
{
  for (int i=0;i<2;i++)
    for (int j=0;j<3;j++)
      SDL_FreeSurface (button_gfx[i][j]);

  for (int i=0;i<1;i++)
    delete imgtext_gfx[i];

  delete ls;
  delete hs;
}

void numpl_screen::activate(int signum)
{

  bool enterpushed=false;

  if (signum == next_connect)
    {
      b1->set_act_screen(ls, 1);
      b2->set_act_screen(ls, 2);
    }
  else if (signum == next_host)
    {
      b1->set_act_screen(hs, 1);
      b2->set_act_screen(hs, 2);
    }
  else assert (0); // shouldn't get here

  re_init(false);
  draw();

  SDL_Event ev;
  while (true)
    {
      draw();
      try {
	while (SDL_PollEvent(&ev))
	  {
	    if (ev.type == SDL_KEYDOWN)
	      {
		if(ev.key.keysym.sym == SDLK_DOWN)
		  focus_down();
		else if(ev.key.keysym.sym == SDLK_UP)
		  focus_up();
		else if (ev.key.keysym.sym == SDLK_ESCAPE)
		  back_one_screen();
		else
		  {
		    gui_focus_component *t = dynamic_cast<gui_focus_component *>(comps[cur_focus]);
		    if (t && (ev.key.keysym.sym == SDLK_RETURN || 
			      dynamic_cast<gui_button *>(t) == 0))
		      {
			click(t);
			enterpushed=true;
		      }
		    if (t)
		      {
			t->action(ev.key.keysym.sym, ev.key.keysym.mod);
			if(enterpushed==true)
			  {
			    this->re_init(true);
			    enterpushed=false;
			  }
		      }
		  }
	      }
	  }
      } // end try
      catch (int i)
	{
	  if (i == 1) return;
	}
    }
};

options_screen::options_screen( SDL_Surface *scr, TTF_Font *fnt, gfx *graphics, starfield_efect *se, audio *au, config *cfg )
: gui_screen( scr, fnt, graphics, se, au, cfg )
{
  /* Load images */
  /* We keep a local copy of the changes and then, if user hits
   * apply, se store the info in 'global' config...
   */
  localconf = *conf;
  
  for ( Uint32 i = 0; i < sizeof( button_files ) / sizeof( button_files[0] ); i++ ) {
    for ( Uint32 j = 0; j < sizeof( button_files[0] ) / sizeof( button_files[0][0] ); j++ ) {
      SDL_Surface   *s = img_load( button_files[i][j] );
      if ( !s ) {
        fprintf( stderr, "Failed to load image %s (%s)\n", button_files[i][j], IMG_GetError() );
        exit( 1 );
      }

      button_gfx[i][j] = s;
    }
  }

  bs = new bind_screen( screen, font, graphix, starfield, aud, &localconf );

  /*
   * TODO:
   * Hoidetaan nm vain yhdess loopissa, jookosta?
   *
   */
  /* Player 1*/
  coord cp1 = coord::pc( 0, 20 );
  gui_textbox *tp1 = new gui_textbox( "PLAYER 1", cp1, font );
  cp1.setPX( SCR_WIDTH / 2 - tp1->get_size().getPX() );
  tp1->set_position( cp1 );

  coord ct1 = coord::pc( 0, cp1.getPY() + tp1->get_size().getPY() + 5 );
  gui_textbox *t1 = new gui_textbox( "UP", ct1, font );
  ct1.setPX( SCR_WIDTH / 2 - t1->get_size().getPX() );
  t1->set_position( ct1 );

  coord cb1 = coord::pc( ct1.getPX() + t1->get_size().getPX() + 10, ct1.getPY() );
  keybtns[config::PLAYER_1][config::_KEY_UP] = new gui_textbutton( cfg->key_to_str( cfg->get_key( config::PLAYER_1, config::_KEY_UP ) ), cb1, font );

  coord ct2 = coord::pc( 0, ct1.getPY() + t1->get_size().getPY() + 5 );
  gui_textbox *t2 = new gui_textbox( "DOWN", ct2, font );
  ct2.setPX( SCR_WIDTH / 2 - t2->get_size().getPX() );
  t2->set_position( ct2 );

  coord cb2 = coord::pc( ct2.getPX() + t2->get_size().getPX() + 10, ct2.getPY() );
  keybtns[config::PLAYER_1][config::_KEY_DOWN] = new gui_textbutton( cfg->key_to_str( cfg->get_key( config::PLAYER_1, config::_KEY_DOWN ) ), cb2, font );

  coord ct3 = coord::pc( 0, ct2.getPY() + t2->get_size().getPY() + 5 );
  gui_textbox *t3 = new gui_textbox( "LEFT", ct3, font );
  ct3.setPX( SCR_WIDTH / 2 - t3->get_size().getPX() );
  t3->set_position( ct3 );

  coord cb3 = coord::pc( ct3.getPX() + t3->get_size().getPX() + 10, ct3.getPY() );
  keybtns[config::PLAYER_1][config::_KEY_LEFT] = new gui_textbutton( cfg->key_to_str( cfg->get_key( config::PLAYER_1, config::_KEY_LEFT ) ), cb3, font );

  coord ct4 = coord::pc( 0, ct3.getPY() + t3->get_size().getPY() + 5 );
  gui_textbox *t4 = new gui_textbox( "RIGHT", ct4, font );
  ct4.setPX( SCR_WIDTH / 2 - t4->get_size().getPX() );
  t4->set_position( ct4 );

  coord cb4 = coord::pc( ct4.getPX() + t4->get_size().getPX() + 10, ct4.getPY() );
  keybtns[config::PLAYER_1][config::_KEY_RIGHT] = new gui_textbutton( cfg->key_to_str( cfg->get_key( config::PLAYER_1, config::_KEY_RIGHT ) ), cb4, font );

  coord ct5 = coord::pc( 0, ct4.getPY() + t4->get_size().getPY() + 5 );
  gui_textbox *t5 = new gui_textbox( "ACTION 1", ct5, font );
  ct5.setPX( SCR_WIDTH / 2 - t5->get_size().getPX() );
  t5->set_position( ct5 );

  coord cb5 = coord::pc( ct5.getPX() + t5->get_size().getPX() + 10, ct5.getPY() );
  keybtns[config::PLAYER_1][config::_KEY_ACTION1] = new gui_textbutton( cfg->key_to_str( cfg->get_key( config::PLAYER_1, config::_KEY_ACTION1 ) ), cb5, font );

  coord ct6 = coord::pc( 0, ct5.getPY() + t5->get_size().getPY() + 5 );
  gui_textbox *t6 = new gui_textbox( "ACTION 2", ct6, font );
  ct6.setPX( SCR_WIDTH / 2 - t6->get_size().getPX() );
  t6->set_position( ct6 );

  coord cb6 = coord::pc( ct6.getPX() + t6->get_size().getPX() + 10, ct6.getPY() );
  keybtns[config::PLAYER_1][config::_KEY_ACTION2] = new gui_textbutton( cfg->key_to_str( cfg->get_key( config::PLAYER_1, config::_KEY_ACTION2 ) ), cb6, font );

  coord ct7 = coord::pc( 0, ct6.getPY() + t6->get_size().getPY() + 5 );
  gui_textbox *t7 = new gui_textbox( "TALK", ct7, font );
  ct7.setPX( SCR_WIDTH / 2 - t7->get_size().getPX() );
  t7->set_position( ct7 );

  coord cb7 = coord::pc( ct7.getPX() + t7->get_size().getPX() + 10, ct7.getPY() );
  keybtns[config::PLAYER_1][config::_KEY_TALK] = new gui_textbutton( cfg->key_to_str( cfg->get_key( config::PLAYER_1, config::_KEY_TALK ) ), cb7, font );

  /* Player 2 */
  coord cp2 = coord::pc( 0, ct6.getPY() + 50 );
  gui_textbox *tp2 = new gui_textbox( "PLAYER 2", cp2, font );
  cp2.setPX( SCR_WIDTH / 2 - tp2->get_size().getPX() );
  tp2->set_position( cp2 );

  coord ct8 = coord::pc( 0, cp2.getPY() + tp2->get_size().getPY() + 5 );
  gui_textbox *t8 = new gui_textbox( "UP", ct8, font );
  ct8.setPX( SCR_WIDTH / 2 - t8->get_size().getPX() );
  t8->set_position( ct8 );

  coord cb8 = coord::pc( ct8.getPX() + t8->get_size().getPX() + 10, ct8.getPY() );
  keybtns[config::PLAYER_2][config::_KEY_UP] = new gui_textbutton( cfg->key_to_str( cfg->get_key( config::PLAYER_2, config::_KEY_UP ) ), cb8, font );

  coord ct9 = coord::pc( 0, ct8.getPY() + t8->get_size().getPY() + 5 );
  gui_textbox *t9 = new gui_textbox( "DOWN", ct9, font );
  ct9.setPX( SCR_WIDTH / 2 - t9->get_size().getPX() );
  t9->set_position( ct9 );

  coord cb9 = coord::pc( ct9.getPX() + t9->get_size().getPX() + 10, ct9.getPY() );
  keybtns[config::PLAYER_2][config::_KEY_DOWN] = new gui_textbutton( cfg->key_to_str( cfg->get_key( config::PLAYER_2, config::_KEY_DOWN ) ), cb9, font );

  coord ct10 = coord::pc( 0, ct9.getPY() + t9->get_size().getPY() + 5 );
  gui_textbox *t10 = new gui_textbox( "LEFT", ct10, font );
  ct10.setPX( SCR_WIDTH / 2 - t10->get_size().getPX() );
  t10->set_position( ct10 );

  coord cb10 = coord::pc( ct10.getPX() + t10->get_size().getPX() + 10, ct10.getPY() );
  keybtns[config::PLAYER_2][config::_KEY_LEFT] = new gui_textbutton( cfg->key_to_str( cfg->get_key( config::PLAYER_2, config::_KEY_LEFT ) ), cb10, font );

  coord ct11 = coord::pc( 0, ct10.getPY() + t10->get_size().getPY() + 5 );
  gui_textbox *t11 = new gui_textbox( "RIGHT", ct11, font );
  ct11.setPX( SCR_WIDTH / 2 - t11->get_size().getPX() );
  t11->set_position( ct11 );

  coord cb11 = coord::pc( ct11.getPX() + t11->get_size().getPX() + 10, ct11.getPY() );
  keybtns[config::PLAYER_2][config::_KEY_RIGHT] = new gui_textbutton( cfg->key_to_str( cfg->get_key( config::PLAYER_2, config::_KEY_RIGHT ) ), cb11, font );

  coord ct12 = coord::pc( 0, ct11.getPY() + t11->get_size().getPY() + 5 );
  gui_textbox *t12 = new gui_textbox( "ACTION 1", ct12, font );
  ct12.setPX( SCR_WIDTH / 2 - t12->get_size().getPX() );
  t12->set_position( ct12 );

  coord cb12 = coord::pc( ct12.getPX() + t12->get_size().getPX() + 10, ct12.getPY() );
  keybtns[config::PLAYER_2][config::_KEY_ACTION1] = new gui_textbutton( cfg->key_to_str( cfg->get_key( config::PLAYER_2, config::_KEY_ACTION1 ) ), cb12, font );

  coord ct13 = coord::pc( 0, ct12.getPY() + t12->get_size().getPY() + 5 );
  gui_textbox *t13 = new gui_textbox( "ACTION 2", ct13, font );
  ct13.setPX( SCR_WIDTH / 2 - t13->get_size().getPX() );
  t13->set_position( ct13 );

  coord cb13 = coord::pc( ct13.getPX() + t13->get_size().getPX() + 10, ct13.getPY() );
  keybtns[config::PLAYER_2][config::_KEY_ACTION2] = new gui_textbutton( cfg->key_to_str( cfg->get_key( config::PLAYER_2, config::_KEY_ACTION2 ) ), cb13, font );

  coord ct14 = coord::pc( 0, ct13.getPY() + t13->get_size().getPY() + 5 );
  gui_textbox *t14 = new gui_textbox( "TALK", ct14, font );
  ct14.setPX( SCR_WIDTH / 2 - t14->get_size().getPX() );
  t14->set_position( ct14 );

  coord cb14 = coord::pc( ct14.getPX() + t14->get_size().getPX() + 10, ct14.getPY() );
  keybtns[config::PLAYER_2][config::_KEY_TALK] = new gui_textbutton( cfg->key_to_str( cfg->get_key( config::PLAYER_2, config::_KEY_TALK ) ), cb14, font );

  coord ca = coord::pc( SCR_WIDTH / 2 + 20, SCR_HEIGHT - button_gfx[0][0]->h - 20 );
  apply = new gui_button( ca, button_gfx[0][0], button_gfx[0][1], button_gfx[0][2] );
  coord cc = coord::pc( SCR_WIDTH / 2 - button_gfx[1][0]->w - 20, SCR_HEIGHT - button_gfx[1][0]->h - 20 );
  cancel = new gui_button( cc, button_gfx[1][0], button_gfx[1][1], button_gfx[1][2] );

  apply->set_act_screen( this );
  apply->set_act_callback( apply_cb );
  cancel->set_act_screen( this );
  cancel->set_act_callback( cancel_cb );

  /* These macros are small helpers so we'll get BOTH playerno and action_id
   * passed to bind_screen :: What's the proper place for 'em???
   */
#define pack( plno, action )    ( ( plno << 16 ) | ( action & 0xFFFF ) )
#define unpack( plno, action, signum )  { plno = ( signum >> 16 ); action = ( signum & 0xFFFF ); }

  for ( int p = 0; p < config::MAX_LOCAL_PLAYERS; p++ ) {
    for ( int a = 0; a < config::_NUM_KEYS; a++ ) {
      keybtns[p][a]->set_act_screen( bs, pack( p, a ) );
    }
  }

  add_component( tp1 );
  add_component( tp2 );
  add_component( t1 );
  add_component( t2 );
  add_component( t3 );
  add_component( t4 );
  add_component( t5 );
  add_component( t6 );
  add_component( t7 );
  add_component( t8 );
  add_component( t9 );
  add_component( t10 );
  add_component( t11 );
  add_component( t12 );
  add_component( t13 );
  add_component( t14 );
  add_component( apply );
  add_component( cancel );

  for ( int p = 0; p < config::MAX_LOCAL_PLAYERS; p++ ) {
    for ( int a = 0; a < config::_NUM_KEYS; a++ ) {
      add_component( keybtns[p][a] );
    }
  }

  focus_first_component();
}

options_screen::~options_screen() {
  /* Do stuff */
}

void options_screen::activate( int signum )
{
  bool enterpushed = false;
  SDL_Event   ev;

  re_init( false );
  draw();

  /* Bunch of this stuff can be removed from the constructor l8r */
  localconf = *conf;
  
   for ( int p = 0; p < config::MAX_LOCAL_PLAYERS; p++ ) {
    for ( int a = 0; a < config::_NUM_KEYS; a++ ) {
      string    s = conf->key_to_str( conf->get_key( p, config::key_id( a ) ) );

      keybtns[p][a]->set_text( s );
    }
  }

  while ( true ) {
    draw();

    try {
      while ( SDL_PollEvent( &ev ) ) {
        if ( ev.type == SDL_KEYDOWN ) {
          switch ( ev.key.keysym.sym ) {
            case SDLK_UP:
              focus_up();
              break;
            case SDLK_DOWN:
              focus_down();
              break;
            case SDLK_ESCAPE:
              /*
              back_one_screen();
              */
              cancel_cb( this );
              break;

            default:
              {
                gui_focus_component *t = dynamic_cast<gui_focus_component *>(comps[cur_focus]);
                if (t && (ev.key.keysym.sym == SDLK_RETURN ||
                      dynamic_cast<gui_button *>(t) == 0))
                {
                  click(t);
                  enterpushed=true;
                }
                if (t)
                {
                  t->action(ev.key.keysym.sym, ev.key.keysym.mod);
                  if(enterpushed==true)
                  {
                    this->re_init(true);
                    enterpushed=false;
                  }
                }
                if ( bs->did_bind() ) {
                  // bs did some binding. Update text box.
                  string    s = conf->key_to_str( bs->get_action_value() );

                  keybtns[bs->get_bound_pl()][bs->get_bound_action()]->set_text( s );
                }
                break;
              }
          }
        }
      }
    }
    catch ( int i ) {
      if ( i == 1 ) return;
    }
  }
}

void options_screen::apply_cb( gui_screen *scr ) {
  options_screen    *oscr = (options_screen *) scr;
  file_config   fc( &oscr->localconf );
  scr->update_config( &oscr->localconf );
  scr->back_one_screen();
}

void options_screen::cancel_cb( gui_screen *scr ) {
  scr->back_one_screen();
}

bind_screen::bind_screen( SDL_Surface *scr, TTF_Font *fnt, gfx *graphics, starfield_efect *se, audio *au, config *cfg )
  : gui_screen( scr, fnt, graphics, se, au, cfg )
{
  banner = new gui_textbox( "", coord::pc( 0, 0 ), font );
  add_component( banner );
}

bind_screen::~bind_screen() {
  /* cleanup */
}

void bind_screen::activate( int signum ) {
  SDL_Event     ev;
  int           playerno, actionno;


  re_init( false );
  draw();

  really_bound = false;

  unpack( playerno, actionno, signum );

  bannertext = "Press a key for action (ESC cancels)";
  banner->set_text( bannertext );

  coord   c = coord::pc( ( SCR_WIDTH - banner->get_size().getPX() ) / 2, ( SCR_HEIGHT - banner->get_size().getPY() ) / 2 );
  banner->set_position( c );

  while ( true ) {
    draw();
    
    try {
      while ( SDL_PollEvent( &ev ) ) {
        if ( ev.type == SDL_KEYDOWN ) {
          switch ( ev.key.keysym.sym ) {
            case SDLK_ESCAPE:
              back_one_screen();
              break;

            default:
              if ( conf->key_to_str( ev.key.keysym.sym ) ) {
                bool    bound = false;
                
                for ( int i = 0; i < config::MAX_LOCAL_PLAYERS && bound == false; i++ ) {
                  for ( int j = 0; j < config::_NUM_KEYS && bound == false; j++ ) {
                    if ( conf->get_key( i, config::key_id( j ) ) == ev.key.keysym.sym ) {
                      string    s = "Key already bound, try again";
                      banner->set_text( s );
                      bound = true;
                    }
                  }
                }

                if ( bound == false ) {
                  bound_pl = playerno;
                  bound_action = actionno;
                  action_value = ev.key.keysym.sym;
                  really_bound = true;
                  
                  conf->set_key( playerno, config::key_id( actionno ), ev.key.keysym.sym );
                  back_one_screen();
                }
              }
              break;
          }
        }
      }
    }
    catch ( int i ) {
      if ( i == 1 ) return;
    }
  }
}

confirm_quit_screen::confirm_quit_screen( SDL_Surface *scr, TTF_Font *fnt, gfx *graphics,
    starfield_efect *se, audio *au, config *cfg )
: gui_screen( scr, fnt, graphics, se, au, cfg )
{
  coord   c = coord::pc( 0, 0 );

  gui_textbox   *question = new gui_textbox( "Are you sure?", c, font );
  gui_textbutton  *yes = new gui_textbutton( "Yes", c, font );
  gui_textbutton  *no = new gui_textbutton( "No", c, font );

  c.setPX( ( SCR_WIDTH - question->get_size().getPX() ) / 2 );
  c.setPY( SCR_HEIGHT / 2 - question->get_size().getPY() );
  question->set_position( c );

  c.setPY( SCR_HEIGHT / 2 + question->get_size().getPY() );

  c.setPX( SCR_WIDTH / 2 + 10 );
  yes->set_position( c );

  c.setPX( SCR_WIDTH / 2 - no->get_size().getPX() - 10 );
  no->set_position( c );

  yes->set_act_screen( this );
  yes->set_act_callback( yes_cb );
  no->set_act_screen( this );
  no->set_act_callback( no_cb );

  add_component( question );
  add_component( no );
  add_component( yes );

  focus_first_component();
}

void confirm_quit_screen::activate( int signum ) {
  SDL_Event   ev;
  bool        enterpushed = false;

  quit = false;

  re_init( false );

  while ( true ) {
    draw();

    try {
      while ( SDL_PollEvent( &ev ) ) {
        if ( ev.type == SDL_KEYDOWN ) {
          switch ( ev.key.keysym.sym ) {
            case SDLK_RIGHT:
            case SDLK_UP:
              focus_up();
              break;

            case SDLK_LEFT:
            case SDLK_DOWN:
              focus_up();
              break;

            case SDLK_ESCAPE:
              back_one_screen();
              break;

            default:
              {
                gui_focus_component *t = dynamic_cast<gui_focus_component *>( comps[cur_focus] );
                if ( t && ( ev.key.keysym.sym == SDLK_RETURN ||
                      dynamic_cast<gui_button *>( t ) == 0 ) ) {
                  click( t );
                  enterpushed = true;
                }
                if ( t ) {
                  t->action( ev.key.keysym.sym, ev.key.keysym.mod );
                  if ( enterpushed ) {
                    this->re_init( false );
                    enterpushed = false;
                  }
                }
              }
              break;
          }
        }
      }
    }
    catch ( int i ) {
      if ( quit )
        throw ( i );
      else if ( i == 1 )
        return;
    }
  }
}

void confirm_quit_screen::yes_cb( gui_screen *screen ) {
  confirm_quit_screen   *qs = (confirm_quit_screen *) screen;
  qs->set_quit( true );
  qs->back_one_screen();
}

void confirm_quit_screen::no_cb( gui_screen *screen ) {
  screen->back_one_screen();
}

