/* matchbox-nest - a graphical wrapper around xnest
 *
 *  Copyright 2004 Matthew Allum
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 */

#include "fakedev.h"

#define XNEST_BIN "/usr/X11R6/bin/Xnest"

FakeApp*
fakeapp_new(void)
{
  FakeApp *app = malloc(sizeof(FakeApp));
  memset(app, 0, sizeof(FakeApp));
  
  if ((app->dpy = XOpenDisplay(getenv("DISPLAY"))) == NULL)
    {
      fprintf(stderr, "Unable to open DISPLAY. Giving up.\n");
      exit(1);
    }
  
  app->screen  = DefaultScreen(app->dpy);
  app->winroot = RootWindow(app->dpy, app->screen);
  app->gc      = XCreateGC(app->dpy, app->winroot, 0, NULL);
  app->pb      = mb_pixbuf_new(app->dpy, app->screen);

  app->xnest_dpy_name = ":1";
  app->xnest_bin_path = XNEST_BIN ;

  app->key_rep_init_timeout.tv_usec = 0;
  app->key_rep_init_timeout.tv_sec  = 1;

  app->key_rep_timeout.tv_usec      = 40000;
  app->key_rep_timeout.tv_sec       = 0;

  app->win_title                    = "matchbox-nest";

  return app;
}

void
fakeapp_create_gui(FakeApp *app)
{
  FakeButton           *button;
  XSetWindowAttributes  attr;
  Pixmap                backing;
  MBPixbufImage        *img;

  attr.event_mask = 
    ButtonPressMask
    |ButtonReleaseMask
    |ExposureMask;

  app->winmain   = XCreateWindow(app->dpy, 
				 app->winroot, 
				 0, 
				 0,
				 app->device_width, 
				 app->device_height, 
				 0,
				 CopyFromParent, 
				 CopyFromParent, 
				 CopyFromParent,
				 CWEventMask,
				 &attr);

  app->winnest   = XCreateWindow(app->dpy, 
				 app->winmain, 
				 app->device_display_x,
				 app->device_display_y,
				 app->device_display_width,
				 app->device_display_height,
				 0,
				 CopyFromParent, 
				 CopyFromParent, 
				 CopyFromParent,
				 0, NULL);


  backing = XCreatePixmap(app->dpy, 
			  app->winroot,
			  app->device_width, 
			  app->device_height, 
			  mb_pixbuf_get_depth(app->pb));

  mb_pixbuf_img_render_to_drawable(app->pb, app->device_img, backing, 0, 0);

  XSetWindowBackgroundPixmap(app->dpy, app->winmain, backing);
  /* XClearWindow(app->dpy, app->winmain); */

  XFreePixmap(app->dpy, backing);

  attr.event_mask = ButtonPressMask|ButtonReleaseMask;

  for (button = app->button_head; button; button=button->next) 
    {
      int           x,y;
      unsigned char r = 0xff, g = 0xff, b = 0x00;

      button->win = XCreateWindow(app->dpy, 
				 app->winmain, 
				  button->x,
				  button->y,
				  button->width,
				  button->height,
				  0,
				  CopyFromParent, 
				  CopyFromParent, 
				  CopyFromParent,
				  CWEventMask,
				  &attr);


      img = mb_pixbuf_img_rgb_new(app->pb, button->width, button->height);

      mb_pixbuf_img_copy (app->pb, 
			  img, 
			  app->device_img, 
			  button->x,
			  button->y,
			  button->width,
			  button->height,
			  0, 0);
      

      button->backing = XCreatePixmap(app->dpy, 
				      app->winroot,
				      button->width, 
				      button->height, 
				      mb_pixbuf_get_depth(app->pb));

      button->backing_active = XCreatePixmap(app->dpy, 
					     app->winroot,
					     button->width, 
					     button->height, 
					     mb_pixbuf_get_depth(app->pb));

      mb_pixbuf_img_render_to_drawable(app->pb, img, button->backing, 0, 0);

      /* Now draw active state - a border */

      if (!button->img)
	{
	  /* No down image, draw a border around original */

	  for (x=0; x<button->width; x++)
	  {
	    mb_pixbuf_img_plot_pixel(app->pb, img, x, 0, r, g, b);
	    mb_pixbuf_img_plot_pixel(app->pb, img, x, button->height-1, r, g, b);
	  }

	  for (y=0; y<button->height; y++)
	    {
	      mb_pixbuf_img_plot_pixel(app->pb, img, 0, y, r, g, b);
	      mb_pixbuf_img_plot_pixel(app->pb, img, button->width-1, y, r, g, b);
	    }
	}
      else 
	{
	  /* Use supplied down image */

	  mb_pixbuf_img_copy_composite (app->pb, 
					img, 
					button->img,
					0,
					0,
					button->width,
					button->height,
					0, 0);
	}

	
      mb_pixbuf_img_render_to_drawable(app->pb, img, button->backing_active, 
				       0, 0);

      XSetWindowBackgroundPixmap(app->dpy, button->win, button->backing);

      mb_pixbuf_img_free(app->pb, img);

    }

  XStoreName(app->dpy, app->winmain, app->win_title);

  /* Sync so we know the windows are created for holding xnest */

  XSync(app->dpy, True);
}

void
fakeapp_start_server(FakeApp *app)
{
  char winid[24];

  sprintf(winid,"%li", app->winnest);

  switch (fork())
    {
    case 0:
      execl(app->xnest_bin_path, "Xnest", app->xnest_dpy_name, "-ac", "-parent", winid, 0) ;
      fprintf(stderr, "Failed to Launch %s\n", app->xnest_bin_path);
      exit(1);
    case -1:
      fprintf(stderr, "Failed to Launch %s\n", app->xnest_bin_path);
      break;
    }

}

static Bool
get_xevent_timed(Display* dpy, XEvent* event_return, struct timeval *tv)
{

  if (tv->tv_usec == 0 && tv->tv_sec == 0)
    {
      XNextEvent(dpy, event_return);
      return True;
    }

  XFlush(dpy);

  if (XPending(dpy) == 0) 
    {
      int fd = ConnectionNumber(dpy);
      fd_set readset;
      FD_ZERO(&readset);
      FD_SET(fd, &readset);

      if (select(fd+1, &readset, NULL, NULL, tv) == 0) 
	return False;
      else {
	XNextEvent(dpy, event_return);
	return True;
      }

    } else {
      XNextEvent(dpy, event_return);
      return True;
    }
}

void
fakeapp_event_loop(FakeApp *app)
{
  XEvent          xev;
  FakeButton     *button, *held_button = NULL;
  struct timeval  tv;

  tv.tv_usec = 0; tv.tv_sec = 0;

  for (;;)
    {
      if (get_xevent_timed(app->dpy, &xev, &tv))
	{
	  switch (xev.type) 
	    {
	    case ButtonPress:
	      if ((button = button_find_from_win(app, xev.xbutton.window)) != NULL)
		{
		  button_press(app, button);
		  if (button->repeat)
		    {
		      held_button = button;
		      tv.tv_usec = app->key_rep_init_timeout.tv_usec; 
		      tv.tv_sec  = app->key_rep_init_timeout.tv_sec; 
		    }
		}
	      
	      XSetInputFocus(app->dpy, app->winnest, 
			     RevertToParent, CurrentTime);

	      break;
	    case ButtonRelease:
	      if ((button = button_find_from_win(app, xev.xbutton.window)) != NULL)
		{
		  button_release(app, button);
		  held_button = NULL;
		  tv.tv_usec = 0; tv.tv_sec = 0;
		}
	      
	      XSetInputFocus(app->dpy, app->winnest, 
			     RevertToParent, CurrentTime);
	      break;
	    }
	}
      else 			/* timeout - repeats */
	{
	  if (held_button)
	    {
	        keys_send_key(app, held_button->keysym, KEYUPDOWN);
		tv.tv_usec = app->key_rep_timeout.tv_usec; 
		tv.tv_sec  = app->key_rep_timeout.tv_sec; 
	    }
	}
    }
}


void 
usage(char *progname)
{
  fprintf(stderr, 
	  "%s " VERSION " usage:\n"
	  "--xnest-dpy, -xd  Display String for Xnest to use. ( default ':1')\n"
          "--xnest-bin, -xn  Location of Xnest binary ( default " XNEST_BIN ")\n"
	  "--title,     -t   Set the window title\n"     
	  "--device,    -d   Device config file to use\n"
	  "--help,      -h   Show this help\n",
	  progname);

  exit(1);
}

int 
main(int argc, char **argv)
{
  FakeApp *app;
  char    *device =   PKGDATADIR "/ipaq3800.xml"; 
  int      i;

  signal(SIGCHLD, SIG_IGN);

  app = fakeapp_new();

  for (i = 1; i < argc; i++) {
    if (!strcmp ("--xnest-dpy", argv[i]) || !strcmp ("-xd", argv[i])) {
      if (++i>=argc) usage (argv[0]);
      app->xnest_dpy_name = argv[i];
      continue;
    }
    if (!strcmp ("--xnest-bin", argv[i]) || !strcmp ("-xn", argv[i])) {
      if (++i>=argc) usage (argv[0]);
      app->xnest_bin_path = argv[i];
      continue;
    }
    if (!strcmp ("--device", argv[i]) || !strcmp ("-d", argv[i])) {
      if (++i>=argc) usage (argv[0]);
      device = argv[i];
      continue;
    }
    if (!strcmp ("--title", argv[i]) || !strcmp ("-t", argv[i])) {
      if (++i>=argc) usage (argv[0]);
      app->win_title = argv[i];
      continue;
    }

    if (!strcmp("--help", argv[i]) || !strcmp("-h", argv[i])) {
      usage(argv[0]);
    }

    usage(argv[0]);
  }
  
  config_init(app, device);

  fakeapp_create_gui(app);

  fakeapp_start_server(app);

  XMapWindow(app->dpy, app->winmain);
  XMapSubwindows(app->dpy, app->winmain);

  XFlush(app->dpy);

  XSetInputFocus(app->dpy, app->winnest, RevertToParent, CurrentTime);

  keys_init(app);

  fakeapp_event_loop(app);

  return 0;
}
