//
// nono
// Copyright (C) 2020 nono project
// Licensed under nono-license.txt
//

#include "mpu680x0.h"
#include "m68030mmu.h"
#include "mainbus.h"

#define TIA		(0)
#define TIB		(1)
#define TIC		(2)
#define TID		(3)
#define FCL		(4)	// 便宜上ね
#define MAX_LV	(5)

// ディスクリプタタイプ
#define GET_DT(x)	(x & 0x03)
#define DT_INVALID	(0)
#define DT_PAGE		(1)
#define DT_VALID4	(2)
#define DT_VALID8	(3)

// ディスクリプタ
#define MMU_DESC_S	(0x00000100)
#define MMU_DESC_CI	(0x00000040)
#define MMU_DESC_M	(0x00000010)
#define MMU_DESC_U	(0x00000008)
#define MMU_DESC_WP	(0x00000004)
#define MMU_DESC_DT	(0x00000003)

// エントリの状態
enum ATCtype {
	INVALID,
	NORMAL,
	EARLY,
	INDIRECT,
};

//
// デバッグ表示
//

// デバッグ表示のため値の表示に必要な桁数を返す
#define NIBBLE(x)		( (x) == 0 ? 1 : ((x) + 3) / 4 )

// インデント付き string_format
static std::string indent_format(int indent, const char *fmt, ...)
	__printflike(2, 3);
static std::string
indent_format(int indent, const char *fmt, ...)
{
	va_list ap;
	char *buf;

	va_start(ap, fmt);
	vasprintf(&buf, fmt, ap);
	va_end(ap);

	std::string rv(indent, ' ');
	rv += buf;
	free(buf);

	return rv;
}

#define MMULOG(lv, indent, arg...) do { \
	if (__predict_false(loglevel >= (lv))) {	\
		cpu->putlogf(lv, [&, lam_func = __func__] { \
			(void)lam_func; \
			return indent_format(indent, arg); \
		});	\
	}	\
} while (0)

// テーブルサーチ操作クラス
class m68030MMU final
{
 public:
	m68030MMU(MPU68030Device *, int loglevel_);
	~m68030MMU();

	// サーチを実行
	void search(int level, busaddr laddr);

	// ATC にエントリを作成
	void make_atc(m68030ATCLine *);

	// ATC にエントリを作成 (デバッガアクセス用)
	busaddr make_atc_debugger() const;

 private:
	// リミットチェック
	bool limitcheck() const;

	// ディスクリプタを取得
	bool fetch_desc(uint32, int);

	// ディスクリプタを表示用に (デバッグ用)
	std::string sprintf_desc();

	// ディスクリプタタイプ文字列 (デバッグ用)
	const char *dtname(uint32 dt);

 public:
	ATCtype m_type {};		// 状態
	bool    m_s {};			// S ビット
	bool    m_wp {};		// WP ビット
	bool    m_ci {};		// CI ビット
	bool    m_m {};			// ページディスクリプタの M が立っているか
	bool    m_lim {};		// サーチ中にリミットを越えた
	bool	m_berr {};		// テーブルサーチ中にバスエラー
	bool	m_ptest {};		// PTEST[RW] 命令中なら真
	bool	m_debugger {};	// デバッガからのアクセスなら true にする
	int     x {};			// 段数 TIA〜TID, FCL
	uint32	m_lastaddr {};	// 最後にアクセスしたディスクリプタアドレス(PTEST)
 private:
	int     m_lv[MAX_LV] {};
	uint32  m_desc1 {};		// ディスクリプタ
	uint32  m_desc2 {};		// ロングディスクリプタの後半部
	int     m_descsize {};	// ディスクリプタサイズ (4 or 8)
	int     m_lastsize {};	// 前段のディスクリプタサイズ

	// デバッグ用文字列
	static const char * const m_lvname[MAX_LV];

	int loglevel {};

	MPU68030Device *cpu;
};

// 現在の TC:E、TT:E の状態に応じてアドレス変換の有無を切り替える。
// mmu_enable ならアドレス変換ありのバスアクセスを使う。see m68030bus.cpp
void
MPU68030Device::mmu_enable_changed()
{
	bool enable;

	// TC、TT0、TT1 のいずれかが有効ならアドレス変換あり
	enable = (mmu_tc.e || mmu_tt[0].e || mmu_tt[1].e);

	// 変化した時だけ
	if (mmu_enable && !enable) {
		// 無効にする
		mmu_enable = false;
		putlog(1, "MMU Translation Disabled");
	} else if (!mmu_enable && enable) {
		// 有効にする
		mmu_enable = true;
		putlog(1, "MMU Translation Enabled");
	}
}


//
// レジスタクラス
//

bool
MPU68030Device::SetSRP(uint32 h, uint32 l)
{
	if ((h & m68030RP::DT_MASK) == 0) {
		return false;
	}
	reg.srp.h = h & m68030RP::H_MASK;
	reg.srp.l = l & m68030RP::L_MASK;
	return true;
}

bool
MPU68030Device::SetCRP(uint32 h, uint32 l)
{
	if ((h & m68030RP::DT_MASK) == 0) {
		return false;
	}
	reg.crp.h = h & m68030RP::H_MASK;
	reg.crp.l = l & m68030RP::L_MASK;
	return true;
}

void
MPU68030Device::SetTT(int n, uint32 data)
{
	uint32 lbase;
	uint32 lmask;
	uint32 sbase;
	uint32 smask;

	// ハッシュから TT#n の成分を一旦クリアする。
	if (mmu_tt[n].e) {
		// 変更前の TT#n にマッチして、現在の TT#(1-n) にマッチしない
		// アドレスブロックに対応する 4096 エントリをクリア。
		for (int fc = 1; fc <= 6; fc++) {
			auto *table = atc.fctables[fc];
			if (table) {
				busaddr laddr = busaddr(0) | busaddr::FC(fc);
				for (int i = 0; i < 256; i++) {
					// XXX R/W は?
					if (mmu_tt[n].Match(laddr) &&
						mmu_tt[1 - n].Match(laddr) == false)
					{
						uint32 baseidx = i << (24 - 12);
						memset(&table->hash[baseidx],
							m68030ATCTable::HASH_NONE, 4096);
					}
					laddr += 0x0100'0000;
				}
			}
		}
	}

	// マスクしてレジスタイメージにセット
	reg.tt[n] = data & m68030TT::MASK;

	// data を分解してメンバにセット。
	//
	//    3                   2                   1                   0
	//  1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0
	// +---------------+---------------+-+-+-+-+-+-+-+-+-+-----+-+-----+
	// |LogicalAddrBase|LogicalAddrMask|E|       |C|R|M| | FCB | | FCM |
	// +---------------+---------------+-+-+-+-+-+-+-+-+-+-----+-+-----+

	mmu_tt[n].e      = (data & m68030TT::E);
	mmu_tt[n].ci     = (data & m68030TT::CI);

	// 下位ロングワード
	// ・lbase は比較先アドレスで、ff000000の位置。
	// ・lmask はアドレスマスク。%1 が無視を表しているので NOT しておく。
	//
	//  target  AAAAAAAA xxxxxxxx xxxxxxxx xxxxxxxx : Aは論理アドレス
	//        ^                                     : xは不問
	//  lbase   BBBBBBBB 00000000 00000000 00000000 : Bは論理アドレスベース
	//        &
	//  lmask   MMMMMMMM 00000000 00000000 00000000 : Mは論理アドレスマスク
	//
	lbase  = (data & m68030TT::LBASE_MASK);
	lmask  = (~data & m68030TT::LMASK_MASK) << 8;

	// 上位ロングワード
	// ・E ビットは位置はどこでもいいので TT:E ビットの
	//   位置をそのまま使用する。target 側のこの位置は必ずビットが落ちている
	//   ため base 側にビットを立てておくと必ず不一致になることを利用して、
	//   E ビットの評価も一度に行う。
	// ・RW ビットは SSW の RW と比較するので、SSW_RW のビット位置を使う
	//   (0x00000040 の位置)。TT:RW も SSW_RW も Read が %1。
	// ・RWM が %1 なら R/W は無視、RWM が %0 なら R/W の一致も検査するので
	//   RWM が %0 の時は RW のマスクビットを立てておく。
	// ・FC は busaddr の配置に合わせて1ビット左シフトしたところを使う。
	// ・FCMask は %1 が無視を表しているので NOT しておく。
	//
	//  target  00000000 00000000 00000000 0R00FFFS : R はアクセス R/~W
	//        ^                                     : F はアクセス FC
	//                                              : S は FC2 のコピー
	//  base    00000000 00000000 E0000000 0R00FFF0 : R は期待する R/~W
	//        &                                     : F は期待する FC
	//                                              : E は ~TT:E
	//  mask    00000000 00000000 10000000 0M00CCC0 : M は R/~W マスク
	//                                              : C は FC マスク
	sbase  = ~data & m68030TT::E;
	sbase |= (data & m68030TT::RW) >> 3;
	sbase |= (data & m68030TT::FCBASE_MASK) >> 3;
	smask  = m68030TT::E;
	if ((data & m68030TT::RWM) == 0) {
		smask |= M68030::SSW_RW;
	}
	smask |= (~data & m68030TT::FCMASK_MASK) << 1;

	mmu_tt[n].base = lbase | (((uint64)sbase) << 32);
	mmu_tt[n].mask = lmask | (((uint64)smask) << 32);

	// 更新後にこの TT が有効ならハッシュも更新。
	if (mmu_tt[n].e) {
		// この TT#n にマッチするアドレスブロックに対応する 4096 エントリを
		// TT 印にする。
		for (uint fc = 1; fc <= 6; fc++) {
			auto *table = atc.fctables[fc];
			if (table) {
				busaddr laddr = busaddr(0) | busaddr::FC(fc);
				for (uint i = 0; i < 256; i++) {
					// XXX R/W は?
					if (mmu_tt[n].Match(laddr)) {
						uint32 baseidx = i << (24 - 12);
						memset(&table->hash[baseidx],
							m68030ATCTable::HASH_TT, 4096);
					}
					laddr += 0x0100'0000;
				}
			}
		}
	}

	mmu_enable_changed();
}

// アクセスがこの TT とマッチするか調べる。
// laddr の上位ワードはそのまま使える。
// 下位32ビットが論理アドレス (下24ビットは呼び出し側でマスクしなくてよい)。
// この TT が有効でアドレスがマッチすれば true を返す。
// (TT:E は演算に織り込んである、このすぐ上の SetTT() のコメント参照)
bool
m68030TT::Match(busaddr laddr) const
{
	bool rv = (((laddr.Get() ^ base) & mask) == 0);
	return rv;
}

// 成功すれば true、MMU 構成例外が起きる場合は false を返す。
// 呼び出し側で例外を起こすこと。
bool
MPU68030Device::SetTC(uint32 data)
{
	// マスクしてレジスタイメージにセット
	reg.tc = data & m68030TC::MASK;

	// data を分解してメンバにセット
	mmu_tc.Set(data);

	// 有効にする場合は一貫性を確認
	if (mmu_tc.e) {
		if (mmu_tc.Check() == false) {
			// エラーなら E をクリアして MMU 構成例外。
			// (例外処理自体は呼び出し側で行う)
			mmu_tc.e = false;
			mmu_enable_changed();
			return false;
		}
	}

	// マスクをあらかじめ用意しておく
	mmu_tc.MakeMask();

	// ATC でも PS を覚えておく必要がある。
	for (int i = 0; i < countof(atc.tables); i++) {
		auto *table = atc.tables[i].get();
		table->ps4k = mmu_tc.ps - 12;
	}

	// TC:E が変更されていればスイッチ更新
	mmu_enable_changed();
	return true;
}

// TC レジスタへの書き込み値を m68030TC クラスにセットする。
void
m68030TC::Set(uint32 data)
{
	e   = data & TC_E;
	sre = data & TC_SRE;
	fcl = data & TC_FCL;
	tid = (data)       & 0x0f;
	tic = (data >>= 4) & 0x0f;
	tib = (data >>= 4) & 0x0f;
	tia = (data >>= 4) & 0x0f;
	is  = (data >>= 4) & 0x0f;
	ps  = (data >>= 4) & 0x0f;
}

// 現在の値が MMU 構成例外になるなら false を返す。
bool
m68030TC::Check() const
{
	if (ps < 8)
		return false;
	if (tia == 0)
		return false;
	if (tib == 0 && (tic > 0 || tid > 0))
		return false;
	if (tic == 0 && tid > 0)
		return false;
	if (ps + is + tia + tib + tic + tid != 32)
		return false;

	return true;
}

// 論理マスク、ページマスクを作成。(テーブルサーチで参照される)
void
m68030TC::MakeMask()
{
	// lmask (論理マスク) は TIA〜TID 部分のマスク。
	// pgmask (ページマスク) は PS 部分のマスク。
	//
	// 例えば IS=$8, TIA=$3, TIB=$4, TIC=$4, TID=$0, PS=$d
	// (X68030 IPL 内の MMU 判定のケース) なら
	//
	//           3          2          1          0
	//          10987654 32109876 54321098 76543210
	//         +--------+--------+--------+--------+
	//         |IIIIIIII AAABBBBC CCCPPPPP PPPPPPPP|
	//         +--------+--------+--------+--------+
	// lmask  = 00000000 11111111 11100000 00000000
	// pgmask = 00000000 00000000 00011111 11111111
	lmask = (1U << (tia + tib + tic + tid)) - 1;
	lmask <<= ps;
	pgmask = (1U << ps) - 1;

	// アドレスを (IS,) TIA 〜 TID に分解。
	//
	// TIA=$C, TIB=$A, (TIC = TID = 0), PS=$A の場合以下のようにする。
	// 使ってない TIC, TID については tix[TIC] == 0 で判断する。
	//
	//                    A              B             PS
	//           +----------------+--------------+--------------+
	// $00A01A00 | 0000 0000 1010 | 0000 0001 10 | xx xxxx xxxx |
	//           +----------------+--------------+--------------+
	//            shift[TIA]=20     shift[TIB]=10
	//            mask[TIA]=0xfff   mask[TIB]=0x3ff
	uint sh = ps;
	for (int i = TID; i >= TIA; i--) {
		if (tix[i] > 0) {
			shift[i] = sh;
			mask[i] = (1U << tix[i]) - 1;
			sh += tix[i];
		}
	}
}

void
MPU68030Device::SetMMUSR(uint16 data)
{
	reg.mmusr = data & m68030MMUSR::MASK;
}

// リセット例外の CPU 固有部分。
void
MPU68030Device::ResetMMU()
{
	SetTC(GetTC() & ~m68030TC::TC_E);
	SetTT(0, GetTT(0) & ~m68030TT::E);
	SetTT(1, GetTT(1) & ~m68030TT::E);
}


//
// ATC
//
// 今となっては独立したクラスである必要はなくなっているが、これを
// MPU68030Device にマージするとパフォーマンスが下がるので維持してある。
//

// コンストラクタ
m68030ATC::m68030ATC(MPU68030Device *cpu_)
{
	cpu = cpu_;

	tables[0].reset(new m68030ATCTable(cpu, 1));
	tables[1].reset(new m68030ATCTable(cpu, 2));
	tables[2].reset(new m68030ATCTable(cpu, 5));
	tables[3].reset(new m68030ATCTable(cpu, 6));

	fctables[1] = tables[0].get();
	fctables[2] = tables[1].get();
	fctables[5] = tables[2].get();
	fctables[6] = tables[3].get();
}

// デストラクタ
m68030ATC::~m68030ATC()
{
}

// ログレベルの設定。
// これは Object のオーバーライドではないが、同じ使い方をする。
void
m68030ATC::SetLogLevel(int loglevel_)
{
	loglevel = loglevel_;
	for (int i = 0; i < countof(tables); i++) {
		auto *table = tables[i].get();
		table->loglevel = loglevel_;
	}
}

// ATC をすべてフラッシュする。命令からコールされる
void
m68030ATC::flush()
{
	for (int i = 0; i < countof(tables); i++) {
		auto *table = tables[i].get();
		table->Flush();
	}
}

// ATC の fc, mask が一致するエントリをフラッシュする。
// 命令からコールされる。
void
m68030ATC::flush(uint32 fc, uint32 mask)
{
	for (uint i = 1; i <= 6; i++) {
		if (((i ^ fc) & mask) == 0) {
			// FC が一致したテーブルを全フラッシュ
			auto *table = fctables[i];
			if (table) {
				table->Flush();
			}
		}
	}
}

// ATC の fc, mask, addr が一致するエントリをフラッシュする。
// 命令からコールされる。
void
m68030ATC::flush(uint32 fc, uint32 mask, uint32 addr)
{
	for (uint i = 1; i <= 6; i++) {
		if (((i ^ fc) & mask) == 0) {
			// FC が一致したテーブルから addr が一致したエントリをフラッシュ
			auto *table = fctables[i];
			if (table) {
				table->Flush(addr);
			}
		}
	}
}


//
// ATC テーブル
//

// アドレス変換の本来のフローはこんな感じだが、
//
//  if (TT*.E || TC.E) {
//  	if (TT*.E が有効) {
//  		laddr,RW が TT[01] とマッチしたら、paddr = laddr,CI
//  	}
//  	if (TC.E が有効) {
//  		ATC とマッチしたら、paddr = atc[n].paddr,CI
//  		そうでなければ、 paddr = テーブルサーチ
//  	}
//  } else {
//  	paddr = laddr (アドレス変換オフ)
//  }
//
// ハッシュで一度に引く。
//	switch (hash[laddr>>12]) {
//   case HASH_TT:
//  	laddr,RW が TT[01] とマッチしたら、paddr = laddr,CI
//   case HASH_ATC:
//  	paddr = atc[n].paddr,CI
//   case HASH_NONE:
//  	TC.E なら paddr = テーブルサーチ
//  	そうでなければ、paddr = laddr (アドレス変換オフ)
//   case HASH_SUB:
//  	PS が 4KB 未満だとハッシュの1エントリに2つ以上が混在するので、
//  	この場合は諦めて ATC を線形探索。通常起きないので性能無視。
//  }

// ATC テーブルは本来は 1つ(FC混合) x 22 ラインだが、
// 高速化のため FC別(4テーブル) x 22 ラインのモードを用意する。
// FC 混合版ではテーブルは tables[0] のみで、FC(1,0,2) は busaddr の
// 34-32 ビット部分として laddr と一緒に検索キーにする。
// FC 分離版ではテーブルは fctables[fc] で検索キーは laddr(下位32ビット部分)。
//
// ハッシュなのでライン間の順序関係はないが、LRU で古いのを捨てる時や
// PS<4K の時に前から探索する時のために、age を使った順序管理をする。
// ラインを使用するたびに age に ++latest_age を代入しておくことで、
// 順序が必要な時には age の大きい順にすれば最近使った順になる。
// 32 ビット符号なし整数が折り返すと事故るがたぶんそれより高い頻度で
// フラッシュが起きるんじゃないかな…

// コンストラクタ
m68030ATCTable::m68030ATCTable(MPU68030Device *cpu_, uint fc_)
{
	cpu = cpu_;
	fc = fc_;

	std::fill(hash.begin(), hash.end(), HASH_NONE);

	latest_age = 0;
	Flush();
}

// デストラクタ
m68030ATCTable::~m68030ATCTable()
{
}

// この ATC テーブルのエントリをすべてフラッシュする。
void
m68030ATCTable::Flush()
{
	for (int i = 0; i < LINES; i++) {
		Invalidate(&line[i]);
		line[i].age = 0;
	}
	latest_age = 0;
}

// この ATC テーブルから addr が一致するエントリをすべてフラッシュする。
void
m68030ATCTable::Flush(uint32 addr)
{
	for (int i = 0; i < LINES; i++) {
		m68030ATCLine *a = &line[i];
		if (a->GetTag() == addr) {
			Invalidate(a);
		}
	}
}

// この要素を先頭に移動。
inline void
m68030ATCTable::MoveHead(m68030ATCLine *a)
{
	a->age = ++latest_age;

	if (__predict_false(a->age == 0)) {
		// latest_age が1周したので、一旦全部リセット。
		for (int i = 0; i < LINES; i++) {
			line[i].age = 0;
		}
		a->age = ++latest_age;
	}
}

// このテーブルのうち最も古い要素を差し出す。
m68030ATCLine *
m68030ATCTable::RemoveTail()
{
	m68030ATCLine *a = NULL;
	uint32 min = (uint32)~0;
	uint32 minidx = 0;

	// 最後に使われたの (あるいは空き) を探す。
	for (int i = 0; i < LINES; i++) {
		if (line[i].IsInvalid()) {
			return &line[i];
		}
		if (line[i].age < min) {
			min = line[i].age;
			minidx = i;
		}
	}
	if (a == NULL) {
		a = &line[minidx];
	}

	// ハッシュから削除。
	Invalidate(a);

	return a;
}

// a からハッシュを作成する。
void
m68030ATCTable::MakeHash(const m68030ATCLine *a)
{
	uint32 addr4k = a->GetLAddr() >> 12;
	int idx = a - &line[0];

	if (__predict_true(ps4k == 0)) {
		hash[addr4k] = idx;
	} else if (ps4k > 0) {
		for (int i = 0; i < (1U << ps4k); i++) {
			hash[addr4k + i] = idx;
		}
	} else {
		hash[addr4k] = HASH_SUB;
	}
}

// ATC a を無効にする。すでに無効なら何もしない。
void
m68030ATCTable::Invalidate(m68030ATCLine *a)
{
	if (a->IsValid()) {
		// 有効ならハッシュから削除。
		if (__predict_true(ps4k == 0)) {
			uint32 addr4k = a->GetLAddr() >> 12;
			hash[addr4k] = HASH_NONE;
		} else if (ps4k > 0) {
			// PS>4KB
			uint32 addr4k = a->GetLAddr() >> 12;
			for (int i = 0; i < (1U << ps4k); i++) {
				hash[addr4k + i] = HASH_NONE;
			}
		} else {
			// PS<4KB、自分が抜けることでこの枠が空くかどうか
			InvalidateSub(a);
		}

		a->Invalidate();
	}
}

// PS<4KB の場合に aa が抜けることでこの addr4k の枠が空けばハッシュから削除。
void
m68030ATCTable::InvalidateSub(const m68030ATCLine *aa)
{
	const uint32 MASK_4K = 0xfffff000 | m68030ATCLine::INVALID;
	uint32 target_laddr = aa->GetTag() & MASK_4K;
	bool in_use = false;

	for (int i = 0; i < LINES; i++) {
		auto *a = &line[i];
		// 自分自身は今から削除予定なので除く。
		if (a == aa) {
			continue;
		}
		// それ以外で有効なものがあるか。
		if ((a->GetTag() & MASK_4K) == target_laddr) {
			in_use = true;
			break;
		}
	}

	// 誰もいなくなれば削除。
	if (in_use == false) {
		hash[target_laddr >> 12] = HASH_NONE;
	}
}

// laddr を ATC から探す。見つからなければ NULL を返す。
m68030ATCLine *
m68030ATCTable::FindSub(const busaddr laddr)
{
	uint32 addr = laddr.Addr() & cpu->mmu_tc.lmask;

	// 同じエントリはないはずなので、順序通りに探さなくてもいいはず。
	for (int i = 0; i < LINES; i++) {
		auto *a = &line[i];
		if (a->GetTag() == addr) {
			return a;
		}
	}
	return NULL;
}

// テーブルサーチを行う。
// テーブルという用語がぶつかっているので MMU Search にしてある。
m68030ATCLine *
m68030ATCTable::MMUSearch(busaddr laddr)
{
	m68030ATCLine *a = RemoveTail();
	a->tag = laddr.Addr() & cpu->mmu_tc.lmask;

	m68030MMU mmu(cpu, loglevel);
	mmu.search(7, laddr);
	mmu.make_atc(a);

	MakeHash(a);
	return a;
}

// デバッグ表示。
std::string
m68030ATCTable::LineToStr(const m68030ATCLine *a) const
{
	uint32 laddr = a->GetLAddr();

	if (!a->IsBusError()) {
		uint32 paddr = a->GetPAddr();
		uint32 ci    = a->IsCInhibit();
		uint32 wp    = a->IsWProtect();
		uint32 mod   = a->IsModified();
		return indent_format(1,
			"ATC: match %u:$%08x -> $%08x CI=%u WP=%u M=%u",
			fc, laddr, paddr, ci, wp, mod);
	} else {
		return indent_format(1,
			"ATC: match %u:$%08x -> BusErr",
			fc, laddr);
	}
}


//
// テーブルサーチ
//

// コンストラクタ
m68030MMU::m68030MMU(MPU68030Device *cpu_, int loglevel_)
{
	cpu = cpu_;
	loglevel = loglevel_;

	m_type = INVALID;
	x = -1;
}

// デストラクタ
m68030MMU::~m68030MMU()
{
}

// テーブルサーチのメイン部分。
// m_type および ATC 作成に必要な情報を埋めてから帰る。
// ATC は帰った先で作成する。
void
m68030MMU::search(int level, busaddr laddr)
{
	const char *rpname;
	uint32 addr;
	uint32 fc;
	int dt;
	bool is_super;

	addr = laddr.Addr();
	fc   = laddr.GetFC();

	is_super = (fc & 0x04);

	// アドレスを (IS,) TIA .. TID に分解。
	//
	// TIA=$C, TIB=$A, (TIC = TID = 0), PS=$A の場合以下のようにする。
	// 使ってない TIC, TID については cpu->mmu_tc->{tic,tid} == 0 で判断する。
	//
	//                    A              B             PS
	//           +----------------+--------------+--------------+
	// $00A01A00 | 0000 0000 1010 | 0000 0001 10 | xx xxxx xxxx |
	//           +----------------+--------------+--------------+
	//            shift[TIA] =20   shift[TIB] =10
	//            lv[TIA].idx=0xA  lv[TIB].idx=0x6
	//
	for (int i = TID; i >= TIA; i--) {
		if (cpu->mmu_tc.tix[i] > 0) {
			uint32 mask = cpu->mmu_tc.mask[i];
			m_lv[i] = (addr >> cpu->mmu_tc.shift[i]) & mask;
		}
	}
	m_lv[FCL] = fc;

	cpu->putlogf(3, [&] {
		std::string buf;

		buf = indent_format(2, "search: laddr=$%08x ->", addr);
		if (cpu->mmu_tc.is > 0)
			buf += string_format(" IS=%u", cpu->mmu_tc.is);
		if (cpu->mmu_tc.fcl)
			buf += string_format(" FC=%u", m_lv[FCL]);
		for (int i = TIA; i <= TID; i++) {
			if (cpu->mmu_tc.tix[i] > 0) {
				buf += string_format(" %s=%u($%0*x)",
					m_lvname[i], cpu->mmu_tc.tix[i],
					NIBBLE(cpu->mmu_tc.tix[i]), m_lv[i]);
			}
		}
		return buf;
	});

	// この辺に図9-26 の初期化だがコンストラクタで初期化済み。

	// ここから図9-25。

	// 使用するルートポインタを決定
	if (__predict_true(cpu->mmu_tc.sre) && is_super) {
		rpname = "SRP";
		m_desc1 = cpu->GetSRPh();
		m_desc2 = cpu->GetSRPl();
	} else {
		rpname = "CRP";
		m_desc1 = cpu->GetCRPh();
		m_desc2 = cpu->GetCRPl();
	}

	// ルートポインタの DT をチェック
	dt = GET_DT(m_desc1);
	MMULOG(3, 2, "%s: %s dt=%s", lam_func, rpname, dtname(dt));
	if (__predict_true(dt == DT_VALID4)) {
		// 有効 4 バイト。基本的にはこれしか使われないので優先。
		m_descsize = 4;
		m_lastsize = 8;
	} else if (dt == DT_VALID8) {	// 有効 (8バイト)
		m_descsize = 8;
		m_lastsize = 8;
	} else if (dt == DT_PAGE) {		// ページディスクリプタ
		m_type = EARLY;
		return;
	} else {
		// 無効は設定できないはずなので、ここには来ないはず。
		PANIC("%s.DT==Invalid", rpname);
	}

	// 必要ならファンクションコード・ルックアップ
	if (__predict_false(cpu->mmu_tc.fcl)) {
		x = FCL;
		// たぶん検索レベルを1つ消費するはず。未確認。
		level--;

		MMULOG(3, 2, "%s: %s tableaddr=$%08x idx=$%x", lam_func,
			m_lvname[x], m_desc2, m_lv[x]);

		// ディスクリプタをフェッチ
		if (fetch_desc(m_desc2, m_lv[x]) == false) {
			return;
		}

		// ディスクリプタタイプをチェック
		dt = GET_DT(m_desc1);
		switch (dt) {
		 case DT_INVALID:	// 無効
			m_type = INVALID;
			return;

		 case DT_PAGE:		// ページディスクリプタ
			m_type = EARLY;
			return;

		 case DT_VALID4:	// 有効4バイト
		 case DT_VALID8:	// 有効8バイト
			m_lastsize = m_descsize;
			m_descsize = 1U << dt;
			break;
		}
	}

	// TIx のテーブルサーチに入る。図9-25 の下半分。

	for (x = TIA; x < level; x++) {
		// リミットチェック
		if (__predict_false(m_lastsize == 8) && limitcheck() == false) {
			m_type = INVALID;
			m_lim = true;
			return;
		}

		// テーブルアドレス
		uint32 tableaddr;
		tableaddr = (m_lastsize == 4) ? m_desc1 : m_desc2;
		tableaddr &= 0xfffffff0;
		MMULOG(3, 2, "%s: %s tableaddr=$%08x idx=$%x", lam_func,
			m_lvname[x], tableaddr, m_lv[x]);

		// ディスクリプタをフェッチ
		if (fetch_desc(tableaddr, m_lv[x]) == false) {
			return;
		}

		// ディスクリプタタイプをチェック
		dt = GET_DT(m_desc1);
		switch (dt) {
		 case DT_INVALID:	// 無効
			m_type = INVALID;
			return;

		 case DT_PAGE:		// ページディスクリプタ
			if (x == TID) {
				m_type = NORMAL;
			} else {
				if (cpu->mmu_tc.tix[x + 1] == 0) {
					m_type = NORMAL;
				} else {
					m_type = EARLY;
				}
			}
			return;

		 case DT_VALID4:	// 有効4バイト
		 case DT_VALID8:	// 有効8バイト
			m_lastsize = m_descsize;
			m_descsize = 1U << dt;

			if (__predict_false(x == TID || cpu->mmu_tc.tix[x + 1] == 0)) {
				// 間接
				m_type = INDIRECT;

				// ディスクリプタをフェッチ
				tableaddr = (m_lastsize == 4) ? m_desc1 : m_desc2;
				tableaddr &= 0xfffffff0;
				if (fetch_desc(tableaddr, 0) == false) {
					return;
				}
				if (GET_DT(m_desc1) != DT_PAGE) {
					m_type = INVALID;
				}
				return;
			}
			// 次段のサーチを繰り返す
			break;
		}
	}

	// レベル上限にひっかかった場合、x は過ぎているので一つ戻しておく。
	x--;
}

// リミットチェックを実行。図9-28。
// 無効のケースに落ちたら false を返す。
bool
m68030MMU::limitcheck() const
{
	uint16 limit = (m_desc1 >> 16) & 0x7fff;

	// 最上位ビットが L/U
	if ((int32)m_desc1 < 0) {
		if (m_lv[x] < limit) {
			MMULOG(3, 2, "%s: $%04x < $%04x", lam_func, m_lv[x], limit);
			return false;
		}
	} else {
		if (m_lv[x] > limit) {
			MMULOG(3, 2, "%s: $%04x > $%04x", lam_func, m_lv[x], limit);
			return false;
		}
	}
	return true;
}

// ディスクリプタ中のフラグの更新 (を予約する)
#define UPDATE_DESC(flag)	do { \
	if ((m_desc1 & (flag)) == 0) { \
		m_desc1 |= (flag); \
		do_write = true; \
	} \
} while (0)

// ディスクリプタのフェッチと、ヒストリおよびステータスの更新。
// 図9-29。
// ディスクリプタを取得できれば true、
// ATC エントリの作成に向かう場合は false。
bool
m68030MMU::fetch_desc(uint32 tableaddr, int idx)
{
	uint32 descaddr;
	bool do_write;
	uint dt;

	// ディスクリプタのアドレスを計算。ここだけマニュアルの説明が雑すぎ。
	// インダイレクトなら idx = 0 でコールしてあるので、
	// tableaddr がそのまま PA になるという寸法。
	descaddr = tableaddr + idx * m_descsize;
	MMULOG(3, 2, "%s: descsize=%u descaddr=$%08x", lam_func,
		m_descsize, descaddr);

	// バスエラー発生フラグをクリアしておく
	m_berr = false;

	// ディスクリプタを取得
	if (__predict_false(m_debugger)) {
		// デバッガアクセス
		uint64 data;
		data = cpu->mainbus->Peek4(descaddr);
		if ((int64)data < 0) {
			goto buserr;
		}
		m_desc1 = data;

		if (m_descsize == 8) {
			data = cpu->mainbus->Peek4(descaddr + 4);
			if ((int64)data < 0) {
				goto buserr;
			}
			m_desc2 = data;
		}
	} else {
		busdata bd;

		bd = cpu->read_phys_4(descaddr);
		if (__predict_false(bd.IsBusErr())) {
			goto buserr;
		}
		m_desc1 = bd.Data();

		if (__predict_false(m_descsize == 8)) {
			bd = cpu->read_phys_4(descaddr + 4);
			if (bd.IsBusErr()) {
				goto buserr;
			}
			m_desc2 = bd.Data();
		}

		m_lastaddr = descaddr;
	}

	// ディスクリプタタイプをチェック
	// DT は必ず最初のロングワードの下位2ビット。
	do_write = false;
	dt = GET_DT(m_desc1);
	MMULOG(3, 2, "%s: desc=%s dt=%s", lam_func,
		sprintf_desc().c_str(), dtname(dt));
	switch (dt) {
	 case DT_INVALID:	// 無効
		return true;

	 case DT_PAGE:		// ページディスクリプタ
		// U ビットを更新
		UPDATE_DESC(MMU_DESC_U);

		// 書き込み操作なら M を更新
		if (cpu->bus.IsWrite() && !m_ptest) {
			UPDATE_DESC(MMU_DESC_M);
		}
		// 図にはないけど、ATC 更新のためと PTEST のために
		// ページディスクリプタの M ビットを記録。
		if ((m_desc1 & MMU_DESC_M))
			m_m = true;
		break;

	 case DT_VALID4:	// 有効4バイト
	 case DT_VALID8:	// 有効8バイト
		// U ビットを更新
		UPDATE_DESC(MMU_DESC_U);
		break;

	 default:
		__unreachable();
	}

	// デバッガ自身がアドレス変換を実施している時は書き込みは行わない
	if (__predict_false(m_debugger)) {
		do_write = false;
	}

	// ライトライクルの実行
	if (do_write) {
		busdata bd = cpu->write_phys_4(descaddr, m_desc1);
		if (__predict_false(bd.IsBusErr())) {
			goto buserr;
		}
		// バス動作正常終了
		MMULOG(3, 2, "%s: desc=%s updated", lam_func, sprintf_desc().c_str());
	}

	// ステータス更新 (OR していく)
	// XXX 実際には CI はページディスクリプタにしかないけど
	m_wp |= (m_desc1 & MMU_DESC_WP);
	m_ci |= (m_desc1 & MMU_DESC_CI);
	if (__predict_false(m_descsize == 8)) {
		m_s |= (m_desc1 & MMU_DESC_S);
	}

	return true;

	// バスエラー
 buserr:
	m_type = INVALID;
	m_berr = true;
	return false;
}

// ATC エントリを作成。実行系用。
// 図9-27 だが雑すぎ。
void
m68030MMU::make_atc(m68030ATCLine *a)
{
	std::string buf;
	uint32 paddr;

	a->data = 0;
	switch (m_type) {
	 case INVALID:	// 無効
		a->data |= m68030ATCLine::BUSERROR;
		MMULOG(3, 2, "%s: type=INVALID -> BusErr", lam_func);
		return;

	 case NORMAL:	// ノーマル
	 case INDIRECT:	// 間接
		paddr = (m_descsize == 4) ? m_desc1 : m_desc2;
		MMULOG(3, 2, "%s: type=NORMAL paddr=$%08x", lam_func,
			paddr & 0xffffff00);
		break;

	 case EARLY:	// アーリーターミネーション
		paddr = (m_descsize == 4) ? m_desc1 : m_desc2;
		buf = string_format("type=EARLY paddr=$%08x", paddr & 0xffffff00);

		// 残ってるビットも足して返す
		for (int i = x + 1; i <= TID; i++) {
			if (cpu->mmu_tc.tix[i] > 0) {
				paddr |= m_lv[i] << cpu->mmu_tc.shift[i];
				buf += string_format(" +%s=%08x", m_lvname[i], paddr);
			}
		}
		MMULOG(3, 2, "%s: %s", lam_func, buf.c_str());
		break;
	 default:
		PANIC("corrupted m_type=%u", (uint)m_type);
	}

	// PS の部分は反映しない
	a->paddr = paddr & cpu->mmu_tc.lmask;

	if (m_ci)
		a->data |= m68030ATCLine::CINHIBIT;
	if (m_wp)
		a->data |= m68030ATCLine::WPROTECT;
	if (m_m)
		a->data |= m68030ATCLine::MODIFIED;
}

// make_atc() と同様にテーブルサーチの結果から物理ページアドレスを返す。
// バスエラーになるケースなら BusErr を返す。
// デバッガアクセスなので、VM に一切干渉しない。
busaddr
m68030MMU::make_atc_debugger() const
{
	busaddr paddr;

	switch (m_type) {
	 case INVALID:	// 無効
		paddr = BusAddr::BusErr;
		break;

	 case NORMAL:	// ノーマル
	 case INDIRECT:	// 間接
	 {
		uint32 descaddr;
		descaddr = (m_descsize == 4) ? m_desc1 : m_desc2;
		descaddr &= 0xffffff00;
		paddr = busaddr(descaddr);
		break;
	 }

	 case EARLY:	// アーリーターミネーション
	 {
		uint32 descaddr;
		descaddr = (m_descsize == 4) ? m_desc1 : m_desc2;
		descaddr &= 0xffffff00;
		// 残ってるビットも足して返す
		for (int i = x + 1; i <= TID; i++) {
			if (cpu->mmu_tc.tix[i] > 0) {
				descaddr |= m_lv[i] << cpu->mmu_tc.shift[i];
			}
		}
		paddr = busaddr(descaddr);
		break;
	 }

	 default:
		PANIC("corrupted m_type=%u", (uint)m_type);
	}

	paddr |= BusAddr::TableSearched;
	return paddr;
}

// ディスクリプタ表示用の文字列を返す。
// m_descsize によって m_desc1, m_desc2 を適切に文字列にする。
std::string
m68030MMU::sprintf_desc()
{
	std::string buf;

	if (m_descsize == 4) {
		buf = string_format("%08x", m_desc1);
	} else {
		buf = string_format("%08x_%08x", m_desc1, m_desc2);
	}

	// フラグの出現条件がややこしいので注意
	int dt = GET_DT(m_desc1);
	buf += string_format(" (%s%s%s%c%c)",
		((m_descsize == 4) ? "x" : ((m_desc1 & MMU_DESC_S) ? "S" : "-")),
		((dt != DT_PAGE)   ? "x" : ((m_desc1 & MMU_DESC_CI) ? "C" : "-")),
		((dt != DT_PAGE)   ? "x" : ((m_desc1 & MMU_DESC_M)  ? "M" : "-")),
		(m_desc1 & MMU_DESC_U)  ? 'U' : '-',
		(m_desc1 & MMU_DESC_WP) ? 'W' : '-'
	);

	return buf;
}

// ディスクリプタタイプ文字列 (デバッグ用)
const char *
m68030MMU::dtname(uint32 dt)
{
	static char buf[32];
	switch (dt) {
	 case DT_INVALID:	return "invalid";
	 case DT_PAGE:		return "page";
	 case DT_VALID4:	return "valid4";
	 case DT_VALID8:	return "valid8";
	}
	snprintf(buf, sizeof(buf), "(invalid DT %u)", dt);
	return buf;
}

// デバッグ用文字列
const char * const m68030MMU::m_lvname[MAX_LV] = {
	"TIA",
	"TIB",
	"TIC",
	"TID",
	"FCL",	// この順ではないが、便宜上ここにおいてある
};

//
// アドレス変換
//

// 論理アドレスが TT0/TT1 どちらかと一致すれば (bus.ci を更新して) true を返す。
// 一致しなければ false を返す。
bool
MPU68030Device::CheckTT()
{
	bool ci;
	int match;

	match = 0;
	ci = false;
	for (int i = 0; i < 2; i++) {
		const m68030TT& tt = mmu_tt[i];
		if (tt.Match(bus.laddr)) {
			// 一致した
			match += i + 1;
			ci |= tt.ci != 0;
			putlog(3, " %s: TT%u match", __func__, i);
		}
	}

	// 一致したら PA <- LA
	if (match != 0) {
		putlog(3, "%s: result %08x (match TT%s)", __func__,
			bus.laddr.Addr(),
			(match == 3) ? "0 and TT1" : ((match == 1) ? "0" : "1"));
		bus.ci = ci;
		return true;
	}
	return false;
}

// リード/フェッチ用のアドレス変換を行う。
// 変換できれば true、バスエラーなら false を返す。
bool
MPU68030Device::TranslateRead()
{
	m68030ATCLine *a;
	busaddr& laddr = bus.laddr;
	uint32 addr = laddr.Addr();
	uint32 fc   = laddr.GetFC();

	// 状態表示
	putlog(3, "%s:  enter  %c:$%08x pc=%c:$%08x", __func__,
		(laddr.IsSuper() ? 'S' : 'U'),
		addr,
		IsSuper() ? 'S' : 'U',
		ppc);

	m68030ATCTable *table = atc.fctables[fc];
	uint8 n = table->hash[addr >> 12];
	if (__predict_true((int8)n >= 0)) {
		// ATC #(n)
		table->atcstat.total++;
		a = &table->line[n];
	} else if (n == m68030ATCTable::HASH_NONE) {
		if (__predict_true(mmu_tc.e)) {
			// TC.E 有効で対応なしなら、テーブルサーチ。
			table->atcstat.total++;
			table->atcstat.miss++;
			a = table->MMUSearch(laddr);
		} else {
			// TC.E 無効。(total もカウントしない)
			return true;
		}
	} else if (n == m68030ATCTable::HASH_TT) {
		// TT。CheckTT で外れたかどうかはカウントしない。ほぼ 0 なので。
		table->atcstat.total++;
		table->atcstat.tthit++;
		return CheckTT();
	} else {
		// PS<4KB でマッチしたので再検査。
		table->atcstat.total++;
		a = table->FindSub(laddr);
		if (a == NULL) {
			table->atcstat.miss++;
			a = table->MMUSearch(laddr);
		}
	}

	// ATC a でマッチした
	table->MoveHead(a);
	putlog(3, "%s", table->LineToStr(a).c_str());
	if (__predict_false(a->IsBusError())) {
		return false;
	}
	bus.paddr.ChangeAddr(a->GetPAddr() | (addr & mmu_tc.pgmask));
	bus.ci = a->IsCInhibit();
	return true;
}

// ライト用のアドレス変換を行う。
// 変換できれば true、バスエラーなら false を返す。
bool
MPU68030Device::TranslateWrite()
{
	m68030ATCLine *a;
	busaddr& laddr = bus.laddr;
	uint32 addr = laddr.Addr();
	uint32 fc   = laddr.GetFC();

	// 状態表示
	putlog(3, "%s: enter  %c:$%08x pc=%c:$%08x", __func__,
		(laddr.IsSuper() ? 'S' : 'U'),
		addr,
		IsSuper() ? 'S' : 'U',
		ppc);

	m68030ATCTable *table = atc.fctables[fc];
	uint8 n = table->hash[addr >> 12];
	if (__predict_true((int8)n >= 0)) {
		// ATC #(n)
		table->atcstat.total++;
		a = &table->line[n];
	} else if (n == m68030ATCTable::HASH_NONE) {
		if (__predict_true(mmu_tc.e)) {
			// TC.E 有効で対応なしなら、テーブルサーチ。
			table->atcstat.total++;
			goto tablesearch;
		} else {
			// TC.E 無効。(total もカウントしない)
			return true;
		}
	} else if (n == m68030ATCTable::HASH_TT) {
		// TT。CheckTT で外れたかどうかはカウントしない。ほぼ 0 なので。
		table->atcstat.total++;
		table->atcstat.tthit++;
		return CheckTT();
	} else {
		// PS<4KB でマッチしたので再検査。
		table->atcstat.total++;
		a = table->FindSub(laddr);
		if (a == NULL) {
			goto tablesearch;
		}
	}

	// ライトアクセスなので、WP でなく Modified == 0 なら
	// Modified を立てるため、このエントリを破棄して改めて
	// テーブルサーチを実行する。
	if (a->IsWProtect() == false && a->IsModified() == false) {
		table->Invalidate(a);
 tablesearch:
		table->atcstat.miss++;
		a = table->MMUSearch(laddr);
	}

	// ATC a でマッチした
	table->MoveHead(a);

	if (__predict_false(a->IsBusError())) {
		return false;
	}
	if (__predict_false(a->IsWProtect())) {
		// Write アクセスで WP が立ってたらバスエラー
		putlog(3, "%s: result Buserr (ATC:WP)", __func__);
		return false;
	}

	putlog(3, "%s", table->LineToStr(a).c_str());
	bus.paddr.ChangeAddr(a->GetPAddr() | (addr & mmu_tc.pgmask));
	bus.ci = a->IsCInhibit();
	return true;
}

// ピーク用のアドレス変換を行う。
// 変換できれば対応する物理アドレスを、バスエラーなら BusErr を返す。
// テーブルサーチを行ったら TableSearched ビットを立てる。
busaddr
MPU68030Device::TranslatePeek(busaddr laddr)
{
	const m68030ATCLine *a;
	busaddr paddr;
	uint32 addr = laddr.Addr();
	uint32 fc   = laddr.GetFC();

	// 状態表示
	putlog(5, "%s:  enter  %c:$%08x", __func__,
		(laddr.IsSuper() ? 'S' : 'U'),
		addr);

	m68030ATCTable *table = atc.fctables[fc];
	uint8 n = table->hash[addr >> 12];
	// PS<4KB のケースを先に解決する。
	if (__predict_false(n == m68030ATCTable::HASH_SUB)) {
		a = table->FindSub(laddr);
		if (a) {
			n = a - &table->line[0];
		} else {
			n = m68030ATCTable::HASH_NONE;
		}
	}
	if (__predict_true((int8)n >= 0)) {
		// ATC #(n)
		a = &table->line[n];
		paddr = busaddr(a->GetPAddr());
		paddr |= addr & mmu_tc.pgmask;
		putlog(5, "%s:  result $%08x (match atc)", __func__, paddr.Addr());
		return paddr;
	} else if (n == m68030ATCTable::HASH_NONE) {
		if (__predict_true(mmu_tc.e)) {
			// TC.E 有効で対応なしなら、こっそりテーブルサーチ。

			// mmu.search() はさすがに bus を使うので事前に退避。
			m68kbus backup = bus;

			m68030MMU mmu(this, loglevel);
			mmu.m_debugger = true;
			mmu.search(7, laddr);
			paddr = mmu.make_atc_debugger();
			// リストア
			bus = backup;

			if (paddr.IsBusErr()) {
				// TC が有効だがヒットしなかったらバスエラー
				putlog(5, "%s:  result BusErr", __func__);
			} else {
				paddr |= addr & mmu_tc.pgmask;
				putlog(5, "%s:  result $%08x (table search)",
					__func__, paddr.Addr());
			}
			return paddr;
		} else {
			// TC.E 無効
			paddr = laddr;
			putlog(5, "%s:  result $%08x (no translation)",
				__func__, paddr.Addr());
			return paddr;
		}
	} else {
		// TT をこっそりチェック。
		for (int i = 0; i < 2; i++) {
			m68030TT& tt = mmu_tt[i];
			if (tt.Match(laddr)) {
				// 一致した。
				// CI はここでは使わないため、一致したことだけわかればよい
				putlog(5, "%s:  result $%08x (match tt)",
					__func__, paddr.Addr());
				return paddr;
			}
		}
		return BusAddr::BusErr;
	}
}

//
// 命令
//

// ir2 の下位5ビットから FC を返す。
// 不当パターンなら C++ の例外をスローする。
uint32
MPU68030Device::get_fc()
{
	uint fcfield = ir2 & 0x1f;

	if (fcfield == 0) {
		return reg.GetSFC();
	} else if (fcfield == 1) {
		return reg.GetDFC();
	} else if (fcfield < 0x08) {
		// break;
	} else if (fcfield < 0x10) {
		return reg.D[fcfield & 7] & 7;
	} else if (fcfield < 0x18) {
		return fcfield & 7;
	}
	// 不当命令
	throw M68K::EXCEP_ILLEGAL;
}

// MMU 不当命令 (2ワード目で確定)
void
MPU68030Device::mmu_op_illg2()
{
	putlog(1, "MMU illegal instruction %04x %04x", ir, ir2);
	Exception(M68K::EXCEP_FLINE);
}

// PFLUSH fc,#mask 命令 (EA を持たない方)
void
MPU68030Device::mmu_op_pflush()
{
	uint32 mask;
	uint32 fc;

	mask = (ir2 >> 5) & 7;
	fc = get_fc();
	atc.flush(fc, mask);
}

// PFLUSH fc,#mask,ea 命令
void
MPU68030Device::mmu_op_pflush_ea()
{
	uint32 ea;
	uint32 mask;
	uint32 fc;

	ea = cea_copro();
	ea &= mmu_tc.lmask;
	mask = (ir2 >> 5) & 7;
	fc = get_fc();
	atc.flush(fc, mask, ea);
}

// PLOADR/PLOADW 命令
void
MPU68030Device::mmu_op_pload()
{
	uint32 ea;
	uint32 fc;
	busaddr laddr;

	if ((ir2 & 0x01e0) != 0) {
		mmu_op_illg2();
		return;
	}
	ea = cea_copro();
	fc = get_fc();

	// 実効アドレスを論理アドレスとする ATC を作成。
	// ってこれでいいのか?
	if ((ir2 & 0x0200)) {
		laddr = busaddr(ea) | busaddr::FC(fc) | BusAddr::R;
	} else {
		laddr = busaddr(ea) | busaddr::FC(fc) | BusAddr::W;
	}

	m68030ATCTable *table = atc.fctables[fc];
	m68030ATCLine *a;
	uint8 n = table->hash[laddr.Addr() >> 12];
	// PS<4KB のケースを先に解決する。
	if (__predict_false(n == m68030ATCTable::HASH_SUB)) {
		a = table->FindSub(laddr);
		if (a) {
			n = a - &table->line[0];
		} else {
			n = m68030ATCTable::HASH_NONE;
		}
	}
	if ((int8)n >= 0) {
		// すでにエントリがある場合
		if (laddr.IsWrite()) {
			// 命令が ploadw でエントリが WProtect==0 && Modified==0 なら、
			// 一回破棄して書き込み用に作り直す。
			a = &table->line[n];
			if (a->IsWProtect() == false && a->IsModified() == false) {
				table->Invalidate(a);
				table->MMUSearch(laddr);
			}
		}
	} else if (n == m68030ATCTable::HASH_NONE) {
		if (__predict_true(mmu_tc.e)) {
			// TC.E 有効で対応なしなら、テーブルサーチを行って
			// ATC を作成。
			table->MMUSearch(laddr);
		} else {
			// TC.E 無効なら何もしない?
		}
	} else if (n == m68030ATCTable::HASH_TT) {
		// TT の領域なら何もしてはいけない。
	}
}

// PTESTR/PTESTW 命令
void
MPU68030Device::mmu_op_ptest()
{
	uint level = (ir2 >> 10) & 7;
	busaddr rw = (ir2 & 0x0200) ? BusAddr::R : BusAddr::W;
	uint rn    = (ir2 >> 5) & 0xf;
	uint32 ea;
	uint32 fc;
	uint16 mmusr;

	ea = cea_copro();
	fc = get_fc();
	mmusr = 0;

	if (level == 0) {
		if (rn != 0) {
			// レベル0で An 指定なら Fライン例外
			mmu_op_illg2();
			return;
		}
		CYCLE(22);

		// laddr は IS と PS のところを落としたもの。
		// ssw_laddr のアドレスは IS を落とさないもの (PSは不問)。
		busaddr laddr = busaddr(ea) | busaddr::FC(fc) | rw;
		uint32 addr = ea & mmu_tc.lmask;

		m68030ATCTable *table = atc.fctables[fc];
		m68030ATCLine *a = NULL;
		uint8 n = table->hash[addr >> 12];
		// PS<4KB のケースを先に解決する。
		if (__predict_false(n == m68030ATCTable::HASH_SUB)) {
			a = table->FindSub(laddr);
			if (a) {
				n = a - &table->line[0];
			} else {
				n = m68030ATCTable::HASH_NONE;
			}
		}
		if (__predict_true((int8)n >= 0)) {
			// ATC #(n) で見付かった。
			a = &table->line[n];
		} else if (n == m68030ATCTable::HASH_NONE) {
			// ATC に見付からなかった。Invalid を立てる。
			mmusr = m68030MMUSR::I;
		} else if (n == m68030ATCTable::HASH_TT) {
			for (int i = 0; i < 2; i++) {
				m68030TT& tt = mmu_tt[i];
				if (tt.Match(laddr)) {
					mmusr |= m68030MMUSR::T;
				}
			}
		}

		if (a) {
			// 見付かれば Modified 等の状態を返す。
			if (a->IsBusError()) {
				mmusr |= m68030MMUSR::B;
			} else if (a->IsWProtect()) {
				mmusr |= m68030MMUSR::W;
			} else if (a->IsModified()) {
				mmusr |= m68030MMUSR::M;
			}
		}
	} else {
		// レベル 1-7 ならテーブルサーチを実行。
		CYCLE(88);

		m68030MMU mmu(this, loglevel);
		mmu.m_ptest = true;
		mmu.search(level, busaddr(ea) | busaddr::FC(fc) | rw);

		if (mmu.m_berr) {
			mmusr |= m68030MMUSR::B;
		}
		if (mmu.m_lim) {
			// XXX not tested
			mmusr |= m68030MMUSR::L;
		}
		if (mmu.m_s && (fc & 4)) {
			// XXX not tested
			mmusr |= m68030MMUSR::S;
		}
		if (mmu.m_wp) {
			mmusr |= m68030MMUSR::W;
		}
		if (mmu.m_type == INVALID) {
			mmusr |= m68030MMUSR::I;
		}
		if (mmu.m_m) {
			mmusr |= m68030MMUSR::M;
		}
		// N を求める。
		// もともとそういう風には出来ていないので逆算している。
		int n;
		if (mmu_tc.fcl) {
			// たぶんファンクションテーブルで1段消費する? (未確認)
			if (mmu.x == FCL) {
				n = 1;
			} else {
				n = mmu.x + 2;
			}
		} else {
			n = mmu.x + 1;
		}
		mmusr |= n;

		// アドレスレジスタ指定があれば、最後のディスクリプタアドレスを返す。
		// rn は有効なら 8-15。
		if (rn != 0) {
			// XXX 途中でバスエラーが起きるケースは未実装
			if (mmu.m_berr == false) {
				reg.R[rn] = mmu.m_lastaddr;
			}
		}
	}

	SetMMUSR(mmusr);
}
