---
authors: "vjeux"
title: "Prettier 1.4: TypeScript and CSS support"
---

This release introduces support for TypeScript, CSS, Less, and SCSS languages to Prettier!

<!-- truncate -->

[![prettier-revolution-conf](./assets/prettier-revolution-conf.png)](https://revolutionconf.com/talk/a-prettier-printer)

## TypeScript Support

This is the most requested feature for prettier. With 1.4.0, you can now use prettier to format your `.ts` and `.tsx` files!

The way prettier works is by using those project to generate an AST representation of the code and print it. Both babylon (the parser that powers babel) and flow are producing an AST approximately following the [estree](https://github.com/estree/estree) format for the JavaScript parts and then have special nodes for Flow-specific ones.

TypeScript, the same way as Flow, introduces special nodes for the syntax it introduces. Unfortunately, it doesn't follow the estree format for the rest of the JavaScript language. This puts us in a rough spot with prettier as we would have to essentially completely fork it in order to print TypeScript.

This incompatibility with the AST is not a new problem and another project struggled with it: ESLint. Because the AST is different, none of the ESLint rules are working. Fortunately for us, [@JamesHenry](https://github.com/JamesHenry) and [@soda0289](https://github.com/soda0289) wrote a project called [typescript-eslint-parser](https://github.com/eslint/typescript-eslint-parser) which takes a TypeScript AST and convert it to an estree one, just what we need for prettier!

After that project got setup inside of prettier, [@azz](https://github.com/azz), [@despairblue](https://github.com/despairblue) and [@Pajn](https://github.com/Pajn) implemented all the TypeScript-specific nodes and ensured that the 13k tests of the TypeScript test suite are correctly passing. This was a huge undertaking and it is finally ready to be used :)

We tested prettier on the biggest TypeScript projects we could find on GitHub to ensure that it prints correct code. We haven't spent a lot of time trying to optimize the way code is formatted yet, so if you see something strange, please raise an issue!

## CSS, Less and SCSS Support

While TypeScript is the most requested feature from open source, CSS is the biggest one from Facebook engineers. Once you are used to pretty print your code in one language, you want to do it everywhere!

It turns out that CSS is a much smaller language than JavaScript and supporting it only took a few days. We are using [postcss](https://github.com/postcss/postcss) by [@ai](https://github.com/ai) as the underlying parser which is able to parse CSS, Less and SCSS. We also depend on [postcss-values-parser](https://github.com/shellscape/postcss-values-parser), [postcss-selector-parser](https://github.com/postcss/postcss-selector-parser) by [@ben](https://github.com/ben)-eb [postcss-media-query-parser](https://github.com/dryoma/postcss-media-query-parser) by [@dryoma](https://github.com/dryoma).

Unfortunately, postcss right now doesn't parse Sass nor Stylus. We'd be happy to support them if someone is willing to do the work of printing them.

Note that prettier is currently just formatting the code, it does not respect any options yet such as `singleQuote` nor is doing any color or number normalization like we do for JavaScript.

## Editor Integration

The first phase of the project was to make prettier output correct and good looking code. Now that it's in a good shape, we can spend time on making the integrations better. We just introduced support for two great features: maintain cursor position and being able to format a range instead of the entire file.

Note that we just landed the support inside of prettier itself, none of the editor integrations are using it yet. Also, we haven't really tried them out in practice so we're likely going to have to fix rough edges with them!

#### Add `cursorOffset` option for cursor translation ([#1637](https://github.com/prettier/prettier/pull/1637)) by [@josephfrazier](https://github.com/josephfrazier)

Right now, we let editors figure out where the cursor should be, which they do an okay job at. But since we are printing the code, we can give the correct position!

#### Add `--range-start/end` options to format only parts of the input ([#1609](https://github.com/prettier/prettier/pull/1609)) by [@josephfrazier](https://github.com/josephfrazier)

This one is a very often requested feature. Right now prettier only formats the entire file. Now it is possible to format a range.

The way it works is by going up through the AST in order to find the closest statement. This way you don't need to select exactly the right range that is valid. You can just drag in the rough place where the code you want to reformat it, and it's going to!

#### Adding filepath option in order to enable filetype inference ([#1835](https://github.com/prettier/prettier/pull/1835)) by [@mitermayer](https://github.com/mitermayer)

Since we are now formatting CSS and TypeScript, it is not convenient to have to specify the parser for every file. You can now pass the filepath of the file you are working on and prettier will read the extension and figure out the right parser to use.

## Highlights

#### Wrap text content inside of JSX ([#1120](https://github.com/prettier/prettier/pull/1120), [#1671](https://github.com/prettier/prettier/pull/1671), [#1827](https://github.com/prettier/prettier/pull/1827), [#1829](https://github.com/prettier/prettier/pull/1829)) by [@karl](https://github.com/karl)

The biggest remaining issue that people have with prettier when printing JSX is when it is used when printing text. The behavior of prettier used to add an ugly `{" "}` before and if a line was too long, just leave it alone. Now we treat each word as a token and are able to make it flow correctly.

This is an awesome piece of work by [@karl](https://github.com/karl) as not only did he implement the feature, but also introduced a new primitive inside of prettier in order to print a sequence of elements and break as soon as one hits the edge.

<!-- prettier-ignore -->
```jsx
// Before
<div>
  Please state your
  {" "}
  <b>name</b>
  {" "}
  and
  {" "}
  <b>occupation</b>
  {" "}
  for the board of directors.
</div>

// After
<div>
  Please state your <b>name</b> and <b>occupation</b> for the board of
  directors.
</div>
```

#### Remove parenthesis for JSX inside of arrow functions ([#1733](https://github.com/prettier/prettier/pull/1733)) by [@xixixao](https://github.com/xixixao)

People writing functional components are going to be happy about this one. We no longer put parens for arrow functions that return JSX.

<!-- prettier-ignore -->
```jsx
// Before
const render1 = ({ styles }) => (
  <div style={styles}>
      Keep the wrapping parens. Put each key on its own line.
  </div>
);

// After
const render1 = ({ styles }) =>
  <div style={styles}>
      Keep the wrapping parens. Put each key on its own line.
  </div>;
```

#### Improve template literal printing ([#1664](https://github.com/prettier/prettier/pull/1664), [#1714](https://github.com/prettier/prettier/pull/1714)) by [@josephfrazier](https://github.com/josephfrazier)

Template literal printing has always caused prettier a lot of difficulties. With 1.3.0 we massively improved the situation and with this release, I believe that we handle all the common situations in a good way.

In order to workaround issues, we added an utility that removes empty lines from the output, but it yielded some really weird results sometimes, this is now gone. Another tweak we've done is instead of indenting when `${` starts, we indent where the line that contains `${` starts.

Let us know if you still have issues with how template literals output after this release!

<!-- prettier-ignore -->
```js
// Before
const Bar = styled.div`
  color: ${props => (props.highlight.length > 0 ? palette([
                 'text',
                 'dark',
                 'tertiary'
               ])(props) : palette([
                 'text',
                 'dark',
                 'primary'
               ])(props))} !important;
`

// After
const Bar = styled.div`
  color: ${props =>
    props.highlight.length > 0
      ? palette(["text", "dark", "tertiary"])(props)
      : palette(["text", "dark", "primary"])(props)} !important;
`
```

#### Use the same breaking rules for assignment and object values ([#1721](https://github.com/prettier/prettier/pull/1721))

We have a lot of fine-tuned logic for how to break things after assignment (eg `a = ...`). We are now using the same one for object values. This should help for multi-line boolean logic, or big conditionals. This is also a good example of how we can create a consistent printer.

<!-- prettier-ignore -->
```js
// Before
const o = {
  somethingThatsAReallyLongPropName: this.props.cardType ===
    AwesomizerCardEnum.SEEFIRST,
};

// After
const o = {
  somethingThatsAReallyLongPropName:
    this.props.cardType === AwesomizerCardEnum.SEEFIRST,
};
```

#### Indent conditions inside of !() ([#1731](https://github.com/prettier/prettier/pull/1731))

There's been a steady stream of people complaining about the way it was rendered and was put on the list of things that are probably hard to do, will check later. It turned out to be super easy, so here you go!

<!-- prettier-ignore -->
```js
// Before
const anyTestFailures = !(aggregatedResults.numFailedTests === 0 &&
  aggregatedResults.numRuntimeErrorTestSuites === 0);

// After
const anyTestFailures = !(
  aggregatedResults.numFailedTests === 0 &&
  aggregatedResults.numRuntimeErrorTestSuites === 0
);
```

## Formatting Fixes

#### Put loop bodies on the same line when possible ([#1498](https://github.com/prettier/prettier/pull/1498))

We were already doing this for if statements, we should be consistent and also do it for loops.

<!-- prettier-ignore -->
```js
// Before
for (a in b)
  var c = {};

// After
for (a in b) var c = {};
```

#### Fix empty line with flow union ([#1511](https://github.com/prettier/prettier/pull/1511)) by [@existentialism](https://github.com/existentialism)

We shouldn't indent things twice ;)

<!-- prettier-ignore -->
```ts
// Before
type Foo = Promise<

    | { ok: true, bar: string, baz: SomeOtherLongType }
    | { ok: false, bar: SomeOtherLongType }
>;

// After
type Foo = Promise<
  { ok: true, bar: string, baz: SomeOtherLongType } |
  { ok: false, bar: SomeOtherLongType }
>;
```

#### Do not put parens for single argument with end of line comment ([#1518](https://github.com/prettier/prettier/pull/1518))

The detection code for whether an arrow function should be written without parenthesis just checked if there was a comment, but instead we only want comments that are inline like `(/* comment */ num)`, not end of line comments.

<!-- prettier-ignore -->
```jsx
// Before
KEYPAD_NUMBERS.map((num) => ( // Buttons 0-9
  <div />
));

KEYPAD_NUMBERS.map(num => ( // Buttons 0-9
  <div />
));
```

#### Do not indent nested ternaries ([#1822](https://github.com/prettier/prettier/pull/1822))

This avoids making it seems like it is indented by 4 characters instead of two. The downside is that if the condition is multi-line it's not going to be properly aligned, but I feel it's a better trade-offs. If you are doing nested ternaries, you usually have small conditions.

<!-- prettier-ignore -->
```js
// Before
aaaaaaaaaaaaaaa
  ? bbbbbbbbbbbbbbbbbb
  : ccccccccccccccc
      ? ddddddddddddddd
      : eeeeeeeeeeeeeee ? fffffffffffffff : gggggggggggggggg;

// After
aaaaaaaaaaaaaaa
  ? bbbbbbbbbbbbbbbbbb
  : ccccccccccccccc
    ? ddddddddddddddd
    : eeeeeeeeeeeeeee ? fffffffffffffff : gggggggggggggggg;
```

#### Inline chained conditionals inside of jsx attribute ([#1519](https://github.com/prettier/prettier/pull/1519))

We don't need to use the indentation to disambiguate another block as nothing can come after.

<!-- prettier-ignore -->
```jsx
// Before
<div
  src={
    !isEnabled &&
      diffUpdateMessageInput != null &&
      this.state.isUpdateMessageEmpty
  }
/>;

// After
<div
  src={
    !isEnabled &&
    diffUpdateMessageInput != null &&
    this.state.isUpdateMessageEmpty
  }
/>;
```

#### Unescape unnecessarily escaped characters in strings ([#1575](https://github.com/prettier/prettier/pull/1575)) by [@josephfrazier](https://github.com/josephfrazier)

We are already trying to cleanup strings in various ways, this is another small addition that's going to remove `\` that are not needed.

<!-- prettier-ignore -->
```js
// Before
a = 'hol\a';

// After
a = 'hola';
```

#### Fix boolean for empty objects ([#1590](https://github.com/prettier/prettier/pull/1590)) by [@dmitrika](https://github.com/dmitrika)

We want to inline objects inside of a boolean expression as it looks weird to have `{` on its own line. But it turns out that it leads to weird behaviors for empty objects. So we keep them on their own line if they are empty.

<!-- prettier-ignore -->
```js
const x = firstItemWithAVeryLongNameThatKeepsGoing ||
secondItemWithALongNameAsWell || {};

// After
const x =
  firstItemWithAVeryLongNameThatKeepsGoing ||
  secondItemWithALongNameAsWell ||
  {};
```

#### Remove Parens from SequenceExpressions in ForStatements ([#1597](https://github.com/prettier/prettier/pull/1597)) by [@k15a](https://github.com/k15a)

It is common to assign multiple values inside of a for loop, now we don't add parenthesis anymore.

<!-- prettier-ignore -->
```js
// Before
for ((i = 0), (len = arr.length); i < len; i++) {

// After
for (i = 0, len = arr.length; i < len; i++) {
```

#### Do not inline arrow when argument has a leading comment ([#1660](https://github.com/prettier/prettier/pull/1660))

If you put block comments inside of arrow functions, we no longer mess everything up!

<!-- prettier-ignore -->
```js
// Before
export const bem = block => /**
   * @param {String} [element] - the BEM Element within that block; if undefined, selects the block itself.
   */
element => /**
     * @param {?String} [modifier] - the BEM Modifier for the Block or Element; if undefined, selects the Block or Element unmodified.
     */
modifier =>

// After
export const bem = block =>
  /**
   * @param {String} [element] - the BEM Element within that block; if undefined, selects the block itself.
   */
  element =>
    /**
     * @param {?String} [modifier] - the BEM Modifier for the Block or Element; if undefined, selects the Block or Element unmodified.
     */
    modifier =>
```

#### Fix last comments of imports ([#1677](https://github.com/prettier/prettier/pull/1677))

Another place where we have to do special logic for comments!

<!-- prettier-ignore -->
```js
// Before
import {
  ExecutionResult,
  DocumentNode,
  /* tslint:disable */
  SelectionSetNode,
} /* tslint:enable */ from 'graphql';

// After
import {
  DocumentNode,
  /* tslint:disable */
  SelectionSetNode,
  /* tslint:enable */
} from 'graphql';
```

#### Handle comments in member chain ([#1686](https://github.com/prettier/prettier/pull/1686), [#1691](https://github.com/prettier/prettier/pull/1691))

We handled some placements before and kept adding places where they could appear, now we switch to a more general approach. Hopefully those issues shouldn't crop up in the future anymore.

<!-- prettier-ignore -->
```js
// Before
const configModel = this.baseConfigurationService.getCache().consolidated // global/default values (do NOT modify)
  .merge(this.cachedWorkspaceConfig);

// After
const configModel = this.baseConfigurationService
  .getCache()
  .consolidated // global/default values (do NOT modify)
  .merge(this.cachedWorkspaceConfig);
```

#### Use expandLast for nested arrow functions ([#1720](https://github.com/prettier/prettier/pull/1720))

<!-- prettier-ignore -->
```js
// Before
f(action => next =>
    next(action));

// After
f(action => next =>
    next(action),
);
```

#### Put JSX comments inside of the parenthesis ([#1712](https://github.com/prettier/prettier/pull/1712))

This mostly affects Facebook engineers where we automatically add `$FlowFixMe` when pushing a new version of flow. Now it no longer messes up those comments.

<!-- prettier-ignore -->
```jsx
// Before
const aDiv = /* $FlowFixMe */
(
  <div className="foo">
    Foo bar
  </div>
);

// After
const aDiv = (
  /* $FlowFixMe */
  <div className="foo">
    Foo bar
  </div>
);
```

#### Force \n for multiple variable declarations ([#1723](https://github.com/prettier/prettier/pull/1723))

This one has been very often requested. We used to only break multiple variable declarations if the line was > 80 columns. Now we do it regardless if there's at least one with an assignment.

<!-- prettier-ignore -->
```js
// Before
var numberValue1 = 1, numberValue2 = 2;

// After
var numberValue1 = 1,
 numberValue2 = 2;
```

#### Inline | null and | void ([#1734](https://github.com/prettier/prettier/pull/1734))

The expanded version of flow union looks good when they are many objects but if it's used for nullability, then it looks very weird. We're now inlining `| null` and `| void`.

<!-- prettier-ignore -->
```ts
// Before
interface RelayProps {
  articles:
    | Array<
      | {
        __id: string,
      }
      | null
    >
    | null
}

// After
interface RelayProps {
  articles: Array<{
    __id: string,
  } | null> | null,
}
```

#### Break on implements instead of extends ([#1730](https://github.com/prettier/prettier/pull/1730))

We no longer break on `extends`. This should make classes with extends that can break look less wonky.

<!-- prettier-ignore -->
```ts
// Before
class MyContractSelectionWidget
  extends React.Component<
    void,
    MyContractSelectionWidgetPropsType,
    void
  > {
  method() {}
}

// After
class MyContractSelectionWidget extends React.Component<
  void,
  MyContractSelectionWidgetPropsType,
  void
> {
  method() {}
}
```

#### Inline single import ([#1729](https://github.com/prettier/prettier/pull/1729))

The same way we don't break long `require` calls, we no longer break `import` statements if there is only a single thing being imported.

<!-- prettier-ignore -->
```js
// Before
import somethingSuperLongsomethingSuperLong
  from "somethingSuperLongsomethingSuperLongsomethingSuperLong";

// After
import somethingSuperLongsomethingSuperLong from "somethingSuperLongsomethingSuperLongsomethingSuperLong";
```

#### Add the ability for SequenceExpression to break ([#1749](https://github.com/prettier/prettier/pull/1749))

Did you know that if none of your code were statements, you could use `()` instead of `{}` and `,` instead of `;`? Now you do. Some people exploit this fact when returning things from arrow functions. This is not recommended but it's easy to support in prettier so might as well ¯\\\_(ツ)\_/¯

<!-- prettier-ignore -->
```js
// Before
const f = (argument1, argument2, argument3) =>
  (doSomethingWithArgument(argument1), doSomethingWithArgument(
    argument2
  ), argument1);

// After
const f = (argument1, argument2, argument3) => (
  doSomethingWithArgument(argument1),
  doSomethingWithArgument(argument2),
  argument1
);
```

#### Don't force line break in empty loop bodies ([#1815](https://github.com/prettier/prettier/pull/1815))

Loops with empty body no longer have their `{}` split into two lines.

<!-- prettier-ignore -->
```js
// Before
while (true) {
}

// After
while (true) {}
```

#### Preserve empty lines between switch cases with comments ([#1708](https://github.com/prettier/prettier/pull/1708))

<!-- prettier-ignore -->
```js
// Before
switch (true) {
  case true:
  // Good luck getting here
  case false:
}

// After
switch (true) {
  case true:

  // Good luck getting here
  case false:
}
```

## Correctness

#### Remove ast-types ([#1743](https://github.com/prettier/prettier/pull/1743), [#1744](https://github.com/prettier/prettier/pull/1744), [#1745](https://github.com/prettier/prettier/pull/1745), [#1746](https://github.com/prettier/prettier/pull/1746), [#1747](https://github.com/prettier/prettier/pull/1747))

We used to find where to put comments by traversing the AST using the definition from ast-types. This occasionally caused issues when some field wasn't declared, we wouldn't find the node and either print comments in an incorrect location or throw an error. It turns out that we don't need to keep this mapping and can just traverse the objects and if a node has a `type` field, then it's a node.

<!-- prettier-ignore -->
```ts
// Before
Error: did not recognize object of type "ObjectTypeSpreadProperty"

// After
type X = {...Y/**/};
type X = {/**/...Y};
```

#### Preserve unusual unicode whitespace ([#1658](https://github.com/prettier/prettier/pull/1658), [#1165](https://github.com/prettier/prettier/pull/1165)) by [@karl](https://github.com/karl) and [@yamafaktory](https://github.com/yamafaktory)

If you were adding invisible characters inside of JSX text, we would replace them by regular spaces. I don't know why anyone would ever want to do that, but now we print it back as is!

#### Don't let trailing template literal comments escape ([#1580](https://github.com/prettier/prettier/pull/1580), [#1713](https://github.com/prettier/prettier/pull/1713), [#1598](https://github.com/prettier/prettier/pull/1598), [#1713](https://github.com/prettier/prettier/pull/1713)) by [@josephfrazier](https://github.com/josephfrazier) and [@k15a](https://github.com/k15a)

We used to have some pretty complicated (and not working well) code to handle comments inside of template literals. We introduced a really nice solution for JSX `{}` expressions. The idea is to introduce a boundary before the end of the `}` and if we still have unprinted comments, then flush them all at once, put a \n and print the `}`. We are now using this logic for template literals :)

<!-- prettier-ignore -->
```js
// Before
`${0} // comment`;

// After
`${
0
// comment
}`;
```

#### Parenthesize await correctly ([#1513](https://github.com/prettier/prettier/pull/1513), [#1595](https://github.com/prettier/prettier/pull/1595), [#1593](https://github.com/prettier/prettier/pull/1593)) by [@bakkot](https://github.com/bakkot) and [@existentialism](https://github.com/existentialism)

We don't have an automated way to put parenthesis, we instead specify all the possible combinations of nodes and when they should or shouldn't have parenthesis. So there's likely a long tail of unusual combinations that are still remaining. In this case, we made `await` handling a lot more robust by both adding parenthesis where they are needed and removing them when they are not.

<!-- prettier-ignore -->
```js
// Before
(await spellcheck) && spellcheck.setChecking(false);
new A((await x));

// After
await (spellcheck && spellcheck.setChecking(false));
new A(await x);
```

#### Preserve getter/setter info on flow ObjectTypeProperty ([#1585](https://github.com/prettier/prettier/pull/1585)) by [@josephfrazier](https://github.com/josephfrazier)

Another long tail option that we haven't got right!

<!-- prettier-ignore -->
```ts
// Before
type T = { method: () => void };

// After
type T = { get method(): void }
```

#### Add parenthesis for single arg types with generics ([#1814](https://github.com/prettier/prettier/pull/1814))

Another case of sneaky parenthesis that we didn't properly add!

<!-- prettier-ignore -->
```ts
// Before
type ExtractType = <A>B<C> => D

// After
type ExtractType = <A>(B<C>) => D
```

#### Fall back to non-strict mode in babylon ([#1587](https://github.com/prettier/prettier/pull/1587), [#1608](https://github.com/prettier/prettier/pull/1608)) by [@josephfrazier](https://github.com/josephfrazier)

We want prettier to be able to parse all the JavaScript out there. For babylon parser, we have to chose whether a file is using strict mode or not. We opted in to use strict mode by default as most files parse that way. But if you have octal literals like `0775`, it would not even parse. Now if it fails to parse in strict mode, we're going to try again in non-strict. We also allow `return` outside of a function as it's valid in node files.

<!-- prettier-ignore -->
```js
// Before
SyntaxError

// After
return 0775;
```

## CLI

#### Allow `--write` to be used with `--list-different` ([#1633](https://github.com/prettier/prettier/pull/1633))

This is useful to combine the two if you are writing a commit hook to tell the user what actually changed in a single command.

#### Ignore node_modules when running prettier from CLI ([#1683](https://github.com/prettier/prettier/pull/1683)) by [@thymikee](https://github.com/thymikee)

It's very easy to run prettier over the `node_modules/` folder by mistake which is something you almost never want to. So now we disable it by default and add a `--with-node-modules` option if you really want to.

#### Traverse dot files for glob ([#1844](https://github.com/prettier/prettier/pull/1844)) by [@jhgg](https://github.com/jhgg)

We enabled the option to go through .dotfiles in the glob parsing library we are using. This means that writing `*` will now catch `.eslintrc`.
