"""
sbftypes_blocks.py

SBF Protocol output payload definitions.

THESE ARE THE PAYLOAD DEFINITIONS FOR MESSAGES _FROM_ THE RECEIVER
(e.g. Periodic Navigation Data; Poll Responses; Info messages).

Created on 19 May 2025

Info sourced from mosaic-X5 Reference Guide v4.14.10 © 2000-2024 Septentrio NV/SA

:author: semuadmin (Steve Smith)
:copyright: semuadmin © 2020
:license: BSD 3-Clause
"""

# pylint: disable=too-many-lines

from pysbf2.sbftypes_core import (
    C1,
    C3,
    C9,
    C10,
    C20,
    C21,
    C32,
    C40,
    C60,
    F4,
    F8,
    I1,
    I2,
    I4,
    PAD,
    PD,
    PD1,
    PD2,
    U1,
    U2,
    U3,
    U4,
    U5,
    U6,
    U7,
    U8,
    U9,
    U12,
    U13,
    U16,
    U20,
    V1,
    V2,
    V4,
    X1,
    X2,
    X4,
)

SBF_MEASUREMENT_BLOCKS = {
    "Meas3CN0HiRes": {},  # definition not in public domain
    "Meas3Doppler": {},  # definition not in public domain
    "Meas3MP": {},  # definition not in public domain
    "Meas3PP": {},  # definition not in public domain
    "Meas3Ranges": {},  # definition not in public domain
    "MeasEpoch": {
        # PRtype1 [m] = (CodeMSB*4294967296+CodeLSB)*0.001
        # Dtype1 [Hz] = Doppler*0.0001
        # Ltype1 [cycles] = PRtype1/λ+(CarrierMSB*65536+CarrierLSB)*0.001
        # where λ = carrier frequency
        # PRtype2 [m] = PRtype1+(CodeOffsetMSB*65536+CodeOffsetLSB)*0.001
        # Ltype2 [cycles] = PRtype2/λ+(CarrierMSB*65536+CarrierLSB)*0.001
        # Dtype2 [Hz] = Dtype1*α+(DopplerOffsetMSB*65536+DopplerOffsetLSB)*1e-4,
        # where α = ratio of type 1 and type 2 carrier frequencies
        "TOW": U4,
        "WNc": U2,
        "N1": U1,
        "SB1Length": U1,
        "SB2Length": U1,
        "CommonFlags": (
            X1,
            {
                "MultipathMitigation": U1,
                "CodeSmoothing": U1,
                "Reserved": U1,
                "ClockSteering": U1,
                "Reserved1": U1,
                "HighDynamics": U1,
                "E6BUsed": U1,
                "Scrambling": U1,
            },
        ),
        "CumClkJumps": U1,  # Rev 1
        "Reserved2": U1,
        "MeasEpochChannelType1": (
            "N1",
            {
                "RxChannel": U1,
                "Type": (
                    X1,
                    {"SigIdxLo": U5, "AntennaID": U3},
                ),
                "SVID": U1,
                "Misc": (
                    X1,
                    {
                        "CodeMSB": U4,
                        "Reserved3": U4,
                    },
                ),
                "CodeLSB": U4,
                "Doppler": I4,
                "CarrierLSB": U2,
                "CarrierMSB": I1,
                "CN0": U1,
                "LockTime": U2,
                "ObsInfo": (
                    X1,
                    {
                        "PRSmoothed": U1,
                        "Reserved4": U1,
                        "HalfCycleAmbiguity": U1,
                        "SigIdxHi": U5,
                    },
                ),
                "N2": U1,
                PAD: PD1,
                "MeasEpochChannelType2": (
                    "N2+1",
                    {
                        "Type": (
                            X1,
                            {"SigIdxLo": U5, "AntennaID": U3},
                        ),
                        "LockTime": U1,
                        "CN0": U1,
                        "OffsetsMSB": (
                            X1,
                            {
                                "CodeOffsetMSB": U3,
                                "DopplerOffsetMSB": U5,
                            },
                        ),
                        "CarrierMSB": I1,
                        "ObsInfo": (
                            X1,
                            {
                                "PRSmoothed": U1,
                                "Reserved6": U1,
                                "HalfCycleAmbiguity": U1,
                                "SigIdxHi": U5,
                            },
                        ),
                        "CodeOffsetLSB": U2,
                        "CarrierLSB": U2,
                        "DopplerOffsetLSB": U2,
                        PAD: PD2,
                    },
                ),
            },
        ),
    },
    "MeasExtra": {
        "TOW": U4,
        "WNc": U2,
        "N": U1,
        "SBLength": U1,
        "DopplerVarFactor": F4,
        "group": (
            "N",
            {
                "RxChannel": U1,
                "Type": (
                    X1,
                    {"SigIdxLo": U5, "AntennaID": U3},
                ),
                "MPCorrection": I2,
                "SmoothingCorr": I2,
                "CodeVar": U2,
                "CarrierVar": U2,
                "LockTime": U2,
                "CumLossCont": U1,  # Rev 1
                "CarMPCorr": I1,
                "Info": U1,  # Rev 2
                "Misc": (  # Rev 3
                    X1,
                    {"CN0HighRes": U3, "SigIdxHi": U5},
                ),
                PAD: PD,
            },
        ),
    },
    "EndOfMeas": {
        "TOW": U4,
        "WNc": U2,
    },
}

SBF_NAVIGATION_PAGE_BLOCKS = {
    "GPSRawCA": {
        "TOW": U4,
        "WNc": U2,
        "SVID": U1,
        "CRCPassed": U1,
        "ViterbiCnt": U1,
        "Source": (
            X1,
            {
                "SigIdx": U5,
                "Reserved1": U3,
            },
        ),
        "FreqNr": U1,
        "RxChannel": U1,
        "group": (
            10,
            {
                "NavBits": U4,
            },
        ),
    },
    "GPSRawL2C": {
        "TOW": U4,
        "WNc": U2,
        "SVID": U1,
        "CRCPassed": U1,
        "ViterbiCnt": U1,
        "Source": (
            X1,
            {
                "SigIdx": U5,
                "Reserved1": U3,
            },
        ),
        "FreqNr": U1,
        "RxChannel": U1,
        "group": (
            10,
            {
                "NavBits": U4,
            },
        ),
    },
    "GPSRawL5": {
        "TOW": U4,
        "WNc": U2,
        "SVID": U1,
        "CRCPassed": U1,
        "ViterbiCnt": U1,
        "Source": (
            X1,
            {
                "SigIdx": U5,
                "Reserved1": U3,
            },
        ),
        "FreqNr": U1,
        "RxChannel": U1,
        "group": (
            10,
            {
                "NavBits": U4,
            },
        ),
    },
    "GLORawCA": {
        "TOW": U4,
        "WNc": U2,
        "SVID": U1,
        "CRCPassed": U1,
        "ViterbiCnt": U1,
        "Source": (
            X1,
            {
                "SigIdx": U5,
                "Reserved1": U3,
            },
        ),
        "FreqNr": U1,
        "RxChannel": U1,
        "group": (
            3,
            {
                "NavBits": U4,
            },
        ),
    },
    "GALRawFNAV": {
        "TOW": U4,
        "WNc": U2,
        "SVID": U1,
        "CRCPassed": U1,
        "ViterbiCnt": U1,
        "Source": (
            X1,
            {
                "SigIdx": U5,
                "Reserved1": U3,
            },
        ),
        "FreqNr": U1,
        "RxChannel": U1,
        "group": (
            8,
            {
                "NavBits": U4,
            },
        ),
    },
    "GALRawINAV": {
        "TOW": U4,
        "WNc": U2,
        "SVID": U1,
        "CRCPassed": U1,
        "ViterbiCnt": U1,
        "Source": (
            X1,
            {
                "SigIdx": U5,
                "L1BCFlag": U1,
                "Reserved1": U2,
            },
        ),
        "FreqNr": U1,
        "RxChannel": U1,
        "group": (
            8,
            {
                "NavBits": U4,
            },
        ),
    },
    "GALRawCNAV": {
        "TOW": U4,
        "WNc": U2,
        "SVID": U1,
        "CRCPassed": U1,
        "ViterbiCnt": U1,
        "Source": (
            X1,
            {
                "SigIdx": U5,
                "Reserved1": U3,
            },
        ),
        "FreqNr": U1,
        "group": (
            16,
            {
                "NavBits": U4,
            },
        ),
    },
    "GEORawL1": {
        "TOW": U4,
        "WNc": U2,
        "SVID": U1,
        "CRCPassed": U1,
        "ViterbiCnt": U1,
        "Source": (
            X1,
            {
                "SigIdx": U5,
                "Reserved1": U3,
            },
        ),
        "FreqNr": U1,
        "RxChannel": U1,
        "group": (
            8,
            {
                "NavBits": U4,
            },
        ),
    },
    "GEORawL5": {
        "TOW": U4,
        "WNc": U2,
        "SVID": U1,
        "CRCPassed": U1,
        "ViterbiCnt": U1,
        "Source": (
            X1,
            {
                "SigIdx": U5,
                "Reserved1": U3,
            },
        ),
        "FreqNr": U1,
        "RxChannel": U1,
        "group": (
            8,
            {
                "NavBits": U4,
            },
        ),
    },
    "BDSRaw": {
        "TOW": U4,
        "WNc": U2,
        "SVID": U1,
        "CRCPassed": U1,
        "ViterbiCnt": U1,
        "Source": (
            X1,
            {
                "SigIdx": U8,
            },
        ),
        "Reserved": U1,
        "RxChannel": U1,
        "group": (
            10,
            {
                "NavBits": U4,
            },
        ),
    },
    "BDSRawB1C": {
        "TOW": U4,
        "WNc": U2,
        "SVID": U1,
        "CRCSF2": U1,
        "CRCSF3": U1,
        "Source": (
            X1,
            {
                "SigIdx": U8,
            },
        ),
        "Reserved": U1,
        "RxChannel": U1,
        "group": (
            57,
            {
                "NavBits": U4,
            },
        ),
    },
    "BDSRawB2a": {
        "TOW": U4,
        "WNc": U2,
        "SVID": U1,
        "CRCPassed": U1,
        "ViterbiCnt": U1,
        "Source": (
            X1,
            {
                "SigIdx": U8,
            },
        ),
        "Reserved": U1,
        "RxChannel": U1,
        "group": (
            18,
            {
                "NavBits": U4,
            },
        ),
    },
    "BDSRawB2b": {
        "TOW": U4,
        "WNc": U2,
        "SVID": U1,
        "CRCPassed": U1,
        "Reserved1": U1,
        "Source": (
            X1,
            {
                "SigIdx": U8,
            },
        ),
        "Reserved2": U1,
        "RxChannel": U1,
        "group": (
            31,
            {
                "NavBits": U4,
            },
        ),
    },
    "QZSRawL1CA": {
        "TOW": U4,
        "WNc": U2,
        "SVID": U1,
        "CRCPassed": U1,
        "Reserved": U1,
        "Source": (
            X1,
            {
                "SigIdx": U8,
            },
        ),
        "Reserved2": U1,
        "RxChannel": U1,
        "group": (
            10,
            {
                "NavBits": U4,
            },
        ),
    },
    "QZSRawL2C": {
        "TOW": U4,
        "WNc": U2,
        "SVID": U1,
        "CRCPassed": U1,
        "ViterbiCnt": U1,
        "Source": (
            X1,
            {
                "SigIdx": U5,
                "Reserved1": U3,
            },
        ),
        "Reserved": U1,
        "RxChannel": U1,
        "group": (
            10,
            {
                "NavBits": U4,
            },
        ),
    },
    "QZSRawL5": {
        "TOW": U4,
        "WNc": U2,
        "SVID": U1,
        "CRCPassed": U1,
        "ViterbiCnt": U1,
        "Source": (
            X1,
            {
                "SigIdx": U5,
                "Reserved1": U3,
            },
        ),
        "Reserved": U1,
        "RxChannel": U1,
        "group": (
            10,
            {
                "NavBits": U4,
            },
        ),
    },
    "NAVICRaw": {
        "TOW": U4,
        "WNc": U2,
        "SVID": U1,
        "CRCPassed": U1,
        "ViterbiCnt": U1,
        "Source": (
            X1,
            {
                "SigIdx": U8,
            },
        ),
        "Reserved": U1,
        "RxChannel": U1,
        "group": (
            10,
            {
                "NavBits": U4,
            },
        ),
    },
}

SBF_GPS_DECODED_MESSAGE_BLOCKS = {
    "GPSNav": {
        "TOW": U4,
        "WNc": U2,
        "PRN": U1,
        "Reserved": U1,
        "WN": U2,
        "CAorPonL2": U1,
        "URA": U1,
        "health": U1,
        "L2DataFlag": U1,
        "IODC": U2,
        "IODE2": U1,
        "IODE3": U1,
        "FitIntFlg": U1,
        "Reserved2": U1,
        "T_gd": F4,
        "t_oc": U4,
        "a_f2": F4,
        "a_f1": F4,
        "a_f0": F4,
        "C_rs": F4,
        "DEL_N": F4,
        "M_0": F8,
        "C_uc": F4,
        "e": F8,
        "C_us": F4,
        "SQRT_A": F8,
        "t_oe": U4,
        "C_ic": F4,
        "OMEGA_0": F8,
        "C_is": F4,
        "i_0": F8,
        "C_rc": F4,
        "omega": F8,
        "OMEGADOT": F4,
        "IDOT": F4,
        "WNt_oc": U2,
        "WNt_oe": U2,
    },
    "GPSAlm": {
        "TOW": U4,
        "WNc": U2,
        "PRN": U1,
        "Reserved": U1,
        "e": F4,
        "t_oa": U4,
        "delta_i": F4,
        "OMEGADOT": F4,
        "SQRT_A": F4,
        "OMEGA_0": F4,
        "omega": F4,
        "M_0": F4,
        "a_f1": F4,
        "a_f0": F4,
        "WN_a": U1,
        "config": U1,
        "health8": U1,
        "health6": U1,
    },
    "GPSIon": {
        "TOW": U4,
        "WNc": U2,
        "PRN": U1,
        "Reserved": U1,
        "alpha_0": F4,
        "alpha_1": F4,
        "alpha_2": F4,
        "alpha_3": F4,
        "beta_0": F4,
        "beta_1": F4,
        "beta_2": F4,
        "beta_3": F4,
    },
    "GPSUtc": {
        "TOW": U4,
        "WNc": U2,
        "PRN": U1,
        "Reserved": U1,
        "A_1": F4,
        "A_0": F8,
        "t_ot": U4,
        "WN_t": U1,
        "DEL_t_LS": I1,
        "WN_LSF": U1,
        "DN": U1,
        "DEL_t_LSF": I1,
    },
    "GPSCNav": {
        "TOW": U4,
        "WNc": U2,
        "PRN": U1,
        "Flags": (
            X1,
            {
                "Alert": U1,
                "Integrity": U1,
                "L2CPhasing": U1,
                "Reserved": U3,
                "L2CUsed": U1,
                "L5Used": U1,
            },
        ),
        "WN": U2,
        "Health": U1,
        "URA_ED": I1,
        "t_op": U4,
        "t_oe": U4,
        "A": F8,
        "A_DOT": F8,
        "DELTA_N": F4,
        "DELTA_N_DOT": F4,
        "M_0": F8,
        "e": F8,
        "omega": F8,
        "OMEGA_0": F8,
        "OMEGADOT": F8,
        "i_0": F8,
        "IDOT": F4,
        "C_is": F4,
        "C_ic": F4,
        "C_rs": F4,
        "C_rc": F4,
        "C_us": F4,
        "C_uc": F4,
        "t_oc": U4,
        "URA_NED0": I1,
        "URA_NED1": U1,
        "URA_NED2": U1,
        "WN_op": U1,
        "a_f2": F4,
        "a_f1": F4,
        "a_f0": F8,
        "T_gd": F4,
        "ISC_L1CA": F4,
        "ISC_L2C": F4,
        "ISC_L5I5": F4,
        "ISC_L5Q5": F4,
    },
}

SBF_GLONASS_DECODED_MESSAGE_BLOCKS = {
    "GLONav": {
        "TOW": U4,
        "WNc": U2,
        "SVID": U1,
        "FreqNr": U1,
        "X": F8,
        "Y": F8,
        "Z": F8,
        "Dx": F4,
        "Dy": F4,
        "Dz": F4,
        "Ddx": F4,
        "Ddy": F4,
        "Ddz": F4,
        "gamma": F4,
        "tau": F4,
        "dtau": F4,
        "t_oe": U4,
        "WN_toe": U2,
        "P1": U1,
        "P2": U1,
        "E": U1,
        "B": U1,
        "tb": U2,
        "M": U1,
        "P": U1,
        "l": U1,
        "P4": U1,
        "N_T": U2,
        "F_T": U2,
        "C": U1,
    },
    "GLOAlm": {
        "TOW": U4,
        "WNc": U2,
        "SVID": U1,
        "FreqNr": U1,
        "epsilon": F4,
        "t_oa": U4,
        "Delta_i": F4,
        "lambda": F4,
        "t_ln": F4,
        "omega": F4,
        "Delta_T": F4,
        "dDelta_T": F4,
        "tau": F4,
        "WN_a": U1,
        "C": U1,
        "N": U2,
        "M": U1,
        "N_4": U1,
    },
    "GLOTime": {
        "TOW": U4,
        "WNc": U2,
        "SVID": U1,
        "FreqNr": U1,
        "N_4": U1,
        "KP": U1,
        "N": U2,
        "tau_GPS": F4,
        "tau_c": F8,
        "B1": F4,
        "B2": F4,
    },
}

SBF_GALILEO_DECODED_MESSAGE_BLOCKS = {
    "GALNav": {
        "TOW": U4,
        "WNc": U2,
        "SVID": U1,
        "Source": U1,
        "SQRT_A": F8,
        "M_0": F8,
        "e": F8,
        "i_0": F8,
        "omega": F8,
        "OMEGA_0": F8,
        "OMEGADOT": F4,
        "IDOT": F4,
        "DEL_N": F4,
        "C_uc": F4,
        "C_us": F4,
        "C_rc": F4,
        "C_rs": F4,
        "C_ic": F4,
        "C_is": F4,
        "t_oe": U4,
        "t_oc": U4,
        "a_f2": F4,
        "a_f1": F4,
        "a_f0": F8,
        "WNt_oe": U2,
        "WNt_oc": U2,
        "IODnav": U2,
        "Health_OSSOL": (
            X2,
            {
                "L1-BHealth": U1,
                "L1-Bdvs": U1,
                "L1-Bhs": U2,
                "E5bHealth": U1,
                "E5bdvs": U1,
                "E5bhs": U2,
                "E5aHealth": U1,
                "E5advs": U1,
                "E5ahs": U2,
                "Reserved": U4,
            },
        ),
        "Health_PRS": U1,
        "SISA_L1E5a": U1,
        "SISA_L1E5b": U1,
        "SISA_L1AE6A": U1,
        "BGD_L1E5a": F4,
        "BGD_L1E5b": F4,
        "BGD_L1AE6A": F4,
        "CNAVenc": U1,
    },
    "GALAlm": {
        "TOW": U4,
        "WNc": U2,
        "SVID": U1,
        "Source": U1,
        "e": F4,
        "t_oa": U4,
        "delta_i": F4,
        "OMEGADOT": F4,
        "SQRT_A": F4,
        "OMEGA_0": F4,
        "omega": F4,
        "M_0": F4,
        "a_f1": F4,
        "a_f0": F4,
        "WN_a": U1,
        "SVID_A": U1,
        "health": (
            X2,
            {
                "L1-BHealth": U1,
                "L1-Bhs": U2,
                "E5bHealth": U1,
                "E5bhs": U2,
                "E5aHealth": U1,
                "E5ahs": U2,
                "Reserved": U7,
            },
        ),
        "IODa": U1,
    },
    "GALIon": {
        "TOW": U4,
        "WNc": U2,
        "SVID": U1,
        "Source": U1,
        "a_i0": F4,
        "a_i1": F4,
        "a_i2": F4,
        "StormFlags": (
            X1,
            {
                "StormSF5": U1,
                "StormSF4": U1,
                "StormSF3": U1,
                "StormSF2": U1,
                "StormSF1": U1,
                "Reserved": U3,
            },
        ),
    },
    "GALUtc": {
        "TOW": U4,
        "WNc": U2,
        "SVID": U1,
        "Source": U1,
        "A_1": F4,
        "A_0": F8,
        "t_ot": U4,
        "WN_ot": U1,
        "DEL_t_LS": I1,
        "WN_LSF": U1,
        "DN": U1,
        "DEL_t_LSF": I1,
    },
    "GALGstGps": {
        "TOW": U4,
        "WNc": U2,
        "SVID": U1,
        "Source": U1,
        "A_1G": F4,
        "A_0G": F4,
        "t_oG": U4,
        "WN_oG": U1,
    },
    "GALSARRLM": {
        "TOW": U4,
        "WNc": U2,
        "SVID": U1,
        "Source": U1,
        "RLMLength": U1,
        "Reserved": U3,
        "group": (
            "RLMLength",  # if 80, N=3, if 160, N=5
            {
                "RLMBits": U4,
            },
        ),
    },
}

SBF_BEIDOU_DECODED_MESSAGE_BLOCKS = {
    "BDSNav": {
        "TOW": U4,
        "WNc": U2,
        "PRN": U1,
        "Reserved": U1,
        "WN": U2,
        "URA": U1,
        "SatH1": U1,
        "IODC": U1,
        "IODE": U1,
        "Reserved2": U2,
        "T_GD1": F4,
        "T_GD2": F4,
        "t_oc": U4,
        "a_f2": F4,
        "a_f1": F4,
        "a_f0": F4,
        "C_rs": F4,
        "DEL_N": F4,
        "M_0": F8,
        "C_uc": F4,
        "e": F8,
        "C_us": F4,
        "SQRT_A": F8,
        "t_oe": U4,
        "C_ic": F4,
        "OMEGA_0": F8,
        "C_is": F4,
        "i_0": F8,
        "C_rc": F4,
        "omega": F8,
        "OMEGADOT": F4,
        "IDOT": F4,
        "WNt_oc": U2,
        "WNt_oe": U2,
    },
    "BDSAlm": {
        "TOW": U4,
        "WNc": U2,
        "PRN": U1,
        "WN_a": U1,
        "t_oa": U4,
        "SQRT_A": F4,
        "e": F4,
        "omega": F4,
        "M_0": F4,
        "OMEGA_0": F4,
        "OMEGADOT": F4,
        "delta_i": F4,
        "a_f0": F4,
        "a_f1": F4,
        "Health": U2,
        "Reserved": U2,
    },
    "BDSIon": {
        "TOW": U4,
        "WNc": U2,
        "PRN": U1,
        "Reserved": U1,
        "alpha_0": F4,
        "alpha_1": F4,
        "alpha_2": F4,
        "alpha_3": F4,
        "beta_0": F4,
        "beta_1": F4,
        "beta_2": F4,
        "beta_3": F4,
    },
    "BDSUtc": {
        "TOW": U4,
        "WNc": U2,
        "PRN": U1,
        "Reserved": U1,
        "A_1": F4,
        "A_0": F8,
        "DEL_t_LS": I1,
        "WN_LSF": U1,
        "DN": U1,
        "DEL_t_LSF": I1,
    },
}

SBF_QZSS_DECODED_MESSAGE_BLOCKS = {
    "QZSNav": {
        "TOW": U4,
        "WNc": U2,
        "PRN": U1,
        "Reserved": U1,
        "WN": U2,
        "CAorPonL2": U1,
        "URA": U1,
        "health": U1,
        "L2DataFlag": U1,
        "IODC": U2,
        "IODE2": U1,
        "IODE3": U1,
        "FitIntFlg": U1,
        "Reserved2": U1,
        "T_gd": F4,
        "t_oc": U4,
        "a_f2": F4,
        "a_f1": F4,
        "a_f0": F4,
        "C_rs": F4,
        "DEL_N": F4,
        "M_0": F8,
        "C_uc": F4,
        "e": F8,
        "C_us": F4,
        "SQRT_A": F8,
        "t_oe": U4,
        "C_ic": F4,
        "OMEGA_0": F8,
        "C_is": F4,
        "i_0": F8,
        "C_rc": F4,
        "omega": F8,
        "OMEGADOT": F4,
        "IDOT": F4,
        "WNt_oc": U2,
        "WNt_oe": U2,
    },
    "QZSAlm": {
        "TOW": U4,
        "WNc": U2,
        "PRN": U1,
        "Reserved": U1,
        "e": F4,
        "t_oa": U4,
        "delta_i": F4,
        "OMEGADOT": F4,
        "SQRT_A": F4,
        "OMEGA_0": F4,
        "omega": F4,
        "M_0": F4,
        "a_f1": F4,
        "a_f0": F4,
        "WN_a": U1,
        "Reserved2": U1,
        "health8": U1,
        "health6": U1,
    },
}

SBF_SBAS_L1_DECODED_MESSAGE_BLOCKS = {
    "GEOMT00": {
        "TOW": U4,
        "WNc": U2,
        "PRN": U1,
    },
    "GEOPRNMask": {
        "TOW": U4,
        "WNc": U2,
        "PRN": U1,
        "IODP": U1,
        "NbrPRNs": U1,
        "group": (
            "NbrPRNs",
            {
                "PRNMask": U1,
            },
        ),
    },
    "GEOFastCorr": {
        "TOW": U4,
        "WNc": U2,
        "PRN": U1,
        "MT": U1,
        "IODP": U1,
        "IODF": U1,
        "N": U1,
        "SBLength": U1,
        "group": (
            "N",
            {
                "PRNMaskNo": U1,
                "UDREI": U1,
                "Reserved": U2,
                "PRC": F4,
                PAD: PD,
            },
        ),
    },
    "GEOIntegrity": {
        "TOW": U4,
        "WNc": U2,
        "PRN": U1,
        "Reserved": U1,
        "group1": (
            4,
            {
                "IODF": U1,
            },
        ),
        "group2": (
            51,
            {
                "UDREI": U1,
            },
        ),
    },
    "GEOFastCorrDegr": {
        "TOW": U4,
        "WNc": U2,
        "PRN": U1,
        "IODP": U1,
        "t_lat": U1,
        "group": (
            51,
            {
                "ai": U1,
            },
        ),
    },
    "GEONav": {
        "TOW": U4,
        "WNc": U2,
        "PRN": U1,
        "Reserved": U1,
        "IODN": U2,
        "URA": U2,
        "t0": U4,
        "Xg": F8,
        "Yg": F8,
        "Zg": F8,
        "Xgd": F8,
        "Ygd": F8,
        "Zgd": F8,
        "Xgdd": F8,
        "Ygdd": F8,
        "Zgdd": F8,
        "aGf0": F4,
        "aGf1": F4,
    },
    "GEODegrFactors": {
        "TOW": U4,
        "WNc": U2,
        "PRN": U1,
        "Reserved": U1,
        "Brrc": F8,
        "Cltc_lsb": F8,
        "Cltc_v1": F8,
        "Iltc_v1": U4,
        "Cltc_v0": F8,
        "Iltc_v0": U4,
        "Cgeo_lsb": F8,
        "Cgeo_v": F8,
        "Igeo": U4,
        "Cer": F4,
        "Ciono_step": F8,
        "Iiono": U4,
        "Ciono_ramp": F8,
        "RSSudre": U1,
        "RSSiono": U1,
        "Reserved2": U2,
        "Ccovariance": F8,
    },
    "GEONetworkTime": {
        "TOW": U4,
        "WNc": U2,
        "PRN": U1,
        "Reserved": U1,
        "A_1": F4,
        "A_0": F8,
        "t_ot": U4,
        "WN_t": U1,
        "DEL_t_LS": I1,
        "WN_LSF": U1,
        "DN": U1,
        "DEL_t_LSF": I1,
        "UTC_std": U1,
        "GPS_WN": U2,
        "GPS_TOW": U4,
        "GlonassID": U1,
    },
    "GEOAlm": {
        "TOW": U4,
        "WNc": U2,
        "PRN": U1,
        "Reserved0": U1,
        "DataID": U1,
        "Reserved1": U1,
        "Health": U2,
        "t_oa": U4,
        "Xg": F8,
        "Yg": F8,
        "Zg": F8,
        "Xgd": F8,
        "Ygd": F8,
        "Zgd": F8,
    },
    "GEOIGPMask": {
        "TOW": U4,
        "WNc": U2,
        "PRN": U1,
        "NbrBands": U1,
        "BandNbr": U1,
        "IODI": U1,
        "NbrIGPs": U1,
        "group": (
            "NbrIGPs",
            {
                "IGPMask": U1,
            },
        ),
    },
    "GEOLongTermCorr": {
        "TOW": U4,
        "WNc": U2,
        "PRN": U1,
        "N": U1,
        "SBLength": U1,
        "Reserved": U3,
        "group": (
            "N",
            {
                "VelocityCode": U1,
                "PRNMaskNo": U1,
                "IODP": U1,
                "IODE": U1,
                "dx": F4,
                "dy": F4,
                "dz": F4,
                "dxRate": F4,
                "dyRate": F4,
                "dzRate": F4,
                "da_f0": F4,
                "da_f1": F4,
                "t_oe": U4,
                PAD: PD,
            },
        ),
    },
    "GEOIonoDelay": {
        "TOW": U4,
        "WNc": U2,
        "PRN": U1,
        "BandNbr": U1,
        "IODI": U1,
        "N": U1,
        "SBLength": U1,
        "Reserved": U1,
        "group": (
            "N",
            {
                "IGPMaskNo": U1,
                "GIVEI": U1,
                "Reserved1": U2,
                "VerticalDelay": F4,
                PAD: PD,
            },
        ),
    },
    "GEOServiceLevel": {
        "TOW": U4,
        "WNc": U2,
        "PRN": U1,
        "Reserved": U1,
        "IODS": U1,
        "nrMessages": U1,
        "MessageNR": U1,
        "PriorityCode": U1,
        "dUDREI_In": U1,
        "dUDREI_Out": U1,
        "N": U1,
        "SBLength": U1,
        "group": (
            "N",
            {
                "Latitude1": I1,
                "Latitude2": I1,
                "Longitude1": I2,
                "Longitude2": I2,
                "RegionShape": U1,
                PAD: PD,
            },
        ),
    },
    "GEOClockEphCovMatrix": {
        "TOW": U4,
        "WNc": U2,
        "PRN": U1,
        "IODP": U1,
        "N": U1,
        "SBLength": U1,
        "Reserved": U2,
        "group": (
            "N",
            {
                "PRNMaskNo": U1,
                "Reserved1": U2,
                "ScaleExp": U1,
                "E11": U2,
                "E22": U2,
                "E33": U2,
                "E44": U2,
                "E12": I2,
                "E13": I2,
                "E14": I2,
                "E23": I2,
                "E24": I2,
                "E34": I2,
                PAD: PD,
            },
        ),
    },
}

SBF_GNSS_POSITION_VELOCITY_TIME_BLOCKS = {
    "PVTCartesian": {
        "TOW": U4,
        "WNc": U2,
        "Mode": (
            X1,
            {
                "Type": U4,  # decode PVT_TYPE
                "Reserved1": U2,
                "AutoSet": U1,
                "2D": U1,
            },
        ),
        "Error": U1,  # decode PVT_ERROR
        "X": F8,
        "Y": F8,
        "Z": F8,
        "Undulation": F4,
        "Vx": F4,
        "Vy": F4,
        "Vz": F4,
        "COG": F4,
        "RxClkBias": F8,
        "RxClkDrift": F4,
        "TimeSystem": U1,  # decode PVT_TIMESYSTEM
        "Datum": U1,  # decode PVT_DATUM
        "NrSV": U1,
        "WACorrInfo": (
            X1,
            {
                "Corr_OrbClkUsed": U1,
                "Corr_RngUsed": U1,
                "Corr_IonoUsed": U1,
                "Corr_OrbAccUsed": U1,
                "Corr_DO229Active": U1,
                "Corr_RTKType": U2,  # decode PVT_RTK_TYPE
                "Reserved2": U1,
            },
        ),
        "ReferenceID": U2,
        "MeanCorrAge": U2,
        "SignalInfo": U4,
        "AlertFlag": (
            X1,
            {
                "RAIMIntegrity": U2,
                "GalHPCAFail": U1,
                "GalIonStorm": U1,
                "Reserved3": U4,
            },
        ),
        "NrBases": U1,
        "PPPInfo": (
            X2,
            {
                "PPPSeedAge": U12,
                "Reserved4": U1,
                "PPPSeedType": U3,
            },
        ),
        "Latency": U2,
        "HAccuracy": U2,
        "VAccuracy": U2,
        "Misc": (
            X1,
            {
                "BaseARP": U1,
                "PhaseCtrOffset": U1,
                "Reserved5": U4,
                "ARPOffset": U2,
            },
        ),
    },
    "PVTGeodetic": {
        "TOW": U4,
        "WNc": U2,
        "Mode": (
            X1,
            {
                "Type": U4,  # decode PVT_TYPE
                "Reserved1": U2,
                "AutoSet": U1,
                "2D": U1,
            },
        ),
        "Error": U1,  # decode PVT_ERROR
        "Latitude": F8,  # radians
        "Longitude": F8,  # radians
        "Height": F8,  # HAE
        "Undulation": F4,
        "Vn": F4,
        "Ve": F4,
        "Vu": F4,
        "COG": F4,
        "RxClkBias": F8,
        "RxClkDrift": F4,
        "TimeSystem": U1,  # decode PVT_TIMESYSTEM
        "Datum": U1,  # decode PVT_DATUM
        "NrSV": U1,  # SIP
        "WACorrInfo": (
            X1,
            {
                "Corr_OrbClkUsed": U1,
                "Corr_RngUsed": U1,
                "Corr_IonoUsed": U1,
                "Corr_OrbAccUsed": U1,
                "Corr_DO229Active": U1,
                "Corr_RTKType": U2,  # decode PVT_RTK_TYPE
                "Reserved2": U1,
            },
        ),
        "ReferenceID": U2,
        "MeanCorrAge": U2,
        "SignalInfo": U4,
        "AlertFlag": (
            X1,
            {
                "RAIMIntegrity": U2,
                "GalHPCAFail": U1,
                "GalIonStorm": U1,
                "Reserved3": U4,
            },
        ),
        "NrBases": U1,  # Rev 1
        "PPPInfo": (
            X2,
            {
                "PPPSeedAge": U12,
                "Reserved4": U1,
                "PPPSeedType": U3,
            },
        ),
        "Latency": U2,
        "HAccuracy": U2,  # cm
        "VAccuracy": U2,  # cm
        "Misc": (
            X1,
            {
                "BaseARP": U1,
                "PhaseCtrOffset": U1,
                "Reserved5": U4,
                "ARPOffset": U2,
            },
        ),
    },
    "PosCovCartesian": {
        "TOW": U4,
        "WNc": U2,
        "Mode": (
            X1,
            {
                "Type": U4,  # decode PVT_TYPE
                "Reserved1": U2,
                "AutoSet": U1,
                "2D": U1,
            },
        ),
        "Error": U1,  # decode PVT_ERROR
        "Cov_xx": F4,
        "Cov_yy": F4,
        "Cov_zz": F4,
        "Cov_bb": F4,
        "Cov_xy": F4,
        "Cov_xz": F4,
        "Cov_xb": F4,
        "Cov_yz": F4,
        "Cov_yb": F4,
        "Cov_zb": F4,
    },
    "PosCovGeodetic": {
        "TOW": U4,
        "WNc": U2,
        "Mode": (
            X1,
            {
                "Type": U4,  # decode PVT_TYPE
                "Reserved1": U2,
                "AutoSet": U1,
                "2D": U1,
            },
        ),
        "Error": U1,  # decode PVT_ERROR
        "Cov_latlat": F4,
        "Cov_lonlon": F4,
        "Cov_hgthgt": F4,
        "Cov_bb": F4,
        "Cov_latlon": F4,
        "Cov_lathgt": F4,
        "Cov_latb": F4,
        "Cov_lonhgt": F4,
        "Cov_lonb": F4,
        "Cov_hb": F4,
    },
    "VelCovCartesian": {
        "TOW": U4,
        "WNc": U2,
        "Mode": (
            X1,
            {
                "Type": U4,  # decode PVT_TYPE
                "Reserved1": U2,
                "AutoSet": U1,
                "2D": U1,
            },
        ),
        "Error": U1,  # decode PVT_ERROR
        "Cov_VxVx": F4,
        "Cov_VyVy": F4,
        "Cov_VzVz": F4,
        "Cov_DtDt": F4,
        "Cov_VxVy": F4,
        "Cov_VxVz": F4,
        "Cov_VxDt": F4,
        "Cov_VyVz": F4,
        "Cov_VyDt": F4,
        "Cov_VzDt": F4,
    },
    "VelCovGeodetic": {
        "TOW": U4,
        "WNc": U2,
        "Mode": (
            X1,
            {
                "Type": U4,  # decode PVT_TYPE
                "Reserved1": U2,
                "AutoSet": U1,
                "2D": U1,
            },
        ),
        "Error": U1,  # decode PVT_ERROR
        "Cov_VnVn": F4,
        "Cov_VeVe": F4,
        "Cov_VuVu": F4,
        "Cov_DtDt": F4,
        "Cov_VnVe": F4,
        "Cov_VnVu": F4,
        "Cov_VnDt": F4,
        "Cov_VeVu": F4,
        "Cov_VeDt": F4,
        "Cov_VuDt": F4,
    },
    "DOP": {
        "TOW": U4,
        "WNc": U2,
        "NrSV": U1,
        "Reserved": U1,
        "PDOP": U2,
        "TDOP": U2,
        "HDOP": U2,
        "VDOP": U2,
        "HPL": F4,
        "VPL": F4,
    },
    "PosCart": {
        "TOW": U4,
        "WNc": U2,
        "Mode": (
            X1,
            {
                "Type": U4,  # decode PVT_TYPE
                "Reserved1": U2,
                "AutoSet": U1,
                "2D": U1,
            },
        ),
        "Error": U1,  # decode PVT_ERROR
        "X": F8,
        "Y": F8,
        "Z": F8,
        "Base2RoverX": F8,
        "Base2RoverY": F8,
        "Base2RoverZ": F8,
        "Cov_xx": F4,
        "Cov_yy": F4,
        "Cov_zz": F4,
        "Cov_xy": F4,
        "Cov_xz": F4,
        "Cov_yz": F4,
        "PDOP": U2,
        "HDOP": U2,
        "VDOP": U2,
        "Misc": (
            X1,
            {
                "BaseARP": U1,
                "PhaseCtrOffset": U1,
                "Reserved2": U4,
                "ARPOffset": U2,
            },
        ),
        "Reserved3": U1,
        "AlertFlag": U1,
        "Datum": U1,  # decode PVT_DATUM
        "NrSV": U1,
        "WACorrInfo": (
            X1,
            {
                "Corr_OrbClkUsed": U1,
                "Corr_RngUsed": U1,
                "Corr_IonoUsed": U1,
                "Corr_OrbAccUsed": U1,
                "Corr_DO229Active": U1,
                "Corr_RTKType": U2,  # decode PVT_RTK_TYPE
                "Reserved4": U1,
            },
        ),
        "ReferenceId": U2,
        "MeanCorrAge": U2,
        "SignalInfo": U4,
    },
    "PosLocal": {
        "TOW": U4,
        "WNc": U2,
        "Mode": (
            X1,
            {
                "Type": U4,  # decode PVT_TYPE
                "Reserved1": U2,
                "AutoSet": U1,
                "2D": U1,
            },
        ),
        "Error": U1,  # decode PVT_ERROR
        "Lat": F8,  # radians
        "Lon": F8,  # radians
        "Alt": F8,
        "Datum": U1,  # decode PVT_DATUM
    },
    "PosProjected": {
        "TOW": U4,
        "WNc": U2,
        "Mode": (
            X1,
            {
                "Type": U4,  # decode PVT_TYPE
                "Reserved1": U2,
                "AutoSet": U1,
                "2D": U1,
            },
        ),
        "Error": U1,  # decode PVT_ERROR
        "Northing": F8,
        "Easting": F8,
        "Alt": F8,
        "Datum": U1,  # decode PVT_DATUM
    },
    "BaseVectorCart": {
        "TOW": U4,
        "WNc": U2,
        "N": U1,
        "SBLength": U1,
        "group": (
            "N",
            {
                "nrSv": U1,
                "Error": U1,  # decode PVT_ERROR
                "Mode": (
                    X1,
                    {
                        "Type": U4,  # decode PVT_TYPE
                        "Reserved1": U2,
                        "AutoSet": U1,
                        "2D": U1,
                    },
                ),
                "Misc": (
                    X1,
                    {
                        "BaseARP": U1,
                        "PhaseCtrOffset": U1,
                        "Reserved2": U4,
                        "ARPOffset": U2,
                    },
                ),
                "DeltaX": F8,
                "DeltaY": F8,
                "DeltaZ": F8,
                "DeltaVx": F4,
                "DeltaVy": F4,
                "DeltaYz": F4,
                "Azimuth": U2,
                "Elevation": I2,
                "ReferenceID": U2,
                "CorrAge": U2,
                "SignalInfo": U4,
                PAD: PD,
            },
        ),
    },
    "BaseVectorGeod": {
        "TOW": U4,
        "WNc": U2,
        "N": U1,
        "SBLength": U1,
        "group": (
            "N",
            {
                "NrSV": U1,
                "Error": U1,  # decode PVT_ERROR
                "Mode": (
                    X1,
                    {
                        "Type": U4,  # decode PVT_TYPE
                        "Reserved1": U2,
                        "AutoSet": U1,
                        "2D": U1,
                    },
                ),
                "Misc": (
                    X1,
                    {
                        "BaseARP": U1,
                        "PhaseCtrOffset": U1,
                        "Reserved2": U4,
                        "ARPOffset": U2,
                    },
                ),
                "DeltaEast": F8,
                "DeltaNorth": F8,
                "DeltaUp": F8,
                "DeltaVe": F4,
                "DeltaVn": F4,
                "DeltaVu": F4,
                "Azimuth": U2,
                "Elevation": I2,
                "ReferenceID": U2,
                "CorrAge": U2,
                "SignalInfo": U4,
                PAD: PD,
            },
        ),
    },
    "PVTSupport": {},  # definition not in public domain
    "PVTSupportA": {},  # definition not in public domain
    "EndOfPVT": {
        "TOW": U4,
        "WNc": U2,
    },
}

SBF_GNSS_ATTITUDE_BLOCKS = {
    "AttEuler": {
        "TOW": U4,
        "WNc": U2,
        "NrSV": U1,
        "Error": U1,
        "Mode": U2,
        "Reserved": U2,
        "Heading": F4,
        "Pitch": F4,
        "Roll": F4,
        "PitchDot": F4,
        "RollDot": F4,
        "HeadingDot": F4,
    },
    "AttCovEuler": {
        "TOW": U4,
        "WNc": U2,
        "Reserved": U1,
        "Error": U1,
        "Cov_HeadHead": F4,
        "Cov_PitchPitch": F4,
        "Cov_RollRoll": F4,
        "Cov_HeadPitch": F4,
        "Cov_HeadRoll": F4,
        "Cov_PitchRoll": F4,
    },
    "AuxAntPositions": {
        "TOW": U4,
        "WNc": U2,
        "N": U1,
        "SBLength": U1,
        "group": (
            "N",
            {
                "NrSV": U1,
                "Error": U1,
                "AmbiguityType": U1,
                "AuxAntID": U1,
                "DeltaEast": F8,
                "DeltaNorth": F8,
                "DeltaUp": F8,
                "EastVel": F8,
                "NorthVel": F8,
                "UpVel": F8,
                PAD: PD,
            },
        ),
    },
    "EndOfAtt": {
        "TOW": U4,
        "WNc": U2,
    },
}

SBF_RECEIVER_TIME_BLOCKS = {
    "ReceiverTime": {
        "TOW": U4,
        "WNc": U2,
        "UTCYear": I1,
        "UTCMonth": I1,
        "UTCDay": I1,
        "UTCHour": I1,
        "UTCMin": I1,
        "UTCSec": I1,
        "DeltaLS": I1,
        "SyncLevel": U1,
    },
    "xPPSOffset": {
        "TOW": U4,
        "WNc": U2,
        "SyncAge": U1,
        "TimeScale": U1,
        "Offset": F4,
    },
}

SBF_EXTERNAL_EVENT_BLOCKS = {
    "ExtEvent": {
        "TOW": U4,
        "WNc": U2,
        "Source": U1,
        "Polarity": U1,
        "Offset": F4,
        "RxClkBias": F8,
        "PVTAge": U2,  # Rev 1
    },
    "ExtEventPVTCartesian": {
        "TOW": U4,
        "WNc": U2,
        "Mode": (
            X1,
            {
                "Type": U4,
                "Reserved1": U2,
                "AutoSet": U1,
                "2D": U1,
            },
        ),
        "Error": U1,
        "X": F8,
        "Y": F8,
        "Z": F8,
        "Undulation": F4,
        "Vx": F4,
        "Vy": F4,
        "Vz": F4,
        "COG": F4,
        "RxClkBias": F8,
        "RxClkDrift": F4,
        "TimeSystem": U1,
        "Datum": U1,
        "NrSV": U1,
        "WACorrInfo": (
            X1,
            {
                "Corr_OrbClkUsed": U1,
                "Corr_RngUsed": U1,
                "Corr_IonoUsed": U1,
                "Corr_OrbAccUsed": U1,
                "Corr_DO229Active": U1,
                "Corr_RTKType": U2,
                "Reserved2": U1,
            },
        ),
        "ReferenceID": U2,
        "MeanCorrAge": U2,
        "SignalInfo": U4,
        "AlertFlag": U1,
        "NrBases": U1,  # Rev 1
        "PPPInfo": U2,
        "Latency": U2,  # Rev 2
        "HAccuracy": U2,
        "VAccuracy": U2,
        "Misc": (
            X1,
            {
                "BaseARP": U1,
                "PhaseCtrOffset": U1,
                "Reserved3": U4,
                "ARPOffset": U2,
            },
        ),
    },
    "ExtEventPVTGeodetic": {
        "TOW": U4,
        "WNc": U2,
        "Mode": (
            X1,
            {
                "Type": U4,
                "Reserved1": U2,
                "AutoSet": U1,
                "2D": U1,
            },
        ),
        "Error": U1,
        "Latitude": F8,
        "Longitude": F8,
        "Height": F8,
        "Undulation": F4,
        "Vn": F4,
        "Ve": F4,
        "Vu": F4,
        "COG": F4,
        "RxClkBias": F8,
        "RxClkDrift": F4,
        "TimeSystem": U1,
        "Datum": U1,
        "NrSV": U1,
        "WACorrInfo": (
            X1,
            {
                "Corr_OrbClkUsed": U1,
                "Corr_RngUsed": U1,
                "Corr_IonoUsed": U1,
                "Corr_OrbAccUsed": U1,
                "Corr_DO229Active": U1,
                "Corr_RTKType": U2,
                "Reserved2": U1,
            },
        ),
        "ReferenceID": U2,
        "MeanCorrAge": U2,
        "SignalInfo": U4,
        "AlertFlag": U1,
        "NrBases": U1,  # Rev 1
        "PPPInfo": U2,
        "Latency": U2,  # Rev 2
        "HAccuracy": U2,
        "VAccuracy": U2,
        "Misc": (
            X1,
            {
                "BaseARP": U1,
                "PhaseCtrOffset": U1,
                "Reserved3": U4,
                "ARPOffset": U2,
            },
        ),
    },
    "ExtEventBaseVectGeod": {
        "TOW": U4,
        "WNc": U2,
        "N": U1,
        "SBLength": U1,
        "group": (
            "N",
            {
                "NrSV": U1,
                "Error": U1,
                "Mode": (
                    X1,
                    {
                        "Type": U4,
                        "Reserved1": U2,
                        "AutoSet": U1,
                        "2D": U1,
                    },
                ),
                "Misc": (
                    X1,
                    {
                        "BaseARP": U1,
                        "PhaseCtrOffset": U1,
                        "Reserved2": U6,
                    },
                ),
                "DeltaEast": F8,
                "DeltaNorth": F8,
                "DeltaUp": F8,
                "DeltaVe": F4,
                "DeltaVn": F4,
                "DeltaVu": F4,
                "Azimuth": U2,
                "Elevation": I2,
                "ReferenceID": U2,
                "CorrAge": U2,
                "SignalInfo": U4,
                PAD: PD,
            },
        ),
    },
    "ExtEventAttEuler": {
        "TOW": U4,
        "WNc": U2,
        "NrSV": U1,
        "Error": U1,
        "Mode": U2,
        "Reserved": U2,
        "Heading": F4,
        "Pitch": F4,
        "Roll": F4,
        "PitchDot": F4,
        "RollDot": F4,
        "HeadingDot": F4,
    },
}

SBF_DIFFERENTIAL_CORRECTION_BLOCKS = {
    "DiffCorrIn": {
        "TOW": U4,
        "WNc": U2,
        "Mode": U1,
        "Source": U1,
        "optionmode0": (  # present if Mode = 0
            ("Mode", 0),
            {
                "RTCM2Words": V4,  # variable by size of RTCM2 message
            },
        ),
        "optionmode1": (  # present if Mode = 1
            ("Mode", 1),
            {
                "CMRMessage": V2,  # variable by size of CMR message
            },
        ),
        "optionmode2": (  # present if Mode = 2
            ("Mode", 2),
            {
                "RTCM3Message": V2,  # variable by size of RTCM3 message
            },
        ),
        "optionmode4": (  # present if Mode = 3
            ("Mode", 3),
            {
                "RTCMVMessage": V2,  # variable by size of RTCMV message
            },
        ),
    },
    "BaseStation": {
        "TOW": U4,
        "WNc": U2,
        "BaseStationID": U2,
        "BaseType": U1,
        "Source": U1,
        "Datum": U1,
        "Reserved": U1,
        "X": F8,
        "Y": F8,
        "Z": F8,
    },
    "RTCMDatum": {
        "TOW": U4,
        "WNc": U2,
        "SourceCRS": C32,
        "TargetCRS": C32,
        "Datum": U1,
        "HeightType": U1,
        "QualityInd": U1,
    },
    "FugroDDS": {},  # definition not in public domain
}

SBF_LBAND_DEMODULATOR_BLOCKS = {
    "LBandTrackerStatus": {
        "TOW": U4,
        "WNc": U2,
        "N": U1,
        "SBLength": U1,
        "group": (
            "N",
            {
                "Frequency": U4,
                "Baudrate": U2,
                "ServiceID": U2,
                "FreqOffset": F4,
                "CN0": U2,
                "AvgPower": I2,
                "AGCGain": I1,
                "Mode": U1,
                "Status": U1,
                "SVID": U1,  # Rev 2
                "LockTime": U2,  # Rev 1
                "Source": U1,  # Rev 3
                PAD: PD,
            },
        ),
    },
    "LBandBeams": {
        "TOW": U4,
        "WNc": U2,
        "N": U1,
        "SBLength": U1,
        "group": (
            "N",
            {
                "SVID": U1,
                "SatName": C9,
                "SatLongitude": I2,
                "BeamFreq": U4,
                PAD: PD,
            },
        ),
    },
    "LBandRaw": {
        "TOW": U4,
        "WNc": U2,
        "N": U2,
        "Frequency": U4,
        "group": (
            "N",
            {
                "UserData": U1,
            },
        ),
        "Channel": U1,  # Rev 1
    },
    "FugroStatus": {
        "TOW": U4,
        "WNc": U2,
        "Reserved": U2,
        "Status": U4,
        "SubStartingTime": I4,
        "SubExpirationTime": I4,
        "SubHourGlass": I4,
        "SubscribedMode": U4,
        "SubCurrentMode": U4,
        "SubLinkVector": U4,
        "CRCGoodCount": U4,
        "CRCBadCount": U4,
        "LbandTrackerStatusIdx": U2,  # Rev 1
    },
}

SBF_STATUS_BLOCKS = {
    "ChannelStatus": {
        "TOW": U4,
        "WNc": U2,
        "N": U1,
        "SB1Length": U1,
        "SB2Length": U1,
        "Reserved": U3,
        "group": (
            "N",
            {
                "SVID": U1,
                "FreqNr": U1,
                "Reserved1": U2,
                "Azimuth/RiseSet": (
                    X2,
                    {
                        "Azimuth": U9,
                        "Reserved5": U5,
                        "RiseSet": U2,
                    },
                ),
                "HealthStatus": U2,
                "Elevation": I1,
                "N2": U1,
                "RxChannel": U1,
                "Reserved2": U1,
                PAD: PD1,
                "group1": (
                    "N2+1",
                    {
                        "Antenna": U1,
                        "Reserved4": U1,
                        "TrackingStatus": U2,
                        "PVTStatus": U2,
                        "PVTInfo": U2,
                        PAD: PD2,
                    },
                ),
            },
        ),
    },
    "ReceiverStatus": {
        "TOW": U4,
        "WNc": U2,
        "CPULoad": U1,
        "ExtError": (
            X1,
            {
                "SISError": U1,
                "DiffCorrError": U1,
                "ExtSensorError": U1,
                "SetupError": U1,
                "Reserved1": U4,
            },
        ),
        "UpTime": U4,
        "RxState": (
            X4,
            {
                "Reserved2": U1,
                "ActiveAntenna": U1,
                "ExtFreq": U1,
                "ExtTime": U1,
                "WnSer": U1,
                "TowSet": U1,
                "FineTime": U1,
                "InternalDiskActivity": U1,
                "InternalDiskFull": U1,
                "InternalDiskMounted": U1,
                "IntAnt": U1,
                "RefoutLocked": U1,
                "Reserved3": U1,
                "ExternalDiskActivity": U1,
                "ExternalDiskFull": U1,
                "ExternalDiskMounted": U1,
                "PpsInCal": U1,
                "DiffCorrIn": U1,
                "Internet": U1,
                "Reserved4": U13,
            },
        ),
        "RxError": (
            X4,
            {
                "Reserved5": U3,
                "SoftwareError": U1,
                "WatchdogError": U1,
                "AntennaError": U1,
                "CongestionError": U1,
                "Reserved6": U1,
                "MissEventError": U1,
                "CPUOverloadError": U1,
                "InvalidConfigError": U1,
                "OutOfGeofenceError": U1,
                "Reserved7": U20,
            },
        ),
        "N": U1,
        "SBLength": U1,
        "CmdCount": U1,
        "Temperature": U1,  # NB: subtract 100 for °C
        "group": (
            "N",
            {
                "FrontEndID": U1,
                "Gain": I1,
                "SampleVar": U1,
                "BlankingStat": U1,
                PAD: PD,
            },
        ),
    },
    "SatVisibility": {
        "TOW": U4,
        "WNc": U2,
        "N": U1,
        "SBLength": U1,
        "group": (
            "N",
            {
                "SVID": U1,
                "FreqNr": U1,
                "Azimuth": U2,
                "Elevation": I2,
                "RiseSet": U1,
                "SatelliteInfo": U1,
                PAD: PD,
            },
        ),
    },
    "InputLink": {
        "TOW": U4,
        "WNc": U2,
        "N": U1,
        "SBLength": U1,
        "group": (
            "N",
            {
                "CD": U1,
                "Type": U1,
                "AgeOfLastMessage": U2,
                "NrBytesReceived": U4,
                "NrBytesAccepted": U4,
                "NrMsgReceived": U4,
                "NrMsgAccepted": U4,
                PAD: PD,
            },
        ),
    },
    "OutputLink": {
        "TOW": U4,
        "WNc": U2,
        "N1": U1,
        "SB1Length": U1,
        "SB2Length": U1,
        "Reserved": U3,
        "group": (
            "N1",
            {
                "CD": U1,
                "N2": U1,
                "AllowedRate": U2,
                "NrBytesProduced": U4,
                "NrBytesSent": U4,
                "NrClients": U1,  # Rev 1
                "Reserved2": U3,
                PAD: PD1,
                "group": (
                    "N2+1",
                    {
                        "Type": U1,
                        "Percentage": U1,
                        PAD: PD2,
                    },
                ),
            },
        ),
    },
    "NTRIPClientStatus": {
        "TOW": U4,
        "WNc": U2,
        "N": U1,
        "SBLength": U1,
        "group": (
            "N",
            {
                "CDIndex": U1,
                "Status": U1,
                "ErrorCode": U1,
                "Info": U1,
                PAD: PD,
            },
        ),
    },
    "NTRIPServerStatus": {
        "TOW": U4,
        "WNc": U2,
        "N": U1,
        "SBLength": U1,
        "group": (
            "N",
            {
                "CDIndex": U1,
                "Status": U1,
                "ErrorCode": U1,
                "Info": U1,
                PAD: PD,
            },
        ),
    },
    "IPStatus": {
        "TOW": U4,
        "WNc": U2,
        "MACAddress": U6,
        "IPAddress": U16,
        "Gateway": U16,
        "Netmask": U1,
        "Reserved": U3,
        "HostName": C32,  # Rev 1
    },
    "DynDNSStatus": {
        "TOW": U4,
        "WNc": U2,
        "Status": U1,
        "ErrorCode": U1,
        "IPAddress": U16,  # Rev 1
    },
    "QualityInd": {
        "TOW": U4,
        "WNc": U2,
        "N": U1,
        "Reserved": U1,
        "group": (
            "N",
            {
                "Indicators": (
                    X2,
                    {
                        "QualityType": U8,
                        "QualityValue": U4,
                        "Reserved1": U4,
                    },
                ),
            },
        ),
    },
    "DiskStatus": {
        "TOW": U4,
        "WNc": U2,
        "N": U1,
        "SBLength": U1,
        "Reserved": U4,
        "group": (
            "N",
            {
                "DiskID": U1,
                "Status": (
                    X1,
                    {
                        "Mounted": U1,
                        "Full": U1,
                        "Activity": U1,
                        "Logging": U1,
                        "Mounting": U1,  # Rev 1
                        "Formatting": U1,
                        "Reserved1": U2,
                    },
                ),
                "DiskUsageMSB": U2,
                "DiskUsageLSB": U4,
                "DiskSize": U4,
                "CreateDeleteCount": U1,
                "Error": U1,  # Rev 1
                PAD: PD,
            },
        ),
    },
    "RFStatus": {
        "TOW": U4,
        "WNc": U2,
        "N": U1,
        "SBLength": U1,
        "Flags": (
            X1,
            {
                "Misleading": U1,
                "Inauthentic": U1,
                "Reserved2": U6,
            },
        ),
        "Reserved": U3,
        "group": (
            "N",
            {
                "Frequency": U4,
                "Bandwidth": U2,
                "Info": (
                    X1,
                    {
                        "Mode": U4,
                        "Reserved3": U2,
                        "AntennaID": U2,
                    },
                ),
                PAD: PD,
            },
        ),
    },
    "P2PPStatus": {
        "TOW": U4,
        "WNc": U2,
        "N": U1,
        "SBLength": U1,
        "group": (
            "N",
            {
                "SessionID": U1,
                "Port": U1,
                "Status": (
                    X1,
                    {
                        "Mode": U1,
                        "P2PPStatus": U7,
                    },
                ),
                "ErrorCode": U1,
                PAD: PD,
            },
        ),
    },
    "GALAuthStatus": {
        "TOW": U4,
        "WNc": U2,
        "OSNMAStatus": (
            X2,
            {
                "Status": U3,
                "OSNMAProgress": U8,
                "TimeSource": U3,
                "MerkleInProgress": U1,
                "Reserved": U1,
            },
        ),
        "TrustedTimeDelta": F4,
        "GalActiveMask": U8,
        "GalAuthenticMask": U8,
        "GpsActiveMask": U8,
        "GpsAuthenticMask": U8,
    },
}

SBF_MISCELLANEOUS_BLOCKS = {
    "ReceiverSetup": {
        "TOW": U4,
        "WNc": U2,
        "Reserved": U2,
        "MarkerName": C60,
        "MarkerNumber": C20,
        "Observer": C20,
        "Agency": C40,
        "RxName": C20,
        "RxVersion": C20,
        "AntSerialNbr": C20,
        "AntType": C20,
        "deltaH": F4,
        "deltaE": F4,
        "deltaN": F4,
        "MarkerType": C20,  # Rev 1
        "GNSSFWVersion": C40,  # Rev 2
        "ProductName": C40,  # Rev 3
        "Latitude": F8,  # radians Rev 4
        "Longitude": F8,  # radians
        "Height": F4,
        "StationCode": C10,
        "MonumentIdx": U1,
        "ReceiverIdx": U1,
        "CountryCode": C3,
        "Reserved1": C21,
    },
    "RxMessage": {
        "TOW": U4,
        "WNc": U2,
        "Type": U1,
        "Severity": U1,
        "MessageID": U4,
        "StringLn": U2,
        "Reserved2": U2,
        "group": (
            "StringLn",
            {
                "Message": C1,
            },
        ),
    },
    "Comment": {
        "TOW": U4,
        "WNc": U2,
        "CommentLn": U2,
        "group": (
            "CommentLn",
            {
                "Comment": C1,
            },
        ),
    },
    "Commands": {
        "TOW": U4,
        "WNc": U2,
        "Reserved": U2,
        "CmdData": V1,  # variable number of U1
    },
    "BBSamples": {
        "TOW": U4,
        "WNc": U2,
        "N": U2,
        "Info": (
            X1,
            {
                "AntennaID": U3,
                "Reserved": U5,
            },
        ),
        "Reserved": U3,
        "SampleFreq": U4,
        "LOFreq": U4,
        "group": (
            "N",
            {
                "Samples": U2,
            },
        ),
    },
    "ASCIIIn": {
        "TOW": U4,
        "WNc": U2,
        "CD": U1,
        "Reserved1": U3,
        "StringLn": U2,
        "SensorModel": C20,
        "SensorType": C20,
        "Reserved2": U20,
        "group": (
            "StringLn",
            {
                "ASCIIString": C1,
            },
        ),
    },
    "EncapsulatedOutput": {
        "TOW": U4,
        "WNc": U2,
        "Mode": U1,
        "Reserved": U1,
        "N": U2,
        "ReservedId": U2,
        "group": (
            "N",
            {
                "Payload": U1,
            },
        ),
    },
}

# TESTING = {
#     # Test Message - Do Not Use
#     "TestOnly": {
#         "TOW": U4,
#         "WNc": U2,
#         "ScaledVal": [U4, 0.0001],
#         "Mode": U1,
#         "optionmode0": (  # present if Mode = 0
#             ("Mode", 0),
#             {
#                 "ModeZERO": U1,
#             },
#         ),
#         "optionmode1": (  # present if Mode = 1
#             ("Mode", 1),
#             {
#                 "ModeONE": U1,
#             },
#         ),
#         "SB1Length": U1,
#         "SB2Length": U1,
#         "N1": U2,
#         "group1": (
#             "N1",
#             {
#                 "test1": U1,
#                 "test2": U2,
#                 "N2": U2,
#                 PAD: PD1,
#                 "group2": (
#                     "N2+1",
#                     {
#                         "test3": U1,
#                         "test4": U2,
#                         PAD: PD2,
#                     },
#                 ),
#             },
#         ),
#     },
#     # Variable Length Test Message - Do Not Use
#     "TestVariable": {
#         "TOW": U4,
#         "WNc": U2,
#         "Mode": U1,
#         "optionmode0": (  # present if Mode = 0
#             ("Mode", 0),
#             {
#                 "VariableZERO": V4,
#             },
#         ),
#         "optionmode1": (  # present if Mode = 1
#             ("Mode", [1, 2]),
#             {
#                 "VariableONE": V2,
#             },
#         ),
#     },
# }

SBF_BLOCKS = {
    **SBF_MEASUREMENT_BLOCKS,
    **SBF_NAVIGATION_PAGE_BLOCKS,
    **SBF_GPS_DECODED_MESSAGE_BLOCKS,
    **SBF_GLONASS_DECODED_MESSAGE_BLOCKS,
    **SBF_GALILEO_DECODED_MESSAGE_BLOCKS,
    **SBF_BEIDOU_DECODED_MESSAGE_BLOCKS,
    **SBF_QZSS_DECODED_MESSAGE_BLOCKS,
    **SBF_SBAS_L1_DECODED_MESSAGE_BLOCKS,
    **SBF_GNSS_POSITION_VELOCITY_TIME_BLOCKS,
    **SBF_GNSS_ATTITUDE_BLOCKS,
    **SBF_RECEIVER_TIME_BLOCKS,
    **SBF_EXTERNAL_EVENT_BLOCKS,
    **SBF_DIFFERENTIAL_CORRECTION_BLOCKS,
    **SBF_LBAND_DEMODULATOR_BLOCKS,
    **SBF_STATUS_BLOCKS,
    **SBF_MISCELLANEOUS_BLOCKS,
    # **TESTING,
}
