/*
 * Copyright (c) 2002 Isao SEKI <iseki@gongon.com>
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote
 *    products derived from this software without specific prior written
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * This file is included in wistumbler package.
 *
 * $Id: stumbler.c,v 1.8 2002/08/16 14:27:13 iseki Exp $
 */

#include <sys/types.h>
#include <sys/cdefs.h>
#include <sys/param.h>
#include <sys/socket.h>
#include <sys/ioctl.h> 
#include <sys/socket.h>
#include <net/if.h>
#include <netinet/in.h>
#include <netinet/if_ether.h>
#include <net/if_ieee80211.h> 
#include <dev/ic/wi_ieee.h>
#include <stdio.h>
#include <string.h> 
#include <ctype.h>
#include <stdlib.h>
#include <unistd.h> 
#include <errno.h>
#include <err.h>
#include <signal.h>
#include <gtk/gtk.h>
#include "gpscontrol.h"

/* import from wicontrol.c */
#define	WI_APRATE_0  0x00	/* NONE */
#define WI_APRATE_1  0x0A	/* 1 Mbps */
#define WI_APRATE_2  0x14	/* 2 Mbps */
#define WI_APRATE_5  0x37	/* 5.5 Mbps */
#define WI_APRATE_11 0x6E	/* 11 Mbps */
#define MAX_APS 1024

struct access_point {
    struct wi_apinfo wi_apinfo;
    long first;
    long last;
    int cur_receive;
    struct dir dir;
};
struct access_point *access_points[MAX_APS];
/* end of import */

extern int wi_init(char *);
extern int wi_finish(char *);
extern int wi_apscan(char *);
extern void print_aps(FILE *);

int wi_exit(int);

struct _app {
    GtkWidget *window;
    GtkWidget *clist;
    GtkWidget *label;
} app;

int gps = 0;
int debug = 0;
char *logfile = NULL;
char *iface1 = NULL;
gint timer;
gchar *titles[12] = {"", "SSID", "BSSID", "Channel", "Signal", "Interval",
                     "Capinfo", "First", "Last", "Rate", "Position"};

void window_setup()
{
    GtkWidget *vbox, *hbox, *quit_button, *scrolled_win;

    app.window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title(GTK_WINDOW (app.window), "wistumbler");
    gtk_signal_connect(GTK_OBJECT(app.window), "destroy",
                       GTK_SIGNAL_FUNC(gtk_main_quit), NULL);

    vbox = gtk_vbox_new(FALSE, 5);
    gtk_container_set_border_width(GTK_CONTAINER(vbox), 5);
    gtk_container_add(GTK_CONTAINER(app.window), vbox);

    /* create scrolled window */
    scrolled_win = gtk_scrolled_window_new(NULL, NULL);
    gtk_container_set_border_width(GTK_CONTAINER(scrolled_win), 0);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled_win),
        GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
    gtk_widget_set_usize(GTK_WIDGET(scrolled_win), 700, 100);
    gtk_box_pack_start(GTK_BOX (vbox), scrolled_win, TRUE, TRUE, 0);

    /* create GtkCList */
    app.clist = gtk_clist_new_with_titles(11, titles);
    gtk_clist_column_titles_passive((GtkCList *)app.clist);
    gtk_clist_set_column_width (GTK_CLIST(app.clist), 0, 10);  /* receive */
    gtk_clist_set_column_width (GTK_CLIST(app.clist), 1, 150); /* name */
    gtk_clist_set_column_width (GTK_CLIST(app.clist), 2, 120); /* BSSID */
    gtk_clist_set_column_width (GTK_CLIST(app.clist), 3, 15);  /* channel */
    gtk_clist_set_column_width (GTK_CLIST(app.clist), 4, 30);  /* signal */
    gtk_clist_set_column_width (GTK_CLIST(app.clist), 5, 30);  /* interval */
    gtk_clist_set_column_width (GTK_CLIST(app.clist), 7, 40);  /* First */
    gtk_clist_set_column_width (GTK_CLIST(app.clist), 8, 40);  /* last */
    gtk_clist_set_column_width (GTK_CLIST(app.clist), 9, 30);  /* rate */
    gtk_clist_set_column_width (GTK_CLIST(app.clist), 10, 170);  /* position */
    gtk_container_add(GTK_CONTAINER(scrolled_win), app.clist);

    /* create Quit button */
    hbox = gtk_hbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, TRUE, 0);
    quit_button = gtk_button_new_with_label("Quit");
    gtk_box_pack_end(GTK_BOX (hbox), quit_button, FALSE, TRUE, 0);
    gtk_signal_connect_object(GTK_OBJECT(quit_button), "clicked",
        GTK_SIGNAL_FUNC(gtk_widget_destroy), (gpointer)app.window);

    gtk_widget_show_all(app.window); 
}

void print_clist()
{
    int i, j;
    float rate;
    struct tm *tm;
    struct access_point *w;
    gchar receive[2], name[33], bssid[18], channel[3], signal[4], interval[6],
          capinfo[4], first[6], last[6], speed[4], position[32];
    gchar *line[12] = {receive, name, bssid, channel, signal, interval, capinfo,
                       first, last, speed, position};

    gtk_clist_clear((GtkCList *)app.clist);

    for (i=0; i < MAX_APS; i++) {
        if (access_points[i] == NULL)
            continue;

        w = access_points[i];

        /* for 1st column, receive status */
        if (w->cur_receive == 0) {
            receive[0] = ' ', receive[1] = '\0';
        } else {
            receive[0] = 'o', receive[1] = '\0';
            w->cur_receive = 0;
        }

        for (j=0; j<w->wi_apinfo.namelen && j < sizeof name - 1; j++) {
            name[j] = w->wi_apinfo.name[j];
        }
        name[j] = '\0';
     
        sprintf(bssid, "%02x:%02x:%02x:%02x:%02x:%02x",
                      w->wi_apinfo.bssid[0]&0xff, w->wi_apinfo.bssid[1]&0xff,
                      w->wi_apinfo.bssid[2]&0xff, w->wi_apinfo.bssid[3]&0xff,
                      w->wi_apinfo.bssid[4]&0xff, w->wi_apinfo.bssid[5]&0xff);

        snprintf(channel, sizeof channel, "%d", w->wi_apinfo.channel);
        snprintf(signal, sizeof signal, "%d", w->wi_apinfo.signal);
        snprintf(interval, sizeof interval, "%d", w->wi_apinfo.interval);
                                
        if (w->wi_apinfo.capinfo & IEEE80211_CAPINFO_PRIVACY)
            sprintf(capinfo, "WEP");
        else if (w->wi_apinfo.capinfo & IEEE80211_CAPINFO_ESS)
            sprintf(capinfo, "ESS");
	else
	    capinfo[0] = '\0';
        
        tm = localtime(&w->first);
        sprintf(first, "%02.2d:%02.2d", tm->tm_hour, tm->tm_min);
        tm = localtime(&w->last);
        sprintf(last, "%02.2d:%02.2d", tm->tm_hour, tm->tm_min);

        switch (w->wi_apinfo.rate) {
            case WI_APRATE_1:
                rate = 1;
                break;
            case WI_APRATE_2:
                rate = 2;
                break;
            case WI_APRATE_5:
                rate = 5.5;
                break;
            case WI_APRATE_11:
                rate = 11;
                break;
            case WI_APRATE_0:
            default:
                rate = 0;
                break;
        }
        if (rate)
            snprintf(speed, sizeof speed, "%g", rate);
        else
	    speed[0] = '\0';

        if (gps)
            snprintf(position, sizeof position, "%f%c/%f%c",
                                w->dir.Ndeg, w->dir.N, w->dir.Edeg, w->dir.E);
        else
            position[0] = '\0';

        gtk_clist_append((GtkCList *)app.clist, line);
    }  
}

gint wi_polling(iface)
char *iface;
{
    gps_read();

    if (wi_apscan(iface) == -1) {
        gtk_timeout_remove(timer);
        wi_exit(1);
    }

    if (debug)
        print_aps(stdout);

    print_clist();
    return(TRUE);
}

int wi_exit(code)
int code;
{
    FILE *fp;

    wi_finish(iface1);		/* power down the card , if needed */
    gps_close();

    if (logfile == NULL)
	exit(code);

    if ((fp = fopen(logfile, "a")) == NULL) {
        perror("fopen");
        exit(1);
    } else {
        printf("Writing log ...\n\n");
        print_aps(fp);
        fclose(fp);
        exit(code);
    }
}

static void cleanup(dummy)
int dummy;
{
    wi_exit(0);
}

void usage()
{
        fprintf(stderr,
            "usage: %s"
            " [-f log file name] [-g gps device name] [-d]"
	    " [ -i ] interface"
	    "\n",
            getprogname());
        exit(1);
}

int main(argc, argv)
int argc;
char *argv[];
{
    int i, c;
    char *gpsdev = NULL;

    gtk_set_locale();
    gtk_init(&argc, &argv);

    if (argc > 1 && argv[1][0] != '-') {
        iface1 = argv[1];
        optind++;
    }

    while ((c = getopt(argc, argv, "df:g:hi:")) != -1) {
        switch (c) {
            case 'd':
                debug = 1;
                break;
            case 'f':
                logfile = optarg;
                break;
            case 'g':
                gps = 1;
                gpsdev = optarg;
                break;
	    case 'i':
		iface1 = optarg;
		break;
            case 'h':
            default:
                usage();
        }
    }
    argc -= optind;
    argv += optind;

    if (iface1 == NULL && argc >= 1) {
	iface1 = argv[0];
	argc--;
	argv++;
    }

    if (iface1 == NULL || argc > 0)
        usage();

    signal(SIGINT, cleanup);
    signal(SIGQUIT, cleanup);
    signal(SIGHUP, cleanup);
    signal(SIGTERM, cleanup);

    if (wi_init(iface1) < 0)			/* power on the card, if down */
	exit(1);

    if (gps)
        if (gps_open(gpsdev) <= 0)
            exit(1);

    for (i = 0; i <= MAX_APS; i++) 
        access_points[i] = NULL;

    window_setup();

    /* call wi_polling per two seconds
     *   the interval must exceed 1000 milliseconds becuase wi_apscan()
     *   blocks GTK event loop 1000 milliseconds maxinum
     */
    timer = gtk_timeout_add(2000, (GtkFunction)wi_polling, (gpointer)iface1);

    gtk_main();        /* event loop of GTK */

    /* don't remove teh following line.
     * If window was destroyed and call timeout routine (draw clist),
     * crash will be occured.
     */
    gtk_timeout_remove(timer);
    wi_exit(0);

    return(0);
}
