package org.mandas.docker.client.messages.swarm;

import com.fasterxml.jackson.annotation.JsonProperty;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import org.mandas.docker.Nullable;

/**
 * Immutable implementation of {@link NodeSpec}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableNodeSpec.builder()}.
 */
@SuppressWarnings({"all"})
final class ImmutableNodeSpec implements NodeSpec {
  private final @Nullable String name;
  private final @Nullable Map<String, String> labels;
  private final String role;
  private final String availability;

  private ImmutableNodeSpec(
      @Nullable String name,
      @Nullable Map<String, String> labels,
      String role,
      String availability) {
    this.name = name;
    this.labels = labels;
    this.role = role;
    this.availability = availability;
  }

  /**
   * @return The value of the {@code name} attribute
   */
  @JsonProperty("Name")
  @Override
  public @Nullable String name() {
    return name;
  }

  /**
   * @return The value of the {@code labels} attribute
   */
  @JsonProperty("Labels")
  @Override
  public @Nullable Map<String, String> labels() {
    return labels;
  }

  /**
   * @return The value of the {@code role} attribute
   */
  @JsonProperty("Role")
  @Override
  public String role() {
    return role;
  }

  /**
   * @return The value of the {@code availability} attribute
   */
  @JsonProperty("Availability")
  @Override
  public String availability() {
    return availability;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link NodeSpec#name() name} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for name (can be {@code null})
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableNodeSpec withName(@Nullable String value) {
    if (Objects.equals(this.name, value)) return this;
    return new ImmutableNodeSpec(value, this.labels, this.role, this.availability);
  }

  /**
   * Copy the current immutable object by replacing the {@link NodeSpec#labels() labels} map with the specified map.
   * Nulls are not permitted as keys or values.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param entries The entries to replace the labels map
   * @return A modified copy or {@code this} if not changed
   */
  public final ImmutableNodeSpec withLabels(@Nullable Map<String, ? extends String> entries) {
    if (this.labels == entries) return this;
    @Nullable Map<String, String> newValue = entries == null ? null : createUnmodifiableMap(true, false, entries);
    return new ImmutableNodeSpec(this.name, newValue, this.role, this.availability);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link NodeSpec#role() role} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for role
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableNodeSpec withRole(String value) {
    String newValue = Objects.requireNonNull(value, "role");
    if (this.role.equals(newValue)) return this;
    return new ImmutableNodeSpec(this.name, this.labels, newValue, this.availability);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link NodeSpec#availability() availability} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for availability
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableNodeSpec withAvailability(String value) {
    String newValue = Objects.requireNonNull(value, "availability");
    if (this.availability.equals(newValue)) return this;
    return new ImmutableNodeSpec(this.name, this.labels, this.role, newValue);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableNodeSpec} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableNodeSpec
        && equalsByValue((ImmutableNodeSpec) another);
  }

  private boolean equalsByValue(ImmutableNodeSpec another) {
    return Objects.equals(name, another.name)
        && Objects.equals(labels, another.labels)
        && role.equals(another.role)
        && availability.equals(another.availability);
  }

  /**
   * Computes a hash code from attributes: {@code name}, {@code labels}, {@code role}, {@code availability}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + Objects.hashCode(name);
    h += (h << 5) + Objects.hashCode(labels);
    h += (h << 5) + role.hashCode();
    h += (h << 5) + availability.hashCode();
    return h;
  }

  /**
   * Prints the immutable value {@code NodeSpec} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "NodeSpec{"
        + "name=" + name
        + ", labels=" + labels
        + ", role=" + role
        + ", availability=" + availability
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link NodeSpec} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable NodeSpec instance
   */
  public static ImmutableNodeSpec copyOf(NodeSpec instance) {
    if (instance instanceof ImmutableNodeSpec) {
      return (ImmutableNodeSpec) instance;
    }
    return ImmutableNodeSpec.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableNodeSpec ImmutableNodeSpec}.
   * <pre>
   * ImmutableNodeSpec.builder()
   *    .name(String | null) // nullable {@link NodeSpec#name() name}
   *    .labels(Map&lt;String, String&gt; | null) // nullable {@link NodeSpec#labels() labels}
   *    .role(String) // required {@link NodeSpec#role() role}
   *    .availability(String) // required {@link NodeSpec#availability() availability}
   *    .build();
   * </pre>
   * @return A new ImmutableNodeSpec builder
   */
  public static ImmutableNodeSpec.Builder builder() {
    return new ImmutableNodeSpec.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableNodeSpec ImmutableNodeSpec}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  static final class Builder implements NodeSpec.Builder {
    private static final long INIT_BIT_ROLE = 0x1L;
    private static final long INIT_BIT_AVAILABILITY = 0x2L;
    private long initBits = 0x3L;

    private String name;
    private Map<String, String> labels = null;
    private String role;
    private String availability;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code NodeSpec} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * Collection elements and entries will be added, not replaced.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(NodeSpec instance) {
      Objects.requireNonNull(instance, "instance");
      @Nullable String nameValue = instance.name();
      if (nameValue != null) {
        name(nameValue);
      }
      @Nullable Map<String, String> labelsValue = instance.labels();
      if (labelsValue != null) {
        putAllLabels(labelsValue);
      }
      this.role(instance.role());
      this.availability(instance.availability());
      return this;
    }

    /**
     * Initializes the value for the {@link NodeSpec#name() name} attribute.
     * @param name The value for name (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Name")
    public final Builder name(@Nullable String name) {
      this.name = name;
      return this;
    }

    /**
     * Put one entry to the {@link NodeSpec#labels() labels} map.
     * @param key The key in the labels map
     * @param value The associated value in the labels map
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addLabel(String key, String value) {
      if (this.labels == null) {
        this.labels = new LinkedHashMap<String, String>();
      }
      this.labels.put(
          Objects.requireNonNull(key, "labels key"),
          Objects.requireNonNull(value, value == null ? "labels value for key: " + key : null));
      return this;
    }

    /**
     * Put one entry to the {@link NodeSpec#labels() labels} map. Nulls are not permitted
     * @param entry The key and value entry
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addLabel(Map.Entry<String, ? extends String> entry) {
      if (this.labels == null) {
        this.labels = new LinkedHashMap<String, String>();
      }
      String k = entry.getKey();
      String v = entry.getValue();
      this.labels.put(
          Objects.requireNonNull(k, "labels key"),
          Objects.requireNonNull(v, v == null ? "labels value for key: " + k : null));
      return this;
    }

    /**
     * Sets or replaces all mappings from the specified map as entries for the {@link NodeSpec#labels() labels} map. Nulls are not permitted as keys or values, but parameter itself can be null
     * @param entries The entries that will be added to the labels map
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Labels")
    public final Builder labels(@Nullable Map<String, ? extends String> entries) {
      if (entries == null) {
        this.labels = null;
        return this;
      }
      this.labels = new LinkedHashMap<String, String>();
      return putAllLabels(entries);
    }

    /**
     * Put all mappings from the specified map as entries to {@link NodeSpec#labels() labels} map. Nulls are not permitted
     * @param entries The entries that will be added to the labels map
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder putAllLabels(Map<String, ? extends String> entries) {
      if (this.labels == null) {
        this.labels = new LinkedHashMap<String, String>();
      }
      for (Map.Entry<String, ? extends String> e : entries.entrySet()) {
        String k = e.getKey();
        String v = e.getValue();
        this.labels.put(
            Objects.requireNonNull(k, "labels key"),
            Objects.requireNonNull(v, v == null ? "labels value for key: " + k : null));
      }
      return this;
    }

    /**
     * Initializes the value for the {@link NodeSpec#role() role} attribute.
     * @param role The value for role 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Role")
    public final Builder role(String role) {
      this.role = Objects.requireNonNull(role, "role");
      initBits &= ~INIT_BIT_ROLE;
      return this;
    }

    /**
     * Initializes the value for the {@link NodeSpec#availability() availability} attribute.
     * @param availability The value for availability 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Availability")
    public final Builder availability(String availability) {
      this.availability = Objects.requireNonNull(availability, "availability");
      initBits &= ~INIT_BIT_AVAILABILITY;
      return this;
    }

    /**
     * Builds a new {@link ImmutableNodeSpec ImmutableNodeSpec}.
     * @return An immutable instance of NodeSpec
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableNodeSpec build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new ImmutableNodeSpec(name, labels == null ? null : createUnmodifiableMap(false, false, labels), role, availability);
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_ROLE) != 0) attributes.add("role");
      if ((initBits & INIT_BIT_AVAILABILITY) != 0) attributes.add("availability");
      return "Cannot build NodeSpec, some of required attributes are not set " + attributes;
    }
  }

  private static <K, V> Map<K, V> createUnmodifiableMap(boolean checkNulls, boolean skipNulls, Map<? extends K, ? extends V> map) {
    switch (map.size()) {
    case 0: return Collections.emptyMap();
    case 1: {
      Map.Entry<? extends K, ? extends V> e = map.entrySet().iterator().next();
      K k = e.getKey();
      V v = e.getValue();
      if (checkNulls) {
        Objects.requireNonNull(k, "key");
        Objects.requireNonNull(v, v == null ? "value for key: " + k : null);
      }
      if (skipNulls && (k == null || v == null)) {
        return Collections.emptyMap();
      }
      return Collections.singletonMap(k, v);
    }
    default: {
      Map<K, V> linkedMap = new LinkedHashMap<>(map.size() * 4 / 3 + 1);
      if (skipNulls || checkNulls) {
        for (Map.Entry<? extends K, ? extends V> e : map.entrySet()) {
          K k = e.getKey();
          V v = e.getValue();
          if (skipNulls) {
            if (k == null || v == null) continue;
          } else if (checkNulls) {
            Objects.requireNonNull(k, "key");
            Objects.requireNonNull(v, v == null ? "value for key: " + k : null);
          }
          linkedMap.put(k, v);
        }
      } else {
        linkedMap.putAll(map);
      }
      return Collections.unmodifiableMap(linkedMap);
    }
    }
  }
}
