package org.mandas.docker.client.messages.swarm;

import com.fasterxml.jackson.annotation.JsonProperty;
import java.util.Objects;
import org.mandas.docker.Nullable;

/**
 * Immutable implementation of {@link UpdateConfig}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableUpdateConfig.builder()}.
 */
@SuppressWarnings({"all"})
final class ImmutableUpdateConfig implements UpdateConfig {
  private final @Nullable Long parallelism;
  private final @Nullable Long delay;
  private final @Nullable String failureAction;
  private final @Nullable String order;

  private ImmutableUpdateConfig(
      @Nullable Long parallelism,
      @Nullable Long delay,
      @Nullable String failureAction,
      @Nullable String order) {
    this.parallelism = parallelism;
    this.delay = delay;
    this.failureAction = failureAction;
    this.order = order;
  }

  /**
   * @return The value of the {@code parallelism} attribute
   */
  @JsonProperty("Parallelism")
  @Override
  public @Nullable Long parallelism() {
    return parallelism;
  }

  /**
   * @return The value of the {@code delay} attribute
   */
  @JsonProperty("Delay")
  @Override
  public @Nullable Long delay() {
    return delay;
  }

  /**
   * @return The value of the {@code failureAction} attribute
   */
  @JsonProperty("FailureAction")
  @Override
  public @Nullable String failureAction() {
    return failureAction;
  }

  /**
   * @return The value of the {@code order} attribute
   */
  @JsonProperty("Order")
  @Override
  public @Nullable String order() {
    return order;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link UpdateConfig#parallelism() parallelism} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for parallelism (can be {@code null})
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableUpdateConfig withParallelism(@Nullable Long value) {
    if (Objects.equals(this.parallelism, value)) return this;
    return new ImmutableUpdateConfig(value, this.delay, this.failureAction, this.order);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link UpdateConfig#delay() delay} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for delay (can be {@code null})
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableUpdateConfig withDelay(@Nullable Long value) {
    if (Objects.equals(this.delay, value)) return this;
    return new ImmutableUpdateConfig(this.parallelism, value, this.failureAction, this.order);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link UpdateConfig#failureAction() failureAction} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for failureAction (can be {@code null})
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableUpdateConfig withFailureAction(@Nullable String value) {
    if (Objects.equals(this.failureAction, value)) return this;
    return new ImmutableUpdateConfig(this.parallelism, this.delay, value, this.order);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link UpdateConfig#order() order} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for order (can be {@code null})
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableUpdateConfig withOrder(@Nullable String value) {
    if (Objects.equals(this.order, value)) return this;
    return new ImmutableUpdateConfig(this.parallelism, this.delay, this.failureAction, value);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableUpdateConfig} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableUpdateConfig
        && equalsByValue((ImmutableUpdateConfig) another);
  }

  private boolean equalsByValue(ImmutableUpdateConfig another) {
    return Objects.equals(parallelism, another.parallelism)
        && Objects.equals(delay, another.delay)
        && Objects.equals(failureAction, another.failureAction)
        && Objects.equals(order, another.order);
  }

  /**
   * Computes a hash code from attributes: {@code parallelism}, {@code delay}, {@code failureAction}, {@code order}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + Objects.hashCode(parallelism);
    h += (h << 5) + Objects.hashCode(delay);
    h += (h << 5) + Objects.hashCode(failureAction);
    h += (h << 5) + Objects.hashCode(order);
    return h;
  }

  /**
   * Prints the immutable value {@code UpdateConfig} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "UpdateConfig{"
        + "parallelism=" + parallelism
        + ", delay=" + delay
        + ", failureAction=" + failureAction
        + ", order=" + order
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link UpdateConfig} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable UpdateConfig instance
   */
  public static ImmutableUpdateConfig copyOf(UpdateConfig instance) {
    if (instance instanceof ImmutableUpdateConfig) {
      return (ImmutableUpdateConfig) instance;
    }
    return ImmutableUpdateConfig.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableUpdateConfig ImmutableUpdateConfig}.
   * <pre>
   * ImmutableUpdateConfig.builder()
   *    .parallelism(Long | null) // nullable {@link UpdateConfig#parallelism() parallelism}
   *    .delay(Long | null) // nullable {@link UpdateConfig#delay() delay}
   *    .failureAction(String | null) // nullable {@link UpdateConfig#failureAction() failureAction}
   *    .order(String | null) // nullable {@link UpdateConfig#order() order}
   *    .build();
   * </pre>
   * @return A new ImmutableUpdateConfig builder
   */
  public static ImmutableUpdateConfig.Builder builder() {
    return new ImmutableUpdateConfig.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableUpdateConfig ImmutableUpdateConfig}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  static final class Builder {
    private Long parallelism;
    private Long delay;
    private String failureAction;
    private String order;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code UpdateConfig} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(UpdateConfig instance) {
      Objects.requireNonNull(instance, "instance");
      @Nullable Long parallelismValue = instance.parallelism();
      if (parallelismValue != null) {
        parallelism(parallelismValue);
      }
      @Nullable Long delayValue = instance.delay();
      if (delayValue != null) {
        delay(delayValue);
      }
      @Nullable String failureActionValue = instance.failureAction();
      if (failureActionValue != null) {
        failureAction(failureActionValue);
      }
      @Nullable String orderValue = instance.order();
      if (orderValue != null) {
        order(orderValue);
      }
      return this;
    }

    /**
     * Initializes the value for the {@link UpdateConfig#parallelism() parallelism} attribute.
     * @param parallelism The value for parallelism (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Parallelism")
    public final Builder parallelism(@Nullable Long parallelism) {
      this.parallelism = parallelism;
      return this;
    }

    /**
     * Initializes the value for the {@link UpdateConfig#delay() delay} attribute.
     * @param delay The value for delay (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Delay")
    public final Builder delay(@Nullable Long delay) {
      this.delay = delay;
      return this;
    }

    /**
     * Initializes the value for the {@link UpdateConfig#failureAction() failureAction} attribute.
     * @param failureAction The value for failureAction (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("FailureAction")
    public final Builder failureAction(@Nullable String failureAction) {
      this.failureAction = failureAction;
      return this;
    }

    /**
     * Initializes the value for the {@link UpdateConfig#order() order} attribute.
     * @param order The value for order (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Order")
    public final Builder order(@Nullable String order) {
      this.order = order;
      return this;
    }

    /**
     * Builds a new {@link ImmutableUpdateConfig ImmutableUpdateConfig}.
     * @return An immutable instance of UpdateConfig
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableUpdateConfig build() {
      return new ImmutableUpdateConfig(parallelism, delay, failureAction, order);
    }
  }
}
