/*
 * Decompiled with CFR 0.152.
 */
package jdk.test.lib.hexdump;

import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.CharArrayWriter;
import java.io.DataInputStream;
import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.io.UncheckedIOException;
import java.nio.ByteBuffer;
import java.util.Objects;

public final class HexPrinter {
    static final String[] CONTROL_MNEMONICS = new String[]{"nul", "soh", "stx", "etx", "eot", "enq", "ack", "bel", "b", "t", "n", "vt", "f", "r", "so", "si", "dle", "dc1", "dc2", "dc3", "dc4", "nak", "syn", "etb", "can", "em", "sub", "esc", "fs", "gs", "rs", "us"};
    private static final int initBytesCount = 16;
    private static final String initBytesFormat = "%02x ";
    private static final int initAnnoWidth = 64;
    private static final String initAnnoDelim = " // ";
    final Appendable dest;
    final String offsetFormat;
    final String bytesFormat;
    final int bytesCount;
    final String annoDelim;
    final int annoWidth;
    final String lineSeparator;
    final Formatter annoFormatter;

    private HexPrinter(Formatter printer, String offsetFormat, String bytesFormat, int bytesCount, String annoDelim, int annoWidth, String lineSeparator, Appendable dest) {
        this.annoFormatter = Objects.requireNonNull(printer, "formatter");
        this.bytesCount = bytesCount;
        this.bytesFormat = Objects.requireNonNull(bytesFormat, bytesFormat);
        this.offsetFormat = Objects.requireNonNull(offsetFormat, "offsetFormat");
        this.annoDelim = Objects.requireNonNull(annoDelim, "annoDelim");
        this.annoWidth = annoWidth;
        this.lineSeparator = Objects.requireNonNull(lineSeparator, "lineSeparator");
        this.dest = Objects.requireNonNull(dest, "dest");
    }

    public static HexPrinter minimal() {
        return new HexPrinter(Formatters.NONE, "", "%02x", 16, "", 64, "", System.out);
    }

    public static HexPrinter canonical() {
        return new HexPrinter(Formatters.PRINTABLE, "%08x  ", initBytesFormat, 16, "|", 31, "|" + System.lineSeparator(), System.out);
    }

    public static HexPrinter simple() {
        return new HexPrinter(Formatters.ASCII, "%04x: ", initBytesFormat, 16, initAnnoDelim, 64, System.lineSeparator(), System.out);
    }

    public static HexPrinter source() {
        return new HexPrinter(Formatters.PRINTABLE, "    ", "(byte)%3d, ", 8, initAnnoDelim, 64, System.lineSeparator(), System.out);
    }

    public HexPrinter dest(Appendable dest) {
        Objects.requireNonNull(dest, "dest");
        return new HexPrinter(this.annoFormatter, this.offsetFormat, this.bytesFormat, this.bytesCount, this.annoDelim, this.annoWidth, this.lineSeparator, dest);
    }

    public HexPrinter format(byte[] source) {
        Objects.requireNonNull(source, "byte array must be non-null");
        return this.format(new ByteArrayInputStream(source));
    }

    public HexPrinter format(byte[] source, int offset, int length) {
        Objects.requireNonNull(source, "byte array must be non-null");
        return this.format(new ByteArrayInputStream(source, offset, length), offset);
    }

    public HexPrinter format(InputStream source) {
        return this.format(source, 0);
    }

    private HexPrinter format(InputStream source, int offset) {
        Objects.requireNonNull(source, "InputStream must be non-null");
        try (AnnotationWriter writer = new AnnotationWriter(this, source, offset, this.dest);){
            writer.flush();
            HexPrinter hexPrinter = this;
            return hexPrinter;
        }
    }

    public HexPrinter format(ByteBuffer source, int index, int length) {
        Objects.requireNonNull(source, "ByteBuffer must be non-null");
        byte[] bytes = new byte[length];
        source.get(index, bytes, 0, length);
        ByteArrayInputStream bais = new ByteArrayInputStream(bytes);
        return this.format(bais, index);
    }

    public HexPrinter format(ByteBuffer source) {
        return this.format(source, source.position(), source.limit() - source.position());
    }

    public String toString(byte[] source) {
        Objects.requireNonNull(source, "byte array must be non-null");
        return this.toString(new ByteArrayInputStream(source));
    }

    public String toString(byte[] source, int offset, int length) {
        Objects.requireNonNull(source, "byte array must be non-null");
        StringBuilder sb = new StringBuilder();
        try (AnnotationWriter writer = new AnnotationWriter(this, new ByteArrayInputStream(source, offset, length), offset, sb);){
            writer.flush();
            String string = sb.toString();
            return string;
        }
    }

    public String toString(InputStream source) {
        Objects.requireNonNull(source, "InputStream must be non-null");
        StringBuilder sb = new StringBuilder();
        try (AnnotationWriter writer = new AnnotationWriter(this, source, 0, sb);){
            writer.flush();
            String string = sb.toString();
            return string;
        }
    }

    public String toString(ByteBuffer source, int index, int length) {
        Objects.requireNonNull(source, "ByteBuffer must be non-null");
        byte[] bytes = new byte[length];
        source.get(index, bytes, 0, length);
        ByteArrayInputStream bais = new ByteArrayInputStream(bytes);
        StringBuilder sb = new StringBuilder();
        try (AnnotationWriter writer = new AnnotationWriter(this, bais, index, sb);){
            writer.flush();
            String string = sb.toString();
            return string;
        }
    }

    public String toString(ByteBuffer source) {
        return this.toString(source, source.position(), source.limit() - source.position());
    }

    public HexPrinter withOffsetFormat(String offsetFormat) {
        Objects.requireNonNull(offsetFormat, "offsetFormat");
        return new HexPrinter(this.annoFormatter, offsetFormat, this.bytesFormat, this.bytesCount, this.annoDelim, this.annoWidth, this.lineSeparator, this.dest);
    }

    public HexPrinter withBytesFormat(String byteFormat, int bytesCount) {
        Objects.requireNonNull(this.bytesFormat, "bytesFormat");
        if (bytesCount <= 0) {
            throw new IllegalArgumentException("bytesCount should be greater than zero");
        }
        return new HexPrinter(this.annoFormatter, this.offsetFormat, byteFormat, bytesCount, this.annoDelim, this.annoWidth, this.lineSeparator, this.dest);
    }

    public HexPrinter withLineSeparator(String separator) {
        return new HexPrinter(this.annoFormatter, this.offsetFormat, this.bytesFormat, this.bytesCount, this.annoDelim, this.annoWidth, separator, this.dest);
    }

    public HexPrinter formatter(Formatter formatter) {
        Objects.requireNonNull(formatter, "Formatter must be non-null");
        return new HexPrinter(formatter, this.offsetFormat, this.bytesFormat, this.bytesCount, this.annoDelim, this.annoWidth, this.lineSeparator, this.dest);
    }

    public HexPrinter formatter(Formatter formatter, String delim, int width) {
        Objects.requireNonNull(formatter, "formatter");
        Objects.requireNonNull(delim, "delim");
        return new HexPrinter(formatter, this.offsetFormat, this.bytesFormat, this.bytesCount, delim, width, this.lineSeparator, this.dest);
    }

    public HexPrinter formatter(Class<?> primClass, String fmtString) {
        Formatter formatter = HexPrinter.getFormatter(primClass, fmtString);
        return new HexPrinter(formatter, this.offsetFormat, this.bytesFormat, this.bytesCount, this.annoDelim, this.annoWidth, this.lineSeparator, this.dest);
    }

    static Formatter getFormatter(Class<?> primClass, String fmtString) {
        return new PrimitiveFormatter(primClass, fmtString);
    }

    public String toString() {
        return "formatter: " + String.valueOf(this.annoFormatter) + ", dest: " + this.dest.getClass().getName() + ", offset: \"" + this.offsetFormat + "\", bytes: " + this.bytesCount + " x \"" + this.bytesFormat + "\", delim: \"" + this.annoDelim + "\", width: " + this.annoWidth + ", nl: \"" + this.expand(this.lineSeparator) + "\"";
    }

    private String expand(String sep) {
        return sep.replace("\n", "\\n").replace("\r", "\\r");
    }

    @FunctionalInterface
    public static interface Formatter {
        public void annotate(DataInputStream var1, Appendable var2) throws IOException;
    }

    public static enum Formatters implements Formatter
    {
        PRINTABLE,
        ASCII,
        UTF8,
        NONE;


        @Override
        public void annotate(DataInputStream in, Appendable out) throws IOException {
            switch (this.ordinal()) {
                case 0: {
                    Formatters.bytePrintable(in, out);
                    break;
                }
                case 1: {
                    Formatters.byteASCII(in, out);
                    break;
                }
                case 2: {
                    Formatters.utf8Parser(in, out);
                    break;
                }
                case 3: {
                    Formatters.byteNoneParser(in, out);
                }
            }
        }

        static void bytePrintable(DataInputStream in, Appendable out) throws IOException {
            int v = in.readUnsignedByte();
            if (!Character.isISOControl(v) && v < 127) {
                out.append((char)v);
            } else {
                out.append('.');
            }
        }

        static void byteASCII(DataInputStream in, Appendable out) throws IOException {
            int v = in.readUnsignedByte();
            if (v < 32) {
                out.append('\\').append(CONTROL_MNEMONICS[v]);
            } else if (v < 127) {
                out.append((char)v);
            } else {
                out.append('\\').append(Integer.toString(v, 10));
            }
        }

        static void utf8Parser(DataInputStream in, Appendable out) throws IOException {
            out.append(in.readUTF()).append(" ");
        }

        static void byteNoneParser(DataInputStream in, Appendable out) throws IOException {
            in.readByte();
        }

        public static Formatter ofPrimitive(Class<?> primClass, String fmtString) {
            Objects.requireNonNull(primClass, "primClass");
            Objects.requireNonNull(fmtString, "fmtString");
            return new PrimitiveFormatter(primClass, fmtString);
        }
    }

    private static final class AnnotationWriter
    extends CharArrayWriter {
        private final transient OffsetInputStream source;
        private final transient DataInputStream in;
        private final transient int baseOffset;
        private final transient HexPrinter params;
        private final transient int bytesSingleWidth;
        private final transient int bytesColWidth;
        private final transient int annoWidth;
        private final transient Appendable dest;

        AnnotationWriter(HexPrinter params, InputStream source, int baseOffset, Appendable dest) {
            this.params = params;
            this.baseOffset = baseOffset;
            Objects.requireNonNull(source, "Source is null");
            this.source = new OffsetInputStream(source);
            this.source.mark(1024);
            this.in = new DataInputStream(this.source);
            this.bytesSingleWidth = String.format(params.bytesFormat, 255).length();
            this.bytesColWidth = params.bytesCount * this.bytesSingleWidth;
            this.annoWidth = params.annoWidth;
            this.dest = dest;
        }

        @Override
        public void write(int c) {
            super.write(c);
            this.checkFlush();
        }

        @Override
        public void write(char[] c, int off, int len) {
            super.write(c, off, len);
            for (int i = 0; i < len; ++i) {
                if (c[off + i] != '\n') continue;
                this.process();
                return;
            }
            this.checkFlush();
        }

        @Override
        public void write(String str, int off, int len) {
            super.write(str, off, len);
            if (str.indexOf(10) >= 0) {
                this.process();
            } else {
                this.checkFlush();
            }
        }

        private void checkFlush() {
            if (this.size() > this.annoWidth) {
                this.process();
            }
        }

        @Override
        public void flush() {
            try {
                while (true) {
                    if (this.source.markedByteCount() >= this.params.bytesCount) {
                        this.process();
                    }
                    this.params.annoFormatter.annotate(this.in, this);
                    if (this.source.markedByteCount() <= 256) continue;
                    this.process();
                }
            }
            catch (IOException ioe) {
                this.process();
                if (!(ioe instanceof EOFException)) {
                    throw new UncheckedIOException(ioe);
                }
            }
            catch (UncheckedIOException uio) {
                this.process();
                throw uio;
            }
        }

        private void process() {
            String info = this.toString();
            this.reset();
            int count = this.source.markedByteCount();
            try {
                this.source.reset();
                int binColOffset = (int)this.source.byteOffset();
                while (count > 0 || info.length() > 0) {
                    int offset = binColOffset + this.baseOffset;
                    this.dest.append(String.format(this.params.offsetFormat, offset));
                    int colOffset = offset % this.params.bytesCount;
                    int colWidth = colOffset * this.bytesSingleWidth;
                    this.dest.append(" ".repeat(colWidth));
                    int byteCount = Math.min(this.params.bytesCount - colOffset, count);
                    for (int i = 0; i < byteCount; ++i) {
                        int b = this.source.read();
                        if (b == -1) {
                            throw new IllegalStateException("BUG");
                        }
                        String s = String.format(this.params.bytesFormat, b);
                        colWidth += s.length();
                        this.dest.append(s);
                    }
                    binColOffset += byteCount;
                    count -= byteCount;
                    this.dest.append(" ".repeat(Math.max(0, this.bytesColWidth - colWidth)));
                    this.dest.append(this.params.annoDelim);
                    if (info.length() > 0) {
                        int nl = info.indexOf(10);
                        if (nl < 0) {
                            this.dest.append(info);
                            info = "";
                        } else {
                            this.dest.append(info, 0, nl > 0 && info.charAt(nl - 1) == '\r' ? nl - 1 : nl);
                            info = info.substring(nl + 1);
                        }
                    }
                    this.dest.append(this.params.lineSeparator);
                }
            }
            catch (IOException ioe) {
                try {
                    this.dest.append("\nIOException during annotations: ").append(ioe.getMessage()).append("\n");
                }
                catch (IOException iOException) {
                    // empty catch block
                }
            }
            this.source.mark(1024);
        }
    }

    private static class PrimitiveFormatter
    implements Formatter {
        private final Class<?> primClass;
        private final String fmtString;

        PrimitiveFormatter(Class<?> primClass, String fmtString) {
            Objects.requireNonNull(primClass, "primClass");
            Objects.requireNonNull(fmtString, "fmtString");
            if (!primClass.isPrimitive()) {
                throw new IllegalArgumentException("Not a primitive type: " + primClass.getName());
            }
            this.primClass = primClass;
            this.fmtString = fmtString;
        }

        @Override
        public void annotate(DataInputStream in, Appendable out) throws IOException {
            if (this.primClass == Byte.TYPE) {
                byte v = in.readByte();
                out.append(String.format(this.fmtString, v));
            } else if (this.primClass == Boolean.TYPE) {
                boolean v = in.readByte() != 0;
                out.append(String.format(this.fmtString, v));
            } else if (this.primClass == Short.TYPE | this.primClass == Character.TYPE) {
                short v = in.readShort();
                out.append(String.format(this.fmtString, v));
            } else if (this.primClass == Float.TYPE) {
                float v = in.readFloat();
                out.append(String.format(this.fmtString, Float.valueOf(v)));
            } else if (this.primClass == Integer.TYPE) {
                int v = in.readInt();
                out.append(String.format(this.fmtString, v));
            } else if (this.primClass == Double.TYPE) {
                double v = in.readDouble();
                out.append(String.format(this.fmtString, v));
            } else if (this.primClass == Long.TYPE) {
                long v = in.readLong();
                out.append(String.format(this.fmtString, v));
            } else {
                throw new AssertionError((Object)"missing case on primitive class");
            }
        }

        public String toString() {
            return "(" + this.primClass.getName() + ", \"" + this.fmtString + "\")";
        }
    }

    private static final class OffsetInputStream
    extends BufferedInputStream {
        private long byteOffset = 0L;
        private long markByteOffset = 0L;

        OffsetInputStream(InputStream in) {
            super(in);
        }

        long byteOffset() {
            return this.byteOffset;
        }

        @Override
        public void reset() throws IOException {
            super.reset();
            this.byteOffset = this.markByteOffset;
        }

        @Override
        public synchronized void mark(int readlimit) {
            super.mark(readlimit);
            this.markByteOffset = this.byteOffset;
        }

        int markedByteCount() {
            if (this.markpos < 0) {
                return 0;
            }
            return this.pos - this.markpos;
        }

        @Override
        public int read() throws IOException {
            int b = super.read();
            if (b >= 0) {
                ++this.byteOffset;
            }
            return b;
        }

        @Override
        public long skip(long n) throws IOException {
            long size = super.skip(n);
            this.byteOffset += size;
            return size;
        }

        @Override
        public int read(byte[] b) throws IOException {
            int size = super.read(b);
            this.byteOffset += (long)Math.max(size, 0);
            return size;
        }

        @Override
        public int read(byte[] b, int off, int len) throws IOException {
            int size = super.read(b, off, len);
            this.byteOffset += (long)Math.max(size, 0);
            return size;
        }
    }
}

