﻿using System;
using System.IO;
using System.Runtime.InteropServices;
using System.Drawing;
using System.Drawing.Imaging;
using System.Windows.Forms;
using NaGet.Packages;
using NaGet.Packages.Install;

namespace AppliStation.Util
{
	/// <summary>
	/// GUIに関する雑多な関数
	/// </summary>
	public sealed class GUIUtils
	{
		#region アイコン関連
		
		#region ExtraIcon関連
		
		/// <summary>
		/// アイコンファイル（実行ファイル・DLL）を開いてアイコンを作る
		/// </summary>
		/// <param name="form">ハンドラ</param>
		/// <param name="lpszExeFileNameAndIndex">対象ファイルとアイコンインデックスの文字列表現</param>
		/// <returns>生成されたアイコン。</returns>
		public static Icon ExtractIcon(Form form, string lpszExeFileNameAndIndex)
		{
			int index = lpszExeFileNameAndIndex.LastIndexOf(',');
			if (index >= 0) {
				uint nIconIndex = uint.Parse(lpszExeFileNameAndIndex.Substring(index+1));
				return NativeMethods.ExtractIcon(form, lpszExeFileNameAndIndex.Substring(0, index), nIconIndex);
			} else {
				return Icon.ExtractAssociatedIcon(lpszExeFileNameAndIndex);
			}
		}
		
		/// <summary>
		/// シェルからフォルダーアイコンを生成して返す
		/// </summary>
		/// <returns>フォルダーアイコン</returns>
		public static Icon ShellIconForFolder
		{
			get {
				// Vista以降ならば、SHGetStockIconInfo(SIID_FOLDER, SHGSI_ICON, &sInfo); をP/Invoke呼び出しするのが王道かと
				string windir = Environment.GetEnvironmentVariable("windir");
				return NativeMethods.ExtractIcon(null, Path.Combine(windir, @"system32\shell32.dll"), 3);
			}
		}
		
		#endregion
		
		/// <summary>
		/// パッケージに対応するアイコンを返す
		/// </summary>
		/// <param name="pkg">パッケージ</param>
		/// <returns>対応するアイコン。検出できなかった場合はnull。</returns>
		public static Icon GetIconForPackage(InstalledPackage pkg)
		{
			Icon ico = null;
			try {
				string iconPath = pkg.UninstallInfo.IconPath;
				if (! string.IsNullOrEmpty(iconPath)) {
					ico = ExtractIcon(null, iconPath);
				} else if ((pkg.Type == InstallerType.ARCHIVE)
				           ||(pkg.Type == InstallerType.ITSELF)) {
					string progGrp = Path.Combine(NaGet.Env.ArchiveProgramGroup, pkg.Name);
					if (Directory.Exists(progGrp)) {
						string[] lnkFiles = Directory.GetFiles(progGrp, "*.lnk");
						
						if (lnkFiles.Length >= 1) {
							using (NaGet.InteropServices.ShellLink link = new NaGet.InteropServices.ShellLink(lnkFiles[0])) {
								if (File.Exists(link.GetPath(0))) {
									ico = Icon.ExtractAssociatedIcon(link.GetPath(0));
								}
							}
						}
					}
				}
			} catch (System.Runtime.InteropServices.COMException) {
				// ShellLinkのオープンあるいは、リンク先解決に失敗した場合
			} catch (Exception) {
			}
			return ico;
		}
		
		/// <summary>
		/// グレーアウトアイコンを作るための ImageAttributes を作って返す。
		/// </summary>
		/// <param name="alpha">透明度。(1.0が不透明、0.0が完全透明)</param>
		/// <returns>生成されたImageAttributes</returns>
		public static ImageAttributes GetImageAttributeToGrayOut(float alpha)
		{
			// RGB比率として、YIQカラーモデルの値を採用する
			const float r = 0.298912f;
			const float g = 0.586611f;
			const float b = 0.114478f;
			
			ColorMatrix cm = new ColorMatrix(new float[][]{
		         	new float[]{r, r, r,     0, 0},
		         	new float[]{g, g, g,     0, 0},
		         	new float[]{b, b, b,     0, 0},
		         	new float[]{0, 0, 0, alpha, 0},
		         	new float[]{0, 0, 0,     0, 1},
		         });
			ImageAttributes ia = new ImageAttributes();
			ia.SetColorMatrix(cm);
			
			return ia;
		}
		
		/// <summary>
		/// 画像を指定領域の真中に描く
		/// </summary>
		/// <param name="g">描画対象のグラフィックス</param>
		/// <param name="img">画像</param>
		/// <param name="b">指定領域</param>
		/// <param name="ia">ImageAttributes。nullでもかまわない</param>
		public static void Graphics_DrawCenterImage(Graphics g, Image img, Rectangle b, ImageAttributes ia)
		{
			int x = b.Left + (b.Width  - img.Width ) / 2;
			int y = b.Top  + (b.Height - img.Height) / 2;
			
			g.DrawImage(img,
			            new Rectangle(x, y, img.Width, img.Height),
			            0, 0, img.Width, img.Height,
			            GraphicsUnit.Pixel, ia);
		}
		
		#endregion
		
		#region CheckedListBox関連
		
		/// <summary>
		/// チェックリストボックスのアイテムをスワップする
		/// </summary>
		/// <param name="checkedListBox">操作対象のCheckedListBox</param>
		/// <param name="indexA">アイテムインデックス</param>
		/// <param name="indexB">アイテムインデックス</param>
		/// <remarks>インデックス値のチェックは本メソッドでは行っていない</remarks>
		public static void CheckedListBox_SwapItems(CheckedListBox checkedListBox, int indexA, int indexB) {
			int itemCount = checkedListBox.Items.Count;
			object tempItem		 = checkedListBox.Items[indexA];
			CheckState tempState = checkedListBox.GetItemCheckState(indexA);
			
			checkedListBox.Items[indexA] = checkedListBox.Items[indexB];
			checkedListBox.SetItemCheckState(indexA, checkedListBox.GetItemCheckState(indexB));
			
			checkedListBox.Items[indexB] = tempItem;
			checkedListBox.SetItemCheckState(indexB, tempState);
		}
		
		#endregion
	}
}
