//-----------------------------------------------------------------------------
// AScript re module
//-----------------------------------------------------------------------------
#include "Module.h"
#include "Object_File.h"

AScript_BeginModule(csv)

#define DEFAULT_FORMAT "%g"

AScript_DeclarePrivSymbol(format)

static const char *GetFormat(Environment &env);

static void PutValue(Environment &env, Signal sig, File &file,
									const char *format, const Value &value);

//-----------------------------------------------------------------------------
// Reader
//-----------------------------------------------------------------------------
class DLLEXPORT Reader {
public:
	bool ReadLine(Environment &env, Signal sig, ValueList &valList);
	virtual char NextChar() = 0;
};

bool Reader::ReadLine(Environment &env, Signal sig, ValueList &valList)
{
	enum {
		STAT_LineTop, STAT_FieldTop, STAT_Field, STAT_Quoted, STAT_QuotedEnd,
	} stat = STAT_LineTop;
	String field;
	char ch = '\0';
	bool rtn = true;
	bool eatNextChar = true;
	for (;;) {
		if (eatNextChar) {
			while ((ch = NextChar()) == '\r') ;
		}
		eatNextChar = true;
		if (stat == STAT_LineTop) {
			if (ch == '\0') {
				rtn = false;
				break;
			}
			eatNextChar = false;
			stat = STAT_FieldTop;
		} else if (stat == STAT_FieldTop) {
			field.clear();
			if (ch == '"') {
				stat = STAT_Quoted;
			} else if (ch == '\n' || ch == '\0') {
				valList.push_back(Value(env, field.c_str()));
				break;
			} else {
				eatNextChar = false;
				stat = STAT_Field;
			}
		} else if (stat == STAT_Field) {
			if (ch == ',') {
				valList.push_back(Value(env, field.c_str()));
				stat = STAT_FieldTop;
			} else if (ch == '\n' || ch == '\0') {
				valList.push_back(Value(env, field.c_str()));
				break;
			} else {
				field.push_back(ch);
			}
		} else if (stat == STAT_Quoted) {
			if (ch == '"') {
				stat = STAT_QuotedEnd;
			} else if (ch == '\0') {
				valList.push_back(Value(env, field.c_str()));
				break;
			} else {
				field.push_back(ch);
			}
		} else if (stat == STAT_QuotedEnd) {
			if (ch == '"') {
				field.push_back(ch);
				stat = STAT_Quoted;
			} else if (ch == '\0') {
				valList.push_back(Value(env, field.c_str()));
				break;
			} else {
				eatNextChar = false;
				stat = STAT_Field;
			}
		}
	}
	return rtn;
}

//-----------------------------------------------------------------------------
// ReaderFile
//-----------------------------------------------------------------------------
class DLLEXPORT ReaderFile : public Reader {
private:
	File &_file;
public:
	ReaderFile(File &file) : _file(file) {}
	virtual char NextChar();
};

char ReaderFile::NextChar()
{
	int ch = _file.GetChar();
	return (ch < 0)? '\0' : static_cast<char>(static_cast<unsigned char>(ch));
}

//-----------------------------------------------------------------------------
// ReaderString
//-----------------------------------------------------------------------------
class DLLEXPORT ReaderString : public Reader {
private:
	const char *_strp;
public:
	ReaderString(const char *str) : _strp(str) {}
	virtual char NextChar();
};

char ReaderString::NextChar()
{
	char ch = *_strp;
	if (ch != '\0') _strp++;
	return ch;
}

//-----------------------------------------------------------------------------
// AScript module functions: csv
//-----------------------------------------------------------------------------
// csv.split(str:string):map
AScript_DeclareFunction(split)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "str", VTYPE_String);
}

AScript_ImplementFunction(split)
{
	Value result;
	ValueList &valList = result.InitAsList(env);
	ReaderString reader(context.GetString(0));
	if (!reader.ReadLine(env, sig, valList) || sig.IsSignalled()) return Value::Null;
	return result;
}

// csv.read(file:file)
AScript_DeclareFunction(read)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "file", VTYPE_File);
}

AScript_ImplementFunction(read)
{
	const Value valueEmpty(env, "");
	Value result;
	ValueList &valListCol = result.InitAsList(env);
	ValueList valListFields;
	size_t nLine = 0;
	ReaderFile reader(context.GetFile(0));
	while (reader.ReadLine(env, sig, valListFields)) {
		ValueList::iterator pValueCol = valListCol.begin();
		ValueList::iterator pValueColEnd = valListCol.end();
		foreach_const (ValueList, pValueField, valListFields) {
			if (pValueCol == pValueColEnd) {
				Value valueCol;
				ValueList &valList = valueCol.InitAsList(env, nLine, valueEmpty);
				valListCol.push_back(valueCol);
				valList.push_back(*pValueField);
			} else {
				pValueCol->GetList().push_back(*pValueField);
				pValueCol++;
			}
		}
		for ( ; pValueCol != pValueColEnd; pValueCol++) {
			pValueCol->GetList().push_back(valueEmpty);
		}
		nLine++;
		valListFields.clear();
	}
	return result;
}

// csv.readline(file:file)
AScript_DeclareFunction(readline)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "file", VTYPE_File);
}

AScript_ImplementFunction(readline)
{
	Value result;
	ValueList &valList = result.InitAsList(env);
	ReaderFile reader(context.GetFile(0));
	if (!reader.ReadLine(env, sig, valList) || sig.IsSignalled()) return Value::Null;
	return result;
}

// csv.readlines(file:file) {block?}
class Iterator_readlines : public Iterator {
private:
	Object_File *_pObj;
	ReaderFile _reader;
public:
	inline Iterator_readlines(Object_File *pObj) :
			Iterator(false), _pObj(pObj), _reader(pObj->GetFile()) {}
	virtual ~Iterator_readlines();
	virtual bool DoNext(Signal sig, Value &value);
	virtual String ToString(Signal sig) const;
};

Iterator_readlines::~Iterator_readlines()
{
	Object::Delete(_pObj);
}

bool Iterator_readlines::DoNext(Signal sig, Value &value)
{
	ValueList &valList = value.InitAsList(*_pObj);
	return _reader.ReadLine(*_pObj, sig, valList);
}

String Iterator_readlines::ToString(Signal sig) const
{
	return String("<iterator:csv.readlines>");
}

AScript_DeclareFunction(readlines)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "file", VTYPE_File);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementFunction(readlines)
{
	Object_File *pObjFile = context.GetFileObj(0);
	Iterator *pIterator =
		new Iterator_readlines(dynamic_cast<Object_File *>(pObjFile->IncRef()));
	return ReturnIterator(env, sig, context, pIterator);
}

// csv.write(file:file, elems[]+)
AScript_DeclareFunction(write)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "file", VTYPE_File);
	DeclareArg(env, "elems", VTYPE_Any, OCCUR_OnceOrMore, true);
}

AScript_ImplementFunction(write)
{
	File &file = context.GetFile(0);
	const ValueList &valList = context.GetList(1);
	if (valList.empty()) return Value::Null;
	const char *format = GetFormat(GetEnvScope());
	IteratorOwner iteratorOwner;
	foreach_const (ValueList, pValue, valList) {
		Iterator *pIterator = pValue->CreateIterator(sig);
		if (pIterator == NULL) return Value::Null;
		iteratorOwner.push_back(pIterator);
	}
	for (;;) {
		bool doneFlag = true;
		ValueList valList;
		foreach_const (IteratorOwner, ppIterator, iteratorOwner) {
			Iterator *pIterator = *ppIterator;
			Value value;
			if (pIterator->Next(sig, value)) {
				if (sig.IsSignalled()) return Value::Null;
				doneFlag = false;
				valList.push_back(value);
			} else {
				valList.push_back(Value::Null);
			}
		}
		if (doneFlag) break;
		foreach (ValueList, pValue, valList) {
			if (pValue != valList.begin()) file.PutChar(',');
			if (pValue->IsValid()) {
				PutValue(env, sig, file, format, *pValue);
				if (sig.IsSignalled()) break;
			}
		}
		file.PutChar('\n');
	}
	return Value::Null;
}

// csv.writeline(file:file, elem[])
AScript_DeclareFunction(writeline)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "file", VTYPE_File);
	DeclareArg(env, "elem", VTYPE_Any, OCCUR_Once, true);
}

AScript_ImplementFunction(writeline)
{
	File &file = context.GetFile(0);
	const char *format = GetFormat(GetEnvScope());
	const ValueList &valList = context.GetList(1);
	foreach_const (ValueList, pValue, valList) {
		if (pValue != valList.begin()) file.PutChar(',');
		PutValue(env, sig, file, format, *pValue);
	}
	file.PutChar('\n');
	return Value::Null;
}

// csv.writelines(file:file, iter:iterator)
AScript_DeclareFunction(writelines)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "file", VTYPE_File);
	DeclareArg(env, "iter", VTYPE_Iterator);
}

AScript_ImplementFunction(writelines)
{
	File &file = context.GetFile(0);
	Iterator *pIterator = context.GetIterator(1);
	const char *format = GetFormat(GetEnvScope());
	Value value;
	while (pIterator->Next(sig, value)) {
		if (!value.IsList()) {
			sig.SetError(ERR_ValueError, "each element must be a list");
			return Value::Null;
		}
		const ValueList &valList = value.GetList();
		foreach_const (ValueList, pValue, valList) {
			if (pValue != valList.begin()) file.PutChar(',');
			if (pValue->IsList()) {
				const ValueList &valListSub = pValue->GetList();
				foreach_const (ValueList, pValueSub, valListSub) {
					if (pValueSub != valListSub.begin()) file.PutChar(',');
					PutValue(env, sig, file, format, *pValueSub);
					if (sig.IsSignalled()) return Value::Null;
				}
			} else {
				PutValue(env, sig, file, format, *pValue);
				if (sig.IsSignalled()) return Value::Null;
			}
		}
		file.PutChar('\n');
	}
	if (sig.IsSignalled()) return Value::Null;
	return Value::Null;
}

// Module entry
AScript_ModuleEntry()
{
	AScript_RealizePrivSymbol(format);
	AScript_AssignFunction(split);
	AScript_AssignFunction(read);
	AScript_AssignFunction(readline);
	AScript_AssignFunction(readlines);
	AScript_AssignFunction(write);
	AScript_AssignFunction(writeline);
	AScript_AssignFunction(writelines);
	AScript_AssignValue(format, Value(env, DEFAULT_FORMAT));
}

AScript_ModuleTerminate()
{
}

//-----------------------------------------------------------------------------
// Utilities
//-----------------------------------------------------------------------------
static const char *GetFormat(Environment &env)
{
	const Value *pValue = env.LookupValue(AScript_PrivSymbol(format), false);
	return (pValue != NULL && pValue->IsString())?
									pValue->GetString() : DEFAULT_FORMAT;
}

static void PutValue(Environment &env, Signal sig, File &file,
										const char *format, const Value &value)
{
	if (value.IsInvalid()) {
		// nothing to do
	} else if (value.IsNumber()) {
		Value result(env, Formatter::Format(sig, format, ValueList(value)).c_str());
		if (sig.IsSignalled()) return;
		file.Print(result.ToString(sig, false).c_str());
	} else if (value.IsComplex()) {
		Value result(env, Formatter::Format(sig, format, ValueList(value)).c_str());
		if (sig.IsSignalled()) return;
		file.Print(result.ToString(sig, false).c_str());
	} else if (value.IsString()) {
		file.PutChar('"');
		for (const char *p = value.GetString(); *p != '\0'; p++) {
			char ch = *p;
			file.PutChar(ch);
			if (ch == '"') file.PutChar(ch);
		}
		file.PutChar('"');
	} else {
		sig.SetError(ERR_TypeError, "can't output in CSV format");
	}
}

AScript_EndModule(csv)

AScript_RegisterModule(csv)
