//
// Object_Dict
//
#include "Object_Dict.h"
#include "Expr.h"

namespace AScript {

//-----------------------------------------------------------------------------
// Object_Dict
//-----------------------------------------------------------------------------
bool Object_Dict::IsDict() const { return true; }

Object *Object_Dict::Clone() const
{
	return new Object_Dict(*this);
}

Value Object_Dict::GetByIndex(Signal sig, const Value &valueIdx)
{
	const Value *pValue = Find(sig, valueIdx);
	if (sig.IsSignalled()) {
		return Value::Null;
	} else if (pValue == NULL) {
		SetError_KeyNotFound(sig, valueIdx);
		return Value::Null;
	}
	return *pValue;
}

void Object_Dict::SetByIndex(Signal sig, const Value &valueIdx, const Value &value)
{
	if (!ValueDict::IsValidKey(valueIdx)) {
		sig.SetError(ERR_KeyError, "invalid value type for key");
		return;
	}
	_valDict[valueIdx] = value;
}

Iterator *Object_Dict::CreateIterator(Signal sig)
{
	return new IteratorItems(dynamic_cast<Object_Dict *>(IncRef()));
}

String Object_Dict::ToString(Signal sig, bool exprFlag)
{
	String str;
	str += "%{";
	foreach_const (ValueDict, iter, _valDict) {
		if (iter != _valDict.begin()) str += ", ";
		const Value &value = iter->first;
		if (value.IsString()) {
			str += '"';
			str += value.GetString();
			str += '"';
		} else {
			str += value.ToString(sig, false);
		}
		str += " => ";
		str += iter->second.ToString(sig);
	}
	str += "}";
	return str;
}

const Value *Object_Dict::Find(Signal sig, const Value &valueIdx) const
{
	if (!ValueDict::IsValidKey(valueIdx)) {
		sig.SetError(ERR_KeyError, "invalid value type for key");
		return NULL;
	}
	ValueDict::const_iterator iter = _valDict.find(valueIdx);
	return (iter == _valDict.end())? NULL : &iter->second;
}

void Object_Dict::SetError_KeyNotFound(Signal sig, const Value &valueIdx)
{
	sig.SetError(ERR_KeyError, "dictionary doesn't have a key '%s'",
										valueIdx.ToString(sig).c_str());
}

//-----------------------------------------------------------------------------
// Object_Dict::IteratorKeys
//-----------------------------------------------------------------------------
Object_Dict::IteratorKeys::IteratorKeys(Object_Dict *pObj) :
				Iterator(false), _pObj(pObj), _pCur(pObj->GetDict().begin())
{
}

Object_Dict::IteratorKeys::~IteratorKeys()
{
	Object::Delete(_pObj);
}

bool Object_Dict::IteratorKeys::DoNext(Signal sig, Value &value)
{
	if (_pCur == _pObj->GetDict().end()) return false;
	value = _pCur->first;
	_pCur++;
	return true;
}

String Object_Dict::IteratorKeys::ToString(Signal sig) const
{
	return String("<iterator:dict#keys>");
}

//-----------------------------------------------------------------------------
// Object_Dict::IteratorValues
//-----------------------------------------------------------------------------
Object_Dict::IteratorValues::IteratorValues(Object_Dict *pObj) :
				Iterator(false), _pObj(pObj), _pCur(pObj->GetDict().begin())
{
}

Object_Dict::IteratorValues::~IteratorValues()
{
	Object::Delete(_pObj);
}

bool Object_Dict::IteratorValues::DoNext(Signal sig, Value &value)
{
	if (_pCur == _pObj->GetDict().end()) return false;
	value = _pCur->second;
	_pCur++;
	return true;
}

String Object_Dict::IteratorValues::ToString(Signal sig) const
{
	return String("<iterator:dict#values>");
}

//-----------------------------------------------------------------------------
// Object_Dict::IteratorItems
//-----------------------------------------------------------------------------
Object_Dict::IteratorItems::IteratorItems(Object_Dict *pObj) :
				Iterator(false), _pObj(pObj), _pCur(pObj->GetDict().begin())
{
}

Object_Dict::IteratorItems::~IteratorItems()
{
	Object::Delete(_pObj);
}

bool Object_Dict::IteratorItems::DoNext(Signal sig, Value &value)
{
	if (_pCur == _pObj->GetDict().end()) return false;
	ValueList &valList = value.InitAsList(*_pObj);
	valList.push_back(_pCur->first);
	valList.push_back(_pCur->second);
	_pCur++;
	return true;
}

String Object_Dict::IteratorItems::ToString(Signal sig) const
{
	return String("<iterator:dict#items>");
}

//-----------------------------------------------------------------------------
// Object_Dict::IteratorGet
//-----------------------------------------------------------------------------
Object_Dict::IteratorGet::IteratorGet(Object_Dict *pObj, Iterator *pIteratorKey,
					const Value &valDefault, bool raiseFlag, bool setDefaultFlag) :
	Iterator(pIteratorKey->IsInfinite()), _pObj(pObj), _pIteratorKey(pIteratorKey),
	_valDefault(valDefault), _raiseFlag(raiseFlag), _setDefaultFlag(setDefaultFlag)
{
}

Object_Dict::IteratorGet::~IteratorGet()
{
	Object::Delete(_pObj);
	Iterator::Delete(_pIteratorKey);
}

bool Object_Dict::IteratorGet::DoNext(Signal sig, Value &value)
{
	Value valueIdx;
	if (!_pIteratorKey->Next(sig, valueIdx)) return false;
	const Value *pValue = _pObj->Find(sig, valueIdx);
	if (pValue != NULL) {
		value = *pValue;
	} else if (_raiseFlag) {
		Object_Dict::SetError_KeyNotFound(sig, valueIdx);
		return false;
	} else {
		value = _valDefault;
		if (_setDefaultFlag) _pObj->SetByIndex(sig, valueIdx, value);
		if (sig.IsSignalled()) return false;
	}
	return true;
}

String Object_Dict::IteratorGet::ToString(Signal sig) const
{
	return String("<iterator:dict#get>");
}

//-----------------------------------------------------------------------------
// AScript Interface
//-----------------------------------------------------------------------------
// dict#len()
AScript_DeclareMethod(Dict, len)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
}

AScript_ImplementMethod(Dict, len)
{
	Object_Dict *pSelf = Object_Dict::GetSelfObj(context);
	return Value(static_cast<Number>(pSelf->GetDict().size()));
}

// dict#get(key, default?):[raise]
AScript_DeclareMethod(Dict, get)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "key", VTYPE_Any);
	DeclareArg(env, "default", VTYPE_Any, OCCUR_ZeroOrOnce);
	DeclareAttr(AScript_Symbol(raise));
}

AScript_ImplementMethod(Dict, get)
{
	Object_Dict *pSelf = Object_Dict::GetSelfObj(context);
	const Value &valueArg = context.GetValue(0);
	bool raiseFlag = context.IsSet(AScript_Symbol(raise));
	if (valueArg.IsList() || valueArg.IsIterator()) {
		Iterator *pIteratorKey = valueArg.CreateIterator(sig);
		if (sig.IsSignalled()) return Value::Null;
		Iterator *pIterator = new Object_Dict::IteratorGet(
						dynamic_cast<Object_Dict *>(pSelf->IncRef()),
						pIteratorKey, context.GetValue(1), raiseFlag, false);
		return ReturnIterator(env, sig, context, pIterator);
	} else {
		const Value &valueIdx = valueArg;
		const Value *pValue = pSelf->Find(sig, valueIdx);
		if (pValue != NULL) {
			return *pValue;
		} else if (raiseFlag) {
			Object_Dict::SetError_KeyNotFound(sig, valueIdx);
			return Value::Null;
		} else {
			const Value &value = context.GetValue(1);
			return value;
		}
	}
}

// dict#setdefault(key, default)
AScript_DeclareMethod(Dict, setdefault)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "key", VTYPE_Any);
	DeclareArg(env, "default", VTYPE_Any);
}

AScript_ImplementMethod(Dict, setdefault)
{
	Object_Dict *pSelf = Object_Dict::GetSelfObj(context);
	const Value &valueArg = context.GetValue(0);
	if (valueArg.IsList() || valueArg.IsIterator()) {
		Iterator *pIteratorKey = valueArg.CreateIterator(sig);
		if (sig.IsSignalled()) return Value::Null;
		Iterator *pIterator = new Object_Dict::IteratorGet(
						dynamic_cast<Object_Dict *>(pSelf->IncRef()),
						pIteratorKey, context.GetValue(1), false, true);
		return ReturnIterator(env, sig, context, pIterator);
	} else {
		const Value &valueIdx = valueArg;
		const Value *pValue = pSelf->Find(sig, valueIdx);
		if (pValue != NULL) {
			return *pValue;
		} else {
			const Value &value = context.GetValue(1);
			pSelf->SetByIndex(sig, valueIdx, value);
			if (sig.IsSignalled()) return Value::Null;
			return value;
		}
	}
}

// dict#haskey(key):map
AScript_DeclareMethod(Dict, haskey)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "key", VTYPE_Any);
}

AScript_ImplementMethod(Dict, haskey)
{
	Object_Dict *pSelf = Object_Dict::GetSelfObj(context);
	const Value &valueIdx = context.GetValue(0);
	const Value *pValue = pSelf->Find(sig, valueIdx);
	return Value(pValue != NULL);
}

// dict#keys() {block?}
AScript_DeclareMethod(Dict, keys)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(Dict, keys)
{
	Object_Dict *pSelf = Object_Dict::GetSelfObj(context);
	Object_Dict *pObj = dynamic_cast<Object_Dict *>(pSelf->IncRef());
	return ReturnIterator(env, sig, context,
							new Object_Dict::IteratorKeys(pObj));
}

// dict#values() {block?}
AScript_DeclareMethod(Dict, values)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(Dict, values)
{
	Object_Dict *pSelf = Object_Dict::GetSelfObj(context);
	Object_Dict *pObj = dynamic_cast<Object_Dict *>(pSelf->IncRef());
	return ReturnIterator(env, sig, context,
							new Object_Dict::IteratorValues(pObj));
}

// dict#items() {block?}
AScript_DeclareMethod(Dict, items)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(Dict, items)
{
	Object_Dict *pSelf = Object_Dict::GetSelfObj(context);
	Object_Dict *pObj = dynamic_cast<Object_Dict *>(pSelf->IncRef());
	return ReturnIterator(env, sig, context,
							new Object_Dict::IteratorItems(pObj));
}

// dict#clear!()
AScript_DeclareMethod(Dict, clear_X)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
}

AScript_ImplementMethod(Dict, clear_X)
{
	Object_Dict *pSelf = Object_Dict::GetSelfObj(context);
	pSelf->GetDict().clear();
	return Value::Null;
}

// dict#add!(elem[])
AScript_DeclareMethod(Dict, add_X)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "elem", VTYPE_Any, OCCUR_Once, true);
}

AScript_ImplementMethod(Dict, add_X)
{
	Object_Dict *pSelf = Object_Dict::GetSelfObj(context);
	if (!pSelf->GetDict().Store(sig, context.GetList(0))) return Value::Null;
	return Value::Null;
}

// dict#erase!(key):map
AScript_DeclareMethod(Dict, erase_X)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "key", VTYPE_Any);
}

AScript_ImplementMethod(Dict, erase_X)
{
	Object_Dict *pSelf = Object_Dict::GetSelfObj(context);
	pSelf->GetDict().erase(context.GetValue(0));
	return Value::Null;
}

// Assignment
Class_Dict::Class_Dict(Environment &env) : Class(env.LookupClass(VTYPE_Object))
{
	AScript_AssignMethod(Dict, len);
	AScript_AssignMethod(Dict, get);
	AScript_AssignMethod(Dict, setdefault);
	AScript_AssignMethod(Dict, haskey);
	AScript_AssignMethod(Dict, keys);
	AScript_AssignMethod(Dict, values);
	AScript_AssignMethod(Dict, items);
	AScript_AssignMethodEx(Dict, clear_X, "clear!");
	AScript_AssignMethodEx(Dict, add_X, "add!");
	AScript_AssignMethodEx(Dict, erase_X, "erase!");
}

Object *Class_Dict::CreateDescendant(Environment &env, Signal sig, Class *pClass)
{
	return new Object_Dict((pClass == NULL)? this : pClass);
}

}
