/*
 * Copyright (c) 2006, team-naver.com
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.aibonware.viewnaver.component;

import java.util.*;
import java.awt.*;

@SuppressWarnings("serial")
public class PageLayout implements LayoutManager {
	private int minTabWidth = 10;

	public static final String GLUE = "GLUE";

	private static interface SizeAccessor {
		public Dimension get(Component c);
	}

	private SizeAccessor minSize = new SizeAccessor() {
		public Dimension get(Component c) {
			return c.getMinimumSize();
		}
	};
	
	private SizeAccessor prefSize = new SizeAccessor() {
		public Dimension get(Component c) {
			return c.getPreferredSize();
		}
	};

/*
	private SizeAccessor maxSize = new SizeAccessor() {
		public Dimension get(Component c) {
			return c.getMaximumSize();
		}
	};
*/
	
	static class NewLine extends Canvas {
		public final int nextLineDistance;

		public NewLine(int nextLineDistance) {
			this.nextLineDistance = nextLineDistance;
			
			setMinimumSize(new Dimension(1,1));
			setPreferredSize(new Dimension(1,1));
			setMaximumSize(new Dimension(1,1));
		}
	}

	static class Tab extends Canvas {
		public Tab() {
			setMinimumSize(new Dimension(1,1));
			setPreferredSize(new Dimension(1,1));
			setMaximumSize(new Dimension(1,1));
		}
	}

	private class Sentence extends Vector<Component> {
		public Line parent = null;
		public int width = 0;
	}

	private class Line extends Vector<Sentence> {
		public Page parent = null;
		public int nextLineDistance = 0;
		
		@Override public void addElement(Sentence sentence) {
			sentence.parent = this;
			super.addElement(sentence);
		}
	}

	private class Page extends Vector<Line> {
		public int pageWidth = 0;
		Component[] glues;
		
		public Component findGlue(Line line) {
			for(Sentence sentence: line) {
				for(Component comp: sentence) {
					for(Component glue: glues) {
						if(comp == glue) return glue;
					}
				}
			}
			
			return null;
		}
		
		@Override public void addElement(Line line) {
			line.parent = this;
			super.addElement(line);
		}

		public Page(Vector<Component> glues) {
			this.glues = new Component[glues.size()];
			glues.toArray(this.glues);
		}

		public void resetSentenceSize(SizeAccessor sizeAccessor) {
			int sentenceNum = 0;
			
			for(Line line: this) {
				sentenceNum = Math.max(sentenceNum, line.size());
				
				for(Sentence sentence: line) {
					sentence.width = 0;
					
					for(Component comp: sentence) {
						sentence.width += sizeAccessor.get(comp).width;
					}
				}
			}

			int[] widthList = new int[sentenceNum];

			for(Line line: this) {
				for(int i=0; i<line.size(); i++) {
					widthList[i] = Math.max(widthList[i], line.elementAt(i).width);
				}
			}

			for(Line line: this) {
				for(int i=0; i<line.size(); i++) {
					line.elementAt(i).width = widthList[i];
				}
			}

			pageWidth = 0;
			for(int width: widthList) pageWidth += width;

			pageWidth += (sentenceNum - 1) * minTabWidth;
			if(pageWidth <= 0) pageWidth = 1;
		}
	}

	Page createPage(Component[] comps, Vector<Component> glues) {
		Page page = new Page(glues);
		Line line = new Line();
		Sentence sentence = new Sentence();

		for(Component comp: comps) {
			if(comp instanceof Tab) {
				line.addElement(sentence);
				sentence = new Sentence();
			} else if(comp instanceof NewLine) {
				line.nextLineDistance = ((NewLine)comp).nextLineDistance;
				line.addElement(sentence);
				sentence = new Sentence();

				page.addElement(line);
				line = new Line();
			} else {
				sentence.addElement(comp);
			}
		}

		if(sentence.size() > 0) line.addElement(sentence);
		if(line.size() > 0) page.addElement(line);

		return page;
	}

	public PageLayout() {}

	private Vector<Component> glues = new Vector<Component>();

	public void addLayoutComponent(String name, Component comp) {
		if(name.equals(GLUE)) {
			glues.addElement(comp);
		}
	}

	public void removeLayoutComponent(Component comp) {
		glues.removeElement(comp);
	}

	private static interface CompTask {
		public void process(Component comp, Rectangle rect);
	}

	private Dimension layoutPage(Container parent, SizeAccessor sizeAccessor, Vector<Component> glues) {
		return layoutPage(parent, sizeAccessor, 
			new CompTask() {
				public void process(Component comp, Rectangle rect) {}
			},
			glues);
	}

	private class ProcessedComponent {
		public final Component comp;
		public final Rectangle rect;
		
		public ProcessedComponent(Component comp, Rectangle rect) {
			this.comp = comp;
			this.rect = rect;
		}
	}
	
	private Dimension layoutPage(Container parent, SizeAccessor sizeAccessor, CompTask compTask, Vector<Component> glues) {
		Page page = createPage(parent.getComponents(), glues);
		page.resetSentenceSize(sizeAccessor);
		Insets insets = parent.getInsets();

		int x = insets.left;
		int y = insets.top;

		int maxX = x;

		Vector<ProcessedComponent> processedComps = new Vector<ProcessedComponent>();
		
		for(Line line: page) {
			int lineHeight = 0;				
			Sentence lastSentence = line.lastElement();
			Component lastComp = null;
			if(lastSentence != null) lastComp = lastSentence.lastElement();
			
			for(Sentence sentence: line) {
				Vector<ProcessedComponent> processedLineComps = new Vector<ProcessedComponent>();
		
				int sentenceWidth = 0;


				for(Component comp: sentence) {
					Dimension size = sizeAccessor.get(comp);
					processedLineComps.addElement(new ProcessedComponent(comp, new Rectangle(x, y, size.width, size.height)));

					x += size.width;
					
					if(comp == lastComp) {
						if(x < parent.getSize().width - insets.right) {
							Component glue = page.findGlue(line);

							boolean before = true;
							int plusX = parent.getSize().width - insets.right - x;

							for(ProcessedComponent proc: processedLineComps) {
								if(proc.comp != glue) {
									if(before) {
										continue;
									} else {
										proc.rect.x += plusX;
									}
								} else {
									proc.rect.width += plusX;
									before = false;
								}
							}

//							size.width = parent.getSize().width - insets.right - x;
						}
					}

					sentenceWidth += size.width;
					lineHeight = Math.max(lineHeight, size.height);
				}

				if(sentence.width > sentenceWidth) x += sentence.width - sentenceWidth;
				x += minTabWidth;

				processedComps.addAll(processedLineComps);
			}

			maxX = Math.max(x, maxX);

			x = insets.left;
			y += lineHeight + line.nextLineDistance;
		}
		
		for(ProcessedComponent proc: processedComps) {
			compTask.process(proc.comp, proc.rect);
		}

		x = maxX + insets.right;
		y += insets.bottom;
		
		return new Dimension(x, y);
	}
	
	public Dimension preferredLayoutSize(Container parent) {
		return layoutPage(parent, prefSize, glues);
	}

	public Dimension minimumLayoutSize(Container parent) {
		return layoutPage(parent, minSize, glues);
	}

	public void layoutContainer(Container parent) {
		layoutPage(parent, prefSize, new CompTask() {
			public void process(Component comp, Rectangle r) {
				comp.setBounds(r.x, r.y, r.width, r.height);
			}
		}, glues);
	}

	public Component createNewLine() {
		return new NewLine(3);
	}

	public Component createNewLine(int nextLineDistance) {
		return new NewLine(nextLineDistance);
	}

	public Component createTab() {
		return new Tab();
	}
	
	public Component createHorizonSpace(final int width) {
		return new Canvas() {{
				setMinimumSize(new Dimension(width,1));
				setPreferredSize(new Dimension(width,1));
				setMaximumSize(new Dimension(width,1));
			}
		};
	}
	
	public static class Glue {
		public final Component comp;

		public Glue(Component comp) {
			this.comp = comp;
		}
	}

	static class EmptyGlueComponent extends Canvas {
		public EmptyGlueComponent() {
			setMinimumSize(new Dimension(1,1));
			setPreferredSize(new Dimension(1,1));
			setMaximumSize(new Dimension(1,1));
		}
	}
}
