/*
 * Copyright (c) 2006, team-naver.com
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.aibonware.viewnaver.config;

import java.awt.event.*;
import java.util.*;

import javax.swing.*;
import javax.swing.table.*;

import org.w3c.dom.*;

import com.aibonware.viewnaver.model.*;

@SuppressWarnings("serial")
public class FilterConfig extends ConfigItem {
	public Map<Poster, FilterType> filters = new TreeMap<Poster, FilterType>();

	public FilterConfig() {
		super("FilterList");
		loadConfig(null);
	}

	@Override public void loadConfig(Element elem) {
		filters =  new TreeMap<Poster, FilterType>();
		
		if(elem == null) return;
		NodeList children = elem.getElementsByTagName("Filter");

		for(int i=0; i<children.getLength(); i++) {
			Element filterElem = (Element)children.item(i);

			Poster poster = Config.getPosterConfig(filterElem, "Poster");
			String filterTypeName = Config.getStringConfig(filterElem, "FilterType", FilterType.Visible.name());

			FilterType filterType = FilterType.valueOf(filterTypeName);

			filters.put(poster, filterType);
		}
	}

	@Override public void saveConfig(Element elem) {
		NodeList children = elem.getElementsByTagName("Filter");

		for(int i=0; i<children.getLength(); i++) {
			elem.removeChild(children.item(i));
		}

		for(Poster poster: filters.keySet()) {
			Document doc = elem.getOwnerDocument();
			Element filterElem = doc.createElement("Filter");
			elem.appendChild(filterElem);

			Config.setPosterConfig(filterElem, "Poster", poster);
			Config.setStringConfig(filterElem, "FilterType", filters.get(poster).name());
		}
	}

	@Override public String getDispPath() {
		return "{.̨ݸ";
	}

	private static class FilterConfigView extends ConfigItemView {
		JTable table;
		JButton deleteButton;
		private FilterConfig filterConfig;
		private DefaultTableModel tableModel;

		public FilterConfigView() {
			tableModel = new DefaultTableModel(new Object[] {"Country", "Poster", "Type"}, 0);
			table = new JTable(tableModel);

			deleteButton = new JButton("폜");

			addLine(
				glue(table.getTableHeader()), newline(0),
				glue(table), "\n",
				glue(), deleteButton);
			
			deleteButton.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					int row = table.getSelectedRow();
					if(row < 0) return;
					
					tableModel.removeRow(row);
				}
			});
		}

		public void setConfig(ConfigItem config) {
			filterConfig = (FilterConfig)config;
			
			for(Poster poster: filterConfig.filters.keySet()) {
				FilterType filterType = filterConfig.filters.get(poster);

				tableModel.addRow(new Object[] {poster.country, poster.name, filterType});
			}
		}

		public void applyConfig() throws ConfigException {
			filterConfig.filters.clear();
			
			for(int i=0; i<tableModel.getRowCount(); i++) {
				Country country = (Country)tableModel.getValueAt(i, 0);
				String name = (String)tableModel.getValueAt(i, 1);
				FilterType filterType = (FilterType)tableModel.getValueAt(i, 2);
				
				filterConfig.filters.put(new Poster(name, country), filterType);
			}
		}
	}

	@Override public ConfigItemView getView() {
		return new FilterConfigView();
	}
}
