/*
 * Copyright (c) 2006, team-naver.com
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.aibonware.viewnaver.content.localsearch;

import java.io.*;
import java.text.*;
import java.awt.*;
import java.awt.event.*;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.border.*;

import javax.xml.parsers.*;
import org.w3c.dom.*;
import org.xml.sax.*;

import com.aibonware.viewnaver.*;
import com.aibonware.viewnaver.model.*;
import com.aibonware.viewnaver.browser.*;
import com.aibonware.viewnaver.command.*;
import com.aibonware.viewnaver.content.*;
import com.aibonware.viewnaver.storage.*;
import com.aibonware.viewnaver.task.*;
import com.aibonware.viewnaver.view.*;
import com.aibonware.viewnaver.window.*;
import com.aibonware.viewnaver.xml.*;

@SuppressWarnings("serial")
public class LocalFoundContentView extends TabContentView {
	public Browser browser;
	private boolean mustCancel = false;
	private String SPAN = ViewNaver.instance.getBrowserFactory().SPAN;
	private String HR = ViewNaver.instance.getBrowserFactory().HR;

	private LocalFoundContent clientContent = null;

	private static SimpleDateFormat df = new SimpleDateFormat("MM/dd HH:mm");

	JButton searchThreadButton = new JButton();
	JButton closeButton = new JButton("");

	AbstractAction searchThreadAction = new AbstractAction("") {
		public void actionPerformed(ActionEvent e) {
			SearchLocalThreadDialog dialog = new SearchLocalThreadDialog(
					ViewNaver.instance.mainWindow, 
					clientContent.category.keyword, 
					clientContent.category.localSearchType);
	
			dialog.setVisible(true);
		}
	};
	
	public LocalFoundContentView(TabContentCategory category) {
		super(category);

		setLayout(new BorderLayout());

		browser =ViewNaver.instance.getBrowserFactory().createBrowser();
		
		browser.beginDocWriting();
		browser.writeDocHtml(
				"<html><head>"
				+ "<meta http-equiv=\"Content-Type\" content=\"text/html\">"
				//+"<style type=\"text/css\">BODY, TH, TD { font-family: 'MS UI Gothic'; }</style>"
				//+ "<link rel=\"stylesheet\" type=\"text/css\" href=\"./cache/enjoykorea.css\">"
				//+ "<link rel=\"stylesheet\" type=\"text/css\" href=\"./cache/layout.css\">"
				+ "</head><body></body></html>");
		browser.endDocWriting();

		JPanel panel = new JPanel(new BorderLayout());
		panel.setBorder(new BevelBorder(BevelBorder.LOWERED));
		panel.add(browser.getComponent());
				
		closeButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				mustCancel = true;
				ViewNaver.instance.mainWindow.container.removeContent(clientContent);
			}
		});

		JToolBar toolbar = new JToolBar();
		toolbar.add(closeButton);
		searchThreadButton.setAction(searchThreadAction);
		toolbar.add(searchThreadButton);

		toolbar.setFloatable(false);

		add(toolbar, BorderLayout.NORTH);
		add(panel, BorderLayout.CENTER);

		browser.addBrowserListener(new BrowserListener() {
			public boolean beforeNavigate(String url) {
				if(url.startsWith("thread:")) {
					url = url.substring("thread:".length());
					int dotPos = url.indexOf('.');

					String boardName = url.substring(0, dotPos);
					int nid = Integer.parseInt(url.substring(dotPos+1));
					
					TaskFlow taskFlow = ViewNaver.instance.taskFlowList.getThreadFlow(boardName, nid);

					taskFlow.execute(new ShowThreadCommand(boardName, nid));
				}
				
				return false;
			}
		});
	}

	public TaskStatusReceiver createTaskStatusReceiver(final TaskFlow taskFlow) {
		return new TaskStatusReceiver() {
			public void statusChanged(int status) {
				if(status == RUNNING) {
					searchThreadButton.setAction(new AbstractAction("~") {
						public void actionPerformed(ActionEvent e) {
							taskFlow.cancel();
						}
					});

					closeButton.setEnabled(false);
					
				}  else if(status == FINISHED) {
					searchThreadButton.setAction(searchThreadAction);
	
					closeButton.setEnabled(true);
				}
			}
		};
	}

	public void setClientContent(TabContent content) {
		clientContent = (LocalFoundContent)content;

		final LocalSearchType localSearchType = clientContent.category.localSearchType;
		final String poster = clientContent.category.poster;
		final String keyword = clientContent.category.keyword;
		count = 0;
		curBoardName = null;
		browser.setBodyHtml("<font size=-1><" + SPAN + " id=\"hit:0\"></" + SPAN + "></font>");

		TaskFlow taskFlow = ViewNaver.instance.taskFlowList.getSearchLocalThreadFlow(localSearchType, poster, keyword);

		taskFlow.addStatusReceiver(createTaskStatusReceiver(taskFlow));
		
		taskFlow.execute(new WorkerTask() {
			public void exec() {
				TaskCanceller canceller = new TaskCanceller() {
					public void cancel() {
						mustCancel = true;
					}
				};
				
				taskFlow.addTaskCanceller(canceller);
				
				try {
					final DocumentBuilderFactory dbfactory = DocumentBuilderFactory.newInstance();
					final DocumentBuilder builder = dbfactory.newDocumentBuilder();
					final CompositeThreadReader compositeThreadReader = new CompositeThreadReader();

					ViewNaver.instance.threadStorage.flash();
					ViewNaver.instance.threadStorage.walkFiles(new ThreadStorageWalker() {
						public void finished() {
							ViewNaver.println("OB");
						}

						public void canceled() {
							ViewNaver.println("OLZB");
						}

						public boolean walk(File file) {
							try {
								if(mustCancel) {
									mustCancel = false;
									return false;
								}

								StringBuffer buf = new StringBuffer();
								BufferedReader reader = new BufferedReader(new InputStreamReader(new FileInputStream(file), "UTF8"));

								String line;
								
								while((line = reader.readLine()) != null) {
									buf.append(line);
								}
								
								reader.close();
								String text = buf.toString();

								boolean preMatch = true;
								
								if(poster != null && !text.contains(poster)) preMatch = false;
								if(keyword != null && preMatch && !text.contains(keyword)) preMatch = false;

								if(!preMatch) return true;

								Document doc = builder.parse(new InputSource(new StringReader(text)));

								Element root = doc.getDocumentElement();
								Element threadElem = BaseXMLReader.getAttrElem(root, "CompositeThread");
						

								CompositeThread compositeThread = compositeThreadReader.parse(threadElem);
								
								if(localSearchType.equals(LocalSearchType.SearchThreadTitle)) {
									if(poster != null && !poster.equals(compositeThread.thread.poster)) return true;
									if(keyword != null && !compositeThread.thread.title.contains(keyword)) return true;

									addThreadHit(taskFlow, compositeThread);
								} else if(localSearchType.equals(LocalSearchType.SearchThreadContent)) {
									if(poster != null && !poster.equals(compositeThread.thread.poster)) return true;
									if(keyword != null && !compositeThread.contents.contains(keyword)) return true;
					
									addThreadHit(taskFlow, compositeThread);
								} else {
									int artNum = compositeThread.articleList.getArticleCount();

									for(int i=artNum-1; i>=0; i--) {
										Article art = compositeThread.articleList.getArticle(i);

										if(poster != null && !poster.equals(art.author)) continue;
										if(keyword != null && !art.content.contains(keyword)) continue;

										addArticleHit(taskFlow, compositeThread, art, keyword);
									}
								}
								
								return true;
							} catch(Exception e) {
								ViewNaver.err(e);
								return false;
							}
						}
					});
					
				} catch(Exception e) {
					ViewNaver.err(e);
				} finally {
					taskFlow.removeTaskCanceller(canceller);
				}
			}
		});
	}

	private int count = 0;
	private String curBoardName = "";
	
	private void addBoardName(String name) {
		count++;

		browser.insertBodyHtml("hit:" + (count-1), 
				HR + "<" + SPAN + " id=\"hit:" + count + "\">"
				+ name + "F<br><br>\n</" + SPAN + ">");
	}
	
	private void addThreadHit(TaskFlow taskFlow, final CompositeThread compositeThread) {
		FilterType filterType = 
			ViewNaver.instance.config.filterConfig.filters.get(new Poster(compositeThread.thread.poster, compositeThread.thread.country));
		
		if(filterType != null) return;

		taskFlow.execute(new GUITask() {
			public void exec() {
				if(!compositeThread.boardName.equals(curBoardName)) {
					curBoardName = compositeThread.boardName;
					addBoardName(curBoardName);
				}
				
				count++;
				
				browser.insertBodyHtml("hit:" + (count-1), 
						"<" + SPAN + " id=\"hit:" + count + "\">"
//						+ compositeThread.boardName + " "
						+ "<a href=\"thread:" + compositeThread.boardName + "." 
						+ compositeThread.thread.nid + "\">"
//						+  "</a> "
						+ compositeThread.thread.dispId + "</a> "
						+ df.format(compositeThread.thread.createDate) + " "
						+ compositeThread.thread.country.toHtml() + " "
						+ " <b>" + compositeThread.thread.poster + "</b> "
						+ ViewNaver.htmlToStr(compositeThread.thread.title)
						+ "<br><br>\n</" + SPAN + ">");
			}
		});
	}

	private void addArticleHit(TaskFlow taskFlow, final CompositeThread compositeThread,final Article article, final String keyword) {
		FilterType filterType = 
			ViewNaver.instance.config.filterConfig.filters.get(new Poster(compositeThread.thread.poster, compositeThread.thread.country));
		
		if(filterType != null) return;

		taskFlow.execute(new GUITask() {
			public void exec() {
				if(!compositeThread.boardName.equals(curBoardName)) {
					curBoardName = compositeThread.boardName;
					addBoardName(curBoardName);
				}

				count++;
				
				String articleContent = ViewNaver.htmlToStr(article.content);
				
				if(keyword != null) {
					articleContent = articleContent.replace(keyword, "<b>" + keyword + "</b>");
				}

				browser.insertBodyHtml("hit:" + (count-1), 
						"<" + SPAN + " id=\"hit:" + count + "\">"
//						+ compositeThread.boardName + " "
						+ "<a href=\"thread:" + compositeThread.boardName + "." 
						+ compositeThread.thread.nid + "\">" 
//						+  "</a> "
						+ compositeThread.thread.dispId + "</a> "
						+ df.format(article.postDate) + " "
						+ article.country.toHtml() + " "
						+ " " + article.author + " "
						+ articleContent
						+ "<br><br>\n</" + SPAN + ">");
			}
		});
	}
}
