/*
 *  Copyright (C) 2006  Takashi Kasuya <kasuya@sfc.keio.ac.jp>
 *
 * This library is free software; you can redistribute it and/or
 *@modify it under the terms of the GNU Lesser General Public
 *@License as published by the Free Software Foundation; either
 *@version 2.1 of the License, or (at your option) any later version.
 *@This library is distributed in the hope that it will be useful,
 *@but WITHOUT ANY WARRANTY; without even the implied warranty of
 *@MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *@Lesser General Public License for more details.
 *
 *@You should have received a copy of the GNU Lesser General Public
 *@License along with this library; if not, write to the Free Software
 *@Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package jp.ac.naka.ec.sip;

import javax.sip.ClientTransaction;
import javax.sip.ResponseEvent;
import javax.sip.address.SipURI;
import javax.sip.header.CSeqHeader;
import javax.sip.header.FromHeader;
import javax.sip.header.ToHeader;
import javax.sip.message.Request;
import javax.sip.message.Response;

import jp.ac.naka.ec.EventDispatcher;
import jp.ac.naka.ec.db.TupleSpaceAdapter;
import jp.ac.naka.ec.entity.Entity;
import jp.ac.naka.ec.entity.EntityEvent;
import jp.ac.naka.ec.entity.EntityImpl;
import jp.ac.naka.ec.entity.EntityEvent.EventType;
import net.sourceforge.jsdp.SdpMessageParser;
import net.sourceforge.jsdp.SessionDescription;

/**
 * 
 * @author Takashi Kasuya
 *
 */
public class ResponseDelegate {

	private EventDispatcher dispatcher;

	private SipCore core;

	public ResponseDelegate() {
		dispatcher = TupleSpaceAdapter.getInstance();
		core = SipCore.getInstance();
	}

	/**
	 * 
	 * @param evt
	 * @return
	 */
	public boolean forward(ResponseEvent evt) {
		Response response = (Response) evt.getResponse();
		StatusCode code = getStatusCode(response.getStatusCode());
		boolean a = false;
		switch (code) {
		case CODE1xx:
			// Trying
			break;
		case CODE2xx:
			// SuccessResponse
			doSuccessResponse(evt);
			a = true;
			break;
		case CODE3xx:
			break;
		case CODE4xx:
			break;
		case CODE5xx:
			break;
		case CODE6xx:
		default:
			assert false;
		}
		return a;
	}

	/**
	 * 
	 * @param evt
	 */
	public void doSuccessResponse(ResponseEvent evt) {
		Response res = evt.getResponse();
		try {
			CSeqHeader cseq = (CSeqHeader) res.getHeader(CSeqHeader.NAME);
			// 200 OK
			if (res.getStatusCode() == Response.OK) {
				if (cseq.getMethod().equals(Request.INVITE)) {
					FromHeader contact = (FromHeader) res
							.getHeader(FromHeader.NAME);
					ToHeader to = (ToHeader) res.getHeader(ToHeader.NAME);
					SipURI source_uri = (SipURI) contact.getAddress().getURI();
					SipURI target_uri = (SipURI) to.getAddress().getURI();

					Entity source = new EntityImpl(source_uri);
					Entity target = new EntityImpl(target_uri);
					target.setLocal(false);
					SessionDescription sdp = null;
					byte[] raw = res.getRawContent();
					if (raw != null) {
						String sdp_str = new String(raw, "UTF-8");
						sdp = SdpMessageParser.getSessionDescription(sdp_str);
					}
					EntityEvent ee = new EntityEvent(sdp, target, source);
					ee.setEventType(EventType.ACK);
					dispatcher.dispatchEvent(ee);
					EntityEvent eee = new EntityEvent(sdp, source, target);
					eee.setEventType(EventType.ANSWER);
					dispatcher.dispatchEvent(eee);
				}

				/** ************ NOTIFY ****************** */
				else if (cseq.getMethod().equals(Request.NOTIFY)) {
					// TODO Expires
					core.refresh();
				}
			}
		} catch (Exception ex) {
			ex.printStackTrace();
		}

	}

	/**
	 * 
	 * @param evt
	 */
	public void doErrorResponse(ResponseEvent evt) {
		System.err.println(evt.getResponse().toString());
		ClientTransaction ct = evt.getClientTransaction();
		if (ct == null) {
			return;
		}
		Response res = evt.getResponse();
		if (res.getStatusCode() == Response.REQUEST_TERMINATED) {
			ct = null;
		}
	}

	private StatusCode getStatusCode(int code) {
		String c = code + "";
		if (c.indexOf("1") == 0) {
			return StatusCode.CODE1xx;
		} else if (c.indexOf("2") == 0) {
			return StatusCode.CODE2xx;
		} else if (c.indexOf("3") == 0) {
			return StatusCode.CODE3xx;
		} else if (c.indexOf("4") == 0) {
			return StatusCode.CODE4xx;
		} else if (c.indexOf("5") == 0) {
			return StatusCode.CODE5xx;
		} else if (c.indexOf("6") == 0) {
			return StatusCode.CODE6xx;
		} else {
			throw new IllegalAccessError("Invalid Response code!");
		}
	}

	enum StatusCode {
		CODE1xx, CODE2xx, CODE3xx, CODE4xx, CODE5xx, CODE6xx;
	}

}
