package jp.sourceforge.foolishmerge.diff;

import java.util.Arrays;

/**
 * O(NP)アルゴリズム[1]で文書を比較するクラス。<BR>
 * <BR>
 * [1] S. Wu, E. Myers, U. Manber, and W. Miller, <A HREF="http://www.cs.arizona.edu/people/gene/PAPERS/np_diff.ps">``An <I>O(NP)</I> Sequence Comparison Algorithm,''</A> <I>Information Processing Letters</I> 35, 6 (1990), 317-323.
 */
public class Diff {

	/**
	 * 文書を比較する。
	 * 
	 * @param doc_a 行A
	 * @param doc_b 行B
	 * @return 差分情報
	 */
	public static DocDifference diff(String[] doc_a, String[] doc_b) {

		// 行が等しい場合は対角線の経路を返す。
		if (Arrays.equals(doc_a, doc_b)) {
			return DocDifference.getEquals(doc_a.length);
		}

		// 差分情報を生成。
		DocDifference doc = new DocDifference(doc_a, doc_b);

		int M = doc.getM();
		int N = doc.getN();
		int DELTA = N - M;
		int[] fp = new int[M + N + 3];

		for (int i = 0; i < fp.length; i++) {
			fp[i] = -1;
		}

		// O(NP)アルゴリズムでLCS/SED経路を取得。
		// O(NP)アルゴリズムについては参考文献[1]を参照のこと。

		int p = -1;

		do {
			p += 1;

			for (int k = -p; k < DELTA; k++) {
				fp[k + M + 1] =
					doc.snake(k, Math.max(fp[k + M] + 1, fp[k + M + 2]));
			}

			for (int k = (DELTA + p); k > DELTA; k--) {
				fp[k + M + 1] =
					doc.snake(k, Math.max(fp[k + M] + 1, fp[k + M + 2]));
			}

			fp[N + 1] = doc.snake(DELTA, Math.max(fp[N] + 1, fp[N + 2]));

		} while (fp[N + 1] < N);

		// 差分情報にLCSをセット。
		doc.setLCS(M - p);

		// 差分情報をクローズ。
		doc.close();

		// LCS経路を返す。
		return doc;
	}

}
