package jp.sourceforge.foolishmerge.diff;

import java.util.List;

/**
 * 行差分クラス。
 */
public class LineDifference {

	/**
	 * 状態 - 差分なし
	 */
	public static final int EQ = 0;

	/**
	 * 状態 - 追加
	 */
	public static final int ADD = 1;

	/**
	 * 状態 - 削除
	 */
	public static final int DEL = -1;

	/**
	 * 状態 - 状態なし
	 */
	public static final int NO_STATE = Integer.MIN_VALUE;

	/**
	 * X座標(0～M)
	 */
	private int x = Integer.MIN_VALUE;

	/**
	 * Y座標(0～N)
	 */
	private int y = Integer.MIN_VALUE;

	/**
	 * 編集元ドキュメントの行インデックス(-1～M-1)
	 */
	private int orgLines = Integer.MIN_VALUE;

	/**
	 * 編集後ドキュメントの行インデックス(-1～N-1)
	 */
	private int modLines = Integer.MIN_VALUE;

	/**
	 * 行
	 */
	private String line = null;

	/**
	 * 状態
	 */
	private int state = NO_STATE;

	/**
	 * 移動元
	 */
	private LineDifference prev = null;

	/**
	 * 指定された座標を使用して、行の差分情報を構築する。
	 * 
	 * @param x X座標
	 * @param y Y座標
	 */
	public LineDifference(int x, int y) {
		// 座標をフィールドにセット。
		this.x = x;
		this.y = y;
		// 行インデックスをセット。
		orgLines = x - 1;
		modLines = y - 1;
	}

	/**
	 * 編集元ドキュメントの行インデックスを取得する。
	 * 
	 * @return 行インデックス(-1～M-1)
	 */
	public int getOrgLineNum() {
		// X座標を返す。
		return orgLines;
	}

	/**
	 * 編集後ドキュメントの行インデックスを取得する。
	 * 
	 * @return 行インデックス(-1～N-1)
	 */
	public int getModLineNum() {
		// Y座標を返す。
		return modLines;
	}

	/**
	 * 行をセットする。
	 * 
	 * @param line 行
	 */
	void setLine(String line) {
		// 行をセット。
		this.line = line;
	}

	/**
	 * 行を取得する。
	 * 
	 * @return 行
	 */
	public String getLine() {
		// 行を返す。
		return line;
	}

	/**
	 * 状態を取得する。
	 * 
	 * @return 状態(EQ/ADD/DEL/NO_STATE)
	 */
	public int getState() {
		// 状態を返す。
		return state;
	}

	/**
	 * パッチを当てる。
	 * 
	 * @param doc 編集元文書
	 * @param offset オフセット
	 */
	public void patch(List doc, int offset) {
		// 編集元文書を修正。
		switch (state) {
			case ADD :
				// 追加の場合
				doc.add(orgLines + 1 + offset, line);
				break;

			case DEL :
				// 削除の場合
				doc.remove(orgLines + offset);
				break;
		}
	}

	// package private ----------------------------------------------

	/**
	 * X座標を取得する。
	 * 
	 * @return X座標(0～M)
	 */
	int getX() {
		// X座標を返す。
		return x;
	}

	/**
	 * Y座標を取得する。
	 * 
	 * @return Y座標(0～N)
	 */
	int getY() {
		// Y座標を返す。
		return y;
	}

	/**
	 * 状態をセットする。
	 * 
	 * @param state 状態
	 */
	void setState(int state) {
		// 状態をセット。
		this.state = state;
	}

	/**
	 * 移動元を取得する。
	 * 
	 * @return 移動元
	 */
	LineDifference getPrev() {
		// 移動元を返す。
		return prev;
	}

	/**
	 * 移動元をセットする。
	 * 
	 * @param prev 移動元
	 */
	void setPrev(LineDifference prev) {
		// 移動元をセット。
		this.prev = prev;
	}

}
