package jp.sourceforge.foolishmerge.diff;

import java.util.List;

import jp.sourceforge.foolishmerge.FoolishMergeConfig;
import jp.sourceforge.foolishmerge.FoolishMergeUtils;

/**
 * マージされた差分クラス。
 */
class MergedDelta {

	/**
	 * マージ開始セパレータ
	 */
	private static final String START_SEPARATOR =
		FoolishMergeConfig.getStartSeparator();

	/**
	 * マージ終了セパレータ
	 */
	private static final String END_SEPARATOR =
		FoolishMergeConfig.getEndSeparator();

	/**
	 * リビジョンセパレータ
	 */
	private static final String REV_SEPARATOR =
		FoolishMergeConfig.getRevisionSeparator();

	/**
	 * 修正開始行数
	 */
	private int firstLineNum = Integer.MIN_VALUE;

	/**
	 * 修正終了行数
	 */
	private int lastLineNum = Integer.MIN_VALUE;

	/**
	 * マージされた差分の行数
	 */
	private int size = Integer.MIN_VALUE;

	/**
	 * ファイル名
	 */
	private String filename = null;

	/**
	 * リビジョン
	 */
	private String revision = null;

	/**
	 * 差分1の行
	 */
	private String[] delta1Lines = null;

	/**
	 * 差分2の行
	 */
	private String[] delta2Lines = null;

	/**
	 * 指定された差分を使用してマージされた差分を構築する。
	 * 
	 * @param delta1 差分1
	 * @param delta2 差分2
	 */
	public MergedDelta(Delta delta1, Delta delta2) {
		// 編集後の行をフィールドにセット。
		delta1Lines = delta1.getModLines();
		delta2Lines = delta2.getModLines();

		// 差分を比較して小さい開始行数をオブジェクトの開始行数とする。
		if (delta1.getOrgFirstLineNum() < delta2.getOrgFirstLineNum()) {
			firstLineNum = delta1.getOrgFirstLineNum();
		} else {
			firstLineNum = delta2.getOrgFirstLineNum();
		}

		// 差分を比較して大きい終了行数をオブジェクトの終了行数とする。
		if (delta1.getOrgLastLineNum() > delta2.getOrgLastLineNum()) {
			lastLineNum = delta1.getOrgLastLineNum();
		} else {
			lastLineNum = delta2.getOrgLastLineNum();
		}

		// オブジェクトのサイズをセット。
		size = delta1Lines.length + delta2Lines.length + 3;
	}

	/**
	 * マージされた行を取得する。
	 * 
	 * @return マージされた行
	 */
	public String[] getMergedLines() {
		// バッファを生成。
		String[] merged = new String[size];

		int idx = 0;

		// バッファに開始セパレータを追加。
		merged[idx] = START_SEPARATOR;

		// ファイル名がある場合は開始セパレータに追加。
		if (filename != null) {
			merged[idx] += " " + filename;
		}

		idx++;

		// バッファに差分1の行を追加。
		for (int i = 0; i < delta1Lines.length; i++, idx++) {
			merged[idx] = delta1Lines[i];
		}

		// バッファにリビジョンセパレータを追加。
		merged[idx++] = REV_SEPARATOR;

		// バッファに差分2の行を追加。
		for (int i = 0; i < delta2Lines.length; i++, idx++) {
			merged[idx] = delta2Lines[i];
		}

		// バッファに終了セパレータを追加。
		merged[idx] = END_SEPARATOR;

		// リビジョンがある場合は終了セパレータに追加。
		if (revision != null) {
			merged[idx] += " " + revision;
		}

		// マージされた行を返す。
		return merged;
	}

	/**
	 * マージされた差分の文字列表現を返す。
	 * 
	 * @return マージされた差分の文字列表現
	 */
	public String toString() {
		// マージされた行を文字列に変換して返す。
		return FoolishMergeUtils.arrayToString(getMergedLines());
	}

	/**
	 * 開始行数を取得する。
	 * 
	 * @return 開始行数
	 */
	public int getFirstLineNum() {
		// 開始行数を返す。
		return firstLineNum;
	}

	/**
	 * 終了行数を取得する。
	 * 
	 * @return 終了行数
	 */
	public int getLastLineNum() {
		// 終了行数を返す。
		return lastLineNum;
	}

	/**
	 * マージされた差分のサイズを取得する。
	 * 
	 * @return マージされた差分のサイズ
	 */
	public int getSize() {
		// マージされた差分のサイズを返す。
		return size;
	}

	/**
	 * 差分1の行を取得する。
	 * 
	 * @return 差分1の行
	 */
	public String[] getDelta1Lines() {
		// 差分1の行を返す。
		return delta1Lines;
	}

	/**
	 * 差分2の行を取得する。
	 * 
	 * @return 差分2の行
	 */
	public String[] getDelta2Lines() {
		// 差分2の行を返す。
		return delta2Lines;
	}

	/**
	 * ファイル名をセットする。
	 * 
	 * @param filename ファイル名
	 */
	public void setFilename(String filename) {
		// ファイル名をセット。
		this.filename = filename;
	}

	/**
	 * リビジョンをセットする。
	 * 
	 * @param revision リビジョン
	 */
	public void setRevision(String revision) {
		// リビジョンをセット。
		this.revision = revision;
	}

	/**
	 * マージされた差分をパッチとして当てる。
	 * 
	 * @param doc 文書
	 * @param offset オフセット
	 * @return パッチサイズ
	 */
	public int patch(List doc, int offset) {
		// 修正前の行を文書から削除。
		for (int i = lastLineNum; i <= firstLineNum; i++) {
			doc.remove(i + offset);
		}

		// 文書の修正箇所にオブジェクトを挿入。
		doc.add(firstLineNum + offset, this);

		// パッチサイズを返す。
		return size - (lastLineNum - firstLineNum + 1);
	}

}
