package jp.sourceforge.foolishmerge.diff;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;

import jp.sourceforge.foolishmerge.FoolishMergeUtils;

/**
 * マージされた文書クラス。
 */
public class MergedDocument {

	/**
	 * 文書
	 */
	private List doc = null;

	/**
	 * コンフリクトフラグ
	 */
	private boolean conflict = false;

	/**
	 * 編集元文書からマージされた文書を構築する。
	 * 
	 * @param org 編集元文書
	 */
	public MergedDocument(String[] org) {
		// 編集元文書をリストに変換してフィールドにセット。
		this.doc = new LinkedList(Arrays.asList(org));
	}

	/**
	 * パッチを当てる。
	 * 
	 * @param delta 差分
	 * @param offset オフセット
	 * @return パッチサイズ
	 */
	public int patch(Delta delta, int offset) {
		// 編集元文書にパッチを当てる。
		int size = delta.patch(doc, offset);
		// パッチサイズを返す。
		return size;
	}

	/**
	 * コンフリクトした差分をマージする。
	 * 
	 * @param delta1 差分1
	 * @param delta2 差分2
	 * @param offset オフセット
	 * @return マージサイズ
	 */
	public int merge(Delta delta1, Delta delta2, int offset) {
		int size = 0;

		if (Arrays.equals(delta1.getModLines(), delta2.getModLines())) {
			// 修正後の文書が同じ場合は、差分1のパッチを当てる。
			// (コンフリクトなし)
			size = delta1.patch(doc, offset);
		} else if (
			delta1.getState() == Delta.ADD && delta2.getState() == Delta.ADD) {
			// 差分がともに追加の場合は、差分1のパッチを当ててから差分2のパッチを当てる。
			// (コンフリクトなし)
			size = delta1.patch(doc, offset);
			size += delta2.patch(doc, offset + size);
		} else {
			// 上記のいずれでもない場合は、差分をマージしてパッチを当てる。
			// (コンフリクトあり)
			conflict = true;
			MergedDelta merged = new MergedDelta(delta1, delta2);
			size = merged.patch(doc, offset);
		}

		// マージサイズを返す。
		return size;
	}

	/**
	 * マージされた行を取得する。
	 * 
	 * @return マージされた行
	 */
	public String[] getLines() {
		// バッファを生成。
		List lines = new ArrayList();

		// 文書のサイズ分、繰り返し。
		for (int i = 0; i < doc.size(); i++) {
			// 文書からオブジェクトを取得。
			Object line = doc.get(i);

			if (line instanceof MergedDelta) {
				// マージされた差分の場合は、行を取り出してからバッファに追加。
				MergedDelta merged = (MergedDelta) line;
				String[] mergedLines = merged.getMergedLines();

				for (int j = 0; j < mergedLines.length; j++) {
					lines.add(mergedLines[j]);
				}
			} else {
				// 文字列の場合は、そのままバッファに追加。
				lines.add(line);
			}
		}

		// バッファを配列に変換して返す。
		return (String[]) lines.toArray(new String[lines.size()]);
	}

	/**
	 * マージされた文書の文字列表現を取得する。
	 * 
	 * @return マージされた文書の文字列表現
	 */
	public String toString() {
		// マージされた行を文字列表現に変換して返す。
		return FoolishMergeUtils.arrayToString(getLines());
	}

	/**
	 * コンフリクトフラグを取得する。
	 * 
	 * @return コンフリクトしている場合はtrue
	 */
	public boolean isConflict() {
		// コンフリクトフラグを返す。
		return conflict;
	}

	/**
	 * ファイル名をセットする。
	 * 
	 * @param filename ファイル名
	 */
	public void setFilename(String filename) {
		// 文書を検索してマージされた差分にファイル名をセット。
		for (int i = 0; i < doc.size(); i++) {
			Object line = doc.get(i);

			if (line instanceof MergedDelta) {
				MergedDelta merged = (MergedDelta) line;
				merged.setFilename(filename);
			}
		}
	}

	/**
	 * リビジョンをセットする。
	 * 
	 * @param revision リビジョン
	 */
	public void setRevision(String revision) {
		// 文書を検索してマージされた差分にリビジョンをセット。
		for (int i = 0; i < doc.size(); i++) {
			Object line = doc.get(i);

			if (line instanceof MergedDelta) {
				MergedDelta merged = (MergedDelta) line;
				merged.setRevision(revision);
			}
		}
	}

}
