#------------------------------------------------------------------------------
#    59Tracker, weblog software for personal publisher.
#    Copyright (C) 2004-2009 Kaga, Hiroaki
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#------------------------------------------------------------------------------

package Lib::App::Search;

use strict;
use warnings;

use constant {
    LOCK_SH => 1,
    LOCK_EX => 2,
    LOCK_NB => 4,
    LOCK_UN => 8,
};

use Lib::Logger;
use Lib::String;
use Lib::DateTime;
use Lib::App::Topic;
use Lib::App::Archive;

my $logger = Lib::Logger->new();

# コンストラクタ
sub new {
    my $self = {};

    use Lib::App::Conf;
    my $conf = Lib::App::Conf->new();

    $self->{system_dir} = $conf->get_system_dir();  # システムディレクトリ
    $self->{data_dir} = $conf->get_data_dir();      # データディレクトリ

    bless($self);
    return $self;
}

# 全文検索
sub search {
    my $self = shift;
    my $q = shift;

    # 検索キーワードをログに記録
    $self->writelog($q);

    # 検索条件の調整
    $q = $self->adjust($q);

    # 半角スペースを区切りとしてキーワードに分解
    my @keywords = split(/ /, $q);
    my @resultset = ();
    foreach my $item (@keywords) {
        my $list = $self->find($item);
        my @idset = split(/:/, $list);
        if (scalar(@resultset) == 0) {
            @resultset = @idset;
            next
        }
        else {  # and 検索
            my @newset = ();
            foreach my $id (@idset) {
                foreach my $tempid (@resultset) {
                    if ($id eq $tempid) {
                        push @newset, $id;
                        last;
                    }
                }
            }
            @resultset = @newset;
        }
    }

    return @resultset;
}

# ログに記録
sub writelog() {
    my $self = shift;
    my $q = shift;

    my $dt = Lib::DateTime->new();
    my $localdt = $dt->local_datetime(0);
    my $yyyy = substr($localdt, 0, 4);
    my $mm = substr($localdt, 5, 2);
    my $dd = substr($localdt, 8, 2);
    my $logfile = "$self->{data_dir}/keyword/slog_$yyyy$mm$dd.data";
    my $datarec = "$localdt,$q";
    open my $lfh, '>>', $logfile;
    print {$lfh} "$datarec\n";
    close $lfh;
}

# インデックスの追加
sub addindex {
    my $self = shift;
    my $id = shift;
    my $srcstr = shift;

    $srcstr =~ s/<.*?>//g;    # HTMLタグを取り除く
    $srcstr = $self->adjust($srcstr);

    my @srcstrings = split(/ /, $srcstr);
    my @singlewords = ();
    my @doublewords = ();
    foreach my $str (@srcstrings) {
        my $ch = substr($str, 0, 1);
        if ($ch =~ /\w/) {  # ASCII文字
            my @result = grep { /\A$str\z/ } @singlewords;
            if (!scalar(@result)) {
                push @singlewords, $str;
            }
        }
        else {
            @doublewords = $self->decompose($str, @doublewords);
        }
    }

    # 半角文字
    foreach my $keyword (@singlewords) {
        $self->registeritem($id, $keyword);
    }

    # 全角文字
    foreach my $keyword (@doublewords) {
        $self->registeritem($id, $keyword);
    }

    return 1;
}

# 入力された2バイト文字からなる文を分解してキーワードの配列として返す
sub decompose {
    my $self = shift;
    my $srcstr = shift;            # 変換前の文字列
    my @keywords = @_;             # 分解後のキーワード配列
    my $len = length($srcstr) / 2; # 変換対象の文字列の文字数
    my $width = 1;

    while ($width <= $len) {
        for my $i (0 .. ($len - 1)) {
            my $pos = $i * 2;
            my $cutsize = $width * 2;
            my $cutstr = substr($srcstr, $pos, $cutsize);	# 文字の切り出し

            # 残りの文字サイズが切り出しサイズより小さくなったらループを抜ける
            last if (length($cutstr) < $cutsize);

            # 既に登録されているかチェック
            my $found = 0;
            foreach my $keyword (@keywords) {
                if ($cutstr eq $keyword) {
                    $found = 1;
                    last;
                }
            }
            # 登録されていないときは
            if (!$found) {
                push @keywords, $cutstr;
            }
        }
        $width++;
    }

    return @keywords;
}

# キーワードをインデックスに登録
sub registeritem {
    my $self = shift;
    my $id = shift;
    my $item = shift;

    my $indexfile = $self->getfilename($item);
    if ($indexfile eq '') {
        return 0;
    }

    my $hitline = 0;
    my $len = length($item);

    if (-f $indexfile) {
        open my $indexfh, '<', $indexfile;
        my $line = 1;
        while (my $data = <$indexfh>) {
            chomp $data;
            my $keyword = substr($data, 0, $len);
            if ($item eq $keyword) {
                $hitline = $line;
                last;
            }
            $line++;
        }
        close $indexfh;
    }

    if (!$hitline) {
        open my $outfh, '>>', $indexfile;
        flock $outfh, LOCK_EX;
        print {$outfh} "$item,$id,1\n";
        flock $outfh, LOCK_UN;
        close $outfh;
    }
    else {
        open my $indexfh, '<', $indexfile;
        my @newdata = ();
        while (my $data = <$indexfh>) {
            chomp $data;
            push @newdata, $data;
        }
        close $indexfh;

        $hitline--;
        my $data = $newdata[$hitline];
        my ($keyword, $topiclist, $topicnum) = split(/\,/, $data);
        if ($item eq $keyword) {
            my @idlist = split(/:/, $topiclist);
            foreach my $topicid (@idlist) {
                if ($topicid == $id) {
                    return 0;
                }
            }
            if ($topicnum > 0) {
                $topiclist .= ':';
            }
            $topiclist .= $id;
            $topicnum++;

            $newdata[$hitline] = "$keyword,$topiclist,$topicnum";

            open my $outfh, '>', $indexfile;
            flock $outfh, LOCK_EX;
            foreach my $data (@newdata) {
                print {$outfh} "$data\n";
            }
            flock $outfh, LOCK_UN;
            close $outfh;
        }
    }

    return 1;
}

# 検索条件の調整（検索クエリーを文字種別毎半角スペースで区切る）
sub adjust {
    my $self = shift;
    my $srcstr = shift;
    my $deststr = '';

    # 文字エンコードをシフトJIS変換
    my $string = Lib::String->new();
    $srcstr = $string->convert($srcstr, 'utf8', 'sjis');

    # 全角英数字を半角に変換
    $srcstr = $string->zen_han($srcstr);

    # 半角カナを全角に変換
    $srcstr = $string->kana_han_zen($srcstr);

    my $len = length($srcstr);
    my $prech = 0;
    my $ch;
    my $ch2;

    for (my $i = 0; $i < $len; $i++) {
        $ch = substr($srcstr, $i, 1);
        # 全角文字
        if ($ch =~ /[\x81-\x9F\xE0-\xEF]/) { 
            $i++;
            $ch2 = substr($srcstr, $i, 1);
            if ($ch =~ /[\x81]/) {
                if ($ch2 =~ /[\x5B]/) { # 長音記号
                    if ($prech != 2 && $deststr ne '') {
                        $deststr .= ' ';
                    }
                    $deststr .= $ch;
                    $deststr .= $ch2;
                    $prech = 2;
                }
                else {
                    $deststr .= ' ';
                    $prech = 0;
                }
            }
            elsif ($ch =~ /[\x82]/) {
                if ($ch2 =~ /[\x9F-\xF1]/) { # ひらがな
                    if ($prech != 1 && $deststr ne '') {
                        $deststr .= ' ';
                    }
                    $deststr .= $ch;
                    $deststr .= $ch2;
                    $prech = 1;
                }
            }
            elsif ($ch =~ /[\x83]/)  {
                if ($ch2 =~ /[\x40-\x96]/) {
                    if ($prech != 2 && $deststr ne '') { # カタカナ（全角）
                        $deststr .= ' ';
                    }
                    $deststr .= $ch;
                    $deststr .= $ch2;
                    $prech = 2;
                }
            }
            elsif ($ch =~ /[\x88-\x9F\xE0-\xEE]/) { # 漢字
                if ($prech != 3 && $deststr ne '') {
                    $deststr .= ' ';
                }
                $deststr .= $ch;
                $deststr .= $ch2;
                $prech = 3;
            }
            else {
                $deststr .= ' ';
                $prech = 0;
            }
            next;
        }

        # 半角英数
        if ($ch =~ /[\x20-\x7E]/) { # ASCII文字
            if ($prech != 0 && $deststr ne '') {
                $deststr .= ' ';
            }
            if ($ch =~ /\w/) {
                $ch = uc($ch); # 大文字に統一
                $deststr .= $ch;
            }
            else {  # 英数字以外のASCII文字は全てスペースに変換
                $deststr .= ' ';
            }
            $prech = 0;
            next;
        }
    }

    # 連続した半角スペースを取り除く
    $deststr =~ s/[ ]+/ /g;

	return $deststr;
}

# インデックスデータを検索
sub find {
    my $self = shift;
    my $item = shift;
    my $list = '';

    my $indexfile = $self->getfilename($item);
    if ($indexfile ne '') {
        # インデックスファイルを検索
        open my $infh, '<', $indexfile;
        while (my $data = <$infh>) {
            chomp $data;
            if ($data ne '') {
                my ($keyword, $topiclist, $topicnum) = split(/\,/, $data);
                if ($keyword eq $item) {
                    $list = $topiclist;
                    last;
                }
            }
        }
        close $infh;
    }

    return $list;
}

# インデックスファイル名を取得
sub getfilename {
    my $self = shift;
    my $item = shift;
    my $filename = '';
    if ($item eq '') {
        return $filename;
    }
    my $len = length($item);
    my $ch = substr($item, 0, 1);
    my $ch2;
    if ($ch !~ /[\x30-\x39\x41-\x5a]/) { # 英数以外
        $ch2 = substr($item, 1, 1);
    }

    if ($ch =~ /[\x30-\x39]/) {     # 数字
        $filename = "num$ch" . "l$len.data";
    }
    elsif ($ch =~ /[\x41-\x5a]/) {  # アルファベット
        $filename = "alpha$ch" . "l$len.data";
    }
    elsif (($ch =~ /[\x82]/) && ($ch2 =~ /[\x9F-\xF1]/)) {	# ひらがな
        if ($ch2 =~ /[\x9F-\xA8]/) {    # 0 829F - 82A8  ぁ あ ぃ い ぅ う ぇ え ぉ お
            $filename = "hiragana0l$len.data";
        }
        elsif ($ch2 =~ /[\xA9-\xB2]/) { # 1 82A9 - 82B2  か が き ぎ く ぐ け げ こ ご
            $filename = "hiragana1l$len.data";
        }
        elsif ($ch2 =~ /[\xB3-\xBC]/) { # 2 82B3 - 82BC  さ ざ し じ す ず せ ぜ そ ぞ
            $filename = "hiragana2l$len.data";
        }
        elsif ($ch2 =~ /[\xBD-\xC7]/) { # 3 82BD - 82C7  た だ ち ぢ っ つ づ て で と ど
            $filename = "hiragana3l$len.data";
        }
        elsif ($ch2 =~ /[\xC8-\xCC]/) { # 4 82C8 - 82CC  な に ぬ ね の
            $filename = "hiragana4l$len.data";
        }
        elsif ($ch2 =~ /[\xCD-\xDB]/) { # 5 82CD - 82DB  は ば ぱ ひ び ぴ ふ ぶ ぷ へ べ ぺ ほ ぼ ぽ
            $filename = "hiragana5l$len.data";
        }
        elsif ($ch2 =~ /[\xDC-\xE0]/) { # 6 82DC - 82E0  ま み む め も
            $filename = "hiragana6l$len.data";
        }
        elsif ($ch2 =~ /[\xE1-\xE6]/) { # 7 82E1 - 82E6  ゃ や ゅ ゆ ょ よ
            $filename = "hiragana7l$len.data";
        }
        elsif ($ch2 =~ /[\xE7-\xEB]/) { # 8 82E7 - 82EB  ら り る れ ろ
            $filename = "hiragana8l$len.data";
        }
        else { # 9 82EC - 82F1  ゎ わ ゐ ゑ を ん
            $filename = "hiragana9l$len.data";
        }
    }
    elsif (($ch =~ /[\x83]/) && ($ch2 =~ /[\x40-\x96]/)) {	# カタカナ（全角）
        if ($ch2 =~ /[\x40-\x49]/) {    # 0 8340 - 8349  ァ ア ィ イ ゥ ウ ェ エ ォ オ
            $filename = "katakana0l$len.data";
        }
        elsif ($ch2 =~ /[\x4A-\x53]/) { # 1 834A - 8353  カ ガ キ ギ ク グ ケ ゲ コ ゴ
            $filename = "katakana1l$len.data";
        }
        elsif ($ch2 =~ /[\x54-\x5D]/) { # 2 8354 - 835D  サ ザ シ ジ ス ズ セ ゼ ソ ゾ
            $filename = "katakana2l$len.data";
        }
        elsif ($ch2 =~ /[\x5E-\x68]/) { # 3 835E - 8368  タ ダ チ ヂ ッ ツ ヅ テ デ ト ド
            $filename = "katakana3l$len.data";
        }
        elsif ($ch2 =~ /[\x69-\x6D]/) { # 4 8369 - 836D  ナ ニ ヌ ネ ノ
            $filename = "katakana4l$len.data";
        }
        elsif ($ch2 =~ /[\x6E-\x7C]/) { # 5 836E - 837C  ハ バ パ ヒ ビ ピ フ ブ プ ヘ ベ ペ ホ ボ ポ
            $filename = "katakana5l$len.data";
        }
        elsif ($ch2 =~ /[\x7D-\x82]/) { # 6 837D - 8382  マ ミ ム メ モ
            $filename = "katakana6l$len.data";
        }
        elsif ($ch2 =~ /[\x83-\x88]/) { # 7 8383 - 8388  ャ ヤ ュ ユ ョ ヨ
            $filename = "katakana7l$len.data";
        }
        elsif ($ch2 =~ /[\x89-\x8D]/) { # 8 8389 - 838D  ラ リ ル レ ロ
            $filename = "katakana8l$len.data";
        }
        else { # 9 838E - 8396  ヮ ワ ヰ ヱ ヲ ン ヴ ヵ ヶ
            $filename = "katakana9l$len.data";
        }
    }
    elsif ($ch =~ /[\x88-\x9F\xE0-\xEE]/) { # 漢字
        if ($ch =~ /[\x88-\x89]/) {    # 0 88 - 89
            $filename = "kanji0l$len.data";
        }
        elsif ($ch =~ /[\x90-\x91]/) { # 1 90 - 91
            $filename = "kanji1l$len.data";
        }
        elsif ($ch =~ /[\x92-\x93]/) { # 2 92 - 93
            $filename = "kanji2l$len.data";
        }
        elsif ($ch =~ /[\x94-\x95]/) { # 3 94 - 95
            $filename = "kanji3l$len.data";
        }
        elsif ($ch =~ /[\x96-\x97]/) { # 4 96 - 97
            $filename = "kanji4l$len.data";
        }
        elsif ($ch =~ /[\x98-\x99]/) { # 5 98 - 99
            $filename = "kanji5l$len.data";
        }
        elsif ($ch =~ /[\x9A-\x9B]/) { # 6 9A - 9B
            $filename = "kanji6l$len.data";
        }
        elsif ($ch =~ /[\x9C-\x9D]/) { # 7 9C - 9D
            $filename = "kanji7l$len.data";
        }
        elsif ($ch =~ /[\x9E-\x9F]/) { # 8 9E - 9F
            $filename = "kanji8l$len.data";
        }
        elsif ($ch =~ /[\xE0-\xE3]/) { # 9 E0 - E3
            $filename = "kanji9l$len.data";
        }
        elsif ($ch =~ /[\xE4-\xE7]/) { # A E4 - E7
            $filename = "kanjiAl$len.data";
        }
        elsif ($ch =~ /[\xE8-\xE9]/) { # B E8 - E9
            $filename = "kanjiBl$len.data";
        }
        elsif ($ch =~ /[\xEA-\xEB]/) { # C EA - EB
            $filename = "kanjiCl$len.data";
        }
        elsif ($ch =~ /[\xEC]/) { # D EC
            $filename = "kanjiDl$len.data";
        }
        elsif ($ch =~ /[\xED]/) { # E ED
            $filename = "kanjiEl$len.data";
        }
        else { # E EE
            $filename = "kanjiFl$len.data";
        }
    }
    if ($filename ne '') {
        $filename = "$self->{data_dir}/search/" . $filename;
    }

    return $filename;
}

1;
# End of Search.pm
