#!/usr/bin/perl
#------------------------------------------------------------------------------
#    59Tracker, weblog software for personal publisher.
#    Copyright (C) 2004-2009 Kaga, Hiroaki
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#------------------------------------------------------------------------------

use strict;
use warnings;

use Lib::Logger;
use Lib::Error;
use Lib::User;
use Lib::DateTime;
use Lib::String;
use Lib::Filter;
use Lib::Mail;
use Lib::App::Topic;
use Lib::App::Comment;
use Lib::App::Archive;
use Lib::App::Conf;

my $logger = Lib::Logger->new();
my $error = Lib::Error->new();
my $topic = Lib::App::Topic->new();
my $conf = Lib::App::Conf->new();

sub act() {
    my ($act, $cgi) = @_;
    my ($operation, $object) = split(/_/, $act);

    my $topicid = $cgi->param('topicid');
    if ($topicid eq '') {
        $error->print_error('00002e');
    }
    $topic->load($topicid, 0);

    if ($act ne 'add_comment') {

        my $user = Lib::User->new();
        my $sessionid = $cgi->cookie('cookie59t');
        if ($sessionid eq '') {
            $error->print_error('00003e');
        }
        my $session_userid = $user->check_session($sessionid);
        if ($session_userid eq '') {
            $error->print_error('00003e');
        }

        my $editor = $topic->get_editor();
        if (($editor ne $session_userid) && (!$user->is_admin($session_userid))) {
            $error->print_error('00004e');
        }
    }

    if ($act eq 'add_comment') {
        _add_comment($cgi);
    }
    elsif ($act eq 'confirm_comment') {
        _confirm_comment($cgi);
    }
    elsif ($act eq 'delete_comment') {
        _delete_comment($cgi);
    }
    elsif ($act eq 'recognize_comment') {
        _recognize_comment($cgi);
    }
    else {
        $error->print_error('00002e');
    }
}

# コメントの追加
sub _add_comment() {
    my ($cgi) = @_;

    my $string = Lib::String->new();
    my $dt = Lib::DateTime->new();

    my $commentdata;
    $commentdata->{topicid}  = $cgi->param('topicid');
    $commentdata->{status}   = 1;
    $commentdata->{postdate} = $dt->local_datetime(0);
    $commentdata->{author}   = $cgi->param('author');
    $commentdata->{mailaddr} = $cgi->param('mailaddr');
    $commentdata->{hpurl}    = $cgi->param('hpurl');
    $commentdata->{text}     = $cgi->param('text');
    $commentdata->{evaluate} = $cgi->param('evaluate');
    $commentdata->{ipaddr}   = $ENV{'REMOTE_ADDR'};
    if ($commentdata->{evaluate} eq '') {
        $commentdata->{evaluate} = 0;
    }

    # 入力チェック
    _check_input($commentdata);

    $commentdata->{text} = $string->tag_invalidate($commentdata->{text}); # タグを無効化
    $commentdata->{text} =~ s/\r?\n/enc_crlf/g;
    $commentdata->{text} =~ s/,/enc_conma/g;
    $commentdata->{author} = $string->tag_invalidate($commentdata->{author});
    $commentdata->{author} =~ s/\r?\n//g;
    $commentdata->{author} =~ s/,/enc_conma/g;
    $commentdata->{mailaddr} = $string->tag_invalidate($commentdata->{mailaddr});
    $commentdata->{mailaddr} =~ s/\r?\n//g;
    $commentdata->{mailaddr} =~ s/,/enc_conma/g;
    $commentdata->{hpurl} = $string->tag_invalidate($commentdata->{hpurl});
    $commentdata->{hpurl} =~ s/\r?\n//g;
    $commentdata->{hpurl} =~ s/,/enc_conma/g;

    # データ読み込み
    $topic->load($commentdata->{topicid}, 1);
    my $topic_status = $topic->get_status();
    if ($topic_status != 1 && $topic_status != 3) {
        $error->print_error('10001e');
    }

    if ($topic_status == 3) {
        $commentdata->{status} = 2;  # 承認待ちステータスを設定
    }

    my $editor = $topic->get_editor();

    # コメントIDの取得
    my @comments = $topic->get_comments();
    my $commentid = 0;
    if (scalar(@comments) > 0) {
        $commentid = (split(/\,/, $comments[$#comments]))[0];
    }
    $commentid++;

    # メール通知先をリストアップ
    my @mailinglist = ();
    foreach my $item (@comments) {
        my ($id, $mailaddr) = (split(/\,/, $item))[0,4];
        next if (!$id);
        next if (!$mailaddr);
        $mailaddr =~ s/enc_conma/,/g;
        push @mailinglist, $mailaddr;
    }

    my $commentrec = "$commentid,$commentdata->{status},$commentdata->{postdate},$commentdata->{author},";
    $commentrec .= "$commentdata->{mailaddr},$commentdata->{hpurl},$commentdata->{text},";
    $commentrec .= "$commentdata->{evaluate},$commentdata->{ipaddr}";
    push @comments, $commentrec;
    $topic->set_comments(@comments);

    # トピックデータを設定
    $topic->set_commentnum(scalar(@comments));
    my $point = $topic->get_point() ;
    if ($commentdata->{evaluate} > 0) {
        $point += $conf->get_good_point();
    }
    else {
        $point += $conf->get_normal_point();
    }
    $topic->set_point($point);

    $topic->update();

    my $title = $topic->get_title();
    my $text = $commentdata->{author} . ' ' . $commentdata->{text};
    my $comment = Lib::App::Comment->new();
    $comment->add($commentdata->{topicid}, $commentid, $title, $commentdata->{postdate}, $text);

    $topic->get_html($conf->get_static_html(), 0);

    # おすすめの場合は更新情報に追加
    if ($commentdata->{evaluate} == 1) {
        my $archive = Lib::App::Archive->new();
        $archive->add($commentdata->{topicid});
    }

    my $topicurl = $conf->get_docroot_dir() . "/?act=view_topic&id=$commentdata->{topicid}";
    my $topictitle = $topic->get_title();

    $commentdata->{author} =~ s/enc_conma/,/g;
    $commentdata->{mailaddr} =~ s/enc_conma/,/g;
    $commentdata->{hpurl} =~ s/enc_conma/,/g;
    $commentdata->{text} =~ s/enc_conma/,/g;
    $commentdata->{text} =~ s/enc_crlf/\n/g;

    my $subject = "[$topictitle] にコメントがつきました";

    my $body = <<"END_BODY";
[$topictitle] にコメントがつきました。
$topicurl

----------

投稿者      ：$commentdata->{author}
メール      ：$commentdata->{mailaddr}
ホームページ：$commentdata->{hpurl}
コメント：
$commentdata->{text}

---
このメールは、トピックの編集者である場合、コメント投稿時に
メールアドレスを入力した場合に送信されます。
----------
END_BODY

    my $mail = Lib::Mail->new();

    # トピック投稿者へのメール通知
    if ($editor ne '') {
        my $user = Lib::User->new();
        $user->load($editor);
        my $mailaddr = $user->get_mailaddr();
        push @mailinglist, $mailaddr;
    }

    my $adminaddr = $conf->get_admin_address();
    if ($conf->get_notify_mail() && $adminaddr) {
        push @mailinglist, $adminaddr;
    }

    # 重複を削除
    my %count;
    @mailinglist = grep {!$count{$_}++} @mailinglist;

    # メール通知
    foreach my $mailaddr (@mailinglist) {
        $mail->send($subject, $body, $mailaddr);
    }

    print "Location: $topicurl", "\n\n";
}

# 入力チェック
sub _check_input() {
    my ($commentdata) = @_;

    my $filter = Lib::Filter->new();
    my $docrootdir = $conf->get_docroot_dir();

    my $referer = $ENV{'HTTP_REFERER'};# 呼び出し元を取得
    if ($referer !~ /$docrootdir/) {
        $error->print_error('10001e');
    }

    if ($filter->check_filter($commentdata->{ipaddr})) {
        $error->print_error('10001e');
    }

    # 名前
    if ($commentdata->{author} ne '') {
        $commentdata->{author} =~ s/[\r\n]//g; # 改行は削除
        if ($filter->check_filter($commentdata->{author})) {
            $error->print_error('10001e');
        }
    }

    # メールアドレス
    if ($commentdata->{mailaddr} ne '') {
        $commentdata->{mailaddr} =~ s/[\r\n]//g;		# 改行は削除
        if ($commentdata->{mailaddr} !~ /.+\@.+[.].+/) {
            $error->print_error('10002e');
        }
        if ($filter->check_filter($commentdata->{mailaddr})) {
            $error->print_error('10001e');
        }
    }

    # ホームページURL
    if ($commentdata->{hpurl} ne '') {
        $commentdata->{hpurl} =~ s/[\r\n]//g;		# 改行は削除
        if ($commentdata->{hpurl} !~ /(http:\/\/[^()<>\[\]{}"'\x7F-\xFF\x00-\x20]+)/) {
            $error->print_error('10003e');
        }
        if ($filter->check_filter($commentdata->{hpurl})) {
            $error->print_error('10001e');
        }
    }

    # コメント
    if ($commentdata->{text} eq '') {
        $error->print_error('10000e');
    }
    else {
        if ($commentdata->{text} =~ /\A[\x20-\x7E\r\n]+\z/) {
            $error->print_error('10001e');
        }
        if ($commentdata->{text} =~ /<[aA] .+>.*<\/[aA]>/) {
            $error->print_error('10001e');
        }
        if ($filter->check_filter($commentdata->{text})) {
            $error->print_error('10001e');
        }
    }
}

# コメント削除画面の表示
sub _confirm_comment() {
    my ($cgi) = @_;

    my $commentdata;
    $commentdata->{topicid}   = $cgi->param('topicid');
    $commentdata->{commentid} = $cgi->param('id');
    $commentdata->{status}    = 0;
    $commentdata->{postdate}  = '';
    $commentdata->{author}    = '';
    $commentdata->{mailaddr}  = '';
    $commentdata->{hpurl}     = '';
    $commentdata->{text}      = '';
    $commentdata->{evaluate}  = '';
    $commentdata->{ipaddr}    = '';

    if ($commentdata->{commentid} eq '') {
        $error->print_error('00002e');
    }

    # 削除対象のコメントを取得
    $topic->load($commentdata->{topicid}, 1);
    my @comments = $topic->get_comments();
    foreach my $comment (@comments) {
        my ($commentid, $status, $postdate, $author, $mailaddr, $hpurl, $text, $evaluate, $ipaddr) = split(/\,/, $comment);
        if ($commentid eq $commentdata->{commentid}) {
            $commentdata->{status}   = $status;
            $commentdata->{postdate} = $postdate;
            $commentdata->{author}   = $author;
            $commentdata->{mailaddr} = $mailaddr;
            $commentdata->{hpurl}    = $hpurl;
            $commentdata->{text}     = $text;
            $commentdata->{evaluate} = $evaluate;
            $commentdata->{ipaddr}   = $ipaddr;

            # デコード
            $commentdata->{author} =~ s/enc_conma/,/g;
            $commentdata->{mailaddr} =~ s/enc_conma/,/g;
            $commentdata->{hpurl} =~ s/enc_conma/,/g;
            $commentdata->{text} =~ s/enc_crlf/<br \/>/g;
            $commentdata->{text} =~ s/enc_conma/,/g;
            last;
        }
    }

    _show_page($commentdata, 'このコメントを削除しますか？');
}

# コメントの削除
sub _delete_comment() {
    my ($cgi) = @_;

    my $commentdata;
    $commentdata->{topicid}   = $cgi->param('topicid');
    $commentdata->{commentid} = $cgi->param('id');

    if ($commentdata->{commentid} eq '') {
        $error->print_error('00002e');
    }

    # データ読み込み
    $topic->load($commentdata->{topicid}, 1);

    my $point = $topic->get_point();

    # コメントを削除
    my @comments = $topic->get_comments();
    my @new_comments = ();
    foreach my $comment (@comments) {
        my ($commentid, $evaluate) = (split(/\,/, $comment))[0,7];
        if ($commentid eq $commentdata->{commentid}) {
            if ($evaluate > 0) {
                $point -= $conf->get_good_point();
            }
            else {
                $point -= $conf->get_normal_point();
            }
            next;
        }
        push @new_comments, $comment;
    }
    $topic->set_comments(@new_comments);

    $topic->set_point($point) ;

    $topic->update();

    my $comment = Lib::App::Comment->new();
    $comment->delete($commentdata->{topicid}, $commentdata->{commentid});

    $topic->get_html($conf->get_static_html(), 0);

    my $nextpage = "./?act=view_topic&id=$commentdata->{topicid}";
    print "Location: $nextpage", "\n\n";
}

# コメントの承認
sub _recognize_comment() {
    my ($cgi) = @_;

#    $logger->write("_recognize_comment");   # for debug

    my $commentdata;
    $commentdata->{topicid}   = $cgi->param('topicid');
    $commentdata->{commentid} = $cgi->param('id');

    if ($commentdata->{commentid} eq '') {
        $error->print_error('00002e');
    }

    # データ読み込み
    $topic->load($commentdata->{topicid}, 1);

    # コメントを承認
    my @comments = $topic->get_comments();
    my @new_comments = ();
    foreach my $comment (@comments) {
        my ($commentid, $status, $postdate, $author, $mailaddr, $hpurl, $text, $evaluate, $ipaddr) = split(/\,/, $comment);
        if ($commentid eq $commentdata->{commentid}) {
            # ステータスを有効に変更
            $comment = "$commentid,1,$postdate,$author,$mailaddr,$hpurl,$text,$evaluate,$ipaddr";
        }
        push @new_comments, $comment;
    }
    $topic->set_comments(@new_comments);

    $topic->update();

    $topic->get_html($conf->get_static_html(), 0);

    my $nextpage = "./?act=view_topic&id=$commentdata->{topicid}";
    print "Location: $nextpage", "\n\n";
}

# 削除確認画面の表示
sub _show_page() {
    my ($commentdata, $msg) = @_;

    my $caption = 'コメント削除の確認';
    my $action = 'delete_comment';

    # パスの作成
    my $topictitle = $topic->get_title();
    my $path = <<"END_PATH";
<p><strong>
<a href="./" class="path">ホーム</a>&nbsp;
&gt;&nbsp;
<a href="./?act=view_topic&id=$commentdata->{topicid}" class="path">$topictitle</a>
&gt;&nbsp;
$caption
</strong></p>
END_PATH

    $msg = "<p><strong><font class=\"message\" size=\"+1\">$msg</font></strong></p>";

    my $status = '無効';
    if ($commentdata->{status} == 1) {
        $status = '有効';
    }
    elsif ($commentdata->{status} == 2) {
        $status = '承認待ち';
    } 

    my $evaluate = 'なし';
    if ($commentdata->{evaluate} == 1) {
        $evaluate = 'おすすめ';
    }

    my $content = <<"END_CONTENT";
<div>
  [$commentdata->{commentid}] &nbsp;ステータス：$status &nbsp;評価：$evaluate
  投稿者：$commentdata->{author}（$commentdata->{ipaddr}）&nbsp;投稿日時：$commentdata->{postdate}
</div>
<div>
  $commentdata->{text}
</div>
END_CONTENT

    my $systemdir = $conf->get_system_dir();

    open my $templatefh, '<', "$systemdir/tmpl/confirm.tmpl";
    my $template = do { local $/; <$templatefh> };
    close $templatefh;

    $template =~ s/\$CAPTION\$/$caption/g;
    $template =~ s/\$PATH\$/$path/g;
    $template =~ s/\$MSG\$/$msg/g;
    $template =~ s/\$ACT\$/$action/g;
    $template =~ s/\$CONTENT\$/$content/g;
    $template =~ s/\$TOPICID\$/$commentdata->{topicid}/g;
    $template =~ s/\$ID\$/$commentdata->{commentid}/g;

    print "Content-Type: text/html\n\n";
    print $template;

    exit;
}

1;
