#!/usr/bin/perl
#------------------------------------------------------------------------------
#    59Tracker, weblog software for personal publisher.
#    Copyright (C) 2004-2009 Kaga, Hiroaki
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#------------------------------------------------------------------------------

use strict;
use warnings;

use Lib::Logger;
use Lib::Error;
use Lib::User;
use Lib::DateTime;
use Lib::App::Topic;
use Lib::App::Conf;

my $logger = Lib::Logger->new();
my $error = Lib::Error->new();
my $user = Lib::User->new();
my $topic = Lib::App::Topic->new();
my $conf = Lib::App::Conf->new();

sub act() {
    my ($act, $cgi) = @_;
    my ($operation, $object) = split(/_/, $act);

    my $topicid = $cgi->param('topicid');
    if ($topicid eq '') {
        $error->print_error('00002e');
    }
    $topic->load($topicid, 0);

    my $sessionid = $cgi->cookie('cookie59t');
    if ($sessionid eq '') {
        $error->print_error('00003e');
    }
    my $session_userid = $user->check_session($sessionid);
    if ($session_userid eq '') {
        $error->print_error('00003e');
    }

    my $editor = $topic->get_editor();
    if (($editor ne $session_userid) && (!$user->is_admin($session_userid))) {
        $error->print_error('00004e');
    }

    if ($act eq 'upload_file') {
        _upload_file($cgi);
    }
    elsif ($act eq 'post_file') {
        _post_file($session_userid, $cgi);
    }
    elsif ($act eq 'confirm_file') {
        _confirm_file($cgi);
    }
    elsif ($act eq 'delete_file') {
        _delete_file($cgi);
    }
    else {
        $error->print_error('00002e');
    }
}

# ファイル指定画面の表示
sub _upload_file() {
    my ($cgi) = @_;
#    $logger->write("_upload_file");   # for debug

    my $filedata;
    $filedata->{topicid}  = $cgi->param('topicid');
    $filedata->{filename} = '';

    # 画面表示
    _show_upload($filedata, '');
}

# 添付ファイルデータに追加
sub _post_file() {
    my ($session_userid, $cgi) = @_;

    my $filedata;
    $filedata->{topicid}  = $cgi->param('topicid');
    $filedata->{filename} = $cgi->param('filename');

    # データ読み込み
    $topic->load($filedata->{topicid}, 1);

    # ファイルIDの取得
    my @files = $topic->get_files();
    my $fileid = 0;
    if (scalar(@files) > 0) {
        $fileid = (split(/\,/, $files[$#files]))[0];
    }
    $fileid++;

    # 入力チェック
    _check_input($filedata);

    my $dt = Lib::DateTime->new();
    $filedata->{lastdate} = $dt->local_datetime(0);
    $filedata->{lastuser} = $session_userid;

    my @paths = split(/\//, $filedata->{filename});
    my ($basename, $ext) = split(/\./, $paths[-1]);
    my $newfile = $filedata->{topicid} . "_" . $fileid . "_$basename.$ext";
    my $filesdir = $conf->get_files_dir();
    my $filepath = "$filesdir/$newfile";

    my $data = '';
    my $buffer = '';
    while(read($filedata->{filename}, $buffer, 1024)) {
        $data .= $buffer;
    }

    # アップロードファイルを保存 
    open my $outfh, '>', $filepath; 
    binmode $outfh; 
    print {$outfh} $data; 
    close $outfh;

    # ファイルサイズの取得
    my $filesize = (stat($filepath))[7];

    my $file = "$fileid,$newfile,$ext,$filesize,$filedata->{lastdate},$filedata->{lastuser}";
    push @files, $file;
    $topic->set_files(@files);
    $topic->update();

    $topic->get_html($conf->get_static_html(), 0);

    my $nextpage = "./?act=view_topic&id=$filedata->{topicid}";
    print "Location: $nextpage", "\n\n";
}

# 入力チェック
sub _check_input() {
    my ($filedata) = @_;

    if ($filedata->{filename} eq '') {
        _show_upload($filedata, 'ファイルが指定されていません');
    }
}

# ファイルアップロード画面表示
sub _show_upload() {
    my ($filedata, $msg) = @_;

    my $caption = 'ファイルアップロード';

    # パスの作成
    my $topictitle = $topic->get_title();
    my $path = <<"END_PATH";
<p><strong>
<a href="./" class="path">ホーム</a>&nbsp;
&gt;&nbsp;
<a href="./?act=view_topic&id=$filedata->{topicid}" class="path">$topictitle</a>
&gt;&nbsp;
$caption
</strong></p>
END_PATH

    $msg = "<p><strong><font size=\"+1\">$msg</font></strong></p>";

    my $systemdir = $conf->get_system_dir();

    open my $templatefh, '<', "$systemdir/tmpl/upload.tmpl";
    my $template = do { local $/; <$templatefh> };
    close $templatefh;

    $template =~ s/\$CAPTION\$/$caption/g;
    $template =~ s/\$PATH\$/$path/g;
    $template =~ s/\$MSG\$/$msg/g;
    $template =~ s/\$TOPICID\$/$filedata->{topicid}/g;

    print "Content-Type: text/html\n\n";
    print $template;

    exit;
}

# 添付ファイル削除確認画面の表示
sub _confirm_file() {
    my ($cgi) = @_;

    my $filedata;
    $filedata->{topicid}  = $cgi->param('topicid');
    $filedata->{fileid}   = $cgi->param('fileid');
    $filedata->{filename}  = '';
    $filedata->{filesize}  = '';
    $filedata->{lastdate}  = '';
    $filedata->{lastuser}  = '';

    if ($filedata->{fileid} eq '') {
        $error->print_error('00002e');
    }

    # 削除対象の添付ファイルデータを取得
    $topic->load($filedata->{topicid}, 1);
    my @files = $topic->get_files();
    foreach my $file (@files) {
        my ($fileid, $filename, $fileext, $filesize, $lastdate, $lastuser) = split(/\,/, $file);
        if ($fileid eq $filedata->{fileid}) {
            $filedata->{filename} = $filename;
            $filedata->{filesize} = $filesize;
            $filedata->{lastdate} = $lastdate;
            $filedata->{lastuser} = $lastuser;
            last;
        }
    }

    _show_confirm($filedata, 'この添付ファイルを削除しますか？');
}

# 添付ファイルの削除
sub _delete_file {
    my ($cgi) = @_;

    my $filedata;
    $filedata->{topicid} = $cgi->param('topicid');
    $filedata->{fileid}  = $cgi->param('id');

    # データ読み込み
    $topic->load($filedata->{topicid}, 1);

    # 添付ファイルを削除
    my @files = $topic->get_files();
    my @new_files = ();
    foreach my $file (@files) {
        my ($fileid, $filename) = (split(/\,/, $file))[0,1];
        if ($fileid eq $filedata->{fileid}) {
            my $filesdir = $conf->get_files_dir();
            my $filepath = "$filesdir/$filename";
            unlink $filepath;
            next;
        }
        push @new_files, $file;
    }
    $topic->set_files(@new_files);

    $topic->update();

    $topic->get_html($conf->get_static_html(), 0);

    my $nextpage = "./?act=view_topic&id=$filedata->{topicid}";
    print "Location: $nextpage", "\n\n";
}

# 削除確認画面の表示
sub _show_confirm() {
    my ($filedata, $msg) = @_;

    my $caption = '添付ファイル削除の確認';
    my $action = 'delete_file';

    # パスの作成
    my $topictitle = $topic->get_title();
    my $path = <<"END_PATH";
<p><strong>
<a href="./" class="path">ホーム</a>&nbsp;
&gt;&nbsp;
<a href="./?act=view_topic&id=$filedata->{topicid}" class="path">$topictitle</a>
&gt;&nbsp;
$caption
</strong></p>
END_PATH

    $msg = "<p><strong><font class=\"message\" size=\"+1\">$msg</font></strong></p>";

    my $content = <<"END_CONTENT";
<div>
<b>$filedata->{filename} &nbsp;&nbsp;$filedata->{filesize}byte
&nbsp;&nbsp;$filedata->{lastdate} &nbsp;&nbsp;$filedata->{lastuser}</b>
</div>
END_CONTENT

    my $systemdir = $conf->get_system_dir();

    open my $templatefh, '<', "$systemdir/tmpl/confirm.tmpl";
    my $template = do { local $/; <$templatefh> };
    close $templatefh;

    $template =~ s/\$CAPTION\$/$caption/g;
    $template =~ s/\$PATH\$/$path/g;
    $template =~ s/\$MSG\$/$msg/g;
    $template =~ s/\$ACT\$/$action/g;
    $template =~ s/\$CONTENT\$/$content/g;
    $template =~ s/\$TOPICID\$/$filedata->{topicid}/g;
    $template =~ s/\$ID\$/$filedata->{fileid}/g;

    print "Content-Type: text/html\n\n";
    print $template;

    exit;
}

1;
