#!/usr/bin/perl
#------------------------------------------------------------------------------
#    59Tracker, weblog software for personal publisher.
#    Copyright (C) 2004-2009 Kaga, Hiroaki
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#------------------------------------------------------------------------------

use strict;
use warnings;

use Lib::Logger;
use Lib::Error;
use Lib::Calendar;
use Lib::DateTime;
use Lib::User;
use Lib::String;
use Lib::App::Ads;
use Lib::App::Archive;
use Lib::App::Keyword;
use Lib::App::Search;
use Lib::App::Conf;

my $logger = Lib::Logger->new();
my $error = Lib::Error->new();
my $user = Lib::User->new();
my $conf = Lib::App::Conf->new();

sub act() {
    my ($act, $cgi) = @_;
    my ($operation, $object) = split(/_/, $act);

    if ($act eq 'input_session') {
        _input_session($cgi);
    }
    elsif ($act eq 'login_session') {
        _login_session($cgi);
    }
    elsif ($act eq 'logout_session') {
        _logout_session($cgi);
    }
    elsif ($act eq 'view_session') {
        _view_list('', $cgi);
    }
    else {
        $error->print_error('00002e');
    }
}

sub _input_session() {
    my ($cgi) = @_;
    my $msg = $cgi->param('msg');
    _show_login($msg);
}

sub _login_session() {
    my ($cgi) = @_;

    # 入力チェック
    my $userid = $cgi->param('userid');
    if ($userid eq '') {
        _show_login('ユーザーIDが未入力です');
    }

    my $password = $cgi->param('password');
    if ($userid eq '') {
        _show_login('パスワードが未入力です');
    }

    my $ipaddr = $ENV{'REMOTE_ADDR'};
    my $sessionid = $user->login($userid, $password, $ipaddr);
    if (!$sessionid) {
        _show_login('ユーザーIDまたはパスワードに誤りがあります');
    }

    _view_list($sessionid, undef);
}

sub _show_login() {
    my ($msg) = @_;

    my $systemdir = $conf->get_system_dir();
    my $sitetitle = $conf->get_site_title();

    $msg = "<p><strong><font class=\"message\" size=\"+1\">$msg</font></strong></p>";

    my $caption = "$sitetitleへログイン";

    my $path = <<"END_PATH";
<p><strong>
<a href="./" class="path">ホーム</a>&nbsp;
&gt;&nbsp;
$caption
</strong></p>
END_PATH

    open my $templatefh, '<', "$systemdir/tmpl/login.tmpl";
    my $template = do { local $/; <$templatefh> };
    close $templatefh;

    $template =~ s/\$SITETITLE\$/$sitetitle/g;
    $template =~ s/\$CAPTION\$/$caption/g;
    $template =~ s/\$PATH\$/$path/g;
    $template =~ s/\$MSG\$/$msg/g;

    print "Content-Type: text/html\n\n";
    print $template;

    exit;
}

# ログアウト
sub _logout_session() {
    my ($cgi) = @_;

    # クッキーからセッションIDを取得
    my $sessionid = $cgi->cookie('cookie59t');
    if ($sessionid ne '') {
        $user->logout($sessionid);
    }

    print "Location: ./", "\n\n";
}

# リストページ表示
sub _view_list() {
    my ($sessionid, $cgi) = @_;

    my $session_userid = '';
    my $cookie = '';
    my $date = '';
    my $q = '';
    my $page = '';
    my $f = '';
    if ($sessionid eq '') {
        # セッションのチェック
        $sessionid = $cgi->cookie('cookie59t');
        if ($sessionid ne  '') {
            $session_userid = $user->check_session($sessionid);
        }
        $date = $cgi->param('date');
        $q = $cgi->param('q');
        $page = $cgi->param('page');
        $f = $cgi->param('f');
        $q =~ s/　/ /g;

        my $string = Lib::String->new();
        $q = $string->tag_invalidate($q);
        # 入力チェック
        if ($date ne '' && $date !~ /\A[0-9]+\z/) {
            $error->print_error('00002e');
        }
        if ($page ne '' && $page !~ /\A[0-9]+\z/) {
            $error->print_error('00002e');
        }
    }
    else {
        $session_userid = $user->check_session($sessionid);
        # クッキー文字列を生成
        $cookie = "Set-Cookie: cookie59t=$sessionid\n";
    }

    $page = ($page eq '') ? 1 : $page;

    my @items = ();
    my $format = 0;
    my ($year, $month, $day) = ('', '', '');
    my $archive = Lib::App::Archive->new();

    if ($date ne '') {
        $format = 1;
        $year = substr($date, 0, 4);
        $month = substr($date, 4, 2);
        if (length($date) > 6) {
            $day = substr($date, 6, 2);
            $format = 2;
        }
        @items = $archive->get_archivetopics($year, $month, $day);
    }
    else {
        # 日時を取得
        my $dt = Lib::DateTime->new();
        $dt->local_datetime(0);
        $year = $dt->get_year();
        $month = $dt->get_month();
        if ($q eq '') {
            # 更新情報
            @items = $archive->get_newtopics();
        }
        else {
            # 検索
            if ($q =~ /\A[0-9\,]+\z/) { # 数字とコンマだけの場合はID指定
                my $datadir = $conf->get_data_dir();
                @items = split(/\,/, $q);
                # データファイルが存在するかチェック
                my @checkeditems = ();
                foreach my $item (@items) {
                    my $filename = "$datadir/$item" . "tp.data";
                    if (-f $filename) {
                        push @checkeditems, $item;
                    }
                }
                @items = @checkeditems;
            }
            else {
                my $search = Lib::App::Search->new();
                @items = $search->search($q);
            }
            $format = 3;
            @items = reverse (sort { $a <=> $b; } @items); # 降順にソート
        }
    }

    if ($f eq 'rss') { # RSS出力指定
        _list_rss($q, @items);
    }
    else {
        my $listhtml = _list_html($session_userid, $format, "$year/$month/$day", $page, $q, @items);
        print "Content-Type: text/html\n";
        print "$cookie\n";
        print $listhtml;
    }
}

# 検索結果をRSS出力
sub _list_rss {
    my $q = shift;
    my @result = @_;

#    $logger->write("_list_rss");   # for debug

    my $maxnum = $conf->get_result_count();
    $maxnum--;
    @result = @result[0 .. $maxnum];

    my $archive = Lib::App::Archive->new();
    my $rssxml = $archive->get_rss2(@result);

    print "Content-Type: text/xml; charset=utf-8\n\n";
    print $rssxml;
}

# listページのhtmlソース作成
sub _list_html {
    my $session_userid = shift;
    my $format = shift;
    my $date = shift;
    my $page = shift;
    my $q = shift;
    my @result = @_;

    my $docrootdir = $conf->get_docroot_dir();
    my $sitetitle = $conf->get_site_title();
    my $subtitle = $conf->get_sub_title();

    my $caption = "$sitetitle";
    my $listtitle = '更新情報';
    my $metadisc = "<meta name=\"description\" content=\"$subtitle\">";
    my $metakeywords = "<meta name=\"keywords\" content=\"ブログ,blog,ソフトウェア,ソーシャル,ブックマーク,メディア\">";

    my $ymd = ''; 
    my $currentq = ''; 
    my ($year, $month, $day) = split(/\//, $date);
    my $currentdate = '';
    if ($format == 1) {
        $listtitle = "$year年$month月";
        $caption = "$listtitle：$sitetitle";
        $ymd = "$year年$month";
        $currentdate = "$year$month";
        $metadisc = "<meta name=\"description\" content=\"$listtitle\">";
    }
    elsif ($format == 2) {
        $listtitle = "$year年$month月$day日";
        $caption = "$listtitle：$sitetitle";
        $currentdate = "$year$month$day";
        $metadisc = "<meta name=\"description\" content=\"$listtitle\">";
    }
    elsif ($format == 3) {
        $listtitle = "$qの検索結果";
        $caption = "$listtitle：$sitetitle";
        $metadisc = "<meta name=\"description\" content=\"$listtitle\">";
        $metakeywords = "<meta name=\"keywords\" content=\"$q\">";
        $currentq = $q; 
        $currentq =~ s/(\W)/'%' . unpack('H2', $1)/eg;  # uriパック
    }

    $sitetitle = "<a class=\"sitetitle\" href=\"$docrootdir/\">$sitetitle</a>";

    my $systemdir = $conf->get_system_dir();
    my $rssurl = "$docrootdir/rss.xml";
    if ($q ne '') {
        $rssurl = "$docrootdir/?q=$currentq&f=rss";
    }

    my $freesignup = $conf->get_free_signup();
    my $usersignup = '';
    if ($freesignup) {
        $usersignup = '<a href="./?act=input_register">ユーザー登録</a>&nbsp;&nbsp;';
    }

    my $menu = <<"END_MENU";
<a href="$rssurl"><img src="feed-icon-16x16.jpg" border="0"></a> <a href="$rssurl">更新情報の購読</a>&nbsp;&nbsp;$usersignup<a href="./?act=input_session">ログイン</a>
END_MENU

    if ($session_userid ne '') { # ログイン中
        $menu = _get_menu($session_userid);
    }

    my $resultlist = '<p>該当するトピックがありません</p>';
    my $pagenavi = '';
    if (scalar(@result) > 0) { # ページナビゲーションを作成

        # 総ページ数の算出
        my $itemnum = $conf->get_result_count();
        my $pagenum = int(scalar(@result) / $itemnum); 
        if (scalar(@result) / $itemnum > $pagenum) {
            $pagenum++;
        }

        # 該当ページのユーザーIDを取り出す
        my @items = ();
        for my $i (0 .. $#result) {
            if (($i >= ($page - 1) * $itemnum) && ($i < $page * $itemnum)) {
                push @items, $result[$i];
            }
        }

        # トピック一覧を取得
        $resultlist = _get_topiclist($session_userid, @items);

        # ページナビゲーションの作成
        my $base = 1;
        my $currentnum = ($pagenum < 5) ? $pagenum : 5;
        if ($page > 5) {
            $base = $page - 4;
        }
        my @pagelist = ();
        for my $pagecount ($base .. ($base + $currentnum - 1)) {
            push @pagelist, $pagecount;
        }
        my $current = '';
        foreach my $pagecount (@pagelist) {
            if ($pagecount != $page) {
                $current .= "<a href=\"./?date=$currentdate&q=$currentq&page=$pagecount\">$pagecount</a>&nbsp;";
            }
            else {
                $current .= "$pagecount&nbsp;";
            }
        }
        my $prev = ''; 
        if ($page > 1) {
            my $prevpage = $page - 1;
            $prev = "<a href=\"./?date=$currentdate&q=$currentq&page=1\">&lt;&lt; 先頭へ</a>&nbsp;&nbsp;";
            $prev .= "<a href=\"./?date=$currentdate&q=$currentq&page=$prevpage\">&lt; 前へ</a>";
        }
        my $next = ''; 
        if ($page < $pagenum) {
            my $nextpage = $page + 1;
            $next = "<a href=\"./?date=$currentdate&q=$currentq&page=$nextpage\">次へ &gt;</a>";
        }
        $pagenavi = <<"END_PAGENAVI";
<p><strong>$prev&nbsp;&nbsp;$current&nbsp;$next</strong></p>
END_PAGENAVI
    }

    # 更新情報を取得
    my $archive = Lib::App::Archive->new();

    my $calendar = '';
    my $archivelist = '';
    my $newtrackback = '';
    my $newcomment = '';
    # カレンダーを取得
    my $cal = Lib::Calendar->new();
    $calendar = $cal->get_calendar($year, $month);

    $archivelist = $archive->archive_list();

    # 広告の取得
    my $ads = Lib::App::Ads->new();
    my $ad1 = $ads->get_ad1();
    my $ad2 = $ads->get_ad2();
    my $ad3 = $ads->get_ad3();
    my $adlist = $ads->get_adlist_table();

    # クラウドを取得
    my $keywordobj = Lib::App::Keyword->new();
    my $clouds = $keywordobj->clouds(1);

    # 検索ボックスの取得
    my $searchbox = _search_box();
    $searchbox =~ s/\$PREVQ\$/$q/g;

    # テンプレートの読み込み
    open my $templatefh, '<', "$systemdir/tmpl/list.tmpl";
    my $template = do { local $/; <$templatefh> };
    close $templatefh;

    $template =~ s/\$METADISC\$/$metadisc/g;
    $template =~ s/\$METAKEYWORDS\$/$metakeywords/g;
    $template =~ s/\$RSSURL\$/$rssurl/g;
    $template =~ s/\$CAPTION\$/$caption/g;
    $template =~ s/\$SITETITLE\$/$sitetitle/g;
    $template =~ s/\$LISTTITLE\$/$listtitle/g;
    $template =~ s/\$SUBTITLE\$/$subtitle/g;
    $template =~ s/\$MENU\$/$menu/g;
    $template =~ s/\$RESULTLIST\$/$resultlist/g;
    $template =~ s/\$PAGENAVI\$/$pagenavi/g;
    $template =~ s/\$AD1\$/$ad1/g;
    $template =~ s/\$AD2\$/$ad2/g;
    $template =~ s/\$AD3\$/$ad3/g;
    $template =~ s/\$ADLIST\$/$adlist/g;
    $template =~ s/\$SEARCHBOX\$/$searchbox/g;
    $template =~ s/\$CALENDAR\$/$calendar/g;
    $template =~ s/\$ARCHIVELIST\$/$archivelist/g;
    $template =~ s/\$CLOUDS\$/$clouds/g;

    return $template;
}

# 指定されたトピックの一覧を作成する
sub _get_topiclist {
    my $session_userid = shift;
    my @topics = @_;

    my $topiclist = '';
    my $topic = Lib::App::Topic->new();

    for my $i (0 .. $#topics) {
        my $topicid = $topics[$i];
        $topic->load($topicid);
        my $open = ($i < $conf->get_open_item_num()) ? 1 : 0;
        $topiclist .= _topic_outline($topic, $open, $session_userid);
    }

    return $topiclist;
}

# トピック概要のHTMLを作成して返す、loadが前の処理で呼び出されていることが前提
sub _topic_outline {
    my ($topic, $open, $session_userid) = @_;
    my $topicid = $topic->get_topicid();
    my $attribute;
    my $status = $topic->get_status();
    my $title = $topic->get_title();
    my $text = $topic->get_text();
    my $keywords = $topic->get_keyword();
    my $url = $topic->get_url();
    my $point = $topic->get_point();
    my $commentnum = $topic->get_commentnum();
    my $trackbacknum = $topic->get_trackbacknum();
    my $psnum = $topic->get_psnum();
    my $lastdate = $topic->get_lastdate();

    my $topicclass = 'A';
    my $bookmark = '';
    if ($url ne '') {
        my $bookmarktarget = $conf->get_bookmark_target();
        $topicclass = 'B';
        $bookmark = "&nbsp;&nbsp;<font size=\"-1\"><a href=\"$url\" $bookmarktarget>ブックマーク先へ</a></font>";
    }

    if ($status == 9) {
        $title = "削除されました";
        $text = "削除されました";
    }

    my $topichtml = "./?act=view_topic&id=$topicid";
    if ($conf->get_static_html() && $session_userid eq '') {
        $topichtml = "detail$topicid.html";
    }

    my $editor = $topic->get_editor();
    my $enceditor = $editor;
    $enceditor =~ s/(\W)/'%' . unpack('H2', $1)/eg; # uriパック
    $editor = "<a href=\"./?q=$enceditor\">$editor</a>";

    my $string = Lib::String->new();
    my $summary = $string->cut($text, $conf->get_summary_length());

    # キーワードをリンク
    my $keyword = Lib::App::Keyword->new();
    $summary = $keyword->link_keyword($keywords, $summary);

    my $psstr = '';
    if ($psnum > 0) {
        $psstr = "<div>(<a href=\"$topichtml#ps\">追記あり</a>)</div>";
    }

    my $showblock = $open ? "display:none"  : "display:block";
    my $hideblock = $open ? "display:block" : "display:none";

    my $item = <<"END_ITEM";
<div id="toggleShow$topicid" style="$showblock">
  <a class="toggle" href="javascript:;" onmousedown="toggleText('toggleText$topicid','toggleSummary$topicid','toggleShow$topicid','toggleHide$topicid');"><font class="toggle">＋</font></a>&nbsp;
  <strong><font class="mark">$topicclass</font> <a href="$topichtml">$title</a></strong>$bookmark
</div>
<div id="toggleHide$topicid" style="$hideblock">
  <a class="toggle" href="javascript:;" onmousedown="toggleText('toggleText$topicid','toggleSummary$topicid','toggleShow$topicid','toggleHide$topicid');"><font class="toggle">－</font></a>&nbsp;
  <strong><font class="mark">$topicclass</font> <a href="$topichtml">$title</a></strong>$bookmark
</div>
<div id="toggleSummary$topicid" style="$showblock">$summary</div>
<div id="toggleText$topicid" style="$hideblock">$text</div>
$psstr
<div>
[$topicid] posted by $editor at <font class="datetime">$lastdate</font> |
 <a href="$topichtml#comment">Comment ($commentnum)</a> |
 <a href="$topichtml#trackback">Trackback ($trackbacknum)</a> |
 <b><font class="point" size="+1">$point</font> point</b>
</div>
<br />
END_ITEM

    return $item;
}

# メニューを取得
sub _get_menu {
    my $session_userid = shift;

    my $menu = '';
    if ($user->is_admin($session_userid)) {
        $menu = <<"END_MENU";
<a href="./?act=input_topic">新規投稿</a>&nbsp;&nbsp;
<a href="./?act=view_userlist">ユーザー管理</a>&nbsp;&nbsp;
<a href="./?act=input_content">再構築</a>&nbsp;&nbsp;
<a href="./?act=logout_session">ログアウト</a><br />
<a href="./?act=input_password">パスワード</a>&nbsp;&nbsp;
<a href="./?act=edit_ads">広告設定</a>&nbsp;&nbsp;
<a href="./?act=edit_template&filecode=1">テンプレート</a>&nbsp;&nbsp;
<a href="./?act=edit_pinglist">Ping設定</a>&nbsp;&nbsp;
<a href="./?act=edit_filter">フィルター</a>
END_MENU
    }
    else {
        $menu = <<"END_MENU";
<a href="./?act=input_topic">新規投稿</a>&nbsp;&nbsp;
<a href="./?act=logout_session">ログアウト</a><br />
<a href="./?act=input_invite">招待</a>&nbsp;&nbsp;
<a href="./?act=edit_user&userid=$session_userid">ユーザー情報</a>&nbsp;&nbsp;
<a href="./?act=confirm_user">退会</a>&nbsp;&nbsp;
<a href="./?act=input_password">パスワード</a>
END_MENU
    }

    return $menu;
}

# リストページ用検索ボックスの作成
sub _search_box {

    my $form = <<"END_FORM";
<div class="searchbox">
<form method="post" action="./">
  <input type="text" name="q" size="60" value="\$PREVQ\$">&nbsp;
  <input type="submit" value="検索">
</form>
</div>
END_FORM

    return $form;
}

1;
