#!/usr/bin/perl
#------------------------------------------------------------------------------
#    59Tracker, weblog software for personal publisher.
#    Copyright (C) 2004-2009 Kaga, Hiroaki
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#------------------------------------------------------------------------------

use strict;
use warnings;

use Lib::Logger;
use Lib::Error;
use Lib::User;
use Lib::DateTime;
use Lib::Ping;
use Lib::String;
use Lib::Filter;
use Lib::App::Topic;
use Lib::App::Archive;
use Lib::App::Search;
use Lib::App::Keyword;
use Lib::App::Conf;

my $logger = Lib::Logger->new();
my $error = Lib::Error->new();
my $user = Lib::User->new();
my $topic = Lib::App::Topic->new();
my $conf = Lib::App::Conf->new();

sub act() {
    my ($act, $cgi) = @_;
    my ($operation, $object) = split(/_/, $act);

    my $sessionid = $cgi->cookie('cookie59t');
    my $session_userid = '';
    if ($sessionid ne '') {
        $session_userid = $user->check_session($sessionid);
    }

    if ($act eq 'view_topic') {
        _view_topic($session_userid, $cgi);
    }
    elsif ($act eq 'input_topic') {
        _input_topic($operation, $session_userid, $cgi);
    }
    elsif ($act eq 'add_topic') {
        _add_topic($operation, $session_userid, $cgi);
    }
    elsif ($act eq 'edit_topic') {
        _edit_topic($operation, $session_userid, $cgi);
    }
    elsif ($act eq 'update_topic') {
        _update_topic($operation, $session_userid, $cgi);
    }
    else {
        $error->print_error('00002e');
    }
}

# トピック詳細ページの表示
sub _view_topic() {
    my ($session_userid, $cgi) = @_;

    my $topicid = $cgi->param('id');

    # 管理者権限の有無を取得
    my $isadmin = 0;
    if ($session_userid ne '') {
        $isadmin = $user->is_admin($session_userid);
    }
    else {
        # ログインされていない状態で、静的生成の場合はリダイレクト
        if ($conf->get_static_html()) {
            my $topicurl = "detail$topicid.html";
            print "Status: 301 Moved Permanently\n";
            print "Location: $topicurl", "\n\n";
            exit;
        }
    }

    # トピックデータを読み込み
    $topic->load($topicid, 1);

    # 管理権限のチェック
    my $editor = $topic->get_editor();
    my $mode = 0;
    if ($session_userid ne '') {
        $mode = ($isadmin || ($session_userid eq $editor)) ? 1 : 2;
    }

    my $topichtml = $topic->get_html(0, $mode);

    if ($session_userid ne '') {
        # ユーザー情報を設定
        my $hpurl = $user->get_hpurl();
        $topichtml =~ s/\$AUTHOR\$/$session_userid/g;
        $topichtml =~ s/\$HPURL\$/$hpurl/g;
    }
    else {
        $topichtml =~ s/\$AUTHOR\$//g;
        $topichtml =~ s/\$HPURL\$//g;
    }

    print "Content-Type: text/html\n\n";
    print $topichtml;
}

# 新規投稿入力画面の表示
sub _input_topic() {
    my ($operation, $session_userid, $cgi) = @_;

    if ($session_userid eq '') {
#        $error->print_error('00003e');
        print "Location: ./?act=input_session", "\n\n";
        exit;
    }

    my $ptopicid = $cgi->param('topicid');
    my $keyword = '';
    my ($ad1, $ad2, $ad3) = ('', '', '');
    if ($ptopicid ne '') {
        $topic->load($ptopicid, 0);
        $keyword = $topic->get_keyword();
        my $editor = $topic->get_editor();
        if ($editor eq $session_userid) {
            $ad1 = $topic->get_ad1;
            $ad2 = $topic->get_ad2;
            $ad3 = $topic->get_ad3;
        }
    }

    my $url = $cgi->param('url');
    my $title = '';
    my $blockquote = '';
    if ($url ne '') {
        use Lib::Util;
        my $util = Lib::Util->new();
        ($title, $blockquote) = $util->get_blockquote($url);
    }

    my $topicdata;
    $topicdata->{title}      = $title;
    $topicdata->{status}     = $conf->get_default_status();
    $topicdata->{updateinfo} = 1;
    $topicdata->{convcrlf}   = 1;
    $topicdata->{updatedate} = 1;
    $topicdata->{text}       = '';
    if ($url ne '') {
        my $alink = "<a href=\"$url\" target=\"_blank\">ブックマーク先へ</a>";
        $topicdata->{text} = "$blockquote\n\n";
        $topicdata->{text} .= "$alink\n";
    }
    $topicdata->{keyword}    = $keyword;
    $topicdata->{url}        = $url;
    $topicdata->{trackback}  = '';
    $topicdata->{ad1}        = $ad1;
    $topicdata->{ad2}        = $ad2;
    $topicdata->{ad3}        = $ad3;

    # 入力画面の表示
    _show_topic($operation, $ptopicid, $topicdata, '');
}

# 新規トピックの追加
sub _add_topic() {
    my ($operation, $session_userid, $cgi) = @_;

    if ($session_userid eq '') {
        $error->print_error('00003e');
    }

    my $ptopicid = $cgi->param('topicid');

    # パラメーターの取得
    my $topicdata;
    $topicdata->{title}      = $cgi->param('title');
    $topicdata->{status}     = $cgi->param('status');
    $topicdata->{updateinfo} = ($cgi->param('updateinfo') eq '') ? 0 : 1;
    $topicdata->{convcrlf}   = ($cgi->param('convcrlf') eq '') ? 0 : 1;
    $topicdata->{updatedate} = 1;
    $topicdata->{text}       = $cgi->param('text');
    $topicdata->{keyword}    = $cgi->param('keyword');
    $topicdata->{url}        = $cgi->param('url');
    $topicdata->{trackback}  = $cgi->param('trackback');
    $topicdata->{ad1}        = $cgi->param('ad1');
    $topicdata->{ad2}        = $cgi->param('ad2');
    $topicdata->{ad3}        = $cgi->param('ad3');

    $topicdata->{keyword} =~ s/\A[ ]+//g;
    $topicdata->{keyword} =~ s/[ ]+\z//g;
    $topicdata->{keyword} =~ s/[ ]+/ /g;

    # トピックデータの読み込み
    $topic->load($ptopicid, 1);

    # 入力チェック
    _check_topic($operation, $ptopicid, $topicdata);

    $topicdata->{editor} = $session_userid;
    _add_topic_data($ptopicid, $topicdata);
}

# トピックデータの追加
sub _add_topic_data() {
    my ($ptopicid, $topicdata) = @_;

    # 新規トピックIDを取得
    my $archive = Lib::App::Archive->new();
    my $newtopicid = $archive->get_counter() + 1;
    $archive->update_counter($newtopicid);

    my $topicurl = "./?act=view_topic&id=$newtopicid";
    my $htmlurl = $topicurl;

    my $string = Lib::String->new();
    my $linknum = 0;
    if ($ptopicid ne '') {
        my @links = $topic->get_links();
        my $linkid = (split(/\,/, $links[$#links]))[0];
        $linkid++;
        my $title = $topicdata->{title};
        my $summary = $string->cut($topicdata->{text}, $conf->get_summary_length());
        $title =~ s/,/enc_conma/g;
        $summary =~ s/,/enc_conma/g;
        my $linkdata = "$linkid,$title,$htmlurl,$summary";
        push @links, $linkdata;
        $topic->set_linknum(scalar(@links));
        $topic->set_links(@links);
        $topic->update();
        # 親トピックのHTMLファイルを生成
        $topic->get_html($conf->get_static_html(), 0);
    }

    my $ipaddr = $ENV{'REMOTE_ADDR'}; # 投稿者のIPアドレス
    my $dt = Lib::DateTime->new();
    my $lastdate = $dt->local_datetime(0);

    # ユーザーデータを追加
    my $newtopic = Lib::App::Topic->new();

    if ($ptopicid ne '') {
        # 新規トピックの関連リンクに親トピックを追加
        my $ptext = $topic->get_text();
        my $ptitle = $topic->get_title();
        my $psummary = $string->cut($ptext, $conf->get_summary_length());
        my $ptopicurl = "./?act=view_topic&id=$ptopicid";
        $ptitle =~ s/,/enc_conma/g;
        $psummary =~ s/,/enc_conma/g;
        my $linkdata = "1,$ptitle,$ptopicurl,$psummary";
        $newtopic->set_links($linkdata);
        $linknum++;
        # 親トピックの広告を引き継ぐ
        if ($topicdata->{ad1} eq 'hidden') {
            $topicdata->{ad1} = $topic->get_ad1();
        }
        if ($topicdata->{ad2} eq 'hidden') {
            $topicdata->{ad2} = $topic->get_ad2();
        }
        if ($topicdata->{ad3} eq 'hidden') {
            $topicdata->{ad3} = $topic->get_ad3();
        }
    }

    # 改行の処理
    if ($topicdata->{convcrlf}) {
        $topicdata->{text} =~ s/\r?\n/<br \/>enc_crlf/g;
    }
    else {
        $topicdata->{text} =~ s/\r?\n/enc_crlf/g;
    }

    $newtopic->set_topicid($newtopicid);
    $newtopic->set_status(1);
    $newtopic->set_editor($topicdata->{editor});
    $newtopic->set_lastdate($lastdate);
    $newtopic->set_title($topicdata->{title});
    $newtopic->set_text($topicdata->{text});
    $newtopic->set_psnum(0);
    $newtopic->set_commentnum(0);
    $newtopic->set_trackbacknum(0);
    $newtopic->set_linknum($linknum);
    $newtopic->set_filenum(0);
    if ($topicdata->{ad1} ne 'hidden') {
        $newtopic->set_ad1($topicdata->{ad1});
    }
    if ($topicdata->{ad2} ne 'hidden') {
        $newtopic->set_ad2($topicdata->{ad2});
    }
    if ($topicdata->{ad3} ne 'hidden') {
        $newtopic->set_ad3($topicdata->{ad3});
    }
    $newtopic->set_point(0);
    $newtopic->set_keyword($topicdata->{keyword});
    $newtopic->set_url($topicdata->{url});
    $newtopic->set_ipaddr($ipaddr);
    $newtopic->update();

    # キーワードが設定されていある場合はキーワードデータに追加
    if ($topicdata->{keyword} ne '') {
        my @newarray = split(/ /, $topicdata->{keyword});
        my $keywordobj = Lib::App::Keyword->new();
        foreach my $keyworditem (@newarray) {
            $keywordobj->add($keyworditem, $newtopicid);
        }
    }

    # 新規に作成したトピックのHTMLファイルを生成
    $newtopic->get_html($conf->get_static_html(), 0);

    # 検索用索引に追加
    my $str = "$topicdata->{title} $topicdata->{text} $topicdata->{editor} $topicdata->{keyword}";
    my $search = Lib::App::Search->new();
    $search->addindex($newtopicid, $str);

    my $ping = Lib::Ping->new();
    if ($topicdata->{updateinfo}) {
        # 更新情報データに追加
        $archive->add_archive($newtopicid, $lastdate);
        $archive->add($newtopicid);
        # 更新Ping送信
        $ping->weblogupdates($conf->get_docroot_dir(), $conf->get_site_title());
    }

    # トラックバックの送信
    if ($topicdata->{trackback} ne '') {
        my $trackback = $topicdata->{trackback};
        $trackback =~ s/\r?\n/,/g; # 改行をコンマに置換
        my @pinglist = split(/\,/, $trackback);
        my $string = Lib::String->new();
        my $pingdata;
        $pingdata->{url} = $conf->get_docroot_dir(). "/?act=view_topic&id=$newtopicid";
        $pingdata->{blog_name} = $conf->get_site_title();
        $pingdata->{title} = $topicdata->{title};
        $topicdata->{text} =~ s/enc_crlf//g;
        $pingdata->{excerpt} = $string->cut($topicdata->{text}, $conf->get_excerpt_length());
        $ping->trackback($pingdata, @pinglist);
    }

    my $url = $topicurl;
#    if ($topicdata->{url} ne '') {
#        $url = $topicdata->{url};
#    }
    print "Location: $url", "\n\n";
}

# 編集入力画面の表示
sub _edit_topic() {
    my ($operation, $session_userid, $cgi) = @_;

    my $topicid = $cgi->param('topicid');
    if ($topicid eq '') {
        $error->print_error('00003e');
    }

    # トピックデータの読み込み
    $topic->load($topicid, 0);
    my $editor = $topic->get_editor();
    if (($editor ne $session_userid) && !$user->is_admin($session_userid)) {
        $error->print_error('00004e');
    }

    my $topicdata;
    $topicdata->{title}      = $topic->get_title();
    $topicdata->{status}     = $topic->get_status();
    $topicdata->{updateinfo} = 0;
    $topicdata->{convcrlf}   = 0;
    $topicdata->{updatedate} = $conf->get_date_update();
    $topicdata->{text}       = $topic->get_text();
    $topicdata->{keyword}    = $topic->get_keyword();
    $topicdata->{url}        = $topic->get_url();
    $topicdata->{trackback}  = '';
    $topicdata->{ad1}        = $topic->get_ad1();
    $topicdata->{ad2}        = $topic->get_ad2();
    $topicdata->{ad3}        = $topic->get_ad3();

    # 入力画面の表示
    _show_topic($operation, $topicid, $topicdata, '');
}

# トピックの更新
sub _update_topic() {
    my ($operation, $session_userid, $cgi) = @_;

    my $topicid = $cgi->param('topicid');
    if ($topicid eq '') {
        $error->print_error('00002e');
    }

    # トピックデータの読み込み
    $topic->load($topicid, 1);
    my $editor = $topic->get_editor();
    if (($editor ne $session_userid) && !$user->is_admin($session_userid)) {
        $error->print_error('00004e');
    }

    # パラメーターの取得
    my $topicdata;
    $topicdata->{title}      = $cgi->param('title');
    $topicdata->{status}     = $cgi->param('status');
    $topicdata->{updateinfo} = ($cgi->param('updateinfo') eq '') ? 0 : 1;
    $topicdata->{convcrlf}   = ($cgi->param('convcrlf') eq '') ? 0 : 1;
    $topicdata->{updatedate} = ($cgi->param('updatedate') eq '') ? 0 : 1;
    $topicdata->{text}       = $cgi->param('text');
    $topicdata->{keyword}    = $cgi->param('keyword');
    $topicdata->{url}        = $cgi->param('url');
    $topicdata->{trackback}  = $cgi->param('trackback');
    $topicdata->{ad1}        = $cgi->param('ad1');
    $topicdata->{ad2}        = $cgi->param('ad2');
    $topicdata->{ad3}        = $cgi->param('ad3');

    $topicdata->{keyword} =~ s/　/ /g;
    $topicdata->{keyword} =~ s/\A[ ]+//g;
    $topicdata->{keyword} =~ s/[ ]+\z//g;
    $topicdata->{keyword} =~ s/[ ]+/ /g;

    # 最初のトピックは削除不可
    if ($topicid == 0 && $topicdata->{status} == 9) {
        _show_topic($operation, $topicid, $topicdata, "[$topicdata->{title}]は削除できません");
    }

    # 入力チェック
    _check_topic($operation, $topicid, $topicdata);

    my $oldlastdate = $topic->get_lastdate();
    my $oldkeyword = $topic->get_keyword();

    # 改行の処理
    if ($topicdata->{convcrlf}) {
        $topicdata->{text}    =~ s/\r?\n/<br \/>enc_crlf/g;
    }
    else {
        $topicdata->{text}    =~ s/\r?\n/enc_crlf/g;
    }

    my $dt = Lib::DateTime->new();
    my $lastdate = $dt->local_datetime(0);

    $topic->set_status($topicdata->{status});
    if ($topicdata->{updatedate}) {
        $topic->set_lastdate($lastdate);
    }
    $topic->set_title($topicdata->{title});
    $topic->set_text($topicdata->{text});

    # 広告が非表示の場合は更新しない
    if ($topicdata->{ad1} ne 'hidden') {
        $topic->set_ad1($topicdata->{ad1});
    }
    if ($topicdata->{ad2} ne 'hidden') {
        $topic->set_ad2($topicdata->{ad2});
    }
    if ($topicdata->{ad3} ne 'hidden') {
        $topic->set_ad3($topicdata->{ad3});
    }
    $topic->set_keyword($topicdata->{keyword});
    $topic->set_url($topicdata->{url});
    $topic->update();

    # キーワードが設定されていある場合はキーワードデータに追加
    if ($topicdata->{keyword} ne $oldkeyword) {
        my $newkeyword = $topicdata->{keyword};
        my @addarray = ();
        my @delarray = ();
        my @newarray = split(/ /, $newkeyword);
        my @oldarray = split(/ /, $oldkeyword);

        # 新しいキーワードに存在して古いキーワードに存在しないものは新規追加
        foreach my $new (@newarray) {
            my $found = 0;
            foreach my $old (@oldarray) {
                if ($new eq $old) {
                    $found = 1;
                    last;
                }
            }
            if (!$found) {
                push @addarray, $new;
            }
        }

        # 古いキーワードに存在して新しいキーワードに存在しないものは削除
        foreach my $old (@oldarray) {
            my $found = 0;
            foreach my $new (@newarray) {
                if ($old eq $new) {
                    $found = 1;
                    last;
                }
            }
            if (!$found) {
                push @delarray, $old;
            }
        }

        my $keywordobj = Lib::App::Keyword->new();
        foreach my $item (@addarray) {
            $keywordobj->add($item, $topicid);  # 追加
        }
        foreach my $item (@delarray) {
            $keywordobj->delete($item, $topicid); # 削除
        }
    }

    my $archive = Lib::App::Archive->new();
    my $ping = Lib::Ping->new();

    # ドラフトまたは削除の場合はアーカイブデータから削除
    if ($topicdata->{status} == 0 || $topicdata->{status} == 9) {
        my $detailhtml = "detail$topicid.html";
        if (-f $detailhtml) {
            unlink $detailhtml;
        }
        $archive->delete($topicid);
        $archive->delete_archive($topicid, $oldlastdate);
    }
    else {
        # トピックのHTMLファイルを生成
        $topic->get_html($conf->get_static_html(), 0);

        my $ping = Lib::Ping->new();
        if ($topicdata->{updateinfo}) {
            # アーカイブデータに追加
            $archive->add($topicid);
            $archive->add_archive($topicid, $lastdate);
            # 更新Ping送信
            $ping->weblogupdates($conf->get_docroot_dir(), $conf->get_site_title());
        }

        # トラックバックの送信
        if ($topicdata->{trackback} ne '') {
            my $trackback = $topicdata->{trackback};
            $trackback =~ s/\r?\n/,/g; # 改行をコンマに置換
            my @pinglist = split(/\,/, $trackback);
            my $string = Lib::String->new();
            my $pingdata;
            $pingdata->{url} = $conf->get_docroot_dir(). "/?act=view_topic&id=$topicid";
            $pingdata->{blog_name} = $conf->get_site_title();
            $pingdata->{title} = $topicdata->{title};
            $topicdata->{text} =~ s/enc_crlf//g;
            $pingdata->{excerpt} = $string->cut($topicdata->{text}, $conf->get_excerpt_length());
            $ping->trackback($pingdata, @pinglist);
         }
    }

    my $topicurl = "./?act=view_topic&id=$topicid";
    print "Location: $topicurl", "\n\n";
}

# トピック入力チェック
sub _check_topic() {
    my ($operation, $topicid, $topicdata) = @_;

    # ステータス
    if ($topicdata->{status} eq '') {
        _show_topic($operation, $topicid, $topicdata, 'ステータスが選択されていません');
    }

    # トピックのタイトル
    if ($topicdata->{title} eq '') {
        _show_topic($operation, $topicid, $topicdata, 'タイトルが未入力です');
    }

    # トピックの本文
    if ($topicdata->{text} eq '') {
        _show_topic($operation, $topicid, $topicdata, '本文が未入力です');
    }

    # キーワードのチェック
    my $keywordobj = Lib::App::Keyword->new();
    if ($keywordobj->check($topicdata->{keyword})) {
        _show_topic($operation, $topicid, $topicdata, 'キーワードは半角英数または全角文字で入力してください');
    }
}

# トピック入力画面表示
sub _show_topic() {
    my ($operation, $topicid, $topicdata, $msg) = @_;

    $msg = "<p><strong><font class=\"message\" size=\"+1\">$msg</font></strong></p>";

    my $caption = '新規投稿';
    my $action = 'add_topic';
    my $button = '投稿';
    my $disabled = 'disabled';
    if (($operation eq 'edit') || ($operation eq 'update')) {
        $caption = 'トピックの編集';
        $action = 'update_topic';
        $button = '更新';
        $disabled = '';
    }
    my $path = '';
    if ($topicid ne '') {
        my $topicurl = "./?act=view_topic&id=$topicid";
        my $topictitle = $topic->get_title();
        $path = <<"END_PATH";
<p><strong>
<a href="./" class="path">ホーム</a>&nbsp;
&gt;&nbsp;
<a href="$topicurl" class="path">$topictitle</a>
&gt;&nbsp;
$caption
</strong></p>
END_PATH
    }
    else {
        $path = <<"END_PATH";
<p><strong>
<a href="./" class="path">ホーム</a>&nbsp;
&gt;&nbsp;
$caption
</strong></p>
END_PATH
    }

    # ステータスの設定
    my $statuslist = '';
    if ($topicdata->{status} == 0) {
        $statuslist = <<"END_STATUS";
<input type="radio" name="status" value="0" checked>ドラフト
<input type="radio" name="status" value="1">オープン
<input type="radio" name="status" value="2">クローズ
<input type="radio" name="status" value="3">制限
<input type="radio" name="status" value="9">削除
END_STATUS
    }
    elsif ($topicdata->{status} == 1) {
        $statuslist = <<"END_STATUS";
<input type="radio" name="status" value="0">ドラフト
<input type="radio" name="status" value="1" checked>オープン
<input type="radio" name="status" value="2">クローズ
<input type="radio" name="status" value="3">制限
<input type="radio" name="status" value="9">削除
END_STATUS
    }
    elsif ($topicdata->{status} == 2) {
        $statuslist = <<"END_STATUS";
<input type="radio" name="status" value="0">ドラフト 
<input type="radio" name="status" value="1">オープン 
<input type="radio" name="status" value="2" checked>クローズ 
<input type="radio" name="status" value="3">制限
<input type="radio" name="status" value="9">削除
END_STATUS
    }
    elsif ($topicdata->{status} == 3) {
        $statuslist = <<"END_STATUS";
<input type="radio" name="status" value="0">ドラフト 
<input type="radio" name="status" value="1">オープン 
<input type="radio" name="status" value="2">クローズ 
<input type="radio" name="status" value="3" checked>制限
<input type="radio" name="status" value="9">削除
END_STATUS
    }
    else {
        $statuslist = <<"END_STATUS";
<input type="radio" name="status" value="0">ドラフト 
<input type="radio" name="status" value="1">オープン 
<input type="radio" name="status" value="2">クローズ 
<input type="radio" name="status" value="3">制限
<input type="radio" name="status" value="9" checked>削除
END_STATUS
    }

    # オプションの設定
    my $optioncb = '';
    if ($topicdata->{updateinfo} == 1) {
        $optioncb = <<"END_OPTION";
<input type="checkbox" name="updateinfo" value="1" checked>更新情報に掲載
END_OPTION
    }
    else {
        $optioncb = <<"END_OPTION";
<input type="checkbox" name="updateinfo" value="1">更新情報に掲載
END_OPTION
    }
    if ($topicdata->{convcrlf} == 1) {
        $optioncb .= <<"END_OPTION";
&nbsp;&nbsp;<input type="checkbox" name="convcrlf" value="1" checked>改行を&lt;br /&gt;に変換
END_OPTION
    }
    else {
        $optioncb .= <<"END_OPTION";
&nbsp;&nbsp;<input type="checkbox" name="convcrlf" value="1">改行を&lt;br /&gt;に変換
END_OPTION
    }
    if ($topicdata->{updatedate} == 1) {
        $optioncb .= <<"END_OPTION";
&nbsp;&nbsp;<input type="checkbox" name="updatedate" value="1" checked $disabled>日時を更新
END_OPTION
    }
    else {
        $optioncb .= <<"END_OPTION";
&nbsp;&nbsp;<input type="checkbox" name="updatedate" value="1" $disabled>日時を更新
END_OPTION
    }

    my $refer = '';
    if ($topicdata->{url} ne '') {
        $refer = "<a href=\"$topicdata->{url}\" target=\"_blank\">参照</a>";
    }

    my $systemdir = $conf->get_system_dir();

    open my $templatefh, '<', "$systemdir/tmpl/topic.tmpl";
    my $template = do { local $/; <$templatefh> };
    close $templatefh;

    $template =~ s/\$CAPTION\$/$caption/g;
    $template =~ s/\$MSG\$/$msg/g;
    $template =~ s/\$PATH\$/$path/g;
    $template =~ s/\$TOPICID\$/$topicid/g;
    $template =~ s/\$ACT\$/$action/g;
    $template =~ s/\$TOPICTITLE\$/$topicdata->{title}/g;
    $template =~ s/\$STATUSLIST\$/$statuslist/g;
    $template =~ s/\$OPTIONCB\$/$optioncb/g;
    $template =~ s/\$TEXT\$/$topicdata->{text}/g;
    $template =~ s/\$KEYWORD\$/$topicdata->{keyword}/g;
    $template =~ s/\$URL\$/$topicdata->{url}/g;
    $template =~ s/\$REFER\$/$refer/g;
    $template =~ s/\$TRACKBACK\$/$topicdata->{trackback}/g;
    $template =~ s/\$AD1\$/$topicdata->{ad1}/g;
    $template =~ s/\$AD2\$/$topicdata->{ad2}/g;
    $template =~ s/\$AD3\$/$topicdata->{ad3}/g;
    $template =~ s/\$BUTTON\$/$button/g;

    print "Content-Type: text/html\n\n";
    print $template;

    exit;
}

1;
