#!/usr/bin/perl
#------------------------------------------------------------------------------
#    59Tracker, weblog software for personal publisher.
#    Copyright (C) 2004-2009 Kaga, Hiroaki
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#------------------------------------------------------------------------------

use strict;
use warnings;

use Lib::Logger;
use Lib::Error;
use Lib::User;
use Lib::App::Conf;

my $logger = Lib::Logger->new();
my $error = Lib::Error->new();
my $user = Lib::User->new();
my $conf = Lib::App::Conf->new();

sub act() {
    my ($act, $cgi) = @_;

    my $sessionid = $cgi->cookie('cookie59t');
    if ($sessionid eq '') {
        $error->print_error('00003e');
    }
    my $session_userid = $user->check_session($sessionid);
    if ($session_userid eq '') {
        $error->print_error('00003e');
    }

    if (!$user->is_admin($session_userid)) {
        $error->print_error('00004e');
    }

    if ($act eq 'view_userlist') {
        _view_userlist($cgi);
    }
    else {
        $error->print_error('00002e');
    }
}

# ユーザー情報一覧の表示
sub _view_userlist() {
    my ($cgi) = @_;

    my $page = $cgi->param('page');
    $page = ($page eq '') ? 1 : $page;

    my $q = $cgi->param('q');
    my $pattern = '';
    if ($q ne '') {
        $pattern = "$q.*";
    }

    my $systemdir = $conf->get_system_dir();

    # 全ユーザーのユーザーIDリストを取得
    my @result = ();
    opendir my $dirh, "$systemdir/user/";
    while (defined(my $dir = readdir $dirh)) {
        next if ($dir =~ /\A[.]+\z/);
        next if ($pattern ne '' && $dir !~ /\A$pattern\z/);
        push @result, $dir;
    }
    closedir $dirh;

    @result = sort @result;
    my $usernum = scalar(@result);

    # メッセージを作成
    my $msg = "<p><strong>ユーザー数：$usernum</strong></p>";

    my $resultlist = '<p>ユーザーが登録されていません</p>';
    my $pagenavi = '';
    if (scalar(@result)) { # ページナビゲーションを作成

        # 総ページ数の算出
        my $itemnum = $conf->get_result_count();
        my $pagenum = int(scalar(@result) / $itemnum); 
        if (scalar(@result) / $itemnum > $pagenum) {
            $pagenum++;
        }

        # 該当ページのユーザーIDを取り出す
        my @items = ();
        for my $i (0 .. $#result) {
            if (($i >= ($page - 1) * $itemnum) && ($i < $page * $itemnum)) {
                push @items, $result[$i];
            }
        }

        # ユーザー情報一覧を取得
        $resultlist = _get_userlist(@items);

        # ページナビゲーションの作成
        my $base = 1;
        my $currentnum = ($pagenum < 5) ? $pagenum : 5;
        if ($page > 5) {
            $base = $page - 4;
        }
        my @pagelist = ();
        for my $pagecount ($base .. ($base + $currentnum - 1)) {
            push @pagelist, $pagecount;
        }
        my $current = '';
        foreach my $pagecount (@pagelist) {
            if ($pagecount != $page) {
                $current .= "<a href=\"./?act=view_userlist&q=$q&page=$pagecount\">$pagecount</a>&nbsp;";
            }
            else {
                $current .= "$pagecount&nbsp;";
            }
        }
        my $prev = '';
        if ($page > 1) {
            my $prevpage = $page - 1;
            $prev = "<a href=\"./?act=view_userlist&q=$q&page=$prevpage\">前へ</a>";
        }
        my $next = '';
        if ($page < $pagenum) {
            my $nextpage = $page + 1;
            $next = "<a href=\"./?act=view_userlist&q=$q&page=$nextpage\">次へ</a>";
        }
        $pagenavi = <<"END_PAGENAVI";
<strong>$prev&nbsp;&nbsp;$current&nbsp;$next</strong>
END_PAGENAVI
    }

    my $caption = 'ユーザー管理';

    my $path = <<"END_PATH";
<p><strong>
<a href="./" class="path">ホーム</a>&nbsp;
&gt;&nbsp;
$caption
</strong></p>
END_PATH

    open my $templatefh, '<', "$systemdir/tmpl/userlist.tmpl";
    my $template = do { local $/; <$templatefh> };
    close $templatefh;

    $template =~ s/\$CAPTION\$/$caption/g;
    $template =~ s/\$PATH\$/$path/g;
    $template =~ s/\$MSG\$/$msg/g;
    $template =~ s/\$RESULTLIST\$/$resultlist/g;
    $template =~ s/\$PAGENAVI\$/$pagenavi/g;
    $template =~ s/\$PREVQ\$/$q/g;

    print "Content-Type: text/html\n\n";
    print $template;

    exit;
}

# ユーザー情報リストの作成
sub _get_userlist() {
    my (@useridlist) = @_;

    my $user = Lib::User->new();

    # ユーザー情報一覧の作成
    my $list = '';
    foreach my $userid (@useridlist) {
        $user->load($userid);
        my $mailaddr = $user->get_mailaddr();
        my $auth     = $user->get_auth();
        my $authname = '編集者';
        if ($auth == 0) {
            $authname = '管理者';
        }
        my $status = $user->get_status();
        my $statusname = '有効';
        if ($status == 0) {
            $statusname = '無効';
        }

        $list .= <<"END_ROW";
<tr>
  <td>$userid</td>
  <td width="200">$mailaddr</td>
  <td width="60" nowrap align="center">$authname</td>
  <td width="100" nowrap align="center">$statusname</td>
  <td width="40" nowrap align="center"><a href="./?act=edit_user&userid=$userid">編集</a></td>
</tr>
END_ROW
    }

    return $list;
}

1;
