unit untHttp;

interface

uses
  Classes, SysUtils, IdHttp, IdComponent, gzip, SyncObjs,
  untStreamTool;

type

  THttpErrpr = (heNoError, heBrokenGzip, heSocketError, heMoved);

  TAsyncHttp = class(TThread)
  private
    FLine: string;
    FIdHttp: TIdHttp;
    FWriteEvent: TMemoryStreamEx;
    FBufferReader: TStreamReader;
    FReadPosition: Integer;
    FAddHeaders: string;
    FBuffer: TMemoryStream;
    FGzipStream: TGzipDecompressStream;
    FIdStatus: TIdStatus;
    FIdStatusText: string;
    procedure HttpReceived(const Buff; Count: int64);
    procedure HttpStatus(ASender: TObject; const AStatus: TIdStatus; const AStatusText: string);
  protected
    procedure SyncOnReceived;
    procedure SyncOnStatus;
    procedure AddReceivedLine;
    procedure Execute; override;
  public
    URL: string;
    LastModified: string;
    UseGzip: Boolean;
    ErrorCode: THttpErrpr;
    OnStatus: TIdStatusEvent;
    OnReceived: TNotifyEvent;
    ContentLength: Integer;
    UserAgent: string;
    StartRange: Integer;
    ResponseCode: integer;
    ReceivedLines: TStringList;
    ProxyHost: string;
    ProxyPort: Integer;
    procedure Head;
    procedure AddHeader(HeaderName, Value: string);
    constructor Create;
    destructor Destroy; override;    
  end;

implementation

uses
  untGlobal, IdException;

{ TAsyncHttp }

constructor TAsyncHttp.Create;
begin
  inherited Create(true);
  FIdHttp := TIdHttp.Create(nil);
  ReceivedLines := TStringList.Create;
end;

destructor TAsyncHttp.Destroy;
begin
  ReceivedLines.Clear;
  ReceivedLines.Free;
  FIdHttp.Free;
  inherited;
end;

procedure TAsyncHttp.Head;
begin
  with FIdHttp do
  begin
    Request.Connection := 'close';
    Request.UserAgent  := UserAgent;
    OnStatus           := HttpStatus;
    if ProxyHost <> '' then
    begin
      ProxyParams.ProxyServer := ProxyHost;
      ProxyParams.ProxyPort   := ProxyPort;
      Request.Pragma          := 'no-cache';
    end;
  end;
  FIdHttp.ConnectTimeout := CONNECT_TIMEOUT;
  try
    FIdHttp.Head(URL);
  except on EIdHTTPProtocolException do;
  end;

  ResponseCode := FIdHttp.Response.ResponseCode;
  if ResponseCode = 200 then
  begin
    // VƂ
    LastModified := FIdHttp.Response.RawHeaders.Values['Last-Modified'];
  end;
end;

procedure TAsyncHttp.Execute;
var
  headers: TStringList;
  i: Integer;

begin
  ErrorCode := heNoError;

  headers := TStringList.Create;
  FWriteEvent := TMemoryStreamEx.Create;
  FBuffer := TMemoryStream.Create;
  FBufferReader := TStreamReader.Create(FBuffer);
  FGzipStream := TGzipDecompressStream.Create(FBuffer);

  try
    FWriteEvent.OnWrite := HttpReceived;
    FReadPosition := 0;

    with FIdHttp do
    begin
      Request.ContentRangeStart := StartRange;
      Request.Connection        := 'close';
      Request.UserAgent         := UserAgent;
      OnStatus                  := HttpStatus;
    end;

    if UseGzip and (StartRange = 0) then
    begin
      FIdHttp.Request.AcceptEncoding := 'gzip';
    end;

    if LastModified <> '' then
    begin
      AddHeader('If-Modified-Since', LastModified);
    end;

    headers.Text := FAddHeaders;
    for i := 0 to headers.Count - 1 do
    begin
      FIdHttp.Request.CustomHeaders.Add(headers[i]);
    end;

    if ProxyHost <> '' then
    begin
      with FIdHttp do
      begin
        ProxyParams.ProxyServer := ProxyHost;
        ProxyParams.ProxyPort   := ProxyPort;
        Request.Pragma          := 'no-cache';
      end;
    end;

    FIdHttp.ConnectTimeout := CONNECT_TIMEOUT;
    if not Terminated then
    begin
      try
        FIdHttp.Get(URL, FWriteEvent);
      except
        on EIdHTTPProtocolException do
          ;
        on EIdException do
          begin
            ErrorCode := heSocketError;
          end;
      end;
    end;

    ResponseCode := FIdHttp.Response.ResponseCode;
    if (ResponseCode = 200) or (ResponseCode = 206) then
    begin
      // VƂ
      LastModified := FIdHttp.Response.RawHeaders.Values['Last-Modified'];
    end;

    if (ResponseCode = 302) then
    begin
      ErrorCode := heMoved;
      if Assigned(OnReceived) and not Terminated then
      begin
        // DATꍇO\
        Synchronize(SyncOnReceived);
      end;
    end;

    if FIdHttp.Response.ContentEncoding = 'gzip' then
    begin
      ContentLength := FBuffer.Size;
    end else
    begin
      ContentLength := FIdHttp.Response.ContentLength;
    end;
  finally
    headers.Free;
    FBufferReader.Free;
    FWriteEvent.Free;
    FGzipStream.Free;
    FBuffer.Clear;
    FBuffer.Free;
  end;
end;

procedure TAsyncHttp.SyncOnReceived;
begin
  OnReceived(self);
end;


procedure TAsyncHttp.SyncOnStatus;
begin
  OnStatus(self, FIdStatus, FIdStatusText);
end;

procedure TAsyncHttp.AddReceivedLine;
begin
  ReceivedLines.Add(FLine);
end;

procedure TAsyncHttp.AddHeader(HeaderName, Value: string);
begin
  FAddHeaders := FAddHeaders + HeaderName + ': ' + Value + #13#10;
end;

procedure TAsyncHttp.HttpReceived(const Buff; Count: int64);
var
  line: string;

begin
  FBuffer.Seek(0, soFromEnd);

  if FIdHttp.Response.ContentEncoding = 'gzip' then
  begin
    try
      FGzipStream.Write(Buff, Count);
    except on Exception do
      begin
        ErrorCode := heBrokenGzip;
        exit;
      end;
    end;
  end else
  begin
    FBuffer.Write(Buff, Count);
  end;

  line := '';

  FBuffer.Seek(FReadPosition, soFromBeginning);
  while FBufferReader.ReadLine(line) do
  begin
    FReadPosition := FBuffer.Position;
    if Terminated then
    begin
      exit;
    end else
    begin
      FLine := line;
      Synchronize(AddReceivedLine);
    end;
  end;

  if Assigned(OnReceived) and
     (ReceivedLines.Count > 0) and
     not Terminated then
  begin
    Synchronize(SyncOnReceived);
  end;
end;

procedure TAsyncHttp.HttpStatus(ASender: TObject;
                                const AStatus: TIdStatus;
                                const AStatusText: string);
begin
  if Assigned(OnStatus) and not Terminated then
  begin
    FIdStatus := AStatus;
    FIdStatusText := AStatusText;
    Synchronize(SyncOnStatus);
  end;
end;

end.
