unit untOnlineBoard;

interface

uses
  Classes, IdComponent, Contnrs,
  untBoard, untHttp, untGlobal, untTopic;

type
  TTopicRef = class of TTopic;

  TOnlineBoard = class(TBoard)
  private
    FTopicErase: TList;
  protected
    FHttp: TAsyncHttp;
    FTopicCache: TObjectList;
    FServer: string;
    function GetBoardUrl:      string;
    function GetSubjectTxtUrl: string;
    function GetBbsCgiUrl:     string;
    procedure HttpReceivedProc(const regPattern: string);
    procedure RaiseError(ErrorCode: TFrameworkErrorType;
                         ErrorString: string);
    procedure SetServer(const serverName: string);
    procedure HttpReceived(Sender: TObject); virtual; abstract;
    procedure HttpComplete(Sender: TObject); virtual;
    procedure HttpStatusChange(axSender: TObject;
                               const axStatus: TIdStatus;
                               const asStatusText: string); virtual;
  public
    StatusText: string;
    BoardName:  string;
    WriteError: string;
    OnStateChange: TIdStatusEvent;
    ModTime: Integer;
    property BoardUrl:      string read GetBoardUrl;
    property SubjectTxtUrl: string read GetSubjectTxtUrl;
    property BbsCgiUrl:     string read GetBbsCgiUrl;
    property Server:        string read FServer write SetServer;
    constructor Create;
    destructor Destroy; override;
    procedure FreeAllTopic; override;
    procedure FreeTopic(Topic: TTopic);
    procedure GC; virtual;
    procedure SendTopicList; override;
    procedure DownloadTopic(tp: TTopic; Sender: TObject); virtual;
    procedure CancelDownload;
    procedure MakeTopic(const Title,
                              PostName,
                              PostEmail,
                              PostBody: string); virtual;
    function PostArticle(Topic: TTopic; const PostName,
                                              PostEmail,
                                              PostBody: string): TThread; virtual;
    function GetTopic(const TopicId: string): TTopic; virtual;
  end;

implementation

uses
  sysUtils, RegExpr,
  untTool, untTopicBrowser, untConfig, untTopicPostThread,
  untTopic2ch, untTopicJBBS, untBoard2ch, untBoardJBBS;

procedure TOnlineBoard.CancelDownload;
begin
  if State <> stIdling then
  begin
    FHttp.Terminate;
  end;
end;

function TOnlineBoard.PostArticle(Topic: TTopic; const PostName, PostEmail, PostBody: string): TThread;
var
  postThread: TTopicPostThread;

begin
  // epXbh𗧂グ
  postThread := TTopicPostThread.Create(self,
                                        Topic,
                                        '',
                                        PostName,
                                        PostEmail,
                                        PostBody);
  postThread.Priority := tpLower;
  postThread.FreeOnTerminate := true;
  result := PostThread;
end;

procedure TOnlineBoard.MakeTopic(const Title,
                                       PostName,
                                       PostEmail,
                                       PostBody: string);
var
  postThread: TTopicPostThread;

begin
  // X
  postThread := TTopicPostThread.Create(self,
                                        nil,
                                        Title,
                                        PostName,
                                        PostEmail,
                                        PostBody);
  postThread.Priority := tpLower;
  postThread.Resume;
end;

procedure TOnlineBoard.GC;
var
  i: integer;
  findIndex: integer;

begin
  // ݔ
  for i := 0 to FTopicErase.Count - 1 do
  begin
    findIndex := FTopicCache.IndexOf(FTopicErase[i]);
    // Ƃ͍폜
    if findIndex <> -1 then
    begin
      FTopicCache.Delete(findIndex);
    end;
  end;
  FTopicErase.Clear;
end;

procedure TOnlineBoard.FreeTopic(Topic: TTopic);
var
  i: integer;

begin
  // JĂgsbN͉Ă͂
  if Topic.IsOpened then
  begin
    exit;
  end;
  // Xbhqɂ̃gsbN͉Ă͂
  if Topic.IsInBox then
  begin
    // bZ[W͉Ă悢
    Topic.FreeMessage;
    exit;
  end;
  i := FTopicCache.IndexOf(Topic);
  if (i > -1) then
  begin
    FTopicErase.Add(Topic);
  end;
end;

procedure TOnlineBoard.FreeAllTopic;
var
  i: integer;

begin
  for i := 0 to FTopicCache.Count - 1 do
  begin
    FreeTopic((FtopicCache[i] as TTopic));
  end;
  GC;
end;

function TOnlineBoard.GetTopic(const TopicId: string): TTopic;
var
  newTopic: TTopic;
  bbsType: TBBSType;
  i: integer;

begin
  result := nil;
  for i := 0 to FTopicCache.Count - 1 do
  begin
    if TopicId = TTopic(FTopicCache[i]).TopicId then
    begin
      // LbVǂݎ
      result := TTopic(FTopicCache[i]);
      result.NewMessageCount := 0;
      break;
    end;
  end;
  if not Assigned(result) then
  begin
    // VgsbN
    bbsType := GetBBSType(self.Server);
    if bbsType = bt2ch then
    begin
      newTopic := TTopic2ch.Create(self, TopicId);
    end else if bbsType = btJBBS then
    begin
      newTopic := TTopicJBBS.Create(self, TopicId);
    end else
    begin
      newTopic := nil;
    end;
    newTopic.LoadIdx;
    FTopicCache.Add(newTopic);
    result := newTopic;
  end;
end;

procedure TOnlineBoard.DownloadTopic(tp: TTopic; Sender: TObject);
var
  topicBrowser: TTopicBrowser;

begin
  if Sender is TTopicBrowser then
  begin
    topicBrowser := Sender as TTopicBrowser;
    topicBrowser.Topic := tp;
    tp.OnStateChange := topicBrowser.Topic_ChangeDownloadState;
    tp.OnReceived    := topicBrowser.Topic_MessageReceived;
  end;
  tp.IsOpened := true;
  tp.Download;
end;

constructor TOnlineBoard.Create;
begin
  inherited;
  FTopicErase := TList.Create;
  FTopicCache := TObjectList.Create;
end;

procedure TOnlineBoard.SetServer(const serverName: string);
begin
  FServer := serverName;
end;

function TOnlineBoard.GetBoardUrl: string;
begin
  result := 'http://' + FServer + '/' + BoardName + '/';
end;

function TOnlineBoard.GetSubjectTxtUrl: string;
begin
  result := GetBoardUrl() + 'subject.txt';
end;

function TOnlineBoard.GetBbsCgiUrl: string;
begin
  result := 'http://' + FServer + '/test/bbs.cgi';
end;

procedure TOnlineBoard.SendTopicList;
begin
  inherited;

  FHttp := TAsyncHttp.Create;
  FHttp.FreeOnTerminate := true;
  FHttp.UserAgent := gConfig.UserAgent;
  FHttp.AddHeader('X-2ch-UA', APP_2chUA);
  gConfig.InitReadProxy(FHttp);
  FHttp.OnStatus := HttpStatusChange;
  TopicList.Clear;
end;

procedure TOnlineBoard.RaiseError(ErrorCode: TFrameworkErrorType;
                                  ErrorString: string);
begin
  if Assigned(OnError) then
  begin
    OnError(self, ErrorCode, ErrorString);
  end;
end;

procedure TOnlineBoard.HttpReceivedProc(const regPattern: string);
var
  line, topicId, topicTitle: string;
  regEx: TRegExpr;
  topic: TTopic;
  i: Integer;
  topicMsgCount: integer;

begin
  regEx := TRegExpr.Create;
  try
    regEx.Expression := regPattern;
    for i := 0 to FHttp.ReceivedLines.Count - 1 do
    begin
      line := FHttp.ReceivedLines[i];

      if regEx.Exec(line) then
      begin
        topicId       := regex.Substitute('$1');
        topicTitle    := regex.Substitute('$2');
        topicMsgCount := StrToIntNeo(Regex.Substitute('$3'));

        // ꕶϊ
        topicTitle := StringReplace(topicTitle, '&gt;', '>', [rfReplaceAll]);
        topicTitle := StringReplace(topicTitle, '&lt;', '<', [rfReplaceAll]);

        // gsbN̐
        topic := GetTopic(topicId);
        topic.Index        := i + ReceivedIndex + 1;
        topic.Title        := topicTitle;
        topic.MessageCount := topicMsgCount;
        topic.NoIndex      := false;
        topic.OnlineBoard  := self;
        TopicList.Add(topic);
      end else
      begin
        RaiseError(etParse, '̓G[');
      end;
    end;
   finally
    regEx.Free;
  end;
  // Cxgs
  if Assigned(OnReceived) then
  begin
    OnReceived(self);
  end;
  ReceivedIndex := TopicList.Count;
  FHttp.ReceivedLines.Clear;
end;

procedure TOnlineBoard.HttpStatusChange(axSender: TObject;
                                        const axStatus: TIdStatus;
                                        const asStatusText: string);
  procedure ChangeStatusText(const NewText: string);
  begin
    StatusText := NewText;
    if Assigned(OnStateChange) then
    begin
      OnStateChange(self, axStatus, NewText);
    end;
  end;

begin
  case axStatus of
    hsConnecting:
    begin
      ChangeStatusText(Server + 'ɐڑ');
    end;
    hsConnected:
    begin
      ChangeStatusText(Server + 'ɐڑ܂');
    end;
    hsDisconnected:
    begin
      ChangeStatusText('');
    end;
  end;
end;

destructor TOnlineBoard.Destroy;
begin
  FTopicErase.Free;
  FTopicCache.Clear;
  FTopicCache.Free;

  inherited;
end;

procedure TOnlineBoard.HttpComplete(Sender: TObject);
begin
  if FHttp.ErrorCode = heNoError then
  begin
    StatusText := IntToStr(TopicList.Count) + '̃XM';
  end else if FHttp.ErrorCode = heMoved then
  begin
    StatusText := 'ړ]܂';
  end else
  begin
    StatusText := 'ڑG[';
  end;
  if Assigned(OnComplete) then
  begin
    OnComplete(self);
  end;
  State := stIdling;
end;

end.
