/*******************************************************************************
 * ManjyuRss
 * Copyright (C) 2012 Toshiki IGA
 * 
 * This library is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library.  If not, see <http://www.gnu.org/licenses/>.
 *******************************************************************************/
/*******************************************************************************
 * Copyright (c) 2012 Toshiki IGA and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *      Toshiki IGA - initial API and implementation
 *******************************************************************************/
/*******************************************************************************
 * Copyright 2012 Toshiki IGA and others.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *******************************************************************************/
package org.manjyu.rss;

import java.io.IOException;
import java.io.OutputStream;

import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.sax.SAXTransformerFactory;
import javax.xml.transform.sax.TransformerHandler;
import javax.xml.transform.stream.StreamResult;

import org.manjyu.rss.util.ManjyuRssDateUtil;
import org.manjyu.rss.util.ManjyuRssSerializerUtil;
import org.manjyu.rss.vo.ManjyuRss;
import org.manjyu.rss.vo.ManjyuRssAttribute;
import org.manjyu.rss.vo.ManjyuRssAttributes;
import org.manjyu.rss.vo.ManjyuRssCategory;
import org.manjyu.rss.vo.ManjyuRssChannel;
import org.manjyu.rss.vo.ManjyuRssCloud;
import org.manjyu.rss.vo.ManjyuRssEnclosure;
import org.manjyu.rss.vo.ManjyuRssImage;
import org.manjyu.rss.vo.ManjyuRssItem;
import org.manjyu.rss.vo.ManjyuRssSource;
import org.xml.sax.SAXException;

/**
 * RSS 2.0 Serializer
 * 
 * @author Toshiki Iga
 */
public abstract class ManjyuRssSerializer {
	protected TransformerHandler saxHandler = null;

	protected int indent = 0;

	/**
	 * User code entry point to build item list.
	 * 
	 * @throws SAXException
	 * @throws IOException
	 */
	protected abstract void processItems() throws IOException;

	/**
	 * Entry point to serialize RSS 2.0.
	 * 
	 * @param rss
	 * @param outStream
	 * @throws IOException
	 */
	public void serialize(final ManjyuRss rss, final OutputStream outStream) throws IOException {
		final TransformerFactory tf = TransformerFactory.newInstance();
		final SAXTransformerFactory saxTf = (SAXTransformerFactory) tf;
		try {
			saxHandler = saxTf.newTransformerHandler();
			saxHandler.getTransformer().setOutputProperty("version", "1.0");
			saxHandler.getTransformer().setOutputProperty("encoding", "UTF-8");
		} catch (TransformerConfigurationException e) {
			throw new IllegalArgumentException("Fail to generate TransformerHandler: " + e.toString(), e);
		}

		saxHandler.setResult(new StreamResult(outStream));

		try {
			ManjyuRssSerializerUtil.newLine(saxHandler);

			saxHandler.startDocument();

			final ManjyuRssAttributes attrs = new ManjyuRssAttributes();
			{
				final ManjyuRssAttribute attr = new ManjyuRssAttribute();
				attrs.getAttrList().add(attr);
				attr.setLocalName("version");
				attr.setValue(rss.getVersion());
			}
			saxHandler.startElement(null, "rss", "rss", attrs);
			ManjyuRssSerializerUtil.newLine(saxHandler);

			serializeChannel(rss.getChannel());

			saxHandler.endElement(null, "rss", "rss");
			ManjyuRssSerializerUtil.newLine(saxHandler);

			saxHandler.endDocument();
		} catch (SAXException ex) {
			throw new IOException(ex.toString(), ex);
		}

		outStream.flush();
	}

	void serializeChannel(final ManjyuRssChannel channel) throws SAXException, IOException {
		indent++;

		{
			ManjyuRssSerializerUtil.indent(indent, saxHandler);
			final char[] commentChars = (" Generated by ManjyuRss (" + ManjyuRssConstants.getVersion() + ") ")
					.toCharArray();
			saxHandler.comment(commentChars, 0, commentChars.length);
			ManjyuRssSerializerUtil.newLine(saxHandler);
		}

		ManjyuRssSerializerUtil.indent(indent, saxHandler);
		saxHandler.startElement(null, "channel", "channel", null);
		ManjyuRssSerializerUtil.newLine(saxHandler);

		indent++;

		if (channel.getTitle() != null) {
			ManjyuRssSerializerUtil.indent(indent, saxHandler);
			saxHandler.startElement(null, "title", "title", null);
			ManjyuRssSerializerUtil.writeCharacters(channel.getTitle(), saxHandler);
			saxHandler.endElement(null, "title", "title");
			ManjyuRssSerializerUtil.newLine(saxHandler);
		}

		if (channel.getLink() != null) {
			ManjyuRssSerializerUtil.indent(indent, saxHandler);
			saxHandler.startElement(null, "link", "link", null);
			ManjyuRssSerializerUtil.writeCharacters(channel.getLink(), saxHandler);
			saxHandler.endElement(null, "link", "link");
			ManjyuRssSerializerUtil.newLine(saxHandler);
		}

		if (channel.getDescription() != null) {
			ManjyuRssSerializerUtil.indent(indent, saxHandler);
			saxHandler.startElement(null, "description", "description", null);
			ManjyuRssSerializerUtil.writeCharacters(channel.getDescription(), saxHandler);
			saxHandler.endElement(null, "description", "description");
			ManjyuRssSerializerUtil.newLine(saxHandler);
		}

		if (channel.getLanguage() != null) {
			ManjyuRssSerializerUtil.indent(indent, saxHandler);
			saxHandler.startElement(null, "language", "language", null);
			ManjyuRssSerializerUtil.writeCharacters(channel.getLanguage(), saxHandler);
			saxHandler.endElement(null, "language", "language");
			ManjyuRssSerializerUtil.newLine(saxHandler);
		}

		if (channel.getCopyright() != null) {
			ManjyuRssSerializerUtil.indent(indent, saxHandler);
			saxHandler.startElement(null, "copyright", "copyright", null);
			ManjyuRssSerializerUtil.writeCharacters(channel.getCopyright(), saxHandler);
			saxHandler.endElement(null, "copyright", "copyright");
			ManjyuRssSerializerUtil.newLine(saxHandler);
		}

		if (channel.getManagingEditor() != null) {
			ManjyuRssSerializerUtil.indent(indent, saxHandler);
			saxHandler.startElement(null, "managingEditor", "managingEditor", null);
			ManjyuRssSerializerUtil.writeCharacters(channel.getManagingEditor(), saxHandler);
			saxHandler.endElement(null, "managingEditor", "managingEditor");
			ManjyuRssSerializerUtil.newLine(saxHandler);
		}

		if (channel.getWebMaster() != null) {
			ManjyuRssSerializerUtil.indent(indent, saxHandler);
			saxHandler.startElement(null, "webMaster", "webMaster", null);
			ManjyuRssSerializerUtil.writeCharacters(channel.getWebMaster(), saxHandler);
			saxHandler.endElement(null, "webMaster", "webMaster");
			ManjyuRssSerializerUtil.newLine(saxHandler);
		}

		if (channel.getPubDate() != null) {
			ManjyuRssSerializerUtil.indent(indent, saxHandler);
			saxHandler.startElement(null, "pubDate", "pubDate", null);
			ManjyuRssSerializerUtil.writeCharacters(ManjyuRssDateUtil.date2Rfc822String(channel.getPubDate()),
					saxHandler);
			saxHandler.endElement(null, "pubDate", "pubDate");
			ManjyuRssSerializerUtil.newLine(saxHandler);
		}

		if (channel.getLastBuildDate() != null) {
			ManjyuRssSerializerUtil.indent(indent, saxHandler);
			saxHandler.startElement(null, "lastBuildDate", "lastBuildDate", null);
			ManjyuRssSerializerUtil.writeCharacters(ManjyuRssDateUtil.date2Rfc822String(channel.getLastBuildDate()),
					saxHandler);
			saxHandler.endElement(null, "lastBuildDate", "lastBuildDate");
			ManjyuRssSerializerUtil.newLine(saxHandler);
		}

		// category
		for (ManjyuRssCategory category : channel.getCategoryList()) {
			serializeCategory(category);
		}

		if (channel.getGenerator() != null) {
			ManjyuRssSerializerUtil.indent(indent, saxHandler);
			saxHandler.startElement(null, "generator", "generator", null);
			ManjyuRssSerializerUtil.writeCharacters(channel.getGenerator(), saxHandler);
			saxHandler.endElement(null, "generator", "generator");
			ManjyuRssSerializerUtil.newLine(saxHandler);
		}

		if (channel.getDocs() != null) {
			ManjyuRssSerializerUtil.indent(indent, saxHandler);
			saxHandler.startElement(null, "docs", "docs", null);
			ManjyuRssSerializerUtil.writeCharacters(channel.getDocs(), saxHandler);
			saxHandler.endElement(null, "docs", "docs");
			ManjyuRssSerializerUtil.newLine(saxHandler);
		}

		// cloud
		if (channel.getCloud() != null) {
			serializeCloud(channel.getCloud());
		}

		if (channel.getTtl() != null) {
			ManjyuRssSerializerUtil.indent(indent, saxHandler);
			saxHandler.startElement(null, "ttl", "ttl", null);
			ManjyuRssSerializerUtil.writeCharacters(channel.getTtl(), saxHandler);
			saxHandler.endElement(null, "ttl", "ttl");
			ManjyuRssSerializerUtil.newLine(saxHandler);
		}

		if (channel.getImage() != null) {
			serializeImage(channel.getImage());
		}

		// rating

		// textInput

		// skipHours

		// skipDays

		/////////////////////////////////////////////////////////
		// Do process items.
		processItems();

		/////////////////////////////////////////////////////////
		// End of channel.

		indent--;

		ManjyuRssSerializerUtil.indent(indent, saxHandler);
		saxHandler.endElement(null, "channel", "channel");
		ManjyuRssSerializerUtil.newLine(saxHandler);
		indent--;
	}

	void serializeCategory(final ManjyuRssCategory category) throws SAXException {
		final ManjyuRssAttributes attrs = new ManjyuRssAttributes();
		{
			final ManjyuRssAttribute attr = new ManjyuRssAttribute();
			attrs.getAttrList().add(attr);
			attr.setLocalName("domain");
			attr.setValue(category.getDomain());
		}

		ManjyuRssSerializerUtil.indent(indent, saxHandler);

		saxHandler.startElement(null, "category", "category", attrs);
		ManjyuRssSerializerUtil.writeCharacters(category.getCharacters(), saxHandler);
		saxHandler.endElement(null, "category", "category");
		ManjyuRssSerializerUtil.newLine(saxHandler);
	}

	void serializeImage(final ManjyuRssImage image) throws SAXException {
		ManjyuRssSerializerUtil.indent(indent, saxHandler);
		saxHandler.startElement(null, "image", "image", null);
		ManjyuRssSerializerUtil.newLine(saxHandler);

		indent++;

		if (image.getUrl() != null) {
			ManjyuRssSerializerUtil.indent(indent, saxHandler);
			saxHandler.startElement(null, "url", "url", null);
			ManjyuRssSerializerUtil.writeCharacters(image.getUrl(), saxHandler);
			saxHandler.endElement(null, "url", "url");
			ManjyuRssSerializerUtil.newLine(saxHandler);
		}

		if (image.getTitle() != null) {
			ManjyuRssSerializerUtil.indent(indent, saxHandler);
			saxHandler.startElement(null, "title", "title", null);
			ManjyuRssSerializerUtil.writeCharacters(image.getTitle(), saxHandler);
			saxHandler.endElement(null, "title", "title");
			ManjyuRssSerializerUtil.newLine(saxHandler);
		}

		if (image.getLink() != null) {
			ManjyuRssSerializerUtil.indent(indent, saxHandler);
			saxHandler.startElement(null, "link", "link", null);
			ManjyuRssSerializerUtil.writeCharacters(image.getLink(), saxHandler);
			saxHandler.endElement(null, "link", "link");
			ManjyuRssSerializerUtil.newLine(saxHandler);
		}

		if (image.getDescription() != null) {
			ManjyuRssSerializerUtil.indent(indent, saxHandler);
			saxHandler.startElement(null, "description", "description", null);
			ManjyuRssSerializerUtil.writeCharacters(image.getDescription(), saxHandler);
			saxHandler.endElement(null, "description", "description");
			ManjyuRssSerializerUtil.newLine(saxHandler);
		}

		if (image.getWidth() != null) {
			ManjyuRssSerializerUtil.indent(indent, saxHandler);
			saxHandler.startElement(null, "width", "width", null);
			ManjyuRssSerializerUtil.writeCharacters(image.getWidth(), saxHandler);
			saxHandler.endElement(null, "width", "width");
			ManjyuRssSerializerUtil.newLine(saxHandler);
		}

		if (image.getHeight() != null) {
			ManjyuRssSerializerUtil.indent(indent, saxHandler);
			saxHandler.startElement(null, "height", "height", null);
			ManjyuRssSerializerUtil.writeCharacters(image.getHeight(), saxHandler);
			saxHandler.endElement(null, "height", "height");
			ManjyuRssSerializerUtil.newLine(saxHandler);
		}

		indent--;

		ManjyuRssSerializerUtil.indent(indent, saxHandler);
		saxHandler.endElement(null, "image", "image");
		ManjyuRssSerializerUtil.newLine(saxHandler);
	}

	void serializeCloud(final ManjyuRssCloud cloud) throws SAXException {
		final ManjyuRssAttributes attrs = new ManjyuRssAttributes();

		if (cloud.getDomain() != null) {
			final ManjyuRssAttribute attr = new ManjyuRssAttribute();
			attrs.getAttrList().add(attr);
			attr.setLocalName("domain");
			attr.setValue(cloud.getDomain());
		}
		if (cloud.getPort() != null) {
			final ManjyuRssAttribute attr = new ManjyuRssAttribute();
			attrs.getAttrList().add(attr);
			attr.setLocalName("port");
			attr.setValue(cloud.getPort());
		}
		if (cloud.getPath() != null) {
			final ManjyuRssAttribute attr = new ManjyuRssAttribute();
			attrs.getAttrList().add(attr);
			attr.setLocalName("path");
			attr.setValue(cloud.getPath());
		}
		if (cloud.getRegisterProcedure() != null) {
			final ManjyuRssAttribute attr = new ManjyuRssAttribute();
			attrs.getAttrList().add(attr);
			attr.setLocalName("registerProcedure");
			attr.setValue(cloud.getRegisterProcedure());
		}
		if (cloud.getProtocol() != null) {
			final ManjyuRssAttribute attr = new ManjyuRssAttribute();
			attrs.getAttrList().add(attr);
			attr.setLocalName("protocol");
			attr.setValue(cloud.getProtocol());
		}

		ManjyuRssSerializerUtil.indent(indent, saxHandler);
		saxHandler.startElement(null, "cloud", "cloud", attrs);
		saxHandler.endElement(null, "cloud", "cloud");
		ManjyuRssSerializerUtil.newLine(saxHandler);
	}

	void serializeEnclosure(final ManjyuRssEnclosure enclosure) throws SAXException {
		final ManjyuRssAttributes attrs = new ManjyuRssAttributes();

		if (enclosure.getUrl() != null) {
			final ManjyuRssAttribute attr = new ManjyuRssAttribute();
			attrs.getAttrList().add(attr);
			attr.setLocalName("url");
			attr.setValue(enclosure.getUrl());
		}
		if (enclosure.getLength() != null) {
			final ManjyuRssAttribute attr = new ManjyuRssAttribute();
			attrs.getAttrList().add(attr);
			attr.setLocalName("length");
			attr.setValue(enclosure.getLength());
		}
		if (enclosure.getType() != null) {
			final ManjyuRssAttribute attr = new ManjyuRssAttribute();
			attrs.getAttrList().add(attr);
			attr.setLocalName("type");
			attr.setValue(enclosure.getType());
		}

		ManjyuRssSerializerUtil.indent(indent, saxHandler);
		saxHandler.startElement(null, "enclosure", "enclosure", attrs);
		saxHandler.endElement(null, "enclosure", "enclosure");
		ManjyuRssSerializerUtil.newLine(saxHandler);
	}

	void serializeSource(final ManjyuRssSource source) throws SAXException {
		final ManjyuRssAttributes attrs = new ManjyuRssAttributes();
		{
			final ManjyuRssAttribute attr = new ManjyuRssAttribute();
			attrs.getAttrList().add(attr);
			attr.setLocalName("url");
			attr.setValue(source.getUrl());
		}

		ManjyuRssSerializerUtil.indent(indent, saxHandler);

		saxHandler.startElement(null, "source", "source", attrs);
		ManjyuRssSerializerUtil.writeCharacters(source.getCharacters(), saxHandler);
		saxHandler.endElement(null, "source", "source");
		ManjyuRssSerializerUtil.newLine(saxHandler);
	}

	protected void serializeItem(final ManjyuRssItem item) throws IOException {
		try {
			ManjyuRssSerializerUtil.indent(indent, saxHandler);
			saxHandler.startElement(null, "item", "item", null);

			ManjyuRssSerializerUtil.newLine(saxHandler);

			indent++;

			if (item.getTitle() != null) {
				ManjyuRssSerializerUtil.indent(indent, saxHandler);
				saxHandler.startElement(null, "title", "title", null);
				ManjyuRssSerializerUtil.writeCharacters(item.getTitle(), saxHandler);
				saxHandler.endElement(null, "title", "title");
				ManjyuRssSerializerUtil.newLine(saxHandler);
			}

			if (item.getLink() != null) {
				ManjyuRssSerializerUtil.indent(indent, saxHandler);
				saxHandler.startElement(null, "link", "link", null);
				ManjyuRssSerializerUtil.writeCharacters(item.getLink(), saxHandler);
				saxHandler.endElement(null, "link", "link");
				ManjyuRssSerializerUtil.newLine(saxHandler);
			}

			if (item.getDescription() != null) {
				ManjyuRssSerializerUtil.indent(indent, saxHandler);
				saxHandler.startElement(null, "description", "description", null);
				ManjyuRssSerializerUtil.writeCharacters(item.getDescription(), saxHandler);
				saxHandler.endElement(null, "description", "description");
				ManjyuRssSerializerUtil.newLine(saxHandler);
			}

			if (item.getAuthor() != null) {
				ManjyuRssSerializerUtil.indent(indent, saxHandler);
				saxHandler.startElement(null, "author", "author", null);
				ManjyuRssSerializerUtil.writeCharacters(item.getAuthor(), saxHandler);
				saxHandler.endElement(null, "author", "author");
				ManjyuRssSerializerUtil.newLine(saxHandler);
			}

			// list of category
			for (ManjyuRssCategory category : item.getCategoryList()) {
				serializeCategory(category);
			}

			if (item.getComments() != null) {
				ManjyuRssSerializerUtil.indent(indent, saxHandler);
				saxHandler.startElement(null, "comments", "comments", null);
				ManjyuRssSerializerUtil.writeCharacters(item.getComments(), saxHandler);
				saxHandler.endElement(null, "comments", "comments");
				ManjyuRssSerializerUtil.newLine(saxHandler);
			}

			if (item.getEnclosure() != null) {
				serializeEnclosure(item.getEnclosure());
			}

			if (item.getGuid() != null) {
				ManjyuRssSerializerUtil.indent(indent, saxHandler);
				saxHandler.startElement(null, "guid", "guid", null);
				ManjyuRssSerializerUtil.writeCharacters(item.getGuid(), saxHandler);
				saxHandler.endElement(null, "guid", "guid");
				ManjyuRssSerializerUtil.newLine(saxHandler);
			}

			if (item.getPubDate() != null) {
				ManjyuRssSerializerUtil.indent(indent, saxHandler);
				saxHandler.startElement(null, "pubDate", "pubDate", null);
				ManjyuRssSerializerUtil.writeCharacters(ManjyuRssDateUtil.date2Rfc822String(item.getPubDate()),
						saxHandler);
				saxHandler.endElement(null, "pubDate", "pubDate");
				ManjyuRssSerializerUtil.newLine(saxHandler);
			}

			if (item.getSource() != null) {
				serializeSource(item.getSource());
			}

			indent--;

			ManjyuRssSerializerUtil.indent(indent, saxHandler);
			saxHandler.endElement(null, "item", "item");
			ManjyuRssSerializerUtil.newLine(saxHandler);
		} catch (SAXException ex) {
			throw new IOException(ex);
		}
	}
}
