/*
 * Copyright (c) 2003, Morpho Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 3. Neither the name of the Morpho Project nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */
package jp.morpho.webapp;

import java.io.InputStream;
import java.io.Serializable;
import java.net.URL;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Vector;
import java.util.logging.Logger;

import javax.servlet.ServletContext;

import jp.morpho.config.Configuration;
import jp.morpho.lang.LifeCycle;
import jp.morpho.lang.LifeCycleException;
import jp.morpho.webapp.impl.PluginContextImpl;

/**
 * <P>
 * FrameworkContext 饹ϥץ饰Ǽޤ
 * ץ饰  add ᥽åɤѤ FrameworkContext ˳ǼǤޤ
 * </P>
 * <P>
 * Web ץꥱñ̤Ǹ硢֥֥åȥƥȡ 1 Ĥˡ1 Ĥ FrameworkContext бޤ
 * </P>
 * @author Kenichi Fukuda
 */
public class FrameworkContext
	extends Vector
	implements LifeCycle, Serializable
{

	/**  */
	private Logger log = Logger.getLogger("jp.morpho");

	/** ֥åȥƥȤ˥Хɤ°̾ */
	private static final String ATTRIBUTE = "jp.morpho.webapp.FrameworkContext";

	/** ե졼һҤΥ꥽ѥ */
	private static final String CONFIG_PATH = "/WEB-INF/morpho-config.xml";

	/** ץ饰Υޥå */
	private HashMap map = new HashMap();

	/** FrameworkContext ȤƤ true */
	private boolean running = false;

	/**
	 * <P>
	 * FrameworkContext ޤ
	 * </P>
	 */
	protected FrameworkContext()
	{
		super();
	}

	/**
	 * <P>
	 * FrameworkContext ޤϼޤ
	 * Ǥ FrameworkContext Ƥϡ줬֤ޤ
	 * Ǥʤϡ FrameworkContext ޤ
	 * </P>
	 * <P>
	 *  FrameworkContext 硢Υե졼եϥ֥åȥƥȤ° jp.morpho.webapp.FrameworkContext ˥Хɤޤ
	 * Ǥ˥֥åȥƥȤ° jp.morpho.webapp.FrameworkContext  FrameworkContext ХɤƤϡ줬֤ޤ
	 * </P>
	 * @param sc  FrameworkContext Ǽ륵֥åȥƥ
	 * @return FrameworkContext ֥
	 */
	public static synchronized FrameworkContext getInstance(ServletContext sc) throws FrameworkException
	{
		try
		{
			//֥åȥƥȤ FrameworkContext ޤ
			FrameworkContext fc =
				(FrameworkContext)sc.getAttribute(ATTRIBUTE);

			//ޤ FrameworkContext Ƥʤϡ
			// FrameworkContext ޤ
			if (fc == null)
			{
				//FrameworkContext ޤ
				fc = new FrameworkContext();

				//ե졼һҤ URL ޤ
				URL configURL = sc.getResource(CONFIG_PATH);

				//ե졼ꤷޤ
				InputStream in = configURL.openStream();
				Configuration config = new Configuration();
				config.configure(fc, in);
				in.close();

				//֥åȥƥȤإХɤޤ
				sc.setAttribute(ATTRIBUTE, fc);
			}

			return fc;
		}
		catch (Exception e)
		{
			throw new FrameworkException(e);
		}
	}

	/**
	 * <P>
	 * ꤵ줿֥Ȥץ饰Ǥ뤫ɤȽꤷޤ
	 * ꤵ줿֥Ȥ jp.morpho.webapp.Plugin Ƥ뤫ɤĴ٤ޤ
	 * </P>
	 * @param obj ץ饰Ǥ뤫ɤĴ٤륪֥
	 * @exception IllegalArgumentException ꤵ줿֥Ȥץ饰Ǥʤ
	 */
	private synchronized void checkInstance(Object obj)
	{
		if (!(obj instanceof Plugin))
		{
			throw new IllegalArgumentException("Object is not a Plugin: " + obj);
		}
	}

	/**
	 * <P>
	 * Ǥ˥ץ饰ɲäƤ뤫ɤȽꤷޤ
	 * ץ饰ΥޥåפΥΥޥåԥ󥰤ݻ 㳰 IllegalArgumentException 򥹥ޤ
	 * </P>
	 * @param key ޥåפˤ뤫ɤȽꤵ륭
	 * @exception IllegalArgumentException ץ饰ΥޥåפΥΥޥåԥ󥰤ݻƤ
	 */
	private synchronized void checkMapping(String key)
	{
		if (map.containsKey(key))
		{
			throw new IllegalArgumentException("Plugin Key already exists: " + key);
		}
	}

	/**
	 * <P>
	 * FrameworkContext λꤵ줿֤˻Υץ饰ޤ
	 * ΰ֤Ȥʹߤ˥ץ饰󤬤Ф򱦤˰ưƥץ饰Υǥå 1 äޤ
	 * </P>
	 * @param index Υץ饰륤ǥå
	 * @param element ץ饰
	 * @exception IndexOutOfBoundsException ǥåϰϳξ (index < 0 || index > size())
	 * @exception IllegalArgumentException ꤵ줿֥Ȥץ饰Ǥʤ硢ޤƱ̾Υץ饰 FrameworkContext ˤä
	 * @see java.util.List#add(int, Object)
	 */
	public synchronized void add(int index, Object element)
	{
		checkInstance(element);

		String key = ((Plugin)element).getPluginKey();
		checkMapping(key);

		map.put(key, element);
		super.add(index, element);
	}

	/**
	 * <P>
	 * FrameworkContext ˻Υץ饰ɲäޤ
	 * </P>
	 * @param element FrameworkContext ɲäץ饰
	 * @return true (Collection.add ѵɤ)
	 * @exception IllegalArgumentException ꤵ줿֥Ȥץ饰Ǥʤ硢ޤƱ̾Υץ饰 FrameworkContext ˤä
	 * @see java.util.Collection#add(Object)
	 */
	public synchronized boolean add(Object element)
	{
		checkInstance(element);

		String key = ((Plugin)element).getPluginKey();
		checkMapping(key);

		map.put(key, element);
		try
		{
			PluginContextImpl ctx = new PluginContextImpl();
			ctx.setFrameworkContext(this);
			((Plugin)element).setPluginContext(ctx);
		}
		catch (PluginException e)
		{
			log.warning(e.getMessage());
		}
		return super.add(element);
	}

	/**
	 * <P>
	 * ꤵ줿 Collection Τ٤ƤΥץ饰򤳤 FrameworkContext ˡ Collection  Iterator ˤä֤ɲäޤ
	 * Υڥ졼ưϡꤵ줿 Collection ڥ졼οʹԻѹޤ
	 * ϡ Collection  FrameworkContext ǤˤθƤӽФư줺FrameworkContext ǤʤȤ򼨤ޤ
	 * </P>
	 * @param c FrameworkContext ץ饰
	 * @return θƤӽФη̡ FrameworkContext ѹ줿 true
	 * @exception IndexOutOfBoundsException ǥåϰϳξ (index < 0 || index > size())
	 * @exception NullPointerException ꤵ줿쥯 null Ǥ
	 * @exception IllegalArgumentException ꤵ줿쥯Ǥץ饰Ǥʤ硢ޤƱ̾Υץ饰 FrameworkContext ˤä
	 * @see java.util.Collection#addAll(Collection)
	 */
	public synchronized boolean addAll(Collection c)
	{
		for (Iterator i = c.iterator(); i.hasNext();)
		{
			Object o = i.next();
			checkInstance(o);

			String key = ((Plugin)o).getPluginKey();
			checkMapping(key);

			map.put(key, o);
		}

		return super.addAll(c);
	}

	/**
	 * <P>
	 * ꤵ줿 Collection Τ٤ƤΥץ饰FrameworkContext λꤵ줿֤ޤ
	 * ΰ֤Ȥʹߤ˥ץ饰󤬤ϡ򱦤˰ươƥץ饰Υǥå 1 äޤ
	 * ƥץ饰ϡꤵ줿 Collection ȿҤˤä֤ FrameworkContext ˳Ǽޤ
	 * </P>
	 * @param index ꤵ줿쥯󤫤ǽΥץ饰֤Υǥå
	 * @param c FrameworkContext ץ饰
	 * @return θƤӽФη̡ FrameworkContext ѹ줿 true
	 * @exception IndexOutOfBoundsException ǥåϰϳξ (index < 0 || index > size())
	 * @exception NullPointerException ꤵ줿쥯 null Ǥ
	 * @exception IllegalArgumentException ꤵ줿쥯Ǥץ饰Ǥʤ硢ޤƱ̾Υץ饰 FrameworkContext ˤä
	 * @see java.util.List#addAll(int, Collection)
	 */
	public synchronized boolean addAll(int index, Collection c)
	{
		for (Iterator i = c.iterator(); i.hasNext();)
		{
			Object o = i.next();
			checkInstance(o);

			String key = ((Plugin)o).getPluginKey();
			checkMapping(key);

			map.put(key, o);
		}

		return super.addAll(index, c);
	}

	/**
	 * <P>
	 * Υץ饰 FrameworkContext κǸɲä 1 䤷ޤ
	 *  FrameworkContext ̤礭ʤȤ̤䤵ޤ
	 * </P>
	 * <P>
	 * Υ᥽åɤϵǽŪ add(Object) ᥽åɤƱǤϡList 󥿥եΰǤ
	 * </P>
	 * @param element ɲäץ饰
	 * @exception IllegalArgumentException ꤵ줿֥Ȥץ饰Ǥʤ硢ޤƱ̾Υץ饰 FrameworkContext ˤä
	 * @see jp.morpho.webapp.FrameworkContext#add(Object)
	 */
	public synchronized void addElement(Object element)
	{
		checkInstance(element);

		String key = ((Plugin)element).getPluginKey();
		checkMapping(key);

		map.put(key, element);
		super.addElement(element);
	}

	/**
	 * <P>
	 * FrameworkContext λꤵ줿 index ˡꤵ줿ץ饰ޤ
	 * ꤵ줿 index 礭ǥåΥץ饰Ϥ٤ 1 ĸΥǥåˤޤ
	 * </P>
	 * <P>
	 * ǥåϡ0 礭ͤǤʤФʤޤ
	 * ޤFrameworkContext θߤΥ꾮ͤǤʤФʤޤ
	 * ǥå FrameworkContext θߤΥ硢ץ饰 FrameworkContext ɲäޤ
	 * </P>
	 * <P>
	 * Υ᥽åɤϵǽŪ add(Object, int) ᥽åɤƱǤ
	 * ϡList 󥿥եΰǤ
	 * λȤˤŬˤ뤿 add ᥽åɤѥ᡼νդˤ뤳ȤդƤ
	 * </P>
	 * @param element ץ饰
	 * @param index ץ饰
	 * @exception ArrayIndexOutOfBoundsException - ǥåϰϳξ (index < 0 || index >= size())
	 * @exception IllegalArgumentException ꤵ줿֥Ȥץ饰Ǥʤ硢ޤƱ̾Υץ饰 FrameworkContext ˤä
	 * @see java.util.Vector#insertElementAt(Object, int)
	 */
	public synchronized void insertElementAt(Object element, int index)
	{
		checkInstance(element);

		String key = ((Plugin)element).getPluginKey();
		checkMapping(key);

		map.put(key, element);
		super.insertElementAt(element, index);
	}

	/**
	 * <P>
	 * FrameworkContext λΰ֤ˤץ饰ޤ
	 * ʹߤ˥ץ饰󤬤Ф򺸤˰ưƥץ饰Υǥå 1 򸺤餷ޤ
	 * FrameworkContext 줿ץ饰֤ޤ
	 * </P>
	 * @param index ץ饰Υǥå
	 * @exception ArrayIndexOutOfBoundsException - ǥåϰϳξ (index < 0 || index >= size())
	 * @see java.util.List#remove(int)
	 */
	public synchronized Object remove(int index)
	{
		Object o = super.remove(index);
		if (o != null)
		{
			String key = ((Plugin)o).getPluginKey();
			map.remove(key);
		}
		return o;
	}

	/**
	 * <P>
	 * FrameworkContext Ǻǽ˸Ф줿Υץ饰ޤ
	 * FrameworkContext ץ饰ݻƤʤϡѹޤ
	 * Ĥޤꡢ(element == null ? get(i) == null : element.equals(get(i))) ǤäȤ⾮ǥå i ĥץ饰󤬤кޤ
	 * </P>
	 * @param element FrameworkContext ץ饰 (Υץ饰󤬤)
	 * @return ꤵ줿ץ饰 FrameworkContext ˤä true
	 * @see java.util.Collection#remove(Object)
	 */
	public synchronized boolean remove(Object element)
	{
		boolean ret = super.remove(element);
		if (ret)
		{
			String key = ((Plugin)element).getPluginKey();
			map.remove(key);
		}

		return ret;
	}

	/**
	 * <P>
	 * FrameworkContext 顢ꤵ줿 Collection ݻƤ뤹٤ƤΥץ饰ޤ
	 * </P>
	 * @param c FrameworkContext ץ饰Υ쥯
	 * @return FrameworkContext ƤӽФη̤Ȥѹ줿 true
	 * @exception NullPointerException ꤵ줿쥯 null Ǥ
	 * @see java.util.Collection#removeAll(Collection)
	 */
	public synchronized boolean removeAll(Collection c)
	{
		boolean retAll = false;
		for (Iterator i = c.iterator(); i.hasNext();)
		{
			Object o = i.next();
			boolean ret = this.remove(o);
			retAll = (ret) ? ret : retAll;
		}
		return retAll;
	}

	/**
	 * <P>
	 * FrameworkContext 餹٤ƤΥץ饰 0 ꤷޤ
	 * </P>
	 * <P>
	 * Υ᥽åɤϵǽŪ clear ᥽åɤƱǤ
	 * ϡList 󥿥եΰǤ
	 * </P>
	 * @see java.util.Vector#removeAllElements()
	 */
	public synchronized void removeAllElements()
	{
		map.clear();
		super.removeAllElements();
	}

	/**
	 * <P>
	 * ǽ˸Ф줿 (Ǿǥå) Υץ饰 FrameworkContext ޤ
	 * ץ饰󤬸ФȡΥץ饰ΥǥåƱ礭ǥåΥץ饰Ϥ٤ 1 Υǥå˵ͤޤ
	 * </P>
	 * <P>
	 * Υ᥽åɤϵǽŪ remove(Object) ᥽åɤƱǤ
	 * ϡList 󥿥եΰǤ
	 * </P>
	 * @param element ץ饰
	 * @return  FrameworkContext Υץ饰Ǥ trueǤʤ false
	 * @see java.util.Vector#removeElement(Object)
	 */
	public synchronized boolean removeElement(Object element)
	{
		boolean ret = super.removeElement(element);
		if (ret)
		{
			String key = ((Plugin)element).getPluginKey();
			map.remove(key);
		}

		return ret;
	}

	/**
	 * <P>
	 * ꤵ줿ǥåΥץ饰ޤ
	 * ꤵ줿 index 礭ǥåΥץ饰ϡ٤ 1 Υǥå˵ͤޤ
	 * FrameworkContext Υ 1 餵ޤ
	 * </P>
	 * <P>
	 * ǥåϡ0 礭ͤǤʤФʤޤ
	 * ޤFrameworkContext θߤΥ꾮ͤǤʤФʤޤ
	 * </P>
	 * <P>
	 * Υ᥽åɤϵǽŪ remove ᥽åɤƱǤ
	 * ϡList 󥿥եΰǤ
	 * remove ᥽åɤΰ֤˳ǼƤŤ֤ͤȤդƤ
	 * </P>
	 * @param index ץ饰Υǥå
	 * @exception ArrayIndexOutOfBoundsException - ǥåϰϳξ (index < 0 || index >= size())
	 * @see java.util.Vector#removeElementAt(int)
	 */
	public synchronized void removeElementAt(int index)
	{
		String key = ((Plugin)get(index)).getPluginKey();
		map.remove(key);
		super.removeElementAt(index);
	}

	/**
	 * <P>
	 * ꤵ줿 Collection ݻƤ롢FrameworkContext Υץ饰ݻޤ
	 * ĤޤꡢFrameworkContext 顢ꤵ줿 Collection ݻƤʤ٤ƤΥץ饰ޤ
	 * </P>
	 * @param c FrameworkContext ݻƤץ饰Υ쥯 (ۤΤ٤ƤΥץ饰Ϻ)
	 * @return FrameworkContext ƤӽФη̤Ȥѹ줿 true
	 * @exception NullPointerException ꤵ줿쥯 null Ǥ
	 * @see java.util.Collection#retainAll(Collection)
	 */
	public synchronized boolean retainAll(Collection c)
	{
		boolean ret = super.retainAll(c);
		Object[] keys = map.keySet().toArray();
		for (int i = 0; i < keys.length; i++)
		{
			String key = (String)keys[i];
			if (!(this.contains(map.get(key))))
			{
				map.remove(key);
			}
		}

		return ret;
	}

	/**
	 * <P>
	 * FrameworkContext λꤵ줿֤ˤץ饰򡢻Υץ饰֤ޤ
	 * </P>
	 * @param index ִץ饰Υǥå
	 * @param element ꤵ줿֤˳Ǽץ饰
	 * @return ꤵ줿֤˰äץ饰
	 * @exception ArrayIndexOutOfBoundsException ǥåϰϳξ (index < 0 || index >= size())
	 * @exception IllegalArgumentException ꤵ줿֥Ȥץ饰Ǥʤ硢ޤƱ̾Υץ饰 FrameworkContext ˤä
	 * @see java.util.List#set(int, Object)
	 */
	public synchronized Object set(int index, Object element)
	{
		checkInstance(element);

		String key = ((Plugin)get(index)).getPluginKey();
		map.remove(key);

		key = ((Plugin)element).getPluginKey();
		map.put(key, element);
		return super.set(index, element);
	}

	/**
	 * <P>
	 * FrameworkContext λꤵ줿 index Υץ饰ˡꤵ줿ץ饰ꤷޤ
	 * Υǥåˤäץ饰˴ޤ
	 * </P>
	 * <P>
	 * ǥåϡ0 礭ͤǤʤФʤޤ
	 * ޤFrameworkContext θߤΥ꾮ͤǤʤФʤޤ
	 * </P>
	 * <P>
	 * Υ᥽åɤϵǽŪ set ᥽åɤƱǤ
	 * ϡList 󥿥եΰǤ
	 * λȤˤŤ뤿ˡset ᥽åɤѥ᡼νդˤ뤳ȤդƤ
	 * ޤset ᥽åɤΰ֤˳ǼƤŤ֤ͤȤˤդƤ
	 * </P>
	 * @param element ץ饰ꤵ륪֥
	 * @param index ǥå
	 * @exception ArrayIndexOutOfBoundsException ǥåϰϳξ (index < 0 || index >= size())
	 * @exception IllegalArgumentException ꤵ줿֥Ȥץ饰Ǥʤ硢ޤƱ̾Υץ饰 FrameworkContext ˤä
	 * @see java.util.Vector#setElementAt(Object, int)
	 */
	public synchronized void setElementAt(Object element, int index)
	{
		checkInstance(element);

		String key = ((Plugin)get(index)).getPluginKey();
		map.remove(key);

		key = ((Plugin)element).getPluginKey();
		map.put(key, element);
		super.setElementAt(element, index);
	}

	/**
	 * <P>
	 * ꤵ줿Υץ饰֤ޤ
	 * FrameworkContext ΥΥץ饰ݻƤʤ null ֤ޤ
	 * </P>
	 * @param key ץ饰Υ
	 * @return ꤵ줿Υץ饰󡣻ꤵ줿Υץ饰󤬤ʤ null
	 */
	public Object get(String key)
	{
		return map.get(key);
	}

	/**
	 * <P>
	 * ٤ƤΥץ饰 FrameworkContext ޤ
	 * θƤӽФ֤ȡ줬㳰򥹥ʤꡢFrameworkContext ϶ˤʤޤ
	 * </P>
	 * @see java.util.Collection#clear()
	 */
	public synchronized void clear()
	{
		map.clear();
		super.clear();
	}

	/**
	 * <P>
	 * FrameworkContext ȤƤ true ֤ޤ
	 * </P>
	 * @return FrameworkContext ȤƤ true
	 * @see jp.morpho.lang.LifeCycle#isRunning()
	 */
	public synchronized boolean isRunning()
	{
		return running;
	}

	/**
	 * <P>
	 * FrameworkContext 򥹥Ȥޤ
	 * </P>
	 * @exception LifeCycleException FrameworkContext ΥȽ꤬ä
	 * @see jp.morpho.lang.LifeCycle#start()
	 */
	public synchronized void start() throws LifeCycleException
	{
		if (isRunning())
		{
			throw new LifeCycleException("Already started");
		}
		running = true;

		Plugin[] plugins = new Plugin[this.size()];
		plugins = (Plugin[])this.toArray(plugins);
		for (int i = 0; i < plugins.length; i++)
		{
			try
			{
				plugins[i].pluginActivate();
			}
			catch (PluginException e)
			{
				//³
			}
		}
	}

	/**
	 * <P>
	 * FrameworkContext 򥹥ȥåפޤ
	 * </P>
	 * @exception LifeCycleException FrameworkContext Υȥå׽꤬ä
	 * @see jp.morpho.lang.LifeCycle#stop()
	 */
	public synchronized void stop() throws LifeCycleException
	{
		running = false;
		Plugin[] plugins = new Plugin[this.size()];
		plugins = (Plugin[])this.toArray(plugins);
		while (this.size() > 0)
		{
			try
			{
				plugins[this.size() - 1].pluginRemove();
			}
			catch (PluginException e)
			{
				//³
			}
			this.remove(this.size() - 1);
		}
	}

}
