/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.awk.nano.expr;

import net.morilib.awk.nano.AwkLvalueException;
import net.morilib.awk.nano.io.AwkFiles;
import net.morilib.awk.nano.namespace.AwkNamespace;
import net.morilib.awk.nano.statement.AwkExpressionMachine;
import net.morilib.awk.nano.value.AwkValue;

/**
 * awkium式の中間表現です。
 * 
 * @author MORIGUCHI, Yuichiro 2013/03
 */
public abstract class AwkExpression {

	/**
	 * 中間表現を評価します。
	 * 
	 * @param namespace 名前空間
	 * @param files     ファイル管理
	 * @return 結果
	 */
	public abstract AwkValue eval(AwkNamespace namespace,
			AwkFiles files);

	/**
	 * 代入の中間表現を評価します。
	 * 
	 * @param namespace 名前空間
	 * @param value     代入する値
	 * @param files     ファイル管理
	 * @param namespace
	 */
	public void assign(AwkNamespace namespace, AwkValue value,
			AwkFiles files) {
		throw new AwkLvalueException("assignment to non-lvalue");
	}

	/**
	 * 中間表現が左辺値のときtrueを得ます。
	 */
	public boolean isLvalue() {
		return false;
	}

	/**
	 * 中間表現が入力リダイレクトの可能性があるときにtrueを返します。
	 */
	public boolean maybeInputRedirect() {
		return false;
	}

	/**
	 * 中間表現が出力リダイレクトの可能性があるときにtrueを返します。
	 */
	public boolean maybeOutputRedirect() {
		return false;
	}

	/**
	 * 中間表現を簡易コンパイルします。<br />
	 * 内部で使用しているので外部からは使用しないでください。
	 */
	public void compileInternally(AwkExpressionMachine.Builder b,
			Object brk, Object cnt) {
		b.add(this);
	}

	/**
	 * 中間表現を簡易コンパイルします。
	 */
	public AwkExpression compileInternally() {
		AwkExpressionMachine.Builder b;

		b = new AwkExpressionMachine.Builder();
		compileInternally(b, null, null);
		return b.get();
	}

}
