/*
 * Copyright 2009 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.lisp.nano;

import java.math.BigDecimal;
import java.math.BigInteger;

/**
 * 
 *
 *
 * @author MORIGUCHI, Yuichiro 2009
 */
public abstract class LispNumber extends Atom {

	/*package*/ static String disp(double n) {
		if(n == Double.POSITIVE_INFINITY) {
			return "+inf.0";
		} else if(n == Double.NEGATIVE_INFINITY) {
			return "-inf.0";
		} else if(Double.isNaN(n)) {
			return "+nan.0";
		} else {
			return Double.toString(n);
		}
	}

	/**
	 * 
	 * @param x
	 * @return
	 */
	public abstract LispNumber add(LispNumber x);

	/**
	 * 
	 * @param x
	 * @return
	 */
	public abstract LispNumber sub(LispNumber x);

	/**
	 * 
	 * @param x
	 * @return
	 */
	public abstract LispNumber mul(LispNumber x);

	/**
	 * 
	 * @param x
	 * @return
	 */
	public abstract LispNumber div(LispNumber x);

	/**
	 * 
	 * @return
	 */
	public abstract LispNumber uminus();

	/**
	 * 
	 * @param x
	 * @return
	 */
	public abstract boolean isEqualTo(LispNumber x);

	/**
	 * 
	 * @return
	 */
	public abstract boolean isInteger();

	/**
	 * 
	 * @return
	 */
	public abstract boolean isRational();

	/**
	 * 
	 * @return
	 */
	public abstract boolean isReal();

	/**
	 * 
	 * @return
	 */
	public abstract boolean isExact();

	/**
	 * 
	 * @return
	 */
	public abstract LispNumber toExact();

	/**
	 * 
	 * @return
	 */
	public abstract LispNumber toInexact();

	/**
	 * 
	 * @param radix
	 * @return
	 */
	public abstract LispString toLispString(int radix);

	/**
	 * 
	 * @param radix
	 * @param precision
	 * @return
	 */
	public abstract LispString toLispString(int radix, int precision);

	/**
	 * 
	 * @return
	 */
	public abstract LispReal norm();

	/**
	 * 
	 * @return
	 */
	public abstract LispNumber conjugate();

	/**
	 * 
	 * @return
	 */
	public abstract boolean isNaN();

	/**
	 * 
	 * @return
	 */
	public abstract boolean isOne();

	/*
	 * (non-Javadoc)
	 * @see net.morilib.lisp.Datum#getInt()
	 */
	public abstract int getInt();

	/* (non-Javadoc)
	 * @see net.morilib.lisp.Datum#getLong()
	 */
	public abstract long getLong();

	/*
	 * (non-Javadoc)
	 * @see net.morilib.lisp.Datum#getBigInteger()
	 */
	public abstract BigInteger getBigInteger();

	/* (non-Javadoc)
	 * @see net.morilib.lisp.Datum#getBigDecimal()
	 */
	public abstract BigDecimal getBigDecimal();

	/* (non-Javadoc)
	 * @see net.morilib.lisp.Datum#getRealDouble()
	 */
	public abstract double getRealDouble();

	/* (non-Javadoc)
	 * @see net.morilib.lisp.Datum#getRealFloat()
	 */
	public float getRealFloat() {
		return (float)getRealDouble();
	}

	/**
	 * 
	 * @return
	 */
	public abstract BigInteger getNumerator();

	/**
	 * 
	 * @return
	 */
	public abstract BigInteger getDenominator();

	/**
	 * 
	 * @return
	 */
	public abstract LispReal[] getImags();

	/**
	 * 
	 * @return
	 */
	public abstract double[] getImagsDouble();

	/**
	 * 
	 * @param n
	 * @return
	 */
	public LispNumber pow(int n) {
		LispNumber r = LispInteger.ONE;

		if(n >= 0) {
			for(int i = 0; i < n; i++) {
				r = r.mul(this);
			}
		} else {
			for(int i = 0; i < -n; i++) {
				r = r.div(this);
			}
		}
		return r;
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.math.algebra.ILispInvertable#invert()
	 */
	public LispNumber inv() {
		return LispInteger.ONE.div(this);
	}

	/**
	 * 
	 * @return
	 */
	public int getExactSmallInt() {
		throw new NumberFormatException();
	}

	/*
	 * (non-Javadoc)
	 * @see net.morilib.lisp.Atom#toLispString()
	 */
	public LispString toLispString() {
		return toLispString(10);
	}

	/**
	 * 
	 * @return
	 */
	public boolean isZero() {
		return isEqualTo(LispInteger.ZERO);
	}

	/**
	 * 
	 * @return
	 */
	public abstract boolean isFinite();

}
