/*
 * Copyright (c) 2003 The Ochusha Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: board_properties.c,v 1.9 2003/12/30 21:53:28 fuyu Exp $
 */

#include "config.h"

#include "ochusha_private.h"

#include "ochusha_ui.h"
#include "bulletin_board_ui.h"
#include "board_properties.h"
#include "boardlist_ui.h"
#include "regex_utils.h"
#include "threadlist_filter.h"

#include <gtk/gtk.h>

#include <pthread.h>
#include <stdlib.h>


typedef struct _DialogStates
{
  OchushaApplication *application;
  OchushaBulletinBoard *board;
  BulletinBoardGUIInfo *info;

  GtkToggleButton *use_ochusha_default_button;
  GtkToggleButton *show_mailto_button;
  GtkToggleButton *hide_mailto_button;

  GtkToggleButton *board_local_a_bone_button;
  GtkToggleButton *a_bone_by_name_button;
  RegexEditor *a_bone_by_name_pattern;
  GtkToggleButton *a_bone_by_id_button;
  RegexEditor *a_bone_by_id_pattern;
  GtkToggleButton *a_bone_by_content_button;
  RegexEditor *a_bone_by_content_pattern;

  ThreadlistFilterSetting *setting;
  GtkToggleButton *use_as_default_button;
  GtkToggleButton *enable_filter_button;

  ThreadlistFilter *last_filter;

  gboolean use_default;
} DialogStates;


static GtkWidget *build_board_info_frame(OchushaApplication *application,
					 OchushaBulletinBoard *board,
					 DialogStates *states);
static GtkWidget *build_thread_view_setting_frame(OchushaApplication *application,
						  OchushaBulletinBoard *board,
						  DialogStates *states);
static GtkWidget *build_filter_setting_frame(OchushaApplication *application,
					     OchushaBulletinBoard *board,
					     DialogStates *states);

static void setting_changed_cb(ThreadlistFilterSetting *setting,
			       ThreadlistFilter *filter, DialogStates *states);
static void restore_default_cb(ThreadlistFilterSetting *setting,
			       DialogStates *states);
static void apply_current_states(GtkWidget *dialog, DialogStates *states);
static void board_properties_response_cb(GtkWidget *dialog, int response_id,
					 DialogStates *states);
static void board_local_a_bone_toggled_cb(GtkToggleButton *button,
					  DialogStates *states);

static pthread_mutex_t board_properties_lock;


#define BOARD_PROPERTIES_LOCK					\
  if (pthread_mutex_lock(&board_properties_lock) != 0)		\
    {								\
      fprintf(stderr, "Couldn't lock a mutex.\n");		\
      abort();							\
    }

#define BOARD_PROPERTIES_UNLOCK					\
  if (pthread_mutex_unlock(&board_properties_lock) != 0)	\
    {								\
      fprintf(stderr, "Couldn't unlock a mutex.\n");		\
      abort();							\
    }


void
initialize_board_properties(OchushaApplication *application)
{
  if (pthread_mutex_init(&board_properties_lock, NULL) != 0)
    {
      fprintf(stderr, "Couldn't init a mutex.\n");
      abort();
    }
}


void
open_board_properties(OchushaApplication *application,
		      OchushaBulletinBoard *board)
{
  GtkWidget *dialog;
  GtkWidget *frame;
  GtkWidget *notebook;
  GtkWidget *vbox;
  GtkWidget *tmp_widget;
  DialogStates *states;
  BulletinBoardGUIInfo *info = ensure_bulletin_board_info(board, application);

  BOARD_PROPERTIES_LOCK
  {
    if (info == NULL || info->properties_dialog != NULL)
      {
	BOARD_PROPERTIES_UNLOCK;
	return;
      }

    states = G_NEW0(DialogStates, 1);
    states->application = application;
    states->board = board;
    states->info = info;

    if (info->filter.rule != NULL)
      {
	states->last_filter = &info->filter;
	states->use_default = FALSE;
      }
    else
      {
	states->last_filter = &application->user_default_filter;
	states->use_default = TRUE;
      }

    dialog = gtk_dialog_new_with_buttons(_("Bulletin Board's Properties"),
					 application->top_level,
					 GTK_DIALOG_DESTROY_WITH_PARENT,

					 GTK_STOCK_CANCEL,
					 GTK_RESPONSE_CANCEL,

					 GTK_STOCK_APPLY,
					 GTK_RESPONSE_APPLY,

					 GTK_STOCK_OK,
					 GTK_RESPONSE_OK,

					 NULL);
    gtk_dialog_set_default_response(GTK_DIALOG(dialog), GTK_RESPONSE_OK);
    g_signal_connect(G_OBJECT(dialog), "response",
		     G_CALLBACK(board_properties_response_cb),
		     states);

    info->properties_dialog = dialog;
  }
  BOARD_PROPERTIES_UNLOCK;

  frame = build_board_info_frame(application, board, states);
  gtk_widget_show(frame);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox),
		     frame, FALSE, FALSE, 5);

  notebook = gtk_notebook_new();
  gtk_widget_show(notebook);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox),
		     notebook, TRUE, TRUE, 0);


  vbox = gtk_vbox_new(FALSE, 0);
  gtk_widget_show(vbox);
  gtk_notebook_append_page(GTK_NOTEBOOK(notebook), vbox,
			   gtk_label_new(_("Threadlist's Filter Setting")));

  frame = build_filter_setting_frame(application, board, states);
  gtk_widget_show(frame);
  gtk_box_pack_start(GTK_BOX(vbox), frame, TRUE, TRUE, 0);

  tmp_widget
    = gtk_check_button_new_with_label(_("Apply filter to this board."));
  gtk_widget_show(tmp_widget);
  gtk_box_pack_end(GTK_BOX(vbox), tmp_widget, FALSE, FALSE, 0);
  states->enable_filter_button = GTK_TOGGLE_BUTTON(tmp_widget);
  gtk_toggle_button_set_active(states->enable_filter_button,
			       info->enable_filter);


  frame = build_thread_view_setting_frame(application, board, states);
  gtk_widget_show(frame);

  gtk_notebook_append_page(GTK_NOTEBOOK(notebook), frame,
			   gtk_label_new(_("Default Setting of Thread Views on This Board")));

  g_signal_connect(G_OBJECT(states->board_local_a_bone_button), "toggled",
		   G_CALLBACK(board_local_a_bone_toggled_cb), states);

  switch (info->show_mailto_mode)
    {
    case THREAD_VIEW_MAILTO_MODE_DEFAULT:
      gtk_toggle_button_set_active(states->use_ochusha_default_button, TRUE);
      break;

    case THREAD_VIEW_MAILTO_MODE_SHOW:
      gtk_toggle_button_set_active(states->show_mailto_button, TRUE);
      break;

    case THREAD_VIEW_MAILTO_MODE_HIDE:
      gtk_toggle_button_set_active(states->hide_mailto_button, TRUE);
      break;
    }

  if (info->board_local_a_bone)
    {
      gtk_toggle_button_set_active(states->a_bone_by_name_button,
				   info->a_bone_by_name);
      regex_editor_set_regex_text(states->a_bone_by_name_pattern,
				  info->a_bone_by_name_pattern);
      gtk_toggle_button_set_active(states->a_bone_by_id_button,
				   info->a_bone_by_id);
      regex_editor_set_regex_text(states->a_bone_by_id_pattern,
				  info->a_bone_by_id_pattern);
      gtk_toggle_button_set_active(states->a_bone_by_content_button,
				   info->a_bone_by_content);
      regex_editor_set_regex_text(states->a_bone_by_content_pattern,
				  info->a_bone_by_content_pattern);
    }
  else
    {
      gtk_toggle_button_set_active(states->a_bone_by_name_button,
				   application->a_bone_by_name);
      regex_editor_set_regex_text(states->a_bone_by_name_pattern,
				  application->a_bone_by_name_pattern);
      gtk_toggle_button_set_active(states->a_bone_by_id_button,
				   application->a_bone_by_id);
      regex_editor_set_regex_text(states->a_bone_by_id_pattern,
				  application->a_bone_by_id_pattern);
      gtk_toggle_button_set_active(states->a_bone_by_content_button,
				   application->a_bone_by_content);
      regex_editor_set_regex_text(states->a_bone_by_content_pattern,
				  application->a_bone_by_content_pattern);
    }

  gtk_toggle_button_set_active(states->board_local_a_bone_button,
			       info->board_local_a_bone);
  /* 礤ȥܤä */
  board_local_a_bone_toggled_cb(states->board_local_a_bone_button, states);

  gtk_widget_show_all(dialog);
}


static void
board_local_a_bone_toggled_cb(GtkToggleButton *button, DialogStates *states)
{
  gboolean sensitive = gtk_toggle_button_get_active(button);

  gtk_widget_set_sensitive(GTK_WIDGET(states->a_bone_by_name_button),
			   sensitive);
  gtk_widget_set_sensitive(GTK_WIDGET(states->a_bone_by_name_pattern),
			   sensitive);
  gtk_widget_set_sensitive(GTK_WIDGET(states->a_bone_by_id_button), sensitive);
  gtk_widget_set_sensitive(GTK_WIDGET(states->a_bone_by_id_pattern),
			   sensitive);
  gtk_widget_set_sensitive(GTK_WIDGET(states->a_bone_by_content_button),
			   sensitive);
  gtk_widget_set_sensitive(GTK_WIDGET(states->a_bone_by_content_pattern),
			   sensitive);
}


static void
setting_changed_cb(ThreadlistFilterSetting *setting, ThreadlistFilter *filter,
		   DialogStates *states)
{
  states->use_default = FALSE;
}


static void
restore_default_cb(ThreadlistFilterSetting *setting, DialogStates *states)
{
  threadlist_filter_setting_set_filter(setting,
				&states->application->user_default_filter);
  states->use_default = TRUE;

  states->info->filter.ignore_threshold
    = states->application->user_default_filter.ignore_threshold;
  gtk_toggle_button_set_active(states->use_as_default_button, TRUE);
}


static void
apply_current_states(GtkWidget *dialog, DialogStates *states)
{
  BulletinBoardGUIInfo *info = states->info;
  const ThreadlistFilter *filter;

  filter = threadlist_filter_setting_get_filter(states->setting);
  info->enable_filter
    = gtk_toggle_button_get_active(states->enable_filter_button);

  if (gtk_toggle_button_get_active(states->show_mailto_button))
    {
      info->show_mailto_mode = THREAD_VIEW_MAILTO_MODE_SHOW;
    }
  else if (gtk_toggle_button_get_active(states->hide_mailto_button))
    {
      info->show_mailto_mode = THREAD_VIEW_MAILTO_MODE_HIDE;
    }
  else
    {
      info->show_mailto_mode = THREAD_VIEW_MAILTO_MODE_DEFAULT;
    }

  info->board_local_a_bone
    = gtk_toggle_button_get_active(states->board_local_a_bone_button);

  if (info->a_bone_by_name_pattern != NULL)
    G_FREE(info->a_bone_by_name_pattern);

  if (info->a_bone_by_id_pattern != NULL)
    G_FREE(info->a_bone_by_id_pattern);

  if (info->a_bone_by_content_pattern != NULL)
    G_FREE(info->a_bone_by_content_pattern);

  if (info->board_local_a_bone)
    {
      info->a_bone_by_name
	= gtk_toggle_button_get_active(states->a_bone_by_name_button);
      info->a_bone_by_name_pattern
	= regex_editor_get_regex_text(states->a_bone_by_name_pattern);
      info->a_bone_by_id
	= gtk_toggle_button_get_active(states->a_bone_by_id_button);
      info->a_bone_by_id_pattern
	= regex_editor_get_regex_text(states->a_bone_by_id_pattern);
      info->a_bone_by_content
	= gtk_toggle_button_get_active(states->a_bone_by_content_button);
      info->a_bone_by_content_pattern
	= regex_editor_get_regex_text(states->a_bone_by_content_pattern);
    }
  else
    {
      info->a_bone_by_name_pattern = NULL;
      info->a_bone_by_id_pattern = NULL;
      info->a_bone_by_content_pattern = NULL;
    }

  if (gtk_toggle_button_get_active(states->use_as_default_button))
    {
      G_FREE(states->application->user_default_filter.rule);
      states->application->user_default_filter.rule = G_STRDUP(filter->rule);
      states->application->user_default_filter.ignore_threshold
	= filter->ignore_threshold;

      if (info->filter.rule != NULL)
	{
	  gchar *tmp = info->filter.rule;
	  info->filter.rule = NULL;
	  G_FREE(tmp);
	}

      info->filter.ignore_threshold = filter->ignore_threshold;
      threadlist_filter_setting_set_restore_default_button_sensitive(
						states->setting, FALSE);
      setup_default_threadlist_filter(states->application);
      redraw_current_threadlist_all(states->application);
    }
  else
    {
      if (info->filter.rule != NULL)
	{
	  gchar *tmp = info->filter.rule;
	  info->filter.rule = NULL;
	  G_FREE(tmp);
	}

      if (!states->use_default)
	{
	  info->filter.rule = G_STRDUP(filter->rule);
	  info->filter.ignore_threshold = filter->ignore_threshold;
	}

      redraw_threadlist_view(states->application, states->board);
      fix_filter_button_state(states->application);
    }
}


static void
board_properties_response_cb(GtkWidget *dialog, int response_id,
			     DialogStates *states)
{
  gboolean close_dialog = TRUE;

  switch (response_id)
    {
    case GTK_RESPONSE_APPLY:
      apply_current_states(dialog, states);
      close_dialog = FALSE;
      break;

    case GTK_RESPONSE_OK:
      apply_current_states(dialog, states);
      break;

    case GTK_RESPONSE_CANCEL:
    case GTK_RESPONSE_DELETE_EVENT:
      break;
    }

  if (close_dialog)
    {
      BOARD_PROPERTIES_LOCK
      {
	states->info->properties_dialog = NULL;
      }
      BOARD_PROPERTIES_UNLOCK;
      gtk_widget_destroy(dialog);
    }
}


static GtkWidget *
build_board_info_frame(OchushaApplication *application,
		       OchushaBulletinBoard *board, DialogStates *states)
{
  GtkWidget *table;
  GtkWidget *tmp_widget;
  GtkWidget *board_frame = gtk_frame_new(NULL);

  gtk_frame_set_shadow_type(GTK_FRAME(board_frame), GTK_SHADOW_ETCHED_OUT);
  gtk_widget_show(board_frame);

  table = gtk_table_new(2, 5, FALSE);
  gtk_widget_show(table);
  gtk_container_add(GTK_CONTAINER(board_frame), table);

  tmp_widget = gtk_label_new(_("Board's Name"));
  gtk_widget_show(tmp_widget);
  gtk_table_attach_defaults(GTK_TABLE(table), tmp_widget, 0, 1, 0, 1);

  tmp_widget = gtk_entry_new();
  gtk_entry_set_text(GTK_ENTRY(tmp_widget), board->name);
  gtk_editable_set_editable(GTK_EDITABLE(tmp_widget), FALSE);
  GTK_WIDGET_UNSET_FLAGS(tmp_widget, GTK_CAN_FOCUS);
  gtk_widget_show(tmp_widget);
  gtk_table_attach_defaults(GTK_TABLE(table), tmp_widget, 1, 5, 0, 1);

  tmp_widget = gtk_label_new(_("Board's URL"));
  gtk_widget_show(tmp_widget);
  gtk_table_attach_defaults(GTK_TABLE(table), tmp_widget, 0, 1, 1, 2);

  tmp_widget = gtk_entry_new();
  gtk_entry_set_text(GTK_ENTRY(tmp_widget), board->base_url);
  gtk_editable_set_editable(GTK_EDITABLE(tmp_widget), FALSE);
  GTK_WIDGET_UNSET_FLAGS(tmp_widget, GTK_CAN_FOCUS);
  gtk_widget_show(tmp_widget);
  gtk_table_attach_defaults(GTK_TABLE(table), tmp_widget, 1, 5, 1, 2);

  return board_frame;
}


static GtkWidget *
build_thread_view_setting_frame(OchushaApplication *application,
				OchushaBulletinBoard *board,
				DialogStates *states)
{
  GtkWidget *tmp_widget;
  GtkWidget *thread_frame = gtk_frame_new(_("Default Setting of Thread Views on This Board"));
  GtkWidget *vbox = gtk_vbox_new(FALSE, 0);
  GtkWidget *hbox = gtk_hbox_new(FALSE, 0);

  gtk_widget_show(vbox);
  gtk_container_add(GTK_CONTAINER(thread_frame), vbox);

  gtk_widget_show(hbox);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

  tmp_widget = gtk_label_new(_("Show mailto literally:"));
  gtk_widget_show(tmp_widget);
  gtk_box_pack_start(GTK_BOX(hbox), tmp_widget, FALSE, FALSE, 5);

  tmp_widget
    = gtk_radio_button_new_with_label(NULL, _("Use Ochusha's Default"));
  gtk_widget_show(tmp_widget);
  gtk_box_pack_start(GTK_BOX(hbox), tmp_widget, FALSE, FALSE, 0);
  states->use_ochusha_default_button = GTK_TOGGLE_BUTTON(tmp_widget);
  
  tmp_widget = gtk_radio_button_new_with_label_from_widget(GTK_RADIO_BUTTON(states->use_ochusha_default_button), _("Show mailto"));
  gtk_widget_show(tmp_widget);
  gtk_box_pack_start(GTK_BOX(hbox), tmp_widget, FALSE, FALSE, 0);
  states->show_mailto_button = GTK_TOGGLE_BUTTON(tmp_widget);

  tmp_widget = gtk_radio_button_new_with_label_from_widget(GTK_RADIO_BUTTON(states->use_ochusha_default_button), _("Hide mailto"));
  gtk_widget_show(tmp_widget);
  gtk_box_pack_start(GTK_BOX(hbox), tmp_widget, FALSE, FALSE, 0);
  states->hide_mailto_button = GTK_TOGGLE_BUTTON(tmp_widget);

  gtk_widget_show(hbox);

  tmp_widget = gtk_check_button_new_with_label(_("Use board specific setting for a bone."));
  gtk_widget_show(tmp_widget);
  gtk_box_pack_start(GTK_BOX(vbox), tmp_widget, FALSE, FALSE, 0);
  states->board_local_a_bone_button = GTK_TOGGLE_BUTTON(tmp_widget);

  tmp_widget = gtk_check_button_new_with_label(_("A bone by name field."));
  gtk_widget_show(tmp_widget);
  gtk_box_pack_start(GTK_BOX(vbox), tmp_widget, FALSE, FALSE, 0);
  states->a_bone_by_name_button = GTK_TOGGLE_BUTTON(tmp_widget);

  tmp_widget = regex_editor_new(NULL);
  gtk_widget_show(tmp_widget);
  gtk_box_pack_start(GTK_BOX(vbox), tmp_widget, TRUE, TRUE, 0);
  states->a_bone_by_name_pattern = REGEX_EDITOR(tmp_widget);

  tmp_widget = gtk_check_button_new_with_label(_("A bone by ID."));
  gtk_widget_show(tmp_widget);
  gtk_box_pack_start(GTK_BOX(vbox), tmp_widget, FALSE, FALSE, 0);
  states->a_bone_by_id_button = GTK_TOGGLE_BUTTON(tmp_widget);

  tmp_widget = regex_editor_new(NULL);
  gtk_widget_show(tmp_widget);
  gtk_box_pack_start(GTK_BOX(vbox), tmp_widget, TRUE, TRUE, 0);
  states->a_bone_by_id_pattern = REGEX_EDITOR(tmp_widget);

  tmp_widget = gtk_check_button_new_with_label(_("A bone by content."));
  gtk_widget_show(tmp_widget);
  gtk_box_pack_start(GTK_BOX(vbox), tmp_widget, FALSE, FALSE, 0);
  states->a_bone_by_content_button = GTK_TOGGLE_BUTTON(tmp_widget);

  tmp_widget = regex_editor_new(NULL);
  gtk_widget_show(tmp_widget);
  gtk_box_pack_start(GTK_BOX(vbox), tmp_widget, TRUE, TRUE, 0);
  states->a_bone_by_content_pattern = REGEX_EDITOR(tmp_widget);

  return thread_frame;
}


static GtkWidget *
build_filter_setting_frame(OchushaApplication *application,
			   OchushaBulletinBoard *board, DialogStates *states)
{
  GtkWidget *filter_frame = gtk_frame_new(_("Threadlist's Filter Setting"));

  GtkWidget *vbox;
  GtkWidget *tmp_widget;

  gtk_frame_set_shadow_type(GTK_FRAME(filter_frame), GTK_SHADOW_ETCHED_OUT);
  gtk_widget_show(filter_frame);
  vbox = gtk_vbox_new(FALSE, 0);
  gtk_widget_show(vbox);
  gtk_container_add(GTK_CONTAINER(filter_frame), vbox);

  tmp_widget = gtk_check_button_new_with_label(_("Use this setting as the default setting."));
  states->use_as_default_button = GTK_TOGGLE_BUTTON(tmp_widget);
  gtk_widget_show(tmp_widget);
  gtk_box_pack_start(GTK_BOX(vbox), tmp_widget, FALSE, FALSE, 0);

  tmp_widget = threadlist_filter_setting_new_with_initial_setting(states->last_filter);
  states->setting = THREADLIST_FILTER_SETTING(tmp_widget);
  g_signal_connect(G_OBJECT(tmp_widget), "changed",
		   G_CALLBACK(setting_changed_cb), states);
  g_signal_connect(G_OBJECT(tmp_widget), "restore_default",
		   G_CALLBACK(restore_default_cb), states);

  if (states->info->filter.rule == NULL)
    {
      threadlist_filter_setting_set_restore_default_button_sensitive(THREADLIST_FILTER_SETTING(tmp_widget), FALSE);
    }

  gtk_widget_show(tmp_widget);
  gtk_box_pack_start(GTK_BOX(vbox), tmp_widget, FALSE, FALSE, 0);

  return filter_frame;
}
