/*
 * Copyright (c) 2003 The Ochusha Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: text_search_window.c,v 1.10 2003/12/30 21:53:28 fuyu Exp $
 */

#include "config.h"

#include "ochusha_private.h"
#include "ochusha.h"

#include "ochusha_ui.h"
#include "text_search_window.h"

#include "marshal.h"

#include <glib.h>
#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>


static void text_search_window_class_init(TextSearchWindowClass *klass);
static void text_search_window_init(TextSearchWindow *window);

static void text_search_window_finalize(GObject *object);
static void text_search_window_destroy(GtkObject *object);

static gboolean text_search_window_delete_event_handler(GtkWidget *widget,
							GdkEventAny *event,
							gpointer user_data);
static void text_search_window_process_query_change(TextSearchWindow *window);
static void text_search_window_process_find_next_response(
						TextSearchWindow *window);
static void text_search_window_emit_cancel_response(TextSearchWindow *window);


GType
text_search_window_get_type(void)
{
  static GType search_window_type = 0;

  if (search_window_type == 0)
    {
      static const GTypeInfo search_window_info =
	{
	  sizeof(TextSearchWindowClass),
	  NULL,	/* base_init */
	  NULL, /* base_finalize */
	  (GClassInitFunc)text_search_window_class_init,
	  NULL,	/* class_finalize */
	  NULL, /* class_data */
	  sizeof(TextSearchWindow),
	  0,	/* n_preallocs */
	  (GInstanceInitFunc)text_search_window_init,
	};

      search_window_type = g_type_register_static(GTK_TYPE_WINDOW,
						  "TextSearchWindow",
						  &search_window_info, 0);
    }

  return search_window_type;
}


enum {
  RESPONSE_SIGNAL,
  FIND_NEXT_SIGNAL,
  QUERY_CHANGED_SIGNAL,
  LAST_SIGNAL
};


static GtkWindowClass *parent_class = NULL;
static int text_search_window_signals[LAST_SIGNAL] = { 0, 0, 0 };


static void
text_search_window_class_init(TextSearchWindowClass *klass)
{
  GObjectClass *o_class = (GObjectClass *)klass;
  GtkObjectClass *object_class = (GtkObjectClass *)klass;
  parent_class = g_type_class_peek_parent(klass);

  /* GObject signals */
  o_class->finalize = text_search_window_finalize;

  /* GtkObject signals */
  object_class->destroy = text_search_window_destroy;


  text_search_window_signals[RESPONSE_SIGNAL] =
    g_signal_new("response",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(TextSearchWindowClass, response),
		 NULL, NULL,
		 ochusha_marshal_VOID__INT,
		 G_TYPE_NONE, 1,
		 G_TYPE_INT);
  text_search_window_signals[QUERY_CHANGED_SIGNAL] =
    g_signal_new("query_changed",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(TextSearchWindowClass, query_changed),
		 NULL, NULL,
		 ochusha_marshal_BOOLEAN__STRING_INT_BOOLEAN_BOOLEAN_BOOLEAN,
		 G_TYPE_BOOLEAN, 5,
		 G_TYPE_STRING,
		 G_TYPE_INT,
		 G_TYPE_BOOLEAN,
		 G_TYPE_BOOLEAN,
		 G_TYPE_BOOLEAN);
  text_search_window_signals[FIND_NEXT_SIGNAL] =
    g_signal_new("find_next",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(TextSearchWindowClass, find_next),
		 NULL, NULL,
		 ochusha_marshal_BOOLEAN__STRING_INT_BOOLEAN_BOOLEAN_BOOLEAN,
		 G_TYPE_BOOLEAN, 5,
		 G_TYPE_STRING,
		 G_TYPE_INT,
		 G_TYPE_BOOLEAN,
		 G_TYPE_BOOLEAN,
		 G_TYPE_BOOLEAN);

  klass->response = NULL;
  klass->query_changed = NULL;
  klass->find_next = NULL;
}


static gboolean
text_search_window_key_press_event_cb(TextSearchWindow *window,
				      GdkEventKey *event,
				      gpointer unused)
{
  g_return_val_if_fail(IS_TEXT_SEARCH_WINDOW(window), FALSE);

  if ((event->keyval == GDK_g && (event->state & GDK_CONTROL_MASK) != 0)
      || (event->keyval == GDK_bracketleft
	  && (event->state & GDK_CONTROL_MASK) != 0))
    {
      if (window->statusbar_message_id != 0)
	gtk_statusbar_remove(window->statusbar, window->statusbar_context_id,
			     window->statusbar_message_id);
      window->statusbar_message_id = 0;

      text_search_window_response(TEXT_SEARCH_WINDOW(window),
				  GTK_RESPONSE_CANCEL);
      return TRUE;
    }

  if (event->keyval == GDK_Up
      || (event->keyval == GDK_r && (event->state & GDK_CONTROL_MASK) != 0))
    {
      gtk_toggle_button_set_active(window->direction_up_button, TRUE);
      text_search_window_process_find_next_response(window);
      return TRUE;
    }

  if (event->keyval == GDK_Down
      || (event->keyval == GDK_s && (event->state & GDK_CONTROL_MASK) != 0))
    {
      gtk_toggle_button_set_active(window->direction_down_button, TRUE);
      text_search_window_process_find_next_response(window);
      return TRUE;
    }

  return FALSE;
}


static void
text_search_window_init(TextSearchWindow *window)
{
  /* ɥΤΥƥ */
  GtkWidget *container_vbox;
  GtkWidget *container_hbox;

  /* container */
  GtkWidget *vbox;
  GtkWidget *button_box;

  /* vbox */
  GtkWidget *main_box;
  GtkWidget *option_box;


  /* option_box */
  GtkWidget *search_option_box;
  GtkWidget *search_direction_frame;

  /* search_direction_frame */
  GtkWidget *direction_box;

  /* button_box */
  GtkWidget *next_button;
  GtkWidget *cancel_button;

  g_signal_connect(window,
		   "delete_event",
		   G_CALLBACK(text_search_window_delete_event_handler),
		   NULL);
  g_signal_connect(window,
		   "key_press_event",
		   G_CALLBACK(text_search_window_key_press_event_cb),
		   NULL);

  container_vbox = gtk_vbox_new(FALSE, 5);
  gtk_container_set_border_width(GTK_CONTAINER(&window->parent_object), 5);
  gtk_container_add(GTK_CONTAINER(&window->parent_object), container_vbox);

  container_hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(container_vbox), container_hbox, TRUE, TRUE, 0);

  window->statusbar = GTK_STATUSBAR(gtk_statusbar_new());
  gtk_box_pack_start(GTK_BOX(container_vbox),
		     GTK_WIDGET(window->statusbar), FALSE, FALSE, 0);
  window->statusbar_context_id
    = gtk_statusbar_get_context_id(window->statusbar, "text-search-result");
  window->statusbar_message_id = 0;


  vbox = gtk_vbox_new(FALSE, 5);
  gtk_box_pack_start(GTK_BOX(container_hbox), vbox, TRUE, TRUE, 0);

  button_box = gtk_vbutton_box_new();
  gtk_button_box_set_layout(GTK_BUTTON_BOX(button_box), GTK_BUTTONBOX_START);
  gtk_box_set_spacing(GTK_BOX(button_box), 5);
  gtk_box_pack_end(GTK_BOX(container_hbox), button_box, FALSE, FALSE, 5);


  /* vbox */
  main_box = gtk_hbox_new(FALSE, 5);
  gtk_box_pack_start(GTK_BOX(vbox), main_box, TRUE, TRUE, 0);

  option_box = gtk_hbox_new(FALSE, 5);
  gtk_box_pack_start(GTK_BOX(vbox), option_box, FALSE, FALSE, 0);


  /* main_box */
  gtk_box_pack_start(GTK_BOX(main_box),
		     gtk_label_new(_("Find What: ")), FALSE, FALSE, 0);

  window->entry = GTK_ENTRY(gtk_entry_new());
  g_signal_connect_swapped(window->entry, "changed",
			G_CALLBACK(text_search_window_process_query_change),
			window);
  gtk_box_pack_start(GTK_BOX(main_box),
		     GTK_WIDGET(window->entry), TRUE, TRUE, 0);


  /* option_box */
  search_option_box = gtk_vbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(option_box), search_option_box, FALSE, FALSE, 0);

  search_direction_frame = gtk_frame_new(_("Direction"));
  gtk_box_pack_start(GTK_BOX(option_box),
		     search_direction_frame, FALSE, FALSE, 0);


  /* search_option_box */
  window->match_case_button
    = GTK_TOGGLE_BUTTON(gtk_check_button_new_with_label(_("Match case")));
  g_signal_connect_swapped(window->match_case_button, "toggled",
			G_CALLBACK(text_search_window_process_query_change),
			window);
  gtk_box_pack_start(GTK_BOX(search_option_box),
		     GTK_WIDGET(window->match_case_button), FALSE, FALSE, 0);

  window->enable_wrap_button
    = GTK_TOGGLE_BUTTON(gtk_check_button_new_with_label(_("Wrap")));
  g_signal_connect_swapped(window->enable_wrap_button, "toggled",
			G_CALLBACK(text_search_window_process_query_change),
			window);
  gtk_box_pack_start(GTK_BOX(search_option_box),
		     GTK_WIDGET(window->enable_wrap_button), FALSE, FALSE, 0);

  window->use_regexp_button
    = GTK_TOGGLE_BUTTON(gtk_check_button_new_with_label(_("Regular expression")));
  g_signal_connect_swapped(window->use_regexp_button, "toggled",
			G_CALLBACK(text_search_window_process_query_change),
			window);
  gtk_box_pack_start(GTK_BOX(search_option_box),
		     GTK_WIDGET(window->use_regexp_button), FALSE, FALSE, 0);


  /* search_direction_frame */
  direction_box = gtk_hbox_new(TRUE, 5);
  gtk_container_add(GTK_CONTAINER(search_direction_frame), direction_box);

  /* direction_box */
  window->direction_up_button
    = GTK_TOGGLE_BUTTON(gtk_radio_button_new_with_label(NULL, _("Up")));
  gtk_box_pack_start(GTK_BOX(direction_box),
		     GTK_WIDGET(window->direction_up_button), FALSE, FALSE, 0);

  window->direction_down_button
    = GTK_TOGGLE_BUTTON(gtk_radio_button_new_with_label_from_widget(
				GTK_RADIO_BUTTON(window->direction_up_button),
				_("Down")));
  gtk_toggle_button_set_active(window->direction_down_button, TRUE);
  g_signal_connect_swapped(window->direction_down_button, "toggled",
			G_CALLBACK(text_search_window_process_query_change),
			window);
  gtk_box_pack_start(GTK_BOX(direction_box),
		     GTK_WIDGET(window->direction_down_button),
		     FALSE, FALSE, 0);


  /* button_box */
  next_button = gtk_button_new_with_mnemonic(_("_Find Next"));
  g_signal_connect_swapped(G_OBJECT(next_button), "clicked",
			   G_CALLBACK(text_search_window_process_find_next_response),
			   window);
  gtk_box_pack_start(GTK_BOX(button_box), next_button, FALSE, FALSE, 0);

  cancel_button = gtk_button_new_with_label(_("Cancel"));
  g_signal_connect_swapped(G_OBJECT(cancel_button), "clicked",
			   G_CALLBACK(text_search_window_emit_cancel_response),
			   window);
  gtk_box_pack_start(GTK_BOX(button_box), cancel_button, FALSE, FALSE, 0);

  gtk_widget_show_all(container_vbox);

  window->key = NULL;
  window->direction = TEXT_SEARCH_DIRECTION_DOWN;
  window->enable_wrap = FALSE;
  window->match_case = FALSE;
  window->use_regexp = FALSE;

  window->query_changed_since_last_search = TRUE;
  window->enable_incremental_search = FALSE;

  gtk_window_set_type_hint(GTK_WINDOW(window), GDK_WINDOW_TYPE_HINT_DIALOG);
  gtk_window_set_position(GTK_WINDOW(window), GTK_WIN_POS_CENTER_ON_PARENT);
}


static void
text_search_window_finalize(GObject *object)
{
#if 0
  TextSearchWindow *window = TEXT_SEARCH_WINDOW(object);
#endif
#if DEBUG_WIDGET_MOST
  fprintf(stderr, "text_search_window_finalize(%p)\n", object);
#endif

  if (G_OBJECT_CLASS(parent_class)->finalize)
    (*G_OBJECT_CLASS(parent_class)->finalize)(object);
}


static void
text_search_window_destroy(GtkObject *object)
{
  TextSearchWindow *window = TEXT_SEARCH_WINDOW(object);
#if DEBUG_WIDGET_MOST
  fprintf(stderr, "text_search_window_destroy(%p)\n", object);
#endif

  if (window->key != NULL)
    {
      G_FREE(window->key);
      window->key = NULL;
    }

  if (GTK_OBJECT_CLASS(parent_class)->destroy)
    (*GTK_OBJECT_CLASS(parent_class)->destroy)(object);
}


static gboolean
text_search_window_delete_event_handler(GtkWidget *widget, GdkEventAny *event,
					gpointer user_data)
{
  /* emit response signal */
  text_search_window_response(TEXT_SEARCH_WINDOW(widget),
			      GTK_RESPONSE_DELETE_EVENT);

  /* Do the destroy by default */
  return FALSE;
}


static void
text_search_window_show_no_match_text(TextSearchWindow *window)
{
  if (window->statusbar_message_id != 0)
    gtk_statusbar_remove(window->statusbar, window->statusbar_context_id,
			 window->statusbar_message_id);
  window->statusbar_message_id
    = gtk_statusbar_push(window->statusbar, window->statusbar_context_id,
			 _("No matching text"));
}


/*
 * 郎ѤäƤ顢windowδϢʬѹTRUE֤
 */
static gboolean
text_search_window_update_query(TextSearchWindow *window)
{
  gboolean result = FALSE;
  const gchar *key;
  TextSearchDirection direction;
  gboolean match_case;
  gboolean enable_wrap;
  gboolean use_regexp;

  if (!window->query_changed_since_last_search)
    return result;

  key = gtk_entry_get_text(window->entry);
  if (window->key == NULL || (key != NULL && strcmp(window->key, key) != 0))
    {
      if (window->key != NULL)
	G_FREE(window->key);
      window->key = G_STRDUP(key);
#if DEBUG_WIDGET
      fprintf(stderr, "new key: %s\n", window->key);
#endif
      result = TRUE;
    }

  direction = (gtk_toggle_button_get_active(window->direction_up_button)
	       ? TEXT_SEARCH_DIRECTION_UP : TEXT_SEARCH_DIRECTION_DOWN);
  if (window->direction != direction)
    {
      window->direction = direction;
#if DEBUG_WIDGET
      fprintf(stderr, "new direction: %d\n", window->direction);
#endif
      result = TRUE;
    }

  match_case = gtk_toggle_button_get_active(window->match_case_button);
  if (window->match_case != match_case)
    {
      window->match_case = match_case;
#if DEBUG_WIDGET
      fprintf(stderr, "new match_case: %d\n", window->match_case);
#endif
      result = TRUE;
    }

  enable_wrap = gtk_toggle_button_get_active(window->enable_wrap_button);
  if (window->enable_wrap != enable_wrap)
    {
      window->enable_wrap = enable_wrap;
#if DEBUG_WIDGET
      fprintf(stderr, "new enable_wrap: %d\n", window->enable_wrap);
#endif
      result = TRUE;
    }

  use_regexp = gtk_toggle_button_get_active(window->use_regexp_button);
  if (window->use_regexp != use_regexp)
    {
      window->use_regexp = use_regexp;
#if DEBUG_WIDGET
      fprintf(stderr, "new use_regexp: %d\n", window->use_regexp);
#endif
      result = TRUE;
    }

  window->query_changed_since_last_search = FALSE;

  return result;
}


static void
text_search_window_emit_search_signal(TextSearchWindow *window, int signal_id)
{
  gboolean result = FALSE;

  g_signal_emit(G_OBJECT(window),
		text_search_window_signals[signal_id],
		0,
		window->key,
		window->direction,
		window->enable_wrap,
		window->match_case,
		window->use_regexp,
		&result);

  if (!result)
    text_search_window_show_no_match_text(window);
  else
    {
      if (window->statusbar_message_id != 0)
	gtk_statusbar_remove(window->statusbar, window->statusbar_context_id,
			     window->statusbar_message_id);
      window->statusbar_message_id = 0;
    }
}


static void
text_search_window_process_query_change(TextSearchWindow *window)
{
  g_return_if_fail(IS_TEXT_SEARCH_WINDOW(window));

  window->query_changed_since_last_search = TRUE;
  if (window->enable_incremental_search)
    {
      text_search_window_update_query(window);
      text_search_window_emit_search_signal(window, QUERY_CHANGED_SIGNAL);
    }
}


static void
text_search_window_process_find_next_response(TextSearchWindow *window)
{
  g_return_if_fail(IS_TEXT_SEARCH_WINDOW(window));

  if (window->query_changed_since_last_search
      && text_search_window_update_query(window))
    text_search_window_emit_search_signal(window, QUERY_CHANGED_SIGNAL);
  else
    text_search_window_emit_search_signal(window, FIND_NEXT_SIGNAL);
}


static void
text_search_window_emit_cancel_response(TextSearchWindow *window)
{
  g_return_if_fail(IS_TEXT_SEARCH_WINDOW(window));

  if (window->statusbar_message_id != 0)
    gtk_statusbar_remove(window->statusbar, window->statusbar_context_id,
			 window->statusbar_message_id);
  window->statusbar_message_id = 0;

  text_search_window_response(TEXT_SEARCH_WINDOW(window),
			      GTK_RESPONSE_CANCEL);
}


GtkWidget *
text_search_window_new(void)
{
  return GTK_WIDGET(g_object_new(TEXT_SEARCH_WINDOW_TYPE, NULL));
}


void
text_search_window_set_enable_incremental_search(TextSearchWindow *window,
						 gboolean enable)
{
  g_return_if_fail(IS_TEXT_SEARCH_WINDOW(window));

  window->enable_incremental_search = enable;
}


gboolean
text_search_window_get_enable_incremental_search(TextSearchWindow *window)
{
  g_return_val_if_fail(IS_TEXT_SEARCH_WINDOW(window), FALSE);

  return window->enable_incremental_search;
}


void
text_search_window_set_enable_wrap(TextSearchWindow *window, gboolean enable)
{
  g_return_if_fail(IS_TEXT_SEARCH_WINDOW(window));

  gtk_toggle_button_set_active(window->enable_wrap_button, enable);
}


gboolean
text_search_window_get_enable_wrap(TextSearchWindow *window)
{
  g_return_val_if_fail(IS_TEXT_SEARCH_WINDOW(window), FALSE);

  return gtk_toggle_button_get_active(window->enable_wrap_button);
}


void
text_search_window_set_match_case(TextSearchWindow *window,
				  gboolean match_case)
{
  g_return_if_fail(IS_TEXT_SEARCH_WINDOW(window));

  gtk_toggle_button_set_active(window->match_case_button, match_case);
}


gboolean
text_search_window_get_match_case(TextSearchWindow *window)
{
  g_return_val_if_fail(IS_TEXT_SEARCH_WINDOW(window), FALSE);

  return gtk_toggle_button_get_active(window->match_case_button);
}


void
text_search_window_set_use_regexp(TextSearchWindow *window,
				  gboolean use_regexp)
{
  g_return_if_fail(IS_TEXT_SEARCH_WINDOW(window));

  gtk_toggle_button_set_active(window->use_regexp_button, use_regexp);
}


gboolean
text_search_window_get_use_regexp(TextSearchWindow *window)
{
  g_return_val_if_fail(IS_TEXT_SEARCH_WINDOW(window), FALSE);

  return gtk_toggle_button_get_active(window->use_regexp_button);
}


void
text_search_window_response(TextSearchWindow *window, int response_id)
{
  g_return_if_fail(IS_TEXT_SEARCH_WINDOW(window));

  g_signal_emit(G_OBJECT(window),
		text_search_window_signals[RESPONSE_SIGNAL],
		0,
		response_id);
}


void
text_search_window_set_key_selected(TextSearchWindow *window)
{
  g_return_if_fail(IS_TEXT_SEARCH_WINDOW(window));

  if (window->statusbar_message_id != 0)
    gtk_statusbar_remove(window->statusbar, window->statusbar_context_id,
			 window->statusbar_message_id);
  window->statusbar_message_id = 0;

  gtk_editable_select_region(GTK_EDITABLE(window->entry), 0, -1);
}
