/**********************************************************************

  iso8859-15.c -  Oniguruma (regular expression library)

  Copyright (C) 2003  K.Kosako (kosako@sofnec.co.jp)

**********************************************************************/
#include "regint.h"

/* reference
   http://en.wikipedia.org/wiki/ISO_8859-15
*/

#define ENC_ISO_8859_15_TO_LOWER_CASE(c) EncISO_8859_15_ToLowerCaseTable[c]
#define ENC_IS_ISO_8859_15_CTYPE(code,ctype) \
  ((EncISO_8859_15_CtypeTable[code] & ctype) != 0)

static UChar EncISO_8859_15_ToLowerCaseTable[] = {
  '\000', '\001', '\002', '\003', '\004', '\005', '\006', '\007',
  '\010', '\011', '\012', '\013', '\014', '\015', '\016', '\017',
  '\020', '\021', '\022', '\023', '\024', '\025', '\026', '\027',
  '\030', '\031', '\032', '\033', '\034', '\035', '\036', '\037',
  '\040', '\041', '\042', '\043', '\044', '\045', '\046', '\047',
  '\050', '\051', '\052', '\053', '\054', '\055', '\056', '\057',
  '\060', '\061', '\062', '\063', '\064', '\065', '\066', '\067',
  '\070', '\071', '\072', '\073', '\074', '\075', '\076', '\077',
  '\100', '\141', '\142', '\143', '\144', '\145', '\146', '\147',
  '\150', '\151', '\152', '\153', '\154', '\155', '\156', '\157',
  '\160', '\161', '\162', '\163', '\164', '\165', '\166', '\167',
  '\170', '\171', '\172', '\133', '\134', '\135', '\136', '\137',
  '\140', '\141', '\142', '\143', '\144', '\145', '\146', '\147',
  '\150', '\151', '\152', '\153', '\154', '\155', '\156', '\157',
  '\160', '\161', '\162', '\163', '\164', '\165', '\166', '\167',
  '\170', '\171', '\172', '\173', '\174', '\175', '\176', '\177',
  '\200', '\201', '\202', '\203', '\204', '\205', '\206', '\207',
  '\210', '\211', '\212', '\213', '\214', '\215', '\216', '\217',
  '\220', '\221', '\222', '\223', '\224', '\225', '\226', '\227',
  '\230', '\231', '\232', '\233', '\234', '\235', '\236', '\237',
  '\240', '\241', '\242', '\243', '\244', '\245', '\250', '\247',
  '\250', '\251', '\252', '\253', '\254', '\255', '\256', '\257',
  '\260', '\261', '\262', '\263', '\270', '\265', '\266', '\267',
  '\270', '\271', '\272', '\273', '\275', '\275', '\377', '\277',
  '\340', '\341', '\342', '\343', '\344', '\345', '\346', '\347',
  '\350', '\351', '\352', '\353', '\354', '\355', '\356', '\357',
  '\360', '\361', '\362', '\363', '\364', '\365', '\366', '\327',
  '\370', '\371', '\372', '\373', '\374', '\375', '\376', '\337',
  '\340', '\341', '\342', '\343', '\344', '\345', '\346', '\347',
  '\350', '\351', '\352', '\353', '\354', '\355', '\356', '\357',
  '\360', '\361', '\362', '\363', '\364', '\365', '\366', '\367',
  '\370', '\371', '\372', '\373', '\374', '\375', '\376', '\377',
};

static unsigned short EncISO_8859_15_CtypeTable[256] = {
  0x1004, 0x1004, 0x1004, 0x1004, 0x1004, 0x1004, 0x1004, 0x1004,
  0x1004, 0x1106, 0x1104, 0x1104, 0x1104, 0x1104, 0x1004, 0x1004,
  0x1004, 0x1004, 0x1004, 0x1004, 0x1004, 0x1004, 0x1004, 0x1004,
  0x1004, 0x1004, 0x1004, 0x1004, 0x1004, 0x1004, 0x1004, 0x1004,
  0x1142, 0x10d0, 0x10d0, 0x10d0, 0x10d0, 0x10d0, 0x10d0, 0x10d0,
  0x10d0, 0x10d0, 0x10d0, 0x10d0, 0x10d0, 0x10d0, 0x10d0, 0x10d0,
  0x1c58, 0x1c58, 0x1c58, 0x1c58, 0x1c58, 0x1c58, 0x1c58, 0x1c58,
  0x1c58, 0x1c58, 0x10d0, 0x10d0, 0x10d0, 0x10d0, 0x10d0, 0x10d0,
  0x10d0, 0x1e51, 0x1e51, 0x1e51, 0x1e51, 0x1e51, 0x1e51, 0x1a51,
  0x1a51, 0x1a51, 0x1a51, 0x1a51, 0x1a51, 0x1a51, 0x1a51, 0x1a51,
  0x1a51, 0x1a51, 0x1a51, 0x1a51, 0x1a51, 0x1a51, 0x1a51, 0x1a51,
  0x1a51, 0x1a51, 0x1a51, 0x10d0, 0x10d0, 0x10d0, 0x10d0, 0x18d0,
  0x10d0, 0x1c71, 0x1c71, 0x1c71, 0x1c71, 0x1c71, 0x1c71, 0x1871,
  0x1871, 0x1871, 0x1871, 0x1871, 0x1871, 0x1871, 0x1871, 0x1871,
  0x1871, 0x1871, 0x1871, 0x1871, 0x1871, 0x1871, 0x1871, 0x1871,
  0x1871, 0x1871, 0x1871, 0x10d0, 0x10d0, 0x10d0, 0x10d0, 0x1004,
  0x1004, 0x1004, 0x1004, 0x1004, 0x1004, 0x1004, 0x1004, 0x1004,
  0x1004, 0x1004, 0x1004, 0x1004, 0x1004, 0x1004, 0x1004, 0x1004,
  0x1004, 0x1004, 0x1004, 0x1004, 0x1004, 0x1004, 0x1004, 0x1004,
  0x1004, 0x1004, 0x1004, 0x1004, 0x1004, 0x1004, 0x1004, 0x1004,
  0x1142, 0x10d0, 0x1050, 0x1050, 0x1050, 0x1050, 0x1a51, 0x1050,
  0x1871, 0x1050, 0x1871, 0x10d0, 0x1050, 0x10d0, 0x1050, 0x1050,
  0x1050, 0x1050, 0x1850, 0x1850, 0x1a51, 0x1871, 0x1050, 0x10d0,
  0x1871, 0x1850, 0x1871, 0x10d0, 0x1a51, 0x1871, 0x1a51, 0x10d0,
  0x1a51, 0x1a51, 0x1a51, 0x1a51, 0x1a51, 0x1a51, 0x1a51, 0x1a51,
  0x1a51, 0x1a51, 0x1a51, 0x1a51, 0x1a51, 0x1a51, 0x1a51, 0x1a51,
  0x1a51, 0x1a51, 0x1a51, 0x1a51, 0x1a51, 0x1a51, 0x1a51, 0x1050,
  0x1a51, 0x1a51, 0x1a51, 0x1a51, 0x1a51, 0x1a51, 0x1a51, 0x1871,
  0x1871, 0x1871, 0x1871, 0x1871, 0x1871, 0x1871, 0x1871, 0x1871,
  0x1871, 0x1871, 0x1871, 0x1871, 0x1871, 0x1871, 0x1871, 0x1871,
  0x1871, 0x1871, 0x1871, 0x1871, 0x1871, 0x1871, 0x1871, 0x1050,
  0x1871, 0x1871, 0x1871, 0x1871, 0x1871, 0x1871, 0x1871, 0x1871
};

static int
iso_8859_15_mbc_to_lower(UChar* p, UChar* lower)
{
  *lower = ENC_ISO_8859_15_TO_LOWER_CASE(*p);
  return 1; /* return byte length of converted char to lower */
}

static int
iso_8859_15_mbc_is_case_ambig(UChar* p)
{
  int v = (EncISO_8859_15_CtypeTable[*p]
	   & (ENC_CTYPE_UPPER | ENC_CTYPE_LOWER));

  if ((v | ENC_CTYPE_LOWER) != 0) {
    /* 0xdf, 0xaa, 0xb5, 0xba are lower case letter, but can't convert. */
    if (*p == 0xdf || *p == 0xaa || *p == 0xb5 || *p == 0xba) return FALSE;
  }
  return (v != 0 ? TRUE : FALSE);
}

static int
iso_8859_15_code_is_ctype(CodePoint code, unsigned int ctype)
{
  if (code < 256)
    return ENC_IS_ISO_8859_15_CTYPE(code, ctype);
  else
    return FALSE;
}

static int
iso_8859_15_get_all_fold_match_code(CodePoint** codes)
{
  static CodePoint list[] = { 0xdf };
  *codes = list;
  return 1;
}

static int
iso_8859_15_get_fold_match_info(UChar* p, UChar* end, EncFoldMatchInfo** info)
{
  /* German alphabet ess-tsett(U+00DF) */
  static EncFoldMatchInfo ss = {
    3,
    { 1, 2, 2 },
    { "\337", "ss", "SS" } /* 0337: 0xdf */
  };

  if (p >= end) return -1;

  if (*p == 0xdf) {
    *info = &ss;
    return 1;
  }
  else if (p + 1 < end) {
    if ((*p == 'S' && *(p+1) == 'S') ||
	(*p == 's' && *(p+1) == 's')) {
      *info = &ss;
      return 2;
    }
  }

  return -1; /* is not a fold string. */
}

RegCharEncodingType RegEncodingISO_8859_15 = {
  {
    1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1
  },
  "ISO 8859-15",  /* name */
  1,              /* max byte length */
  TRUE,           /* is_fold_match */
  ENC_CTYPE_SUPPORT_LEVEL_SB_ONLY,  /* ctype_support_level */
  TRUE,           /* is continuous sb mb codepoint */
  enc_single_byte_mbc_to_code,
  enc_single_byte_code_to_mbclen,
  enc_single_byte_code_to_mbc_first,
  enc_single_byte_code_to_mbc,
  iso_8859_15_mbc_to_lower,
  iso_8859_15_mbc_is_case_ambig,
  iso_8859_15_code_is_ctype,
  enc_nothing_get_ctype_code_range,
  enc_single_byte_left_adjust_char_head,
  enc_single_byte_is_allowed_reverse_match,
  iso_8859_15_get_all_fold_match_code,
  iso_8859_15_get_fold_match_info
};


#if 0
/* cc -I. -I.. -DNOT_RUBY iso8859_15.c */
#include <stdio.h>
#include "regint.h"

static int IsAlpha(int c)
{
  if (c >= 0x41 && c <= 0x5a) return 1;
  if (c >= 0x61 && c <= 0x7a) return 1;
  if (c == 0xaa) return 1;
  if (c == 0xb5) return 1;
  if (c == 0xba) return 1;
  if (c >= 0xc0 && c <= 0xd6) return 1;
  if (c >= 0xd8 && c <= 0xf6) return 1;
  if (c >= 0xf8 && c <= 0x220) return 1;

  if (c == 0xa6) return 1;
  if (c == 0xa8) return 1;
  if (c == 0xb4) return 1;
  if (c == 0xb8) return 1;
  if (c == 0xbc) return 1;
  if (c == 0xbd) return 1;
  if (c == 0xbe) return 1;

  return 0;
}

static int IsBlank(int c)
{
  if (c == 0x09	|| c == 0x20 || c == 0xa0) return 1;
  return 0;
}

static int IsCntrl(int c)
{
  if (c >= 0x00	&& c <= 0x1F) return 1;
  if (c >= 0x7f	&& c <= 0x9F) return 1;
  return 0;
}

static int IsDigit(int c)
{
  if (c >= 0x30 && c <= 0x39) return 1;
  return 0;
}

static int IsGraph(int c)
{
  if (c >= 0x21 && c <= 0x7e) return 1;
  if (c >= 0xa1 && c <= 0x220) return 1;
  return 0;
}

static int IsLower(int c)
{
  if (c >= 0x61 && c <= 0x7a) return 1;
  if (c == 0xaa) return 1;
  if (c == 0xb5) return 1;
  if (c == 0xba) return 1;
  if (c >= 0xdf && c <= 0xf6) return 1;
  if (c >= 0xf8 && c <= 0xff) return 1;

  if (c == 0xa8) return 1;
  if (c == 0xb8) return 1;
  if (c == 0xbd) return 1;

  return 0;
}

static int IsPrint(int c)
{
  if (c >= 0x20 && c <= 0x7e) return 1;
  if (c >= 0xa0 && c <= 0x220) return 1;
  return 0;
}

static int IsPunct(int c)
{
#ifndef BY_UNICODE_PROPERTY
  if (c == 0x24 || c == 0x2b || c == 0x5e || c == 0x60 ||
      c == 0x7c || c == 0x7e) return 1;
  if (c >= 0x3c && c <= 0x3e) return 1;
#endif

  if (c >= 0x21 && c <= 0x23) return 1;
  if (c >= 0x25 && c <= 0x2a) return 1;
  if (c >= 0x2c && c <= 0x2f) return 1;
  if (c >= 0x3a && c <= 0x3b) return 1;
  if (c >= 0x3f && c <= 0x40) return 1;
  if (c >= 0x5b && c <= 0x5d) return 1;
  if (c == 0x5f) return 1;
  if (c == 0x7b) return 1;
  if (c == 0x7d) return 1;
  if (c == 0xa1) return 1;
  if (c == 0xab) return 1;
  if (c == 0xad) return 1;
  if (c == 0xb7) return 1;
  if (c == 0xbb) return 1;
  if (c == 0xbf) return 1;

  return 0;
}

static int IsSpace(int c)
{
  if (c >= 0x09 && c <= 0x0d) return 1;
  if (c == 0x20) return 1;
  if (c == 0xa0) return 1;
  return 0;
}

static int IsUpper(int c)
{
  if (c >= 0x41 && c <= 0x5a) return 1;
  if (c >= 0xc0 && c <= 0xd6) return 1;
  if (c >= 0xd8 && c <= 0xde) return 1;

  if (c == 0xa6) return 1;
  if (c == 0xb4) return 1;
  if (c == 0xbc) return 1;
  if (c == 0xbe) return 1;

  return 0;
}

static int IsXDigit(int c)
{
  if (c >= 0x30 && c <= 0x39) return 1;
  if (c >= 0x41 && c <= 0x46) return 1;
  if (c >= 0x61 && c <= 0x66) return 1;
  return 0;
}

static int IsWord(int c)
{
  if (c >= 0x30 && c <= 0x39) return 1;
  if (c >= 0x41 && c <= 0x5a) return 1;
  if (c == 0x5f) return 1;
  if (c >= 0x61 && c <= 0x7a) return 1;
  if (c == 0xaa) return 1;
  if (c >= 0xb2 && c <= 0xb3) return 1;
  if (c == 0xb5) return 1;
  if (c >= 0xb9 && c <= 0xba) return 1;
  if (c >= 0xbc && c <= 0xbe) return 1;
  if (c >= 0xc0 && c <= 0xd6) return 1;
  if (c >= 0xd8 && c <= 0xf6) return 1;
  if (c >= 0xf8 && c <= 0x220) return 1;

  if (c == 0xa6) return 1;
  if (c == 0xa8) return 1;
  if (c == 0xb4) return 1;
  if (c == 0xb8) return 1;

  return 0;
}

static int IsAscii(int c)
{
  if (c >= 0x00 && c <= 0x127) return 1;
  return 0;
}

static int exec(FILE* fp)
{
#define N  8

  int c, val;

  fprintf(fp, "{\n");
  for (c = 0; c < 256; c++) {
    val = 0;
    if (IsAlpha(c))  val |= ENC_CTYPE_ALPHA;
    if (IsBlank(c))  val |= ENC_CTYPE_BLANK;
    if (IsCntrl(c))  val |= ENC_CTYPE_CNTRL;
    if (IsDigit(c))  val |= ENC_CTYPE_DIGIT;
    if (IsGraph(c))  val |= ENC_CTYPE_GRAPH;
    if (IsLower(c))  val |= ENC_CTYPE_LOWER;
    if (IsPrint(c))  val |= ENC_CTYPE_PRINT;
    if (IsPunct(c))  val |= ENC_CTYPE_PUNCT;
    if (IsSpace(c))  val |= ENC_CTYPE_SPACE;
    if (IsUpper(c))  val |= ENC_CTYPE_UPPER;
    if (IsXDigit(c)) val |= ENC_CTYPE_XDIGIT;
    if (IsWord(c))   val |= ENC_CTYPE_WORD;
    if (IsAscii(c))  val |= ENC_CTYPE_ASCII;

    if (c % N == 0) fputs("  ", fp);
    fprintf(fp, "0x%04x", val);
    if (c != 255) fputs(",", fp);
    if (c != 0 && c % N == (N-1))
      fputs("\n", fp);
    else
      fputs(" ", fp);
  }
  fprintf(fp, "};\n");
  return 0;
}

extern int main(int argc, char* argv[])
{
  exec(stdout);
}
#endif /* 0 */
