/*
 * Printer Status Utility Type A Program
 *
 * This program displays the printer status information the backend of
 * CUPS1.1 with OpenPrinting Bi-di plugin Type A capability has issued.
 *
 * Copyright (c) 2003, 2004
 *  E&D Inc. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <getopt.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>
#include <iconv.h>

#include "cups/cups.h"
#include "cups/http.h"
#include "cups/ipp.h"
#include "cups/language.h"

#include "psta_callbacks.h"
#include "psta_interface.h"
#include "psta_support.h"

extern GtkWidget *NameValue;
extern GtkWidget *StateValue;
extern GtkWidget *LocationValue;
extern GtkWidget *DescriptionValue;
extern GtkWidget *InkLevelTextView;
extern GtkWidget *WarningTextView;
extern GtkWidget *InkLevelFrame;


#define CONV_BUFLEN		4096
#define MAX_MESSAGE_SIZE	1024

typedef struct {
	char *name;
	char *message;
} nameMessage;

typedef struct {
	int  number;
	char *message;
} numberMessage;

nameMessage *inkcol;
nameMessage *inkmsg;
numberMessage *sttmsg;


#define IPP_REASONS_MAX_NUM	16
#define is_set(s) ((s) && (*s))

static const char *host_;
static const char *login_;
static const char *password_;
static int port_;
static char *printer_name;
static int jpnmsg;

/*
 * status messages
 */

nameMessage en_inkcol[] = {
	{ "c", "Cyan" },
	{ "m", "Magenta" },
	{ "y", "Yellow" },
	{ "k", "Black" },
	{ NULL },
};

nameMessage en_inkmsg[] = {
	{ "report",      "" },
	{ "low-warning", "Ink level is low." },
	{ "empty",       "Ink is empty." },
	{ NULL },
};

numberMessage en_sttmsg[] = {
	{ IPP_PRINTER_IDLE,       "Idle" },
	{ IPP_PRINTER_PROCESSING, "Printing" },
	{ IPP_PRINTER_STOPPED,    "Stopped" },
	{ -1 , "***" }
};

nameMessage ja_inkcol[] = {
	{ "c", "" },
	{ "m", "ޥ" },
	{ "y", "" },
	{ "k", "֥å" },
	{ NULL },
};

nameMessage ja_inkmsg[] = {
	{ "report",      "" },
	{ "low-warning", "󥯤ʤʤäƤޤ" },
	{ "empty",       "󥯤ʤʤޤ" },
	{ NULL },
};

numberMessage ja_sttmsg[] = {
	{ IPP_PRINTER_IDLE,       "Ե" },
	{ IPP_PRINTER_PROCESSING, "" },
	{ IPP_PRINTER_STOPPED,    "" },
	{ -1 , "***" }
};


/*
 * convert EUC to UTF8
 */
static size_t
conv_utf8(iconv_t conv,
	  char *src,
	  char *dst,
	  size_t dstlen)
{
	size_t ib, ob, n;
	char *p = dst;

	ib = strlen(src);
	ob = dstlen;
	n = iconv(conv, &src, &ib, &dst, &ob);
	if(n >= 0) *dst = '\0';
	return n;
}

/*
 * get ink color message from ink code 
 */
static char *
ink_color(char *cname)
{
	int i = 0;
	char *p;

	while((p = inkcol[i].name)){
		if(!strcmp(cname, p)){
			return inkcol[i].message;
		}
		i++;
	}
	return cname;
}

/*
 * get ink warning message from keyword of printer-state-reasons
 */
static char *
ink_msg(char *msg)
{
	int i = 0;
	char *p;

	while((p = inkmsg[i].name)){
		if(!strcmp(msg, p)){
			return inkmsg[i].message;
		}
		i++;
	}
	return msg;
}

/*
 * get printer status message 
 */
static char *
stt_msg(int val)
{
	int i = 0;
	int n;

	while((n = sttmsg[i].number) >= 0){
		if(val == n){
			return sttmsg[i].message;
		}
		i++;
	}
	return "***";
}



/*
 * display ink warning messages
 */
static void
disp_state_reasons(GtkTextBuffer *buffer,
		   GtkTextIter *iter,
		   char *reasons[],
		   int n)
{
	int i, c, lev;
	char *p;
	char col[8], msg[40], msgbuf[512];

	for(i=0; i<n; i++){
		p = strstr(reasons[i], "opbidi-ink-");
		if(p){
			c = sscanf(p,"opbidi-ink-%[^-]-%i-%s",col,&lev,msg);
			if(c == 3){
				char *inkcol = ink_color(col);
				if(jpnmsg){
					sprintf(msgbuf, "%-14s\t%d\%\t%s\n", inkcol, lev, ink_msg(msg));
				}else{
					sprintf(msgbuf, "%s%s%d\%\t%s\n", inkcol, (strlen(inkcol) < 8)? "\t": " ", lev, ink_msg(msg));
				}
				gtk_text_buffer_insert(buffer, iter, " ", -1);
				gtk_text_buffer_insert(buffer, iter, msgbuf, -1);
			}
		}
	}


}

/*
 * display Warning messages
 */
static void
disp_state_message(GtkTextBuffer *buffer, char *message)
{
	int l;
	char *p, *q;
	char buff[MAX_MESSAGE_SIZE+1], msgbuf[512];

	l = strlen(message);
	if(l > MAX_MESSAGE_SIZE){
		return;
	}

	strcpy(buff, message);
	p = buff;

	while(p){
		if((q = index(p, '\t'))){
			*q++ = '\0';
		}
		sprintf(msgbuf, "%s\n", p);
		p = q;

		gtk_text_buffer_insert_at_cursor(buffer, msgbuf, -1);		
	}
}

/*------------------------------------------------------------------
 * CUPS
 *------------------------------------------------------------------*/

/*
 * Callback for password authentication
 */
static const char*
cupsGetPasswordCB(const char* prompt)
{
	printf("cupsGetPasswordCB:\n");
	return cupsGetPassword("Please enter a password:");
}

/*
 * initialize IPP request
 */
static ipp_t*
newIppRequest(void)
{
	ipp_t	*request = ippNew();
	cups_lang_t	*lang = cupsLangDefault();

	request->request.op.request_id = 1;

	ippAddString(request,IPP_TAG_OPERATION,IPP_TAG_CHARSET,
		     "attributes-charset",NULL,cupsLangEncoding(lang));

	ippAddString(request,IPP_TAG_OPERATION,IPP_TAG_LANGUAGE,
		     "attributes-natural-language",NULL,lang->language);

	return request;
}

/*
 * setup for CUPS server connection
 */
static void
setup_cups_conn(void)
{
	host_  = cupsServer();
	login_ = cupsUser();
	port_  = ippPort();
	cupsSetPasswordCB(cupsGetPasswordCB);
}

/*
 * execute IPP request
 */
static ipp_t*
processRequest(ipp_t *req,
	       const char *res)
{
	http_t	*HTTP;
	ipp_t	*answer;

	HTTP = httpConnect(host_,port_);
	if(!HTTP){
		ippDelete(req);
		return 0;
	}
	answer = cupsDoRequest(HTTP,req,res);
	httpClose(HTTP);
	if(!answer)
		return 0;
	if(answer->state == IPP_ERROR || answer->state == IPP_IDLE){
		ippDelete(answer);
		return 0;
	}
	return answer;
}

/*
 * get numerical value from specified attribute name
 */
static int
get_tag_integer(ipp_t *request,
		char *attr,
		int *num)
{
	ipp_attribute_t	*ipp_attr;
	ipp_value_t *ipp_value;

	ipp_attr = ippFindAttribute(request, attr, IPP_TAG_ZERO);
	if(!ipp_attr)
		return 0;
	ipp_value = &ipp_attr->values[0];
	*num = ipp_value->integer;
	return 1;
}

/*
 * get message string from specified attribute name
 */
static int
get_tag_string(ipp_t *request,
	       char *attr,
	       char **charset,
	       char **text)
{
	ipp_attribute_t	*ipp_attr;
	ipp_value_t *ipp_value;

	ipp_attr = ippFindAttribute(request, attr, IPP_TAG_ZERO);
	if(!ipp_attr) return 0;
	ipp_value = &ipp_attr->values[0];
	if(charset)
		*charset = ipp_value->string.charset;
	*text = ipp_value->string.text;
	return 1;
}

/*
 * get message string array from specified attribute name
 */
static int
get_tag_strings(ipp_t *request,
		char *attr,
		int size,
		char **charset,
		char **text)
{
	ipp_attribute_t	*ipp_attr;
	int i, max;

	ipp_attr = ippFindAttribute(request, attr, IPP_TAG_ZERO);
	if(!ipp_attr) return 0;
	max = (ipp_attr->num_values < size)? ipp_attr->num_values: size;
	for(i=0; i<max; i++){
		if(charset)
			charset[i] = ipp_attr->values[i].string.charset;
		text[i] = ipp_attr->values[i].string.text;
	}
	return i;
}

/*
 * display printer status
 */
static gboolean
updateStatus(gpointer *data)
{
  ipp_t	*request = newIppRequest();
  char	uri[256];
  char	prn[256];
  int	value = -1;
  
  request->request.op.operation_id = IPP_GET_PRINTER_ATTRIBUTES;
  
  sprintf(uri, "ipp://%s:%d/printers/%s", host_, port_, printer_name);
  ippAddString(request,IPP_TAG_OPERATION,IPP_TAG_URI,
	       "printer-uri",NULL,uri);
  
  sprintf(prn, "/printers/");
  request = processRequest(request, prn);
  
  if (request && request->curtag == IPP_TAG_PRINTER) {
    int r, num;
    char *str;
    char *reasons[IPP_REASONS_MAX_NUM];
    
    r = get_tag_string(request, "printer-name", NULL, &str);
    if(r){
      gtk_label_set_text((GtkLabel *)NameValue,  str);
    }
    
    r = get_tag_integer(request, "printer-state", &num);
    if(r){
      gtk_label_set_text((GtkLabel *)StateValue,  stt_msg(num));
    }
    
    r = get_tag_string(request, "printer-location", NULL, &str);
    if(r){
      gtk_label_set_text((GtkLabel *)LocationValue,  str);
    }
    
    r = get_tag_string(request, "printer-info", NULL, &str);
    if(r){
      gtk_label_set_text((GtkLabel *)DescriptionValue,  str);
    }
    
    /* printer-state-message */
    r = get_tag_string(request, "printer-state-message", NULL, &str);
    if(r){
      GtkTextBuffer *buffer;
      GtkTextIter start, end;
      
      /* display warning message */
      buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (WarningTextView));
      gtk_text_buffer_get_start_iter (buffer, &start);
      gtk_text_buffer_get_end_iter(buffer, &end);
      gtk_text_buffer_delete (buffer, &start, &end);
      disp_state_message(buffer, str);
    }
    
    /* printer-state-reasons */
    r = get_tag_strings(request, "printer-state-reasons",
			IPP_REASONS_MAX_NUM, NULL, reasons);
    if(r){
      int i;
      GtkTextBuffer *buffer;
      GtkTextIter start, end;

      /* display warning message */
      buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (InkLevelTextView));
      gtk_text_buffer_get_start_iter (buffer, &start);
      gtk_text_buffer_get_end_iter(buffer, &end);
      gtk_text_buffer_delete (buffer, &start, &end);

      disp_state_reasons(buffer, &start, reasons, r);
    }
    
  }
  ippDelete(request);
  return value;
}

/*
 * display command usage
 */
static void
usage(void)
{
  fprintf(stderr, "\nUsage: printerStatusUtilityTypeA [-P name]\n\n");
}

/*
 * setup for printer name
 */
static void
setup_printer_name(int argc,
		   char **argv)
{
  int	num, c;
  char *prtname = NULL;
  cups_dest_t *dests;
  cups_dest_t *dest;

  num = cupsGetDests(&dests);

  if(num<1){
    /* Can't not find avaiable printer */
    return;
  }

  /* check '-P' command option */
  while(1){
    c = getopt(argc, argv, "P:");
    if(c < 0)
      break;
    switch (c) {
    case 'P':
      prtname = optarg;
      break;
    default:
      usage();
      exit(1);
    }
  }

  /* Does CUP server knows the specified printer name? */
  dest = cupsGetDest(prtname, NULL, num, dests);
  if(dest)
    if(prtname)
      printer_name = prtname;
    else
      printer_name = strdup(dest->name);
  cupsFreeDests(num, dests);

  if(!printer_name){
    puts("No default destination.");
    exit(1);
  }

}

/*
 * initialization of display messages
 */
static void
init_disp_message(void)
{
	nameMessage *_inkcol;
	nameMessage *_inkmsg;
	numberMessage *_sttmsg;

	int n, i;
	iconv_t conv;
	char dest[CONV_BUFLEN];
	char *lang;

	lang = getenv("LANGUAGE");
	if(!is_set(lang))
		lang = getenv("LC_ALL");
	if(!is_set(lang))
		lang = getenv("LC_MESSAGES");
	if(!is_set(lang))
		lang = getenv("LANG");
	if(!is_set(lang))
		lang = "C";

	jpnmsg = FALSE;
	if(index(lang,'.') && !strncasecmp(lang,"ja_JP",5))
		jpnmsg = TRUE;
	else if(!strncasecmp(lang,"ja",2))
		jpnmsg = TRUE;
	
	if(!jpnmsg){
		inkcol = en_inkcol;
		inkmsg = en_inkmsg;
		sttmsg = en_sttmsg;
		return;
	}

	/*
	 * setup for japanese message
	 */
	_inkcol = ja_inkcol;
	_inkmsg = ja_inkmsg;
	_sttmsg = ja_sttmsg;

	conv = iconv_open ("UTF-8","EUC-JP");

	/* initialize Ink color display messages */
	n = sizeof(ja_inkcol) / sizeof(nameMessage);
	inkcol = (nameMessage *)calloc(n, sizeof(nameMessage));
	for(i=0; i<n; i++){
		if(_inkcol[i].name){
			inkcol[i].name = strdup(_inkcol[i].name);
			conv_utf8(conv, _inkcol[i].message, dest, CONV_BUFLEN);
			inkcol[i].message = strdup(dest);
		}
	}

	/* initialize Warning messages */
	n = sizeof(ja_inkmsg) / sizeof(nameMessage);
	inkmsg = (nameMessage *)calloc(n, sizeof(nameMessage));
	for(i=0; i<n; i++){
		if(_inkmsg[i].name){
			inkmsg[i].name = strdup(_inkmsg[i].name);
			inkmsg[i].message = strdup(_inkmsg[i].message);
			conv_utf8(conv, _inkmsg[i].message, dest, CONV_BUFLEN);
			inkmsg[i].message = strdup(dest);
		}
	}

	/* initialize Printer status messages */
	n = sizeof(ja_sttmsg) / sizeof(numberMessage);
	sttmsg = (numberMessage *)calloc(n, sizeof(nameMessage));
	for(i=0; i<n; i++){
	        sttmsg[i].number = _sttmsg[i].number;
		sttmsg[i].message = strdup(_sttmsg[i].message);
		conv_utf8(conv, _sttmsg[i].message, dest, CONV_BUFLEN);
		sttmsg[i].message = strdup(dest);
	}

	iconv_close(conv);
}

/*=========================== PUBLIC FUNCTIONS ===========================*/

void
init_status_proc(int argc, char **argv)
{
  /* setup for printer name */
  setup_printer_name(argc, argv);

  /* connection to CUPS server */
  setup_cups_conn();

  /* initialization of display messages */
  init_disp_message();

  /* display printer status at first time */
  updateStatus(NULL);

  /* registration of timer process handler */
  gtk_timeout_add(2500,(GtkFunction)updateStatus,NULL);
}
