/** @file
 * @brief Process type BiDi-module data(XML) handling utility.
 *
 * This file is part of the `Printer Status Utility Type B' program.
 *
 * @author Copyright (C) 2003,2004 EPSON KOWA Corporation
 * @date 2004/02/06
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <string.h>
#include <libxml/tree.h>

#include "psub_common.h"
#include "psub_xmlutil.h"
#include "psub_debug.h"

/* Common Define... */
#ifdef  OK
#if   ( OK != 0 )
#error  OK is already defined in another include file.
#endif
#else
#define OK 0
#endif

#ifdef  ERROR
#if   ( ERROR != -1 )
#error  ERROR is already defined in another include file.
#endif
#else
#define ERROR (-1)
#endif

#ifndef FUNCTION
#define FUNCTION
#endif

#ifdef  LOCAL_FUNC
#if   ( LOCAL_FUNC != static )
#error  LOCAL_FUNC is already defined in another include file.
#endif
#else
#define LOCAL_FUNC static
#endif

#ifdef  TEST_FUNC
#error  TEST_FUNC is already defined.
#else
#define TEST_FUNC
#endif

#define DEFAULT_LANG NULL	/* UTF-8Τ̵Ѵ */

/* ELEMENT ʸ */
#define ELM_PRT_LABEL  "Printer"
#define ELM_MIB_LABEL  "Printer-MIB"

#define ELM_PST_LABEL  "PrtStatus"
#define ELM_PST_LV1    "PrtStatusTable"
#define ELM_PST_LV2    "PrtStatusEntry"
#define ELM_PST_INDEX  "PrtStatusIndex"
#define ELM_PST_TYPE   "PrtStatusType"
#define ELM_PST_DESC   "PrtStatusDescription"

#define ELM_ALT_LV1    "PrtAlertTable"
#define ELM_ALT_LV2    "PrtAlertEntry"
#define ELM_ALT_DESC   "PrtAlertDescription"

#define ELM_SPL_LV1    "PrtMarkerSuppliesTable"
#define ELM_SPL_LV2    "PrtMarkerSuppliesEntry"
#define ELM_SPL_TYPE   "PrtMarkerSuppliesType"
#define ELM_SPL_CAPA   "PrtMarkerSuppliesMaxCapacity"
#define ELM_SPL_LEVEL  "PrtMarkerSuppliesLevel"
#define ELM_SPL_DESC   "PrtMarkerSuppliesDescription"


typedef struct pair_int_str_tag {
        int n;
        char *s;
} int_str_t;

/* node  ե饰 */
typedef enum {
        NEXT   = 0x0001,
        PREV   = 0x0002,
        CHILD  = 0x0010,
        PARENT = 0x0020
} search_vector_t;


/* ֥ 80%פΤ褦ʸbufɲä롣 */
LOCAL_FUNC int add_percent_str( xmlNodePtr nod, char *buf, int buf_size);

/* ʸؤʸɲ */
LOCAL_FUNC int str_add_str(char *buf, int buf_size, char *str);

/* XMLꥭ element  node pointerõ */
LOCAL_FUNC xmlNodePtr find_node (xmlNodePtr nod, const char *key, search_vector_t flag);

/* nodecontent valueФ */
LOCAL_FUNC int get_string (xmlNodePtr nod, const char *lang, char *buf, size_t buf_size);

/* nodecontent valueintǼФ */
LOCAL_FUNC int get_value (xmlNodePtr nod, int *value);

/* UTF-8langѴ */
LOCAL_FUNC int UTF8to(const char *lang, char *from, char *to, int buf_size);

/* ʸ󤬳ǼƤнü˲Ԥɲä롣 */
LOCAL_FUNC int str_add_NUL(char *buf, int buf_size);

/* for DEBUG: nodeʲξɽ */
TEST_FUNC int show_node_list (xmlNodePtr nod, int level, int num);


FUNCTION void *xml_start( const char *buf, int size )
{
        LIBXML_TEST_VERSION ;
        return (void *) xmlParseMemory(buf, size);
}



FUNCTION void xml_end( const void *doc )
{
        xmlDocPtr  p = (xmlDocPtr)doc;

        xmlFreeDoc(p);

        /*
         *Free the global variables that may
         *have been allocated by the parser.
         */
        xmlCleanupParser();
}


FUNCTION int get_printer_status(void *doc, char *buf, int buf_size)
{
        xmlDocPtr  p = (xmlDocPtr)doc;
        xmlNodePtr nod1, top;
        int st = 0;

        buf[0] = '\0';

        top = xmlDocGetRootElement(p);
        if ( NULL == top || NULL == top->children ){
                msgp(Noisy,"XML ᥨ顼\n");
                strncpy( buf, "Error", buf_size );
                return ERROR;
        }

        /* ץ󥿥ơõ */
        nod1 = find_node(top->children, ELM_PST_LABEL, NEXT);
        if ( NULL == nod1 ) {
                msgp(Noisy,"ơʤ\n");
                strncpy( buf, "Error", buf_size );
                return ERROR;
        }

        /* ץ󥿥ơLabelֹ */
        if ( get_value(nod1->children, &st) != OK || 0 == st ) {
                msgp(Noisy,"ơֹʤ\n");
                strncpy( buf, "Error", buf_size );
                return ERROR;
        }

        /* ץ󥿥ơơ֥򸡺 */
        nod1 = find_node(top->children, ELM_PST_LV1, NEXT);
        if ( NULL == nod1 ) {
                msgp(Noisy,"ơơ֥ʤ\n");
                strncpy( buf, "Error", buf_size );
                return ERROR;

        } else {

                xmlNodePtr nod2;

                for ( nod2 = find_node(nod1->children, ELM_PST_LV2, NEXT);
                      NULL != nod2;
                      nod2 = find_node(nod2->next, ELM_PST_LV2, NEXT) ) {

                        xmlNodePtr nod3;

                        nod3 = find_node(nod2->children, ELM_PST_INDEX, NEXT);
                        if ( NULL != nod3 ) {

                                int index = 0;

                                if ( get_value(nod3->children, &index) == OK &&
                                     index == st ) {

                                        xmlNodePtr nod4;

                                        nod4 = find_node(nod2->children, ELM_PST_DESC, NEXT);

                                        /* DESCRIPTIONθ */
                                        if ( NULL != nod4 ) {
                                                if ( get_string( nod4->children, DEFAULT_LANG, buf, buf_size ) <= 0 ){
                                                        msgp(Noisy,"ơơ֥ʤ\n");
                                                        strncpy( buf, "Error", buf_size );
                                                        return ERROR;
                                                }
                                                return OK;

                                        } else {
                                                /* FIXME: DESCRIPTIONꤵƤʤν̤ */
                                                msgp(Noisy,"ơʸˤʤ\n");
                                                strncpy( buf, "Error", buf_size );
                                                return ERROR;
                                        }
                                }
                        }
                }
        }
        msgp(Info,"ơμ˼Ԥ\n");
        strncpy( buf, "Error", buf_size );
        return ERROR;
}

FUNCTION int get_supplies_status(void *doc, char *buf, int buf_size)
{
        xmlDocPtr  p = (xmlDocPtr)doc;

        int len = 0;
        xmlNodePtr nod, top;

        buf[0] = '\0';

        top = xmlDocGetRootElement(p);
        if ( NULL == top || NULL == top->children )
                return ERROR;

        /* ץMIBõ */
        nod = find_node(top->children, ELM_MIB_LABEL, NEXT);
        if ( NULL == nod ) {
                msgp(Noisy,"ץMIBʤ\n");
                return ERROR;
        }

        /* ʾõ */
        for ( nod = find_node(nod->children, ELM_SPL_LV1, NEXT);
              NULL != nod && len >= 0;
              nod = find_node(nod->next, ELM_SPL_LV1, NEXT) ) {

                xmlNodePtr sub_nod;

                for ( sub_nod = find_node(nod->children, ELM_SPL_LV2, NEXT);
                      NULL != sub_nod && len >= 0;
                      sub_nod = find_node(sub_nod->next, ELM_SPL_LV2, NEXT) ) {

                        /* Supply typeɽʸbufɲä */
                        xmlNodePtr n = find_node(sub_nod->children, ELM_SPL_TYPE, NEXT);
                        if ( NULL != n ) {

                                int type = 0;

                                if ( get_value(n->children, &type) == OK ) {
                                        if ( 6 == type ) {	/* inkCartridge */
                                                len = add_percent_str(sub_nod->children, buf, buf_size);
                                        }
                                } else {
                                        /* FIXME: Supply TypeꤵƤʤν̤ */
                                }
                        }
                }
        }

        return OK;
}

FUNCTION int get_printer_alert(void *doc, char *buf, int buf_size)
{
        xmlDocPtr  p = (xmlDocPtr)doc;

        int len = 0;
        xmlNodePtr top, nod;

        buf[0] = '\0';

        top = xmlDocGetRootElement(p);
        if ( NULL == top || NULL == top->children )
                return ERROR;

        /* ץMIBõ */
        nod = find_node(top->children, ELM_MIB_LABEL, NEXT);
        if ( NULL == nod ) {
                msgp(Noisy,"ץMIBʤ\n");
                return ERROR;
        }

        /* ٹõ */
        for ( nod = find_node(nod->children, ELM_ALT_LV1, NEXT);
              NULL != nod && len >= 0;
              nod = find_node(nod->next, ELM_ALT_LV1, NEXT) ) {

                xmlNodePtr sub_nod;

                for ( sub_nod = find_node(nod->children, ELM_ALT_LV2, NEXT);
                      NULL != sub_nod && len >= 0;
                      sub_nod = find_node(sub_nod->next, ELM_ALT_LV2, NEXT) ) {

                        xmlNodePtr n;

                        /* BiDi֤ʸõ */
                        for ( n = find_node(sub_nod->children, ELM_ALT_DESC, NEXT);
                              NULL != n && len >= 0;
                              n = find_node(n->next, ELM_ALT_DESC, NEXT) ) {

                                /* ʸ󤬳ǼƤнü˲Ԥɲä롣 */
                                len = str_add_NUL(buf, buf_size);
                                if ( len < 0 )
                                        return ERROR;

                                /* BiDi֤ʸХåեɲä롣 */
                                len = get_string(n->children, DEFAULT_LANG, &buf[len], (buf_size - len));
                                if ( len < 0 )
                                        return ERROR;
                        }
                }
        }

        return OK;
}


#include <stdio.h>		/* for snprintf() */

/**  ͡׷ʸbufɲ
 *
 *	bufʸäƤСԤǥѥ졼ȤƤ
 *	ʸɲä롣
 *
 * @param nod Current node pointer.
 * @param buf ʸХåե
 * @param buf_size bufΥ(in Byte)
 *
 * @retval int
 * bufʸΤ礭(in Byte)֤\n
 * 顼λ֤ͤ
 */
LOCAL_FUNC int add_percent_str( xmlNodePtr nod, char *buf, int buf_size)
{
        int capa = 0;
        int level = 0;
        int len = 0;
        int wsize;
        xmlNodePtr cur;

        cur = find_node( nod, ELM_SPL_CAPA, NEXT );
        if ( !cur )
                return -2;

        if ( get_value( cur->children, &capa ) != OK )
                return -3;

        cur = find_node( nod, ELM_SPL_LEVEL, NEXT );
        if ( !cur )
                return -4;

        if ( get_value( cur->children, &level ) != OK )
                return -5;

        cur = find_node( nod, ELM_SPL_DESC, NEXT );
        if ( !cur )
                return -6;

        len = str_add_NUL(buf, buf_size);
        if ( len < 0 )
                return -7;

        if ( get_string( cur->children, DEFAULT_LANG, &buf[len], (buf_size - len) ) <= 0 )
                return -8;

        len = str_add_str(buf, buf_size, " ");
        if ( len < 0 )
                return -9;

        wsize = snprintf(&buf[len], (buf_size - len), "%d%%", (int)(level * 100 / capa));
        if ( ( wsize + len + 1) > buf_size )
                return -10;

        return strlen(buf);
}




/** XMLꥭ element  node pointerõ
 *
 *	nodǻꤵ줿node flagǻꤵ줿
 *	keyǻꤵ줿element nodeõpointer֤
 *
 *	flagǻǤ븡 next or child Τߤǡ
 *		next FLATʤΥ٥Τߡ
 *		child Υ٥ޤ಼إ٥⸡
 *	Ȥʤ롣\n
 *	prevˡparentʾإ٥˸ϡʤ
 *
 *	childꤵ줿硢Υ٥nextʸˤ
 *	children(إ٥)ͥȤʤ롣
 *
 *	nod⸡оݤȤʤ롣Ĥޤnodkeyelement
 *	äƤꤵ줿nod֤
 *
 * @param nod  current node pointer.
 * @param key  search element keyword.
 * @param flag  search vector flag ( recursive? or flat? )
 *
 * @retval xmlNodePtr ȯ xmlNodePtr
 * @retval NULL Ĥʤä
 *
 * ---------------------------------------------------------------- */
LOCAL_FUNC xmlNodePtr find_node (xmlNodePtr nod, const char *key, search_vector_t flag)
{
        /* Check miss call */
        if ( !(flag & CHILD) && !(flag & NEXT) )
                return NULL;

        /* Check terminator */
        if ( NULL == nod )
                return NULL;

        /* Check element */
        if ( XML_ELEMENT_NODE == nod->type ) {
                if ( xmlStrcmp( nod->name, key ) == 0 ) {
                        return nod;
                }
        }

        /* Set vector and recursive call */
        if ( flag & CHILD ) {
                if ( NULL == nod->children )
                        return NULL;

                return find_node ( nod->children, key, flag );
        }
        if ( flag & NEXT ) {
                if ( NULL == nod->next )
                        return NULL;

                return find_node ( nod->next, key, flag );
        }
        return NULL;
}



/** nodecontent valueФ
 *
 *	nodǻꤵ줿nodetext element content value
 *	buf_sizeʲlang˥󥳡ǥ󥰤줿ʸȤ
 *	bufɲä롣
 *
 *	nod->nextEntityTypeTEXTǤСnextʸ
 *	󵢸ƤӽФˤbufɲä롣
 *
 * @note	bufʸ󤬳ǼƤ硢ü '\\n'
 *		ɲä³˼ʸɲä롣
 *		lang  NULLꤵ줿ϡUTF-8֤
 *
 * @param nod  current node pointer.
 * @param lang  Ф󥳡ǥ󥰷
 * @param buf  ФʸγǼХåե
 * @param buf_size  bufΥ(in Bytes)
 *
 * @retval 0 ʸ󤬤ʤä
 * @retval + ֤ʸĹ
 * @retval - 顼ȯλbufʸݾڤʤ
 * ---------------------------------------------------------------- */
LOCAL_FUNC int get_string (xmlNodePtr nod, const char *lang, char *buf, size_t buf_size)
{
        int ret = 0;

        if ( NULL == nod )
                return 0;

        if ( nod->content ) {
                int i = 0;
                while ( (  nod->content[i] != '\0' ) &&
                        ( (nod->content[i] == '\n') ||
                          (nod->content[i] == '\t') ||
                          (nod->content[i] == ' ') ) ) {
                        i++;
                }
                if ( nod->content[i] != '\0' ) {
                        size_t size = strlen(buf);

                        if ( (buf_size -1) <= size ) {
                                return (-1);
                        }

                        if ( 0 != size ) {
                                buf[size++] = '\n';
                                buf[size] = '\0';
                        }

                        if ( NULL == lang ) {
                                strncpy(&buf[size], nod->content, (buf_size - size));
                                if ( strlen(nod->content) + size >= buf_size ) {
                                        return (-1);
                                }
                        } else {
                                ret = UTF8to(lang, nod->content, &buf[size], (buf_size - size));
                                if ( 0 != ret ) {
                                        return (-2);
                                }
                        }
                }
        }

        if ( (NULL != nod->next) && (XML_TEXT_NODE == nod->next->type) )
                return get_string ( nod->next, lang, buf, buf_size );

        return strlen(buf);
}


#include <stdlib.h>		/* for atoi() */
/** nodecontent valueintǼФ
 *
 *	nodǻꤵ줿nodetext element content value
 *	ɤߤȤintѴ֤
 *
 * @param nod  current node pointer.
 * @param value  ɤߤȤäͤγǼݥ
 *
 * @retval 0 ｪλ
 * 顼ȯ֤ͤ롣λvalueͤݾڤʤ
 * ---------------------------------------------------------------- */
LOCAL_FUNC int get_value (xmlNodePtr nod, int *value)
{
        int ret = 0;
        char buf[32] = { '\0', };
        int buf_size = 32;

        ret = get_string(nod, NULL, buf, buf_size);

        if ( ret <= 0 )
                return (-1);

        *value = atoi(buf);

        return 0;
}


#include <iconv.h>
/** UTF-8langѴ
 *
 *	UTF-8ʸfromlangǻꤷ󥳡ǥ󥰤Ѵ
 *	buf_size礭ΥХåեto֤
 *      Note:	langiconv --list ɽ륨󥳡ɤΰĤꡣ
 *		toϸƤӸǳݤΥbuf_sizeǻꤹ롣
 *		
 * example\n
 *	UTF8to(DEFAULT_LANG, xml, buf, sizeof(buf));
 *
 * @param lang - UTF-8Ѵ󥳡ǥ
 * @param from - ѴʸäХåե
 * @param to   - ѴʸǼѥХåե
 * @param buf_size - toΥХåե
 *
 * @retval 0 ｪλ
 * @retval ʳ 顼ȯλtoη̤ݾڤʤ
 *
 * ---------------------------------------------------------------- */
LOCAL_FUNC int UTF8to(const char *lang, char *from, char *to, int buf_size)
{
        iconv_t icv;
        size_t tr = 0;
        size_t in = strlen(from);
        size_t capa = buf_size;
        char  *buf = to;

        icv = iconv_open(lang, "UTF-8");
        if ( icv == (iconv_t)(-1) ) {
                return 1;
        }

        while ( tr != (size_t)(-1) && in > 0 && capa > 0 ) {
                tr = iconv(icv, &from, &in, &buf, &capa);
        }
        if ( 0 == capa ) {
                to[buf_size - 1] = '\0';
        } else {
                to[buf_size - capa] = '\0';
        }

        iconv_close(icv);
        if ( (tr == (size_t)(-1) ) || (in > 0) ) {
                return 2;
        }
        return 0;
}


/** ʸؤβɲ
 *
 *	ʸ󤬳ǼƤнü˲Ԥɲä롣
 *	ХåեƬ '\\0'ξʸκǸʸ
 *	'\n'ξϡ⤷ʤǵ롣(ξｪλ)
 *	Ԥɲä̤buf_sizeۤϥ顼Ȥʤ롣
 *	顼ξbufξ֤ݾڤʤ
 *
 * @param buf  ʸХåե
 * @param buf_size  ХåեΥ(in Byte)
 *
 * @retval 
 * ʸĹ֤ͤϥ顼Ȥ򼨤
 */

LOCAL_FUNC int str_add_NUL(char *buf, int buf_size)
{
        int len = strlen(buf);

        if ( (len + 2) > buf_size )
                return (-1);

        if ( 0 == len )
                return 0;

        if ( buf[(len - 1)] != '\n' ) {
                buf[len] = '\n';
                len ++;
                buf[len] = '\0';
        }
        return len;
}


/** ʸؤʸɲ
 *
 *	ʸǼХåեʸɲä롣
 *	ɲä̤buf_sizeۤϥ顼Ȥʤ롣
 *	顼ξbufξ֤ݾڤʤ
 *
 * 
 * @param buf  ʸХåե
 * @param buf_size  ХåեΥ(in Byte)
 * @param str  ɲäʸ
 *
 * @retval 
 * ʸĹ֤ͤϥ顼Ȥ򼨤
 */

LOCAL_FUNC int str_add_str(char *buf, int buf_size, char *str)
{
        int len = strlen(buf);

        if ( (len + strlen(str) ) >= buf_size )
                return (-1);

        if ( 0 == strlen(str) )
                return len;

        strcat(buf, str);
        len = strlen(buf);

        return len;
}



#define SHOW_BUFF_SIZE 1024
/** for DEBUG: nodeʲξɽ
 *
 *	nodǻꤵ줿nodeʲξɽ롣
 *
 * @param nod   current node pointer.
 * @param level  ߤγإ٥
 * @param num    ߤnodeֹ
 *
 * @retval 0 ｪλ
 * @retval 0ʳ 顼
 */
TEST_FUNC int show_node_list (xmlNodePtr nod, int level, int num)
{
        int ret = 0;
        char buf[SHOW_BUFF_SIZE] = { '\0', };

        if ( XML_DOCUMENT_NODE == nod->type )
                nod = xmlDocGetRootElement((xmlDocPtr)nod);
        if ( nod == NULL )  return 0;

        if ( XML_TEXT_NODE != nod->type)   printf("\n");

        if ( nod->content ) {
                int i = 0;
                while ( (  nod->content[i] != '\0' ) &&
                        ( (nod->content[i] == '\n') ||
                          (nod->content[i] == '\t') ||
                          (nod->content[i] == ' ') ) ) {
                        i++;
                }
                if ( nod->content[i] != '\0' ) {
                        UTF8to(DEFAULT_LANG, nod->content, buf, SHOW_BUFF_SIZE);
                        printf("\t\"%s\"", buf);
                }
        } else {
                printf(" %d: %2d: %s", level, num, nod->name);
        }

        if ( nod->children )
                ret = show_node_list ( nod->children, (level + 1), 0 );

        if ( nod->next )
                ret = show_node_list ( nod->next, level, (num + ((nod->type != XML_TEXT_NODE)? 1: 0)) );

        if ( 0 == level )  printf("\n");
        return ret;
}


#if 0				/* for TEST */
const int_str_t PrtAlertSeverityLevelTC[] = {
        { (1),"other" },
        { (3),  "critical" },
        { (4),  "warning" },
        { (5),  "warningBinaryChangeEvent" },
        { 0, NULL }
};

const int_str_t PrtAlertGroupTC[] = {
        { (1), "other" },
        { (3), "hostResourcesMIBStorageTable" },
        { (4), "hostResourcesMIBDeviceTable" },
        { (5), "generalPrinter" },
        { (6), "cover" },
        { (7), "localization" },
        { (8), "input" },
        { (9), "output" },
        { (10), "marker" },
        { (11), "markerSupplies" },
        { (12), "markerColorant" },
        { (13), "mediaPath" },
        { (14), "channel" },
        { (15), "interpreter" },
        { (16), "consoleDisplayBuffer" },
        { (17), "consoleLights" },
        { (18), "alert" },
        { (30), "finDevice" },
        { (31), "finSupply" },
        { (32), "finSupplyMediaInput" },
        { (33), "finAttribute" },
        { 0, NULL }
};

const int_str_t PrtAlertCodeTC[] = {
        { (1),  "other" },
        { (2),  "unknown" },
        { (3),  "coverOpen" },
        { (4),  "coverClosed" },
        { (5),  "interlockOpen" },
        { (6),  "interlockClosed" },
        { (7),  "configurationChange" },
        { (8),  "jam" },
        { (9),  "subunitMissing" },
        { (10), "subunitLifeAlmostOver" },
        { (11),  "subunitLifeOver" },
        { (12),  "subunitAlmostEmpty" },
        { (13),  "subunitEmpty" },
        { (14),  "subunitAlmostFull" },
        { (15),  "subunitFull" },
        { (16),  "subunitNearLimit" },
        { (17),  "subunitAtLimit" },
        { (18),  "subunitOpened" },
        { (19),  "subunitClosed" },
        { (20),  "subunitTurnedOn" },
        { (21),  "subunitTurnedOff" },
        { (22),  "subunitOffline" },
        { (23),  "subunitPowerSaver" },
        { (24),  "subunitWarmingUp" },
        { (25),  "subunitAdded" },
        { (26),  "subunitRemoved" },
        { (27),  "subunitResourceAdded" },
        { (28),  "subunitResourceRemoved" },
        { (29),  "subunitRecoverableFailure" },
        { (30),  "subunitUnrecoverableFailure" },
        { (31),  "subunitRecoverableStorageError" },
        { (32),  "subunitUnrecoverableStorageError" },
        { (33),  "subunitMotorFailure" },
        { (34),  "subunitMemoryExhausted" },
        { (35),  "subunitUnderTemperature" },
        { (36),  "subunitOverTemperature" },
        { (37),  "subunitTimingFailure" },
        { (38),  "subunitThermistorFailure" },
        { (501),  "doorOpen" },
        { (502),  "doorClosed" },
        { (503),  "powerUp" },
        { (504),  "powerDown" },
        { (505),  "printerNMSReset" },
        { (506),  "printerManualReset" },
        { (507),  "printerReadyToPrint" },
        { (801),  "inputMediaTrayMissing" },
        { (802),  "inputMediaSizeChange" },
        { (803),  "inputMediaWeightChange" },
        { (804),  "inputMediaTypeChange" },
        { (805),  "inputMediaColorChange" },
        { (806),  "inputMediaFormPartsChange" },
        { (807),  "inputMediaSupplyLow" },
        { (808),  "inputMediaSupplyEmpty" },
        { (809),  "inputMediaChangeRequest" },
        { (810),  "inputManualInputRequest" },
        { (811),  "inputTrayPositionFailure" },
        { (812),  "inputTrayElevationFailure" },
        { (813),  "inputCannotFeedSizeSelected" },
        { (901),  "outputMediaTrayMissing" },
        { (902),  "outputMediaTrayAlmostFull" },
        { (903),  "outputMediaTrayFull" },
        { (904),  "outputMailboxSelectFailure" },
        { (1001),  "markerFuserUnderTemperature" },
        { (1002),  "markerFuserOverTemperature" },
        { (1003),  "markerFuserTimingFailure" },
        { (1004),  "markerFuserThermistorFailure" },
        { (1005),  "markerAdjustingPrintQuality" },
        { (1101),  "markerTonerEmpty" },
        { (1102),  "markerInkEmpty" },
        { (1103),  "markerPrintRibbonEmpty" },
        { (1104),  "markerTonerAlmostEmpty" },
        { (1105),  "markerInkAlmostEmpty" },
        { (1106),  "markerPrintRibbonAlmostEmpty" },
        { (1107),  "markerWasteTonerReceptacleAlmostFull" },
        { (1108),  "markerWasteInkReceptacleAlmostFull" },
        { (1109),  "markerWasteTonerReceptacleFull" },
        { (1110),  "markerWasteInkReceptacleFull" },
        { (1111),  "markerOpcLifeAlmostOver" },
        { (1112),  "markerOpcLifeOver" },
        { (1113),  "markerDeveloperAlmostEmpty" },
        { (1114),  "markerDeveloperEmpty" },
        { (1115),  "markerTonerCartridgeMissing" },
        { (1301),  "mediaPathMediaTrayMissing" },
        { (1302),  "mediaPathMediaTrayAlmostFull" },
        { (1303),  "mediaPathMediaTrayFull" },
        { (1304),  "mediaPathCannotDuplexMediaSelected" },
        { (1501),  "interpreterMemoryIncrease" },
        { (1502),  "interpreterMemoryDecrease" },
        { (1503),  "interpreterCartridgeAdded" },
        { (1504),  "interpreterCartridgeDeleted" },
        { (1505),  "interpreterResourceAdded" },
        { (1506),  "interpreterResourceDeleted" },
        { (1507),  "interpreterResourceUnavailable" },
        { (1509),  "interpreterComplexPageEncountered" },
        { (1801),  "alertRemovalOfBinaryChangeEntry" },
        { 0, NULL }
};

const int_str_t PrtMarkerSuppliesType[] = {
        { 1, "other" },
        { 2, "unknown" },
        { 3, "toner" },
        { 4, "wasteToner" },
        { 5, "ink" },
        { 6, "inkCartridge" },
        { 7, "inkRibbon" },
        { 8, "wasteInk" },
        { 9, "opc" },
        { 10, "developer" },
        { 11, "fuserOil" },
        { 12, "solidWax" },
        { 13, "ribbonWax" },
        { 14, "wasteWax" },
        { 15, "fuser" },
        { 16, "coronaWire" },
        { 17, "fuserOilWick" },
        { 18, "cleanerUnit" },
        { 19, "fuserCleaningPad" },
        { 20, "transferUnit" },
        { 21, "tonerCartridge" },
        { 22, "fuserOiler" },
        { 23, "water" },
        { 24, "wasteWater" },
        { 25, "glueWaterAdditive" },
        { 26, "wastePaper" },
        { 27, "bindingSupply" },
        { 28, "bandingSupply" },
        { 29, "stitchingWire" },
        { 30, "shrinkWrap" },
        { 31, "paperWrap" },
        { 32, "staples" },
        { 33, "inserts" },
        { 34, "covers" },
        { 0, NULL }
};

const int_str_t PrtMarkerSuppliesSupplyUnitTC[] = {
        { (1),  "other" },
        { (2),  "unknown" },
        { (3),  "tenThousandthsOfInches" },
        { (4),  "micrometers" },
        { (7),  "impressions" },
        { (8),  "sheets" },
        { (11),  "hours" },
        { (12),  "thousandthsOfOunces" },
        { (13),  "tenthsOfGrams" },
        { (14),  "hundrethsOfFluidOunces" },
        { (15),  "tenthsOfMilliliters" },
        { (16),  "feet" },
        { (17),  "meters" },
        { (18),  "items" },
        { (19),  "percent" },
        { 0, NULL }
};

#define ELM_ST_DEV "hrDeviceEntry"
const int_str_t hrDeviceStatus[] = {
        { (1), "unknown" },
        { (2), "running" },
        { (3), "warning" },
        { (4), "testing" },
        { (5), "down" },
        { 0, NULL }
};

#define ELM_ST_PRT "hrPrinterEntry"
const int_str_t hrPrinterStatus[] = {
        { (1), "other" },
        { (2), "unknown" },
        { (3), "idle" },
        { (4), "printing" },
        { (5), "warmup" },
        { 0, NULL }
};

#define ELM_ER_PRT "hrPrinterEntry"
const int_str_t hrPrinterDetectedErrorState[] = {
        { (1 << 0),  "lowPaper" },
        { (1 << 1),  "noPaper" },
        { (1 << 2),  "lowToner" },
        { (1 << 3),  "noToner" },
        { (1 << 4),  "doorOpen" },
        { (1 << 5),  "jammed" },
        { (1 << 6),  "offline" },
        { (1 << 7),  "serviceRequested" },
        { (1 << 8),  "inputTrayMissing" },
        { (1 << 9),  "outputTrayMissing" },
        { (1 << 10),  "markerSupplyMissing" },
        { (1 << 11),  "outputNearFull" },
        { (1 << 12),  "outputFull" },
        { (1 << 13),  "inputTrayEmpty" },
        { (1 << 14),  "overduePreventMaint" },
        { 0, NULL }
};

#endif

/* End of file */
