#ifndef MERGER_H
#define MERGER_H

#include "SyncList.h"

template <class T> class Merger {
 private:
  SyncList<T> **current;
  int sem_id;
  int refcount_sem_id;
  int *refcount;
  int created_by_new;
  void init(SyncList<T>*, int);
 public:
  static void* operator new(size_t size);
  static void operator delete(void* obj);
  Merger(SyncList<T>*);
  Merger(SyncList<T>*,int);
  ~Merger();
  void free();
  void release();
  void put(T);
  void timedput(T, struct timeval*,pards_status*);
  void intrput(T,pards_status*);
  void increase_referer(){increase_referer(1);}
  void increase_referer(int i);
};

template <class T> void* Merger<T>::operator new(size_t size)
{
  void* ptr = pards_shmalloc(size);

#ifdef PARDS_USE_EXCEPTION  
  if(ptr == 0) throw MemoryException();
  else 
#endif
    return ptr;
}

template <class T> void Merger<T>::operator delete(void* obj)
{
  if(*(((Merger<T>*)obj)->refcount) == -1)
    pards_shmfree(obj);
  else
    pards_error("Merger: deleted when reference count is set. Use release() instead.",CRITICAL);
  return;
}

template <class T> Merger<T>::Merger(SyncList<T>* c)
{
  init(c, -1);
}

template <class T> Merger<T>::Merger(SyncList<T>* c,int rc)
{
  init(c, rc);
}

template <class T> void Merger<T>::init(SyncList<T>* c,int rc)
{
  current = (SyncList<T>**)pards_shmalloc(sizeof(SyncList<T>*));
  *current = c;
  sem_id = pards_semget();
  created_by_new = pards_is_in_shm((char*)this);
  refcount_sem_id = pards_semget();
  refcount = (int*)pards_shmalloc(sizeof(int));
  *refcount = rc;
}

template <class T> void Merger<T>::release()
{
  if(*refcount != -1){
    pards_lock(refcount_sem_id);
    if(*refcount > 0){
      (*refcount)--;
      if((*refcount) == 0) {
        pards_unlock(refcount_sem_id);
        free();
        if(created_by_new) pards_shmfree(this);
        return;
      } 
    } else {
      pards_error("Merger released more than the initial reference count.",CRITICAL);
    }
    pards_unlock(refcount_sem_id);
  }
}

template <class T> Merger<T>::~Merger()
{
  if(created_by_new && *refcount == -1) free();
}

template <class T> void Merger<T>::free()
{
  if(sem_id != -1){
    if(*refcount != -1){
      pards_semfree(refcount_sem_id);
      *refcount = -1;
      refcount_sem_id = -1;
    }
    pards_semfree(sem_id);
    sem_id = -1;
    pards_shmfree(refcount);
    pards_shmfree(current);
  }
  (*current)->writecdr(0);
}

template <class T> void Merger<T>::put(T val)
{
  pards_lock(sem_id);
  
  if((*current)->is_written()){
    *current = (*current)->create();
    (*current)->write(val);
  } else { // first time
    (*current)->write(val);
  }

  pards_unlock(sem_id);
}


template <class T> void Merger<T>::timedput(T val, struct timeval* tv,
					    pards_status* st)
{
  pards_lock(sem_id);
  
  if((*current)->is_written()){
    SyncQueue<T>* queue_current;
#ifdef PARDS_USE_DYNAMIC_CAST
    queue_current = dynamic_cast<SyncQueue<T>*>(*current);
    if(queue_current == 0){
      pards_error("Merger::timedput/intrput is called for not SyncQueue merger",CRITICAL);
      *st = SUCCESS;
      *current = (*current)->create();
      (*current)->write(val);
      pards_unlock(sem_id);      
      return;
    }
#else
    queue_current = (SyncQueue<T>*)(*current);
#endif
    *current = queue_current->timedcreate(tv,st);// it's ok even if not success
    if(*st == SUCCESS)
      (*current)->write(val);
  } else { // first time
    *st = SUCCESS;
    (*current)->write(val);
  }

  pards_unlock(sem_id);
}

template <class T> void Merger<T>::intrput(T val, pards_status* st)
{
  struct timeval tv = {0,0};
  timedput(val,&tv,st);
}

template <class T> void Merger<T>::increase_referer(int i)
{
  pards_lock(refcount_sem_id);
  (*refcount) += i;
  pards_unlock(refcount_sem_id);
}

#endif
