#ifndef WORKPOOL_H
#define WORKPOOL_H

#include "SyncList.h"

template <class T1, class T2> class WorkItem{
 public:
  T1 work;
  SyncList<T2>* output;
};

template <class T1, class T2> class WorkPool {
 private:
  SyncList<T1>** work;
  SyncList<T2>** output;
  int sem_id;
  int refcount_sem_id;
  int *refcount;
  int created_by_new;
  void init(SyncList<T1>*, SyncList<T2>*, int);
 public:
  static void* operator new(size_t size);
  static void operator delete(void* obj);
  WorkPool(SyncList<T1>*, SyncList<T2>*);
  WorkPool(SyncList<T1>*, SyncList<T2>*, int);
  ~WorkPool();
  void free();
  void release();
  WorkItem<T1,T2> getwork();
};

template <class T1, class T2> void* WorkPool<T1,T2>::operator new(size_t size)
{
  void* ptr = pards_shmalloc(size);

#ifdef PARDS_USE_EXCEPTION  
  if(ptr == 0) throw MemoryException();
  else 
#endif
    return ptr;
}

template <class T1, class T2> void WorkPool<T1,T2>::operator delete(void* obj)
{
  if(*(((WorkPool<T1,T2>*)obj)->refcount) == -1)
    pards_shmfree(obj);
  else
    pards_error("WorkPool: deleted when reference count is set. Use release() instead.",CRITICAL);
  return;
}

template <class T1, class T2> WorkPool<T1,T2>::WorkPool(SyncList<T1>* w,
							SyncList<T2>* o)
{
  init(w, o, -1);
}

template <class T1, class T2> WorkPool<T1,T2>::WorkPool(SyncList<T1>* w,
							SyncList<T2>* o,
							int rc)
{
  init(w, o, rc);
}

template <class T1, class T2> void WorkPool<T1,T2>::init(SyncList<T1>* w,
							 SyncList<T2>* o,
							 int rc)
{
  work = (SyncList<T1>**)pards_shmalloc(sizeof(SyncList<T1>*));
  *work = w;
  output = (SyncList<T2>**)pards_shmalloc(sizeof(SyncList<T2>*));
  *output = o;
  sem_id = pards_semget();
  created_by_new = pards_is_in_shm((char*)this);
  refcount_sem_id = pards_semget();
  refcount = (int*)pards_shmalloc(sizeof(int));
  *refcount = rc;
}

template <class T1, class T2> void WorkPool<T1,T2>::release()
{
  if(*refcount != -1){
    pards_lock(refcount_sem_id);
    if(*refcount > 0){
      (*refcount)--;
      if((*refcount) == 0) {
	pards_unlock(refcount_sem_id);
	free();
	if(created_by_new) pards_shmfree(this);
	return;
      } 
    } else {
      pards_error("Merger released more than the initial reference count.",CRITICAL);
    }
    pards_unlock(refcount_sem_id);
  }
}

template <class T1, class T2> WorkPool<T1,T2>::~WorkPool()
{
  if(created_by_new && *refcount == -1) free();
}

template <class T1, class T2> void WorkPool<T1,T2>::free()
{
  if(sem_id != -1){
    if(*refcount != -1){
      pards_semfree(refcount_sem_id);
      *refcount = -1;
      refcount_sem_id = -1;
    }
    pards_semfree(sem_id);
    sem_id = -1;
    pards_shmfree(refcount);
    pards_shmfree(work);
    pards_shmfree(output);
  }
}

template <class T1, class T2> WorkItem<T1,T2> WorkPool<T1,T2>::getwork()
{
  pards_lock(sem_id);

  WorkItem<T1,T2> workItem;
  if(*work == 0) {
    workItem.output = 0; // workItem.work is "don't care"
  } else {
    workItem.work = (*work)->read();
    workItem.output = *output;
    *work = (*work)->release();
    if(*work == 0)
      (*output)->writecdr(0);
    else
      *output = (*output)->create();
  }

  pards_unlock(sem_id);

  return workItem;
}

#endif
