#ifndef SYNC_H
#define SYNC_H

#include "libpards.h"

#ifdef PARDS_USE_EXCEPTION
#include "PardsException.h"
#endif

template <class T> class Sync {
 private:
  int read_sem_id;
  int write_sem_id;
  int *is_written_val; // 0: value is not written, 1: written
  T *value;
  void free_sem();
  void free_shm();
 protected:
  int created_by_new;  
 public:
  static void* operator new(size_t size);
  static void operator delete(void* obj);
  Sync();
  virtual ~Sync();
  T read();
  T* readptr();
  T intrread(pards_status*);
  T* intrreadptr(pards_status*);
  T timedread(struct timeval *, pards_status*);
  T* timedreadptr(struct timeval *, pards_status*);
  int is_written();
  void write(T val);
  virtual void free();
};

template <class T> void* Sync<T>::operator new(size_t size)
{
  void* ptr = pards_shmalloc(size);

#ifdef PARDS_USE_EXCEPTION  
  if(ptr == 0) throw MemoryException();
  else 
#endif
    return ptr;
}

template <class T> void Sync<T>::operator delete(void* obj)
{
  pards_shmfree(obj);
  return;
}

template <class T> Sync<T>::~Sync()
{
  if(created_by_new) free();
}

template <class T> Sync<T>::Sync()
{
  created_by_new = pards_is_in_shm((char*)this);
  read_sem_id = -1;
  write_sem_id = -1;
  is_written_val = 0;
  value = 0;
  read_sem_id = pards_semget();
  if(read_sem_id == -1){
#ifdef PARDS_USE_EXCEPTION
    throw SemaphoreException();
#endif
    return;
  }
  write_sem_id = pards_semget();
  if(write_sem_id == -1){
    pards_semfree(read_sem_id);
    read_sem_id = -1; // avoid multiple free at delete 
#ifdef PARDS_USE_EXCEPTION
    throw SemaphoreException();
#endif
    return;
  }

  is_written_val = (int*)pards_shmalloc(sizeof(int));
  if(is_written_val == 0){
    pards_semfree(read_sem_id);
    pards_semfree(write_sem_id);
    read_sem_id = -1; // avoid multiple free at delete 
    write_sem_id = -1; 
#ifdef PARDS_USE_EXCEPTION  
    throw MemoryException();
#endif
    return;
  }
  *is_written_val = 0;

  value = (T*)pards_shmalloc(sizeof(T));
  if(value == 0){
    pards_semfree(read_sem_id);
    pards_semfree(write_sem_id);
    pards_shmfree(is_written_val);
    read_sem_id = -1; // avoid multiple free at delete 
    write_sem_id = -1;
    is_written_val = 0;
#ifdef PARDS_USE_EXCEPTION  
    throw MemoryException();
#endif
    return;
  }

  pards_lock(read_sem_id);
}

template <class T> T Sync<T>::read()
{
#if 0  // deletion after read might overtake writer's unlock
  if(*is_written_val){
#ifndef NO_EXTEND_SHM
    attach_shm();
#endif
    return *value; // already set
  } else
#endif
    pards_lock(read_sem_id); // wait for write

  pards_unlock(read_sem_id); // for other waiting processes

#ifndef NO_EXTEND_SHM
  attach_shm();
#endif
  return *value;  
}

template <class T> T* Sync<T>::readptr() // almost same as read()
{
#if 0  // deletion after read might overtake writer's unlock
  if(*is_written_val){
#ifndef NO_EXTEND_SHM
    attach_shm();
#endif
    return value; // already set
  }
  else
#endif
    pards_lock(read_sem_id); // wait for write

  pards_unlock(read_sem_id); // for other waiting processes
  
#ifndef NO_EXTEND_SHM
  attach_shm();
#endif

  return value;  
}

template <class T> T Sync<T>::intrread(pards_status* st)
{
  struct timeval tv = {0,0}; // does not time out
  return timedread(&tv, st);
}

template <class T> T Sync<T>::timedread(struct timeval *tv,
					pards_status* is_timedout)
{
  *is_timedout = pards_timedlock(read_sem_id, tv); // wait for write
  if(*is_timedout){
#ifndef NO_EXTEND_SHM
    attach_shm();
#endif
    return *value; // contains nothing; to avoid compiler warning
  } else {
    pards_unlock(read_sem_id); // for other waiting processes
#ifndef NO_EXTEND_SHM
    attach_shm();
#endif
    return *value;  
  }
}

template <class T> T* Sync<T>::intrreadptr(pards_status* st)
{
  struct timeval tv = {0,0}; // does not time out
  return timedreadptr(&tv, st);
}

template <class T> T* Sync<T>::timedreadptr(struct timeval *tv,
					    pards_status* is_timedout)
{
  *is_timedout = pards_timedlock(read_sem_id, tv); // wait for write
  if(*is_timedout){
#ifndef NO_EXTEND_SHM
    attach_shm();
#endif
    return value; // contains nothing; to avoid compiler warning
  } else {
    pards_unlock(read_sem_id); // for other waiting processes
#ifndef NO_EXTEND_SHM
    attach_shm();
#endif
    return value;  
  }
}

template <class T> void Sync<T>::write(T val)
{
  pards_lock(write_sem_id); // lock for changing *is_written_val
  
  if(*is_written_val){
    pards_error("Sync::write: Value is already written",INFO);
    pards_unlock(write_sem_id);
    return;
  } else {
    *value = val; // This is "copy"
    *is_written_val = 1;
    pards_unlock(write_sem_id);
  }

  pards_unlock(read_sem_id); // wake up readers
}

template <class T> void Sync<T>::free_sem()
{
  if(read_sem_id != -1) {pards_semfree(read_sem_id); read_sem_id = -1;}
  if(write_sem_id != -1) {pards_semfree(write_sem_id); write_sem_id = -1;}
}

template <class T> void Sync<T>::free_shm()
{
  if(is_written_val) {pards_shmfree(is_written_val); is_written_val = 0;}
  if(value) {pards_shmfree(value); value = 0;}
}

template <class T> void Sync<T>::free()
{
  free_sem();
  free_shm();
}

template <class T> int Sync<T>::is_written()
{
  pards_lock(write_sem_id); // need lock to get fresh value
  int val = *is_written_val;
  pards_unlock(write_sem_id);
  return val;
}
#endif
