#ifndef SYNCLIST_H
#define SYNCLIST_H

#include "Sync.h"

template <class T> class SyncList : public Sync<T> {
 private:
  int readcdr_sem_id;
  int writecdr_sem_id;
  void init();
  void free_sem();
  void free_shm();
 protected:
  int *is_cdr_written_val; // 0: not written, 1: written
  SyncList<T> **cdr;
 public:
  SyncList();
  SyncList(SyncList<T>*);
  int is_cdr_written();
  virtual ~SyncList();
  virtual SyncList<T> *readcdr();
  virtual SyncList<T> *intrreadcdr(pards_status*);
  virtual SyncList<T> *timedreadcdr(struct timeval*, pards_status*);
  virtual void writecdr(SyncList<T>* cdr);
  virtual void free();
  virtual SyncList<T> *release();
  virtual SyncList<T> *intrrelease(pards_status*);
  virtual SyncList<T> *timedrelease(struct timeval*, pards_status*);
  virtual SyncList<T> *create();
};

template <class T> SyncList<T>::~SyncList()
{
  if(Sync<T>::created_by_new){
    free_sem();
    free_shm();
  }
}

template <class T> void SyncList<T>::init()
{
  readcdr_sem_id = -1;
  writecdr_sem_id = -1;
  is_cdr_written_val = 0;
  cdr = 0;
  readcdr_sem_id = pards_semget();
  if(readcdr_sem_id == -1){
#ifdef PARDS_USE_EXCEPTION
    throw SemaphoreException();
#endif
    return;
  }
  writecdr_sem_id = pards_semget();
  if(writecdr_sem_id == -1){
    pards_semfree(readcdr_sem_id);
    readcdr_sem_id = -1; // avoid multiple free at delete 
#ifdef PARDS_USE_EXCEPTION
    throw SemaphoreException();
#endif
    return;
  }

  is_cdr_written_val = (int*)pards_shmalloc(sizeof(int));
  if(is_cdr_written_val == 0){
    pards_semfree(readcdr_sem_id);
    pards_semfree(writecdr_sem_id);
    readcdr_sem_id = -1; // avoid multiple free at delete 
    writecdr_sem_id = -1;
#ifdef PARDS_USE_EXCEPTION
    throw MemoryException();
#endif
  }
  *is_cdr_written_val = 0;

  cdr = (SyncList<T>**)pards_shmalloc(sizeof(SyncList<T>*));
  if(cdr == 0){
    pards_semfree(readcdr_sem_id);
    pards_semfree(writecdr_sem_id);
    pards_shmfree(is_cdr_written_val);
    readcdr_sem_id = -1; // avoid multiple free at delete
    writecdr_sem_id = -1;
    is_cdr_written_val = 0;
#ifdef PARDS_USE_EXCEPTION
    throw MemoryException();
#endif
  }

  pards_lock(readcdr_sem_id);
}

template <class T> SyncList<T>::SyncList()
{
  init();
}

template <class T> SyncList<T>::SyncList(SyncList<T>* prev)
{
  init();
  prev->writecdr(this);
}

template <class T> SyncList<T>* SyncList<T>::readcdr()
{
#if 0  // deletion after read might overtake writer's unlock
  if(*is_cdr_written_val){
#ifndef NO_EXTEND_SHM
    attach_shm();
#endif
    return *cdr; // already set
  } else 
#endif
    pards_lock(readcdr_sem_id); // wait for write

  pards_unlock(readcdr_sem_id); // for other waiting processes

#ifndef NO_EXTEND_SHM
  attach_shm();
#endif

  return *cdr;
}

template <class T> SyncList<T>* SyncList<T>::intrreadcdr(pards_status* st)
{
  struct timeval tv = {0,0};
  return timedreadcdr(&tv, st);
}

template <class T> SyncList<T>* SyncList<T>::timedreadcdr(struct timeval* tv,
							  pards_status* is_timedout)
{
  *is_timedout = pards_timedlock(readcdr_sem_id,tv); // wait for write

  if(*is_timedout){
#ifndef NO_EXTEND_SHM
    attach_shm();
#endif
    return *cdr; // contains nothing; 
  } else {
    pards_unlock(readcdr_sem_id); // for other waiting processes
#ifndef NO_EXTEND_SHM
    attach_shm();
#endif
    return *cdr;
  }
}

template <class T> void SyncList<T>::writecdr(SyncList<T>* c)
{
  pards_lock(writecdr_sem_id); // lock for *is_cdr_written_val
  
  if(*is_cdr_written_val){
    pards_error("SyncList::writecdr: cdr is already written",INFO);
    pards_unlock(writecdr_sem_id);
    return;
  } else {
    *cdr = c;
    *is_cdr_written_val = 1;
    pards_unlock(writecdr_sem_id);
  }

  pards_unlock(readcdr_sem_id); // wake up readers
}

template <class T> void SyncList<T>::free_sem()
{
  if(readcdr_sem_id != -1){
    pards_semfree(readcdr_sem_id);
    readcdr_sem_id = -1;
  }
  if(writecdr_sem_id != -1){
    pards_semfree(writecdr_sem_id);
    writecdr_sem_id = -1;
  }
}

template <class T> void SyncList<T>::free_shm()
{
  if(is_cdr_written_val)
    {pards_shmfree(is_cdr_written_val); is_cdr_written_val = 0;}
  if(cdr) {pards_shmfree(cdr); cdr = 0;}
}

template <class T> void SyncList<T>::free()
{
  free_sem();
  free_shm();
  Sync<T>::free();
}

template <class T> SyncList<T>* SyncList<T>::release()
{
  SyncList<T>* crntcdr;
  crntcdr = readcdr();
  delete this;
  return crntcdr;
}

template <class T> SyncList<T>* SyncList<T>::intrrelease(pards_status *st)
{
  struct timeval tv = {0,0};
  return timedrelease(&tv, st);
}

template <class T> SyncList<T>* SyncList<T>::timedrelease(struct timeval *tv,
							  pards_status *is_timedout)
{
  SyncList<T>* crntcdr;
  crntcdr = timedreadcdr(tv, is_timedout);
  if(*is_timedout != SUCCESS) return this;
  else {delete this; return crntcdr;}
}

template <class T> SyncList<T>* SyncList<T>::create()
{
  SyncList<T> *new_cell = new SyncList<T>;
  writecdr(new_cell);
  return new_cell;
}

template <class T> int SyncList<T>::is_cdr_written()
{
  pards_lock(writecdr_sem_id);
  int val = *is_cdr_written_val;
  pards_unlock(writecdr_sem_id);
  return val;
}

#endif
