package jp.coocan.la.aide.android.snapcard;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;

import android.content.Context;
import android.content.SharedPreferences;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.Paint.Style;
import android.net.Uri;
import android.os.Environment;
import android.preference.PreferenceManager;
import android.util.DisplayMetrics;
import android.util.Log;
import android.view.MotionEvent;
import android.view.View;
import android.view.animation.Animation;
import android.view.animation.Animation.AnimationListener;
import android.view.animation.AnimationUtils;
import android.widget.Toast;


public class MainView extends View implements AnimationListener {
	private static final String TAG = "SnapCard_MainView";

	private Paint  paint;

	private Bitmap cardBitmap;

	private float offsetX = 0f;
	private float offsetY = 0f;

	private float dragX = 0f;
	private float dragY = 0f;

	/** 表示している名刺の状態 */
	private int cardStatus = AppConst.CardStatus.TAPABLE;

	/** 名刺の幅（ピクセル） */
	private float cardPixelWidth  = 0f;
	/** 名刺の高さ（ピクセル） */
	private float cardPixelHeight = 0f;
	/** 名刺の幅／高さの比（ピクセル） */
	private float cardPixelXYAvg = 0f;
	/** 名刺の対角線（ピクセル） */
	private float cardPixelDiagonal = 0f;

	/** SnapOutを開始するドラッグ距離（ピクセル）*/
	private float snapOutStartDistance = 0f;
	/** SnapOut時のフレームあたりの移動距離（ピクセル）*/
	private float snapOutSteps = 0f;
	/** SnapOutの方向 */
	private int snapOutDirection = AppConst.SnapOut.TO_RIGHT;

	SnapinView snapinView = null;
	private Animation snapinAnimation = null;


	public MainView(Context context, SnapinView snapinView) {
		super(context);
		this.snapinView = snapinView;

		paint = new Paint();
		paint.setAntiAlias( true );
		paint.setColor( Color.BLACK );
		paint.setStyle( Style.FILL );

		//画面DPIから画面上の名刺のピクセルサイズを算出
		DisplayMetrics displayMetrics = AppContainer.getDisplayMetrics();
		cardPixelWidth  = (float)Math.floor( displayMetrics.xdpi * displayMetrics.scaledDensity * AppConst.Scale.CARD_INCH_X );
		cardPixelHeight = (float)Math.floor( displayMetrics.ydpi * displayMetrics.scaledDensity * AppConst.Scale.CARD_INCH_Y );
		cardPixelXYAvg  = cardPixelHeight / cardPixelWidth;
		cardPixelDiagonal = (float)Math.sqrt( cardPixelWidth * cardPixelWidth + cardPixelHeight * cardPixelHeight );
		Log.v( TAG, "cardPixelWidth     = " + cardPixelWidth );
		Log.v( TAG, "cardPixelHeight    = " + cardPixelHeight );
		Log.v( TAG, "cardPixelXYAvg     = " + cardPixelXYAvg );
		Log.v( TAG, "cardPixcelDiagonal = " + cardPixelDiagonal );

		//SnapOut属性を算出
		snapOutStartDistance = cardPixelDiagonal * AppConst.SnapOut.STARTAVG;
		snapOutSteps = cardPixelDiagonal * AppConst.SnapOut.STEPAVG;
		Log.v( TAG, "snapOutStartDistance = " + snapOutStartDistance );
		Log.v( TAG, "snapOutSteps         = " + snapOutSteps );

		//名刺画像をロード
		String dftFilename  = Environment.getExternalStorageDirectory() + "/SnapCard/card.jpg";
		String dftUriString = Uri.fromFile( new File( dftFilename ) ).toString();
		SharedPreferences pref = PreferenceManager.getDefaultSharedPreferences(context);
		String uriString = pref.getString( AppConst.Pref.CARDIMAGE_URI, dftUriString );
		Uri uri = Uri.parse( uriString );
		loadCardBitmap( uri );

		//SnapInアニメーションのロード
		snapinAnimation  = AnimationUtils.loadAnimation( context, R.anim.snapin  );
		snapinAnimation.setAnimationListener( this );
	}

	public void loadCardBitmap( Uri uri ){

		Bitmap src = null;

		//現在の名刺画像を除去
		if( cardBitmap != null ){
			cardBitmap.recycle();
			cardBitmap = null;
		}

		//URIから元となる名刺画像を取得
		if( uri != null ){
			try{
				BitmapFactory.Options opts = new BitmapFactory.Options();
				InputStream is;

				opts.inJustDecodeBounds = true;
				is = AppContainer.getContentResolver().openInputStream( uri );
				try{
					BitmapFactory.decodeStream( is, null, opts );
				}finally{
					is.close();
				}

				//名刺のピクセルサイズからとBitmapサイズからスケールを計算
				int scaleX = (int)Math.floor( (double)opts.outWidth  / (double)cardPixelWidth );
				int scaleY = (int)Math.floor( (double)opts.outHeight / (double)cardPixelHeight );

				opts.inJustDecodeBounds = false;
				opts.inSampleSize = Math.min( scaleX, scaleY );

				is = AppContainer.getContentResolver().openInputStream( uri );
				try{
					src = BitmapFactory.decodeStream( is, null, opts );
				}finally{
					is.close();
				}

			}catch(IOException e){
				src = null;
			}
		}

		//URIから取得できない場合はリソースから取得
		if( src == null ){
			//取得できない旨を表示
			Toast.makeText( getContext(), R.string.message_invarid_image, Toast.LENGTH_LONG ).show();

			//リソースから画像を取得
			src = BitmapFactory.decodeResource(
					getResources(), 
					R.drawable.dftcard
					);
		}

		//取得した画像を名刺の実サイズに拡大/縮小画像を作成
		float srcWidth  = (float)src.getWidth();
		float srcHeight = (float)src.getHeight();
		float srcXYAvg  = srcHeight / srcWidth;
		float scale = 0f;
		if( cardPixelXYAvg <= srcXYAvg ){
			scale = cardPixelWidth  / srcWidth;
		} else {
			scale = cardPixelHeight / srcHeight;
		}
		Log.v( TAG, "srcWidth  = " + srcWidth );
		Log.v( TAG, "srcHeight = " + srcHeight );
		Log.v( TAG, "srcXYAvg  = " + srcXYAvg );
		Log.v( TAG, "scale     = " + scale );

		Bitmap scaled = Bitmap.createScaledBitmap(
				src, 
				(int)Math.ceil(srcWidth  * scale),
				(int)Math.ceil(srcHeight * scale),
				true
				);
		Log.v( TAG, "scaled.getWidth()  = " + scaled.getWidth() );
		Log.v( TAG, "scaled.getHeight() = " + scaled.getHeight() );

		src.recycle();
		src = null;

		//拡大/縮小済み画像の中央部分を切り出して最終名刺画像を作成
		if( cardPixelXYAvg <= srcXYAvg ){
			cardBitmap = Bitmap.createBitmap( 
					scaled, 
					0, 
					(int)((scaled.getHeight() - cardPixelHeight) / 2), 
					(int)cardPixelWidth, 
					(int)cardPixelHeight
					);
		} else {
			cardBitmap = Bitmap.createBitmap( 
					scaled, 
					(int)((scaled.getWidth() - cardPixelWidth) / 2), 
					0, 
					(int)cardPixelWidth, 
					(int)cardPixelHeight
					);
		}

		scaled.recycle();
		scaled = null;

		snapinView.setCardBitmap( cardBitmap );

		resetPosition(); 
		cardStatus = AppConst.CardStatus.TAPABLE;
		invalidate();
	}

	@Override
	protected void onDraw(Canvas canvas) {

		boolean isDrawCard = false;
		float x = 0f;
		float y = 0f;

		switch( cardStatus ){

		case AppConst.CardStatus.TAPABLE : {
			x = (canvas.getWidth()  - cardPixelWidth ) / 2 + (dragX - offsetX);
			y = (canvas.getHeight() - cardPixelHeight) / 2 + (dragY - offsetY);
			isDrawCard = true;
			break;
		}

		case AppConst.CardStatus.SNAPOUT : {
			//名刺画像をSnapOut方向に移動
			switch( snapOutDirection ){
			case AppConst.SnapOut.TO_RIGHT : dragX += snapOutSteps; break;
			case AppConst.SnapOut.TO_LEFT  : dragX -= snapOutSteps; break;
			case AppConst.SnapOut.TO_UP    : dragY -= snapOutSteps; break;
			case AppConst.SnapOut.TO_DOWN  : dragY += snapOutSteps; break;
			}
			x = (canvas.getWidth()  - cardPixelWidth ) / 2 + (dragX - offsetX);
			y = (canvas.getHeight() - cardPixelHeight) / 2 + (dragY - offsetY);

			//移動した名刺画像が画面外だったら状態をINVISIBLEに移行
			if( (x > canvas.getWidth())||(x < - cardPixelWidth)||
					(y > canvas.getHeight())||(y < - cardPixelHeight) ){

				cardStatus = AppConst.CardStatus.INVISIBLE; 
			}

			isDrawCard = true;

			invalidate();
			break;
		}

		case AppConst.CardStatus.SNAPIN :
		case AppConst.CardStatus.INVISIBLE : {
			isDrawCard = false;
		}

		}

		//名刺画像描画
		if( isDrawCard ){
			canvas.drawColor ( Color.BLACK );
			canvas.drawBitmap( cardBitmap, x, y, paint );
		}
	}

	@Override 
	public boolean onTouchEvent(MotionEvent event) {

		switch( event.getAction() ){

		//触る
		case MotionEvent.ACTION_DOWN : {
			offsetX = dragX = event.getX();
			offsetY = dragY = event.getY();
			break;
		}

		//触ったままスライド
		case MotionEvent.ACTION_MOVE : {
			dragX = event.getX();
			dragY = event.getY();
			break;
		}

		//離す
		case MotionEvent.ACTION_UP :
		case MotionEvent.ACTION_CANCEL : {
			//タップ位置からの移動距離を算出
			float dragDistance = (float)Math.sqrt( Math.pow( dragX - offsetX, 2 ) + Math.pow( dragY - offsetY, 2) );

			if( dragDistance >= snapOutStartDistance ){
				startSnapOut();
			}
			else{
				resetPosition();
			}

			break;
		}

		}    	

		// 再描画の指示
		invalidate();

		return true;
	}


	/**
	 * 名刺表示位置のリセット
	 */
	private void resetPosition(){
		offsetX = dragX = 0;
		offsetY = dragY = 0;
	}

	/**
	 * 名刺のSnapOut（画面外へ飛び出す動作）の開始
	 */
	public void startSnapOut(){
		if( cardStatus != AppConst.CardStatus.TAPABLE )  return;
		cardStatus = AppConst.CardStatus.SNAPOUT;

		//移動方向の決定
		float dx = dragX - offsetX;
		float dy = dragY - offsetY;
		if( Math.abs( dx ) >= Math.abs( dy ) ){
			if( dx >= 0 ){
				snapOutDirection = AppConst.SnapOut.TO_RIGHT;
				snapOutSteps = cardPixelWidth * AppConst.SnapOut.STEPAVG;
			} else {
				snapOutDirection = AppConst.SnapOut.TO_LEFT;
			}
		} else {
			if( dy >= 0 ){
				snapOutDirection = AppConst.SnapOut.TO_DOWN;
			} else {
				snapOutDirection = AppConst.SnapOut.TO_UP;
			}
		}


		snapOutSteps = cardPixelWidth * AppConst.SnapOut.STEPAVG;
		invalidate();
	}

	/**
	 * 名刺のSnapIn（画面外から飛び込んで来る動作）の開始
	 */
	public void startSnapIn(){
		if( cardStatus != AppConst.CardStatus.INVISIBLE )  return;
		cardStatus = AppConst.CardStatus.SNAPIN;

		snapinView.startAnimation( snapinAnimation );
		invalidate();
	}


	public void onAnimationStart(Animation animation) {
		this.setVisibility( INVISIBLE );
		snapinView.setVisibility( VISIBLE );
	}

	public void onAnimationEnd(Animation animation) {
		if( animation.equals( snapinAnimation ) ){
			this.setVisibility( VISIBLE );
			snapinView.setVisibility( INVISIBLE );

			resetPosition(); 
			cardStatus = AppConst.CardStatus.TAPABLE;
			invalidate();
		}

	}

	public void onAnimationRepeat(Animation animation) {}

}
