/********************************************************************************************************
		* PROGRAM      : QSerialPortTerminal
		* DATE - TIME  : vendredi 03 octobre 2008 - 11h15
		* AUTHOR       : VIANNEY-LIAUD Philippe ( philippe.vianney.liaud gmail.com )
			* FILENAME     : ManageSerialPort.cpp
			* LICENSE      : GPL
			* COMMENTARY   :
********************************************************************************************************/
#include <QtDebug>
#include "ManageSerialPort.h"

/********************************************************************************************************
		* Classe ManageSerialPort
		*****************************************************************************************************/

//Constructeur
ManageSerialPort::ManageSerialPort()
{
	//Init pointeur a NULL
	threadSend = NULL;
	threadReceive = NULL;
	
	//Init des bool
	sendingEnabled = false;
	receivingEnabled = false;
	closeCalled = false;
	saveStateSendEnabled = false;
	saveStateReceivedEnabled = false;
	saveStateReceiveData = false;
}

ManageSerialPort::ManageSerialPort(const QString &name, const BaudRateType baudRate, \
		const DataBitsType dataBits, const ParityType parity, \
		const StopBitsType stopBits, const FlowType flowControl, \
		ulong seconds, ulong milliseconds)
{
	//Init pointeur a NULL
	threadSend = NULL;
	threadReceive = NULL;
	
	//Init des bool
	sendingEnabled = false;
	receivingEnabled = false;
	closeCalled = false;
	saveStateSendEnabled = false;
	saveStateReceivedEnabled = false;
	saveStateReceiveData = false;
	
	setPort(name);
	setBaudRate(baudRate);
	setDataBits(dataBits);
	setParity(parity);
	setStopBits(stopBits);
	setFlowControl(flowControl);
	setTimeout(seconds, milliseconds);
}

//Destructeur
ManageSerialPort::~ManageSerialPort()
{
	if (threadSend != NULL)
	{
		delete threadSend;
		threadSend = NULL;
	}
	
	if (threadReceive != NULL)
	{
		delete threadReceive;
		threadReceive = NULL;
	}
	
	if (isOpen())
		extSerialPort.close();
}

bool ManageSerialPort::open()
{
	bool res = extSerialPort.open(QIODevice::ReadWrite);
	
	if (closeCalled)
	{
		if (saveStateSendEnabled)
			enableSending();
		
		if (saveStateReceivedEnabled)
			enableReceiving();
		
		if (saveStateReceiveData)
			receiveData();
		closeCalled = false;
	}
	
	return res;
}

bool ManageSerialPort::open(const QString &name, const BaudRateType baudRate, \
		const DataBitsType dataBits,const ParityType parity, \
		const StopBitsType stopBits, const FlowType flowControl, \
		ulong seconds, ulong milliseconds)
{
	setPort(name);
	setBaudRate(baudRate);
	setDataBits(dataBits);
	setParity(parity);
	setStopBits(stopBits);
	setFlowControl(flowControl);
	setTimeout(seconds, milliseconds);
	
	bool res = extSerialPort.open(QIODevice::ReadWrite);
	
	
	return res;
}

bool ManageSerialPort::isOpen()
{
	return extSerialPort.isOpen();
}

void ManageSerialPort::close()
{
	closeCalled = true;
	saveStateSendEnabled = isSendingEnabled();
	saveStateReceivedEnabled = isReceivingEnabled();
	
	disableSending();
	disableReceiving();
	
	extSerialPort.close();
}

void ManageSerialPort::setPort(const QString &name)
{
	extSerialPort.setPortName(name);
}

QString ManageSerialPort::getPort()
{
	return extSerialPort.portName();
}


void ManageSerialPort::setBaudRate(const BaudRateType baudRate)
{
	extSerialPort.setBaudRate(baudRate);
}

QString ManageSerialPort::getBaudRate()
{
	switch (extSerialPort.baudRate())
	{
		case BAUD50:
			return QString("50");
		case BAUD75:
			return QString("75");
		case BAUD110:
			return QString("110");
		case BAUD134:
			return QString("134");
		case BAUD150:
			return QString("150");
		case BAUD200:
			return QString("200");
		case BAUD300:
			return QString("300");
		case BAUD600:
			return QString("600");
		case BAUD1200:
			return QString("1200");
		case BAUD1800:
			return QString("1800");
		case BAUD2400:
			return QString("2400");
		case BAUD4800:
			return QString("4800");
		case BAUD9600:
			return QString("9600");
		case BAUD14400:
			return QString("14400");
		case BAUD19200:
			return QString("19200");
		case BAUD38400:
			return QString("38400");
		case BAUD56000:
			return QString("56000");
		case BAUD57600:
			return QString("57600");
		case BAUD76800:
			return QString("76800");
		case BAUD115200:
			return QString("115200");
		case BAUD128000:
			return QString("128000");
		case BAUD256000:
			return QString("256000");
	}
	return 0;
}


void ManageSerialPort::setDataBits(const DataBitsType dataBits)
{
	extSerialPort.setDataBits(dataBits);
}

QChar ManageSerialPort::getDataBits()
{
	switch (extSerialPort.dataBits())
	{
		case DATA_5:
			return QChar('5');
		case DATA_6:
			return QChar('6');
		case DATA_7:
			return QChar('7');
		case DATA_8:
			return QChar('8');
		}
		return 0;
}


void ManageSerialPort::setParity(const ParityType parity)
{
	extSerialPort.setParity(parity);
}

QString ManageSerialPort::getParity()
{
	switch (extSerialPort.parity())
	{
		case PAR_NONE:
			return QString(tr("None"));
		case PAR_ODD:
			return QString(tr("Odd"));
		case PAR_EVEN:
			return QString(tr("Even"));
		case PAR_MARK:
			return QString(tr("Mark"));
		case PAR_SPACE:
			return QString(tr("Space"));
		}
		return 0;
}


void ManageSerialPort::setStopBits(const StopBitsType stopBits)
{
	extSerialPort.setStopBits(stopBits);
}

QString ManageSerialPort::getStopBit()
{
	switch (extSerialPort.stopBits())
	{
		case STOP_1:
			return QString("1");
		case STOP_1_5:
			return QString("1.5");
		case STOP_2:
			return QString("2");
		}
		return 0;
}


void ManageSerialPort::setFlowControl(const FlowType flowControl)
{
	extSerialPort.setFlowControl(flowControl);
}

QString ManageSerialPort::getFlowControl()
{
	switch (extSerialPort.flowControl())
	{
		case FLOW_OFF:
			return QString(tr("None"));
		case FLOW_HARDWARE 	:
			return QString(tr("Hardware"));
		case FLOW_XONXOFF :
			return QString(tr("Xon/Xoff"));
		}
		return 0;
}


void ManageSerialPort::setTimeout(ulong seconds, ulong milliseconds)
{
	extSerialPort.setTimeout(seconds,milliseconds);
}

/*
QString ManageSerialPort::getLastErrorToString()
{
	ulong res = extSerialPort.lastError();
	switch (res)
	{
		case E_NO_ERROR:
			return QString(tr("No Error has occured"));
		case E_INVALID_FD:
			return QString(tr("Invalid file descriptor (port was not opened correctly)"));
		case E_NO_MEMORY:
			return QString(tr("Unable to allocate memory tables (POSIX)"));
		case E_CAUGHT_NON_BLOCKED_SIGNAL:
			return QString(tr("Caught a non-blocked signal (POSIX)"));
		case E_PORT_TIMEOUT:
			return QString(tr("Operation timed out (POSIX)"));
		case E_INVALID_DEVICE:
			return QString(tr("The file opened by the port is not a character device (POSIX)"));
		case E_BREAK_CONDITION:
			return QString(tr("The port detected a break condition"));
		case E_FRAMING_ERROR:
			return QString(tr("The port detected a framing error (usually caused by incorrect baud rate settings)"));
		case E_IO_ERROR:
			return QString(tr("There was an I/O error while communicating with the port"));
		case E_BUFFER_OVERRUN:
			return QString(tr("Character buffer overrun"));
		case E_RECEIVE_OVERFLOW:
			return QString(tr("Receive buffer overflow"));
		case E_RECEIVE_PARITY_ERROR:
			return QString(tr("The port detected a parity error in the received data"));
		case E_TRANSMIT_OVERFLOW:
			return QString(tr("Transmit buffer overflow"));
		case E_READ_FAILED:
			return QString(tr("General read operation failure"));
		case E_WRITE_FAILED:
			return QString(tr("General write operation failure"));
		}
		return 0;
}*/

/*
ulong ManageSerialPort::getLastError()
{
	return extSerialPort.lastError();
}
*/


void ManageSerialPort::enableSending()
{
	if (!sendingEnabled && threadSend == NULL) //Si l'envoi n'est pas active && si threadSend n'est pas alloue
	{
		threadSend = new ThreadSend(extSerialPort);
		sendingEnabled = true;
	}
}

void ManageSerialPort::disableSending()
{
	if (sendingEnabled && threadSend != NULL) //Si l'envoi est active && si threadSend est alloue
	{
		delete (threadSend);
		threadSend = NULL;
		sendingEnabled = false;
	}
}

bool ManageSerialPort::isSendingEnabled()
{
	return sendingEnabled;
}

uchar ManageSerialPort::sendData(QByteArray &dataToSend)
{
	if (!isOpen()) //Si le port n'est pas ouvert
		return 2;
	
	if (!sendingEnabled || threadSend == NULL) //Si l'envoi n'est pas active || si threadSend n'est pas alloue
		return 3;
	
	threadSend->addDataToSend(dataToSend); //Ajout des donnees a envoyer
	return 1;
}

void ManageSerialPort::stopSending()
{
	if (!sendingEnabled || threadSend == NULL) //Si l'envoi n'est pas active || si threadSend n'est pas été alloue
		return;
	
	if (threadSend->isRunning()) //si un envoi est en cour
	{
		threadSend->stopSending(); //on donne l'ordre d'arreter l'envoi
		threadSend->wait(); //on attend l'arret
	}
}



void ManageSerialPort::enableReceiving()
{
	if (!receivingEnabled && threadReceive == NULL) //Si la reception n'est pas active && si threadReceive n'est pas alloue
	{
		threadReceive = new ThreadReceive(extSerialPort);
		connect(threadReceive, SIGNAL(newDataReceived(const QByteArray &)), this, SIGNAL(newDataReceived(const QByteArray &)));
		receivingEnabled = true;
	}
}

void ManageSerialPort::disableReceiving()
{
	if (receivingEnabled && threadReceive != NULL) //Si la reception est pas active && si threadReceive est alloue
	{
		delete (threadReceive);
		threadReceive = NULL;
		receivingEnabled = false;
	}
}

bool ManageSerialPort::isReceivingEnabled()
{
	return receivingEnabled;
}

uchar ManageSerialPort::receiveData()
{
	if (!isOpen()) //Si le port n'est pas ouvert
		return 2;
	if (!receivingEnabled || threadReceive == NULL) //Si la reception n'est pas active || si threadReceive n'est pas été alloue
		return 3;
	
	if (!threadReceive->isRunning())
	{
		saveStateReceiveData = true;
		threadReceive->start(); //Demarrage du thread de reception
	}
	return 1;
}

void ManageSerialPort::stopReceiving()
{
	if (!receivingEnabled || threadReceive == NULL) //Si la reception n'est pas active || si threadReceive n'est pas alloue
		return;
	
	if (threadReceive->isRunning()) //Si le thread de reception est en fonctionnement
	{
		saveStateReceiveData = false;
		threadReceive->stopReceiving(); //on donne l'ordre d'arreter la reception
		threadReceive->wait(); //on attend l'arret
	}
}

// Transfer File by XMODEM protcol
void ManageSerialPort::startXferFile_XMODEM(QString filepath)
{
    // Enable Thread Send&Receive
    this->stopSending();
    this->stopReceiving();

    this->disableReceiving();
    this->disableSending();

    char resp[1024];
    if (flagKOZOS) {
        extSerialPort.write("load\n", 5);
        extSerialPort.readLine(resp, sizeof(resp)); // read echo back, prompt
    }

    thSendXModem = new ThreadSendXModem(extSerialPort);

    connect(thSendXModem, SIGNAL(sendEnded(int)), this, SLOT(slot_sendEnded(int)));
    connect(thSendXModem, SIGNAL(progress_status(int)), this, SIGNAL(progress_status(int)));

    thSendXModem->setSendFilePath(filepath);

    thSendXModem->start();

}

// Transfer File by XMODEM protcol
void ManageSerialPort::stopXferFile_XMODEM()
{
    thSendXModem->stopSendXModem();
}

 // send KOZOS load command, before FileSend
void ManageSerialPort::setFlagKOZOS(bool flag)
{
    this->flagKOZOS = flag; // true is KOZOS option enable
}

void ManageSerialPort::slot_sendEnded(int result)
{
    qDebug() << QString("slot_SendEnded. code = %1").arg(result);

    // Enable Thread Send&Receive
    this->enableSending();
    this->enableReceiving();
    this->receiveData();

    thSendXModem->stopSendXModem();
    thSendXModem->wait();
    delete (thSendXModem);
    thSendXModem = NULL;

    emit sendEnded(result);
}
//void ManageSerialPort::slot_progress_status(int progress)
//{
//    qDebug() << QString("slot_progress_status : pregress = %1").arg(progress);
//}


/********************************************************************************************************
		* Classe ThreadSend
		*****************************************************************************************************/

ThreadSend::ThreadSend(QextSerialPort &addressSerialPort) : extSerialPort(addressSerialPort)
{
	dataToSend.clear();
	stopped=false;
}

ThreadSend::~ThreadSend()
{
	if (isRunning())
	{
		stopSending();
		wait();
	}
}


void ThreadSend::addDataToSend(QByteArray &dataToAdd)
{
	QMutexLocker locker(&mutexSend);
	for (int i=0; i<dataToAdd.size(); i++)
		dataToSend.enqueue(QByteArray(1,dataToAdd.at(i)));
	
	if (!isRunning())
		start();
}

void ThreadSend::stopSending()
{
	stopped=true;
}

void ThreadSend::run()
{
	QByteArray byteArray;
	
	forever
	{
		if (dataToSend.isEmpty() || stopped)
		{
			stopped = false;
			break;
		}
		mutexSend.lock();
		byteArray = dataToSend.dequeue();
		mutexSend.unlock();
		
		extSerialPort.write(byteArray, 1);
	}
}





/********************************************************************************************************
		* Classe ThreadReceive - A TERMINER
		*****************************************************************************************************/

ThreadReceive::ThreadReceive(QextSerialPort &addressSerialPort) : extSerialPort(addressSerialPort)
{
	stopped=false;
}

ThreadReceive::~ThreadReceive()
{
	if (isRunning())
	{
		stopReceiving();
		wait();
	}
}

void ThreadReceive::stopReceiving()
{
	stopped = true;
}

void ThreadReceive::run()
{
	int numBytes=0;
	char data[1024];
	QByteArray dataReceived;
	
	forever
	{
		if (stopped)
		{
			stopped = false;
			break;
		}
		numBytes = extSerialPort.bytesAvailable();
		if (numBytes > 0)
		{
			mutexReceive.lock();
			
			extSerialPort.read(data, numBytes);
			data[numBytes]='\0';
			
			dataReceived = data;
			
			mutexReceive.unlock();
			
			emit newDataReceived(dataReceived);
                } else {
                    this->msleep(100);
                }
	}
}


/********************************************************************************************************
                * Classe ThreadSendXModem
                *****************************************************************************************************/

ThreadSendXModem::ThreadSendXModem(QextSerialPort &addressSerialPort) : extSerialPort(addressSerialPort)
{
    stopped=false;
    filepath = "";
}

ThreadSendXModem::~ThreadSendXModem()
{
    if (isRunning())
    {
        stopSendXModem();
        wait();
    }
}

void ThreadSendXModem::stopSendXModem()
{
    stopped = true;
}

void ThreadSendXModem::setSendFilePath(QString path)
{
    filepath = path;
}

bool ThreadSendXModem::XMODEM_DataBlock(char *buff, unsigned char blknum, char *data, int datalen)
{
    memset(&buff[3], 0x1A, 128); // Padding EOF(0x1F)
    buff[0] = 0x01; // SOH
    buff[1] = blknum;
    buff[2] = ~blknum;
    memcpy(&buff[3], data, datalen);
    unsigned char chksum = 0;
    for (int i = 0; i < 128; i++) {
        chksum += buff[i+3];
    }
    buff[128+3] = chksum;

    return false;
}

void ThreadSendXModem::run()
{
    enum {
        CODE_SOH = 0x01,
        CODE_STX = 0x02,
        CODE_ETX = 0x03,
        CODE_EOT = 0x04,
        CODE_ACK = 0x06,
        CODE_NAK = 0x15,
        CODE_CAN = 0x18,
        BLOCK_SIZE = 128
    };
    int ret = 0;
    int numBytes=0;

    char buff[BLOCK_SIZE];
    char data[BLOCK_SIZE+4]; // SOH(1) + BLK#(1) + ~BLK#(1) + DATA(128) + CHKSUM(1)
    char senddata[BLOCK_SIZE];
    int  senddata_len;

    int i;

    int BlockCount = 1;
    int LeastBytes = 0;
    unsigned char BlockNumber;
    bool flag_retry = false;

    // fileopen failed then return true;
    qDebug() << filepath;
    QFile file(filepath);
    qint64 filesize = file.size();

    qDebug() << QString("FileSize is %1 bytes").arg(filesize);
    BlockCount = (int)((int)filesize / BLOCK_SIZE);
    LeastBytes = (int)((int)filesize % BLOCK_SIZE);
    if (LeastBytes) BlockCount++;
    qDebug() << QString("BlockCount = %1, LeastBytes = %2").arg(BlockCount).arg(LeastBytes);

    if (!(file.open(QIODevice::ReadOnly))) {
        qDebug() << "FileOpenError";
        buff[0] = CODE_CAN; // CAN
        extSerialPort.write(buff, 1); // SendCAN
        ret = 1; // FileOpen error
        goto failure;
    }

    // 1.Wait for NAK
    while (1) {
        if (stopped) {
            buff[0] = CODE_CAN; // CAN
            extSerialPort.write(buff, 1); // SendCAN
            ret = 2; // 1stNAK error
            goto failure;
        }
        numBytes = extSerialPort.bytesAvailable();
        if (numBytes > 0) {
            data[0] = 0x00;
            extSerialPort.read(data, 1);
            if (data[0] == CODE_NAK) break; // Rcv NAK
        }
        this->msleep(100);
    }
    // this->sleep(1);
    extSerialPort.flush(); // Flush Recv NAK

    // 2.SendData, Wait for ACK
    flag_retry = false;
    BlockNumber = 1;
    for (i = 0; i < BlockCount;) {
        if (stopped) {
            buff[0] = CODE_CAN; // CAN
            extSerialPort.write(buff, 1); // SendCAN
            ret = 3; // Cancel
            goto failure;
        }
        if (!flag_retry) {
            if (LeastBytes != 0 && (i + 1) == BlockCount) { // LastData?
                senddata_len = LeastBytes;
            } else {
                senddata_len = BLOCK_SIZE;
            }
            file.read(senddata, senddata_len);
            this->XMODEM_DataBlock(data, BlockNumber, senddata, senddata_len);
        }
        extSerialPort.write(data, BLOCK_SIZE+4);   // Send BlockData

        this->msleep(200);

        // WaitFor ACK
        while (1) {
            if (stopped) {
                buff[0] = CODE_CAN; // CAN
                extSerialPort.write(buff, 1); // SendCAN
                ret = 4; // DataACK error
                goto failure;
            }
            numBytes = extSerialPort.bytesAvailable();
            if (numBytes > 0) {
                data[0] = 0x00;
                extSerialPort.read(data, 1);
                if (data[0] == CODE_ACK || data[0] == CODE_NAK) break; // Rcv ACK or NAK
            }
            this->msleep(100);
        }

        if (data[0] == CODE_ACK) { // Rcv Ack, goto NextBlock
            emit progress_status((int)((float)(i + 1)/(float)BlockCount*100.0));
            qDebug() << QString("Rcv DataAck, BlkNum = %1").arg(BlockNumber);
            flag_retry = false;
            BlockNumber++;
            i++;
        } else if (data[0] == CODE_NAK) {
            qDebug() << "Rcv DataNAK";
            flag_retry = true;
        }
    }

    // 3.Send EOT, Wait for ACKorNAK
    buff[0] = CODE_EOT; // EOT
    extSerialPort.write(buff, 1); // SendEOT
    qDebug() << QString("Send EOT");
    this->msleep(100);

    // WaitFor EndACK
    while (1) {
        if (stopped) {
            buff[0] = CODE_CAN; // CAN
            extSerialPort.write(buff, 1); // SendCAN
            ret = 5; // EndACK error
            goto failure;
        }
        numBytes = extSerialPort.bytesAvailable();
        if (numBytes > 0) {
            extSerialPort.read(data, 1);
            break;
        }
        this->msleep(100);
    }
    qDebug() << QString("RcvEOT Ack");

    // File Close
    file.close();
    extSerialPort.flush();

    emit sendEnded(0); // NoError
    stopped = false;
    return;

failure:
    // File Close
    file.close();
    extSerialPort.flush();

    emit sendEnded(ret); // Error
    stopped = false;
}

