#include "logthread.h"

#include <QFile>
#include <QDir>
#include <QTextStream>

#include <QDateTime>

#include <QMessageBox>
#include <QDebug>

LogThread::LogThread(QObject *parent) :
    QThread(parent)
{
    stopped = false;

    logging = false;
    logsavefolderpath = ""; // log save folder
    logfilepath = ""; // log file path

    EvtQue = new QQueue<EventData>();
    EvtQue->clear();
}

LogThread::~LogThread()
{
    EvtQue->clear();
    delete EvtQue;

}

void LogThread::stop()
{
    stopped = true;
}


void LogThread::slotLogStart(QString logfolderpath)
{
    logsavefolderpath = logfolderpath;
    EventData evt;
    evt.id = Ev_Log_Start;
    EvtQue->enqueue(evt);
}

void LogThread::slotLogStop()
{
    EventData evt;
    evt.id = Ev_Log_Stop;
    EvtQue->enqueue(evt);
}

void LogThread::LogData(quint64 timestamp, double tmpr, double humid)
{
    EventData evt;
    evt.id = Ev_Log_Data;
    evt.timestamp_ms = timestamp;
    evt.tmpr = tmpr;
    evt.humid = humid;
    EvtQue->enqueue(evt);
}


void LogThread::run()
{

    qDebug() << "logTh is started.";
    EventData event;
    QFile csvfile;
    QTextStream out;
    QDateTime timestamp;
    int prev_Minute; // prev minutes for timestamp for 1 minutes interval
    int prev_Day;    // prev Day for Change logfile.

    while(!stopped) {
        if (EvtQue->isEmpty()) {
            msleep(100);
        } else {
            event = EvtQue->dequeue();
            if (!logging) { // idle
                if (event.id == Ev_Log_Start) {
                    // then file open
                    prev_Minute = -1;
                    QDateTime today = QDateTime::currentDateTime();
                    logfilepath = logsavefolderpath + QDir::separator() + "USBRHLog_" + today.toString("yyyyMMdd") + ".csv";
                    csvfile.setFileName(logfilepath);
                    if (csvfile.open(QIODevice::WriteOnly | QIODevice::Append)) {
                        logging = true;
                        qDebug() << "logTh log started. logfilename is " << logfilepath;
                        //out.flush();
                        out.setDevice(&csvfile);
                        out.setRealNumberNotation(QTextStream::FixedNotation);
                        out.setRealNumberPrecision(2);
                        if (csvfile.size() == 0) {
                            out << "timestamp, tempr(C), humid(%)" << endl; // append header
                        }
                        prev_Day = today.date().day();
                        emit LogStarted();
                        emit ChangeLogFilename(logfilepath);
                    } else {
                        emit LogEnded(1);
                        qDebug() << "open error";
                    }
                }
            } else { // logging
                if (event.id == Ev_Log_Stop) {
                    // then file close
                    out.flush();
                    if (csvfile.isOpen()) csvfile.close();
                    logsavefolderpath = "";
                    logfilepath = "";
                    logging = false;
                    qDebug() << "logTh log ended.";
                    emit LogEnded(0);
                } else if (event.id == Ev_Log_Data) {
                    // if data; // write data to file.
                    timestamp.setMSecsSinceEpoch(event.timestamp_ms);
                    if (timestamp.time().minute() != prev_Minute) { // save interval is 1 minute
                        if (timestamp.date().day() != prev_Day) {
                            out.flush();
                            csvfile.close();
                            logfilepath = logsavefolderpath + QDir::separator() + "USBRHLog_" + timestamp.toString("yyyyMMdd") + ".csv";
                            csvfile.setFileName(logfilepath);
                            if (csvfile.open(QIODevice::WriteOnly | QIODevice::Append)) {
                                logging = true;
                                qDebug() << "logTh log started. logfilename is " << logfilepath;
                                out.setDevice(&csvfile);
                                out.setRealNumberNotation(QTextStream::FixedNotation);
                                out.setRealNumberPrecision(2);
                                if (csvfile.size() == 0) {
                                    out << "timestamp, tempr(C), humid(%)" << endl; // append header
                                }
                                prev_Day = timestamp.date().day();
                                emit ChangeLogFilename(logfilepath);
                            }
                        }
                        // qDebug() << "timestamp: "  << timestamp.toString("yyyy/MM/dd hh:mm:ss") << "," << "tmpr = " << event.tmpr << ", humid = " << event.humid;
                        out << timestamp.toString("yyyy/MM/dd hh:mm") << ", " << event.tmpr << ", " << event.humid << endl;
                        imgfilepath = logsavefolderpath + QDir::separator() + "USBRHLog_" + timestamp.toString("yyyyMMddhhmm") + ".JPG";
                        QString text = timestamp.toString("yyyy/MM/dd hh:mm ") + QString("Tempr = %1 'C, Humid = %2 %").arg(event.tmpr, 0, 'f', 2).arg(event.humid, 0, 'f', 2);
                        emit SaveCurrentImage(imgfilepath, text);
                        prev_Minute = timestamp.time().minute();
                    }
                }
            }
        }
    }

    if (logging) {
        logging = false;
        logsavefolderpath = "";
        emit LogEnded(0);
    }
    if (csvfile.isOpen()) {
        out.flush();
        csvfile.close();
    }

    qDebug() << "logTh is stopped.";
}
