﻿using System;
using System.IO;

namespace Azyobuzi.UserStreamEx
{
    /// <summary>
    /// 接続されていない状態の<see cref="UserStream"/>クラスを表します。
    /// </summary>
    public interface INotStartedUserStream
    {
        /// <summary>
        /// 接続を開始します。
        /// </summary>
        /// <param name="track">検索キーワードっぽいもの。null可。</param>
        /// <param name="allReplies">friendsに対する返信も含めるかどうか</param>
        IStartedUserStream Start(string track, bool allReplies);

        /// <summary>
        /// 取得したデータを書き込む<see cref="System.IO.TextWriter"/>
        /// </summary>
        TextWriter StreamingWriter { set; get; }

        #region Events
        /// <summary>
        /// 接続を開始するときに発生します。
        /// </summary>
        event EventHandler Started;
        /// <summary>
        /// 切断されたときに発生します。
        /// </summary>
        event EventHandler<StoppedEventArgs> Stopped;
        /// <summary>
        /// friends(following)のID一覧を受け取ったときに発生します。通常は開始直後に一回だけ発生します。
        /// </summary>
        event EventHandler<ReceiveFriendsEventArgs> ReceiveFriends;
        /// <summary>
        /// 新しいツイートを受け取ったときに発生します。
        /// </summary>
        event EventHandler<NewTweetEventArgs> NewTweet;
        /// <summary>
        /// 新しいダイレクトメッセージを受信・送信したときに発生します。
        /// </summary>
        event EventHandler<NewDirectMessageEventArgs> NewDirectMessage;
        /// <summary>
        /// ツイートまたはダイレクトメッセージが削除されたときに発生します。ハンドラでは直ちに表示から消すような処理をしてください。
        /// </summary>
        event EventHandler<DeleteStatusEventArgs> DeleteStatus;
        /// <summary>
        /// イベントを受信したときに発生します。
        /// </summary>
        event EventHandler<ReceiveEventEventArgs> ReceiveEvent;
        /// <summary>
        /// Twitter側で処理しきれないツイートがあったときに発生します。
        /// </summary>
        /// <remarks>詳細不明</remarks>
        event EventHandler<TrackLimitEventArgs> TrackLimit;
        /// <summary>
        /// 対応していないデータを受け取ったときに発生します。
        /// </summary>
        event EventHandler<ReceiveJsonEventArgs> ReceiveUnsupportedData;
        #endregion

        #region AddHandlerMethod
        /// <summary>
        /// <see cref="Started"/>イベントにイベントハンドラを追加します。
        /// </summary>
        /// <param name="handler">イベントハンドラ</param>
        INotStartedUserStream SetStartedEvent(EventHandler handler);
        /// <summary>
        /// <see cref="Stopped"/>イベントにイベントハンドラを追加します。
        /// </summary>
        /// <param name="handler">イベントハンドラ</param>
        INotStartedUserStream SetStoppedEvent(EventHandler<StoppedEventArgs> handler);
        /// <summary>
        /// <see cref="ReceiveFriends"/>イベントにイベントハンドラを追加します。
        /// </summary>
        /// <param name="handler">イベントハンドラ</param>
        INotStartedUserStream SetReceiveFriendsEvent(EventHandler<ReceiveFriendsEventArgs> handler);
        /// <summary>
        /// <see cref="NewTweet"/>イベントにイベントハンドラを追加します。
        /// </summary>
        /// <param name="handler">イベントハンドラ</param>
        INotStartedUserStream SetNewTweetEvent(EventHandler<NewTweetEventArgs> handler);
        /// <summary>
        /// <see cref="NewDirectMessage"/>イベントにイベントハンドラを追加します。
        /// </summary>
        /// <param name="handler">イベントハンドラ</param>
        INotStartedUserStream SetNewDirectMessageEvent(EventHandler<NewDirectMessageEventArgs> handler);
        /// <summary>
        /// <see cref="DeleteStatus"/>イベントにイベントハンドラを追加します。
        /// </summary>
        /// <param name="handler">イベントハンドラ</param>
        INotStartedUserStream SetDeleteStatusEvent(EventHandler<DeleteStatusEventArgs> handler);
        /// <summary>
        /// <see cref="ReceiveEvent"/>イベントにイベントハンドラを追加します。
        /// </summary>
        /// <param name="handler">イベントハンドラ</param>
        INotStartedUserStream SetReceiveEventEvent(EventHandler<ReceiveEventEventArgs> handler);
        /// <summary>
        /// <see cref="TrackLimit"/>イベントにイベントハンドラを追加します。
        /// </summary>
        /// <param name="handler">イベントハンドラ</param>
        INotStartedUserStream SetTrackLimitEvent(EventHandler<TrackLimitEventArgs> handler);
        /// <summary>
        /// <see cref="ReceiveUnsupportedData"/>イベントにイベントハンドラを追加します。
        /// </summary>
        /// <param name="handler">イベントハンドラ</param>
        INotStartedUserStream SetReceiveUnsupportedDataEvent(EventHandler<ReceiveJsonEventArgs> handler);
        #endregion
    }

    /// <summary>
    /// 接続されていて、Timeoutが設定された<see cref="UserStream"/>クラスを表します。
    /// </summary>
    public interface ISetedTimeoutUserStream
    {
        /// <summary>
        /// 切断してStoppedイベントを発生させます。
        /// </summary>
        INotStartedUserStream Stop();
    }

    /// <summary>
    /// 接続されている状態の<see cref="UserStream"/>クラスを表します。
    /// </summary>
    public interface IStartedUserStream : ISetedTimeoutUserStream
    {
        /// <summary>
        /// 指定した時間が経過したら切断するように設定します。
        /// </summary>
        /// <param name="waitTime">切断するまでの時間（ミリ秒単位）。0以下を指定するとすぐに切断します。</param>
        ISetedTimeoutUserStream SetTimeout(int waitTime);
    }
}
