﻿using System;
using System.Text.RegularExpressions;
using System.Collections.Generic;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Input;
using FooEditEngine;
using FooEditEngine.WPF;
using Prop = FooEditor.Properties;

namespace FooEditor
{
    /// <summary>
    /// FindReplaceWindow.xaml の相互作用ロジック
    /// </summary>
    public partial class FindReplaceWindow : UserControl
    {
        private IEnumerator<SearchResult> sr;

        public FindReplaceWindow()
        {
            InitializeComponent();
            this.Title = "検索と置き換え";

            this.CommandBindings.Add(new CommandBinding(FindReplaceCommands.FindStart, FindStartCommand));
            this.CommandBindings.Add(new CommandBinding(FindReplaceCommands.ReplaceStart, ReplaceStartCommand));
            this.CommandBindings.Add(new CommandBinding(FindReplaceCommands.ReplaceAllStart, ReplaceAlStartlCommand));

            this.FindText.TextChanged += new TextChangedEventHandler(FindText_TextChanged);
        }

        public string Title
        {
            get;
            set;
        }

        public FooTextBox TextBox
        {
            get;
            set;
        }

        void FindText_TextChanged(object sender, TextChangedEventArgs e)
        {
            this.sr = null;
        }

        private RegexOptions DecideRegexOpt()
        {
            RegexOptions opt = RegexOptions.None;
            if (this.RestrictSearch.IsChecked == true)
                opt |= RegexOptions.IgnoreCase;
            return opt;
        }

        void FindStartCommand(object sender, ExecutedRoutedEventArgs e)
        {
            if (this.TextBox.RectSelectMode)
            {
                MessageBox.Show(Prop.Resources.MustBeDisableRectSelect);
                return;
            }

            try
            {
                if (this.sr == null)
                {
                    this.TextBox.Document.SetFindParam(this.FindText.Text, (bool)this.UseRegEx.IsChecked, DecideRegexOpt());
                    this.sr = this.TextBox.Document.Find();
                }
            }
            catch (ArgumentException ex)
            {
                MessageBox.Show(ex.Message);
                return;
            }

            if (sr.MoveNext() == false)
            {
                this.sr = null;
                MessageBox.Show(Prop.Resources.FindDialogNotFound);
                return;
            }

            SearchResult current = sr.Current;
            this.TextBox.JumpCaret(current.Start);
            this.TextBox.Select(current.Start, current.End - current.Start + 1);
            this.TextBox.Refresh();
            this.ReplaceButton.IsEnabled = true;
        }

        void ReplaceStartCommand(object sender, ExecutedRoutedEventArgs e)
        {
            if(this.UseGroup.IsChecked == true)
                this.TextBox.SelectedText = this.sr.Current.Result(this.ReplaceText.Text);
            else
                this.TextBox.SelectedText = this.ReplaceText.Text;
            this.TextBox.DeSelectAll();
            this.TextBox.Refresh();
            this.FindStartCommand(sender, e);
        }
        
        void ReplaceAlStartlCommand(object sender, ExecutedRoutedEventArgs e)
        {
            if (this.TextBox.RectSelectMode)
            {
                MessageBox.Show(Prop.Resources.MustBeDisableRectSelect);
                return;
            }

            this.TextBox.DeSelectAll();
            this.TextBox.Document.SetFindParam(this.FindText.Text, (bool)this.UseRegEx.IsChecked, DecideRegexOpt());
            this.TextBox.Document.ReplaceAll(this.ReplaceText.Text,(bool)this.UseGroup.IsChecked);
            this.TextBox.Refresh();
        }
    }

    public static class FindReplaceCommands
    {
        public static RoutedCommand FindStart = new RoutedCommand("FindStart", typeof(FindReplaceWindow));
        public static RoutedCommand ReplaceStart = new RoutedCommand("ReplaceStart", typeof(FindReplaceWindow));
        public static RoutedCommand ReplaceAllStart = new RoutedCommand("ReplaceAllStart", typeof(FindReplaceWindow));
    }
}
