﻿/*@@@*
The MIT License (MIT)

Copyright (c) 2013 GMA0BN

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
**@@@*/
#include "stdafx.h"
#include "sockutil.h"

#undef NUMELEMS
#define NUMELEMS(a) (sizeof(a) / sizeof(a[0]))

typedef struct tagSymInfo {
    const char* sym;
    int val;
} SymInfo;

#define TABENT(s)   { #s, (s) },

// ai_flagsフィールド
static const SymInfo tab_ai_flags[] = {
    TABENT(AI_PASSIVE)
    TABENT(AI_PASSIVE)
    TABENT(AI_CANONNAME)
    TABENT(AI_NUMERICHOST)
    TABENT(AI_NUMERICSERV)
    TABENT(AI_ALL)
    TABENT(AI_ADDRCONFIG)
    TABENT(AI_V4MAPPED)
    TABENT(AI_NON_AUTHORITATIVE)
    TABENT(AI_SECURE)
    TABENT(AI_RETURN_PREFERRED_NAMES)
    TABENT(AI_FQDN)
    TABENT(AI_FILESERVER)
};

// ai_familyフィールド
static const SymInfo tab_ai_family[] = {
    TABENT(PF_UNSPEC)
    TABENT(PF_UNIX)
    TABENT(PF_INET)
    TABENT(PF_IMPLINK)
    TABENT(PF_PUP)
    TABENT(PF_CHAOS)
    TABENT(PF_NS)
    TABENT(PF_IPX)
    TABENT(PF_ISO)
    TABENT(PF_OSI)
    TABENT(PF_ECMA)
    TABENT(PF_DATAKIT)
    TABENT(PF_CCITT)
    TABENT(PF_SNA)
    TABENT(PF_DECnet)
    TABENT(PF_DLI)
    TABENT(PF_LAT)
    TABENT(PF_HYLINK)
    TABENT(PF_APPLETALK)
    TABENT(PF_VOICEVIEW)
    TABENT(PF_FIREFOX)
    TABENT(PF_UNKNOWN1)
    TABENT(PF_BAN)
    TABENT(PF_ATM)
    TABENT(PF_INET6)
#if _WIN32_WINNT >= 0x0600
    TABENT(PF_BTH)
#endif  /* _WIN32_WINNT >= 0x0600 */
    TABENT(PF_MAX)
};

// ai_socktypeフィールド
static const SymInfo tab_ai_socktype[] = {
    TABENT(SOCK_STREAM)
    TABENT(SOCK_DGRAM)
    TABENT(SOCK_RAW)
    TABENT(SOCK_RDM)
    TABENT(SOCK_SEQPACKET)
};

// ai_protocolフィールド
static const SymInfo tab_ai_protocol[] = {
#if _WIN32_WINNT >= 0x0501
    TABENT(IPPROTO_HOPOPTS)
#endif  /* _WIN32_WINNT >= 0x0501 */
    TABENT(IPPROTO_ICMP)
    TABENT(IPPROTO_IGMP)
    TABENT(IPPROTO_GGP)
#if _WIN32_WINNT >= 0x0501
    TABENT(IPPROTO_IPV4)
#endif  /* _WIN32_WINNT >= 0x0501 */
#if _WIN32_WINNT >= 0x0600
    TABENT(IPPROTO_ST)
#endif  /* _WIN32_WINNT >= 0x0600 */
    TABENT(IPPROTO_TCP)
#if _WIN32_WINNT >= 0x0600
    TABENT(IPPROTO_CBT)
    TABENT(IPPROTO_EGP)
    TABENT(IPPROTO_IGP)
#endif  /* _WIN32_WINNT >= 0x0600 */
    TABENT(IPPROTO_PUP)
    TABENT(IPPROTO_UDP)
    TABENT(IPPROTO_IDP)
#if _WIN32_WINNT >= 0x0600
    TABENT(IPPROTO_RDP)
#endif  /* _WIN32_WINNT >= 0x0600 */
#if _WIN32_WINNT >= 0x0501
    TABENT(IPPROTO_IPV6)
    TABENT(IPPROTO_ROUTING)
    TABENT(IPPROTO_FRAGMENT)
    TABENT(IPPROTO_ESP)
    TABENT(IPPROTO_AH)
    TABENT(IPPROTO_ICMPV6)
    TABENT(IPPROTO_NONE)
    TABENT(IPPROTO_DSTOPTS)
#endif  /* _WIN32_WINNT >= 0x0501 */
    TABENT(IPPROTO_ND)
#if _WIN32_WINNT >= 0x0501
    TABENT(IPPROTO_ICLFXBM)
#endif  /* _WIN32_WINNT >= 0x0501 */
#if _WIN32_WINNT >= 0x0600
    TABENT(IPPROTO_PIM)
    TABENT(IPPROTO_PGM)
    TABENT(IPPROTO_L2TP)
    TABENT(IPPROTO_SCTP)
#endif  /* _WIN32_WINNT >= 0x0600 */
    TABENT(IPPROTO_RAW)
    TABENT(IPPROTO_MAX)
    TABENT(IPPROTO_RESERVED_RAW)
    TABENT(IPPROTO_RESERVED_IPSEC)
    TABENT(IPPROTO_RESERVED_IPSECOFFLOAD)
    TABENT(IPPROTO_RESERVED_MAX)
};

/// ai_flagsをデコードします。
static BOOL decode_ai_flags(FILE* fp, int flags)
{
    int i, cnt;
    const SymInfo* p_si;

    cnt = 0;
    for (i = 0; i < NUMELEMS(tab_ai_flags); ++i) {
        p_si = &(tab_ai_flags[i]);
        if ((flags & p_si->val) != 0) {
            if (cnt > 0) fputs("|", fp);
            fputs(p_si->sym, fp);
            ++cnt;
        }
    }

    return !ferror(fp);
}

// 数値をデコードします。
static BOOL decode_numeric_item(FILE* fp, int x, const SymInfo* tab, int ne)
{
    int i, cnt;
    const SymInfo* p_si;
    BOOL suc;

    suc = TRUE;
    cnt = 0;
    for (i = 0; i < ne; ++i) {
        p_si = &(tab[i]);
        if (x == p_si->val) {
            if (cnt > 0) suc = FALSE;
            fputs(p_si->sym, fp);
            ++cnt;
        }
    }

    return !ferror(fp) && suc;
}

/// アドレフファミリをデコードします。
BOOL decode_addr_family(FILE* fp, int family)
{
    return decode_numeric_item(fp, family, tab_ai_family, NUMELEMS(tab_ai_family));
}

/// addrinfo構造体の中身をダンプします。
BOOL dump_addrinfo(FILE* fp, struct addrinfo* p, const char* indent)
{
    BOOL suc;
    char indent2[128];

    if (indent == NULL) {
        indent = "";
    }
    if (sprintf_s(indent2, NUMELEMS(indent2), "%s  ", indent) < 0) {
        perror(__FUNCTION__);
        indent2[0] = '\0';
    }

    suc = TRUE;

    // ai_flags field
    fputs(indent, fp);
    fprintf(fp, "ai_flags   =0x%08x (", p->ai_flags);
    if (!decode_ai_flags(fp, p->ai_flags)) {
        suc = FALSE;
    }
    fputs(")\n", fp);

    // ai_family field
    fputs(indent, fp);
    fprintf(fp, "ai_family  =0x%08x (", p->ai_family);
    if (!decode_addr_family(fp, p->ai_family)) {
        suc = FALSE;
    }
    fputs(")\n", fp);

    // ai_protocol field
    fputs(indent, fp);
    fprintf(fp, "ai_protocol=0x%08x (", p->ai_protocol);
    if (!decode_numeric_item(fp, p->ai_protocol, tab_ai_protocol, NUMELEMS(tab_ai_protocol))) {
        suc = FALSE;
    }
    fputs(")\n", fp);

    // ai_addrlen field
    fputs(indent, fp);
    fprintf(fp, "ai_addrlen =%lu byte\n", p->ai_addrlen);

    // ai_canonname field
    fputs(indent, fp);
    if (p->ai_canonname != NULL) {
        fprintf(fp, "ai_canonname=\"%s\"\n", p->ai_canonname);
    } else {
        fprintf(fp, "ai_canonname=(NULL)\n");
    }

    // ai_addr field
    fputs(indent, fp);
    fprintf(fp, "ai_addr    =0x%08p\n", p->ai_addr);
    dump_sockaddr(fp, p->ai_addr, p->ai_addrlen, indent2);

    // ai_next field
    fputs(indent, fp);
    fprintf(fp, "ai_next    =0x%08p\n", p->ai_next);

    return !ferror(fp) && suc;
}
