﻿/*@@@*
The MIT License (MIT)

Copyright (c) 2013 GMA0BN

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
**@@@*/
#include "stdafx.h"
#include "sockutil.h"
#include "sockutil_cl.h"

static ClientCB g_aClientCB[MAXCONN];

static unsigned g_uSessionID;

/// クライアント機能を初期化します。
void cli_init_sockutil(void)
{
    memset(g_aClientCB, 0, sizeof(g_aClientCB));
    init_session_res(g_aClientCB, sizeof(g_aClientCB[0]), MAXCONN);
}

/// クライアント機能を後始末します。
void cli_term_sockutil(void)
{
    release_session_res(g_aClientCB, sizeof(g_aClientCB[0]), MAXCONN, TRUE);
}

/// ユーザー関数のlauncher。
static unsigned __stdcall SessionThreadLauncher(void* exinf)
{
    ClientCB* p;
    unsigned uRet;

    p = (ClientCB*)exinf;
    uRet = p->startAddress(exinf);
    closesocket(p->res.socket);
    p->res.socket = INVALID_SOCKET;
#if USE_BARRIER
    _WriteBarrier();
#endif
    /*= ソケットのcloseは、接続断を通信相手に知らせる意味もあるので
     *  ユーザー関数終了直後に速やかに行う必要がある。
     *  スレッドオブジェクトの開放は、release_cb()でスレッド終了確認後に行う。
     */

    return uRet;
}

/// CBが握っている資源を解放します。
static BOOL release_cb(BOOL bFull)
{
    return release_session_res(g_aClientCB, sizeof(g_aClientCB[0]), MAXCONN, bFull);
}

/// 接続手段を列挙します。
int cli_enum_means(PCSTR pNodeName, PCSTR pServiceName, ClientCB cb[], int n)
{
    struct addrinfo hints;
    struct addrinfo* pRes;
    struct addrinfo* pRes0;
    INT nErr;
    int cnt;
    ClientCB* p;

    // 資源開放
    release_cb(TRUE);

    // address/portを解決
    memset(&hints, 0, sizeof(hints));
#if USE_CANONNAME
    hints.ai_flags |= AI_CANONNAME;
#endif
#if USE_NUMERICHOST
    hints.ai_flags |= AI_NUMERICHOST;
#endif
    hints.ai_family = AF_UNSPEC;
    hints.ai_socktype = SOCK_STREAM;
    nErr = getaddrinfo(pNodeName, pServiceName, &hints, &pRes0);
    if (nErr != 0) {
        fprintf(stderr, "\nERROR: " __FUNCTION__ ": %s\n", gai_strerrorA(nErr));
        pRes0 = NULL;
        goto on_error;
    }

    // 接続手段を記憶
    n = min(n, MAXCONN);
    cnt = 0;
    for (pRes = pRes0; pRes != NULL; pRes = pRes->ai_next) {
#if VERBOSE
        // 情報表示
        printf("addrinfo[%d]:\n", cnt);
        dump_addrinfo(stdout, pRes, "  ");
#endif

        // コピー先オーバーランチェック
        if (cnt >= n) {
            fprintf(stderr, "\nWarning: " __FUNCTION__ ": Too many connection means.\n");
            break;
        }
        p = &(cb[cnt]);
        p->sessionID = g_uSessionID++;
        p->res.socket = INVALID_SOCKET;
        p->res.hThread = NULL;

        // 情報取得(のテスト)
        nErr = getnameinfo(
            pRes->ai_addr,
            pRes->ai_addrlen,
            p->hostName, sizeof(p->hostName),
            p->serviceName, sizeof(p->serviceName),
            NI_NUMERICHOST | NI_NUMERICSERV);
        if (nErr != 0) {
            int err = WSAGetLastError();
            fprintf(stderr, "\nERROR: " __FUNCTION__ ": getnameinfo() failed. (err=%d)\n", err);
            goto on_error;
        }

        // 呼び出し元に渡すためコピー
        memcpy(&(p->to), pRes, sizeof(struct addrinfo));

        // 次へ
        ++cnt;
    }

    return cnt;

on_error:
    if (pRes0 != NULL) {
        freeaddrinfo(pRes0);
    }
    return -1;
}

/// 接続します。
BOOL cli_connect(
    ClientCB* p_cb,
    unsigned (__stdcall *startAddress)(void* exinf),
    BOOL bBlocking
    )
{
    struct addrinfo* pRes;
    SOCKET s;
    int nRet;
    uintptr_t upRet;
    DWORD dwRet;

    assert(p_cb->res.socket == INVALID_SOCKET);
    pRes = &(p_cb->to);

    // ソケット生成
    s = socket(pRes->ai_family, pRes->ai_socktype, pRes->ai_protocol);
    if (s == INVALID_SOCKET) {
        int err = WSAGetLastError();
        fprintf(stderr, "\nWaring: " __FUNCTION__ ": socket() failed. (err=%d)\n", err);
        goto on_error;
    }
    p_cb->res.socket = s;

    // 接続
    nRet = connect(s, pRes->ai_addr, pRes->ai_addrlen);
    if (nRet < 0) {
        int err = WSAGetLastError();
        fprintf(stderr, "\nERROR: " __FUNCTION__ ": connect() failed. (err=%d)\n", err);
        goto on_error;
    }

    // セッションのスレッド起動
    p_cb->startAddress = startAddress;
    upRet = _beginthreadex(NULL, THREAD_STKSZ, SessionThreadLauncher, p_cb, 0, NULL);
    if (upRet == 0) {
        perror(__FUNCTION__);
        goto on_error;
    } else {
        p_cb->res.hThread = (HANDLE)upRet;  // hThread更新
    }

    // 完了待ち
    if (bBlocking) {
        dwRet = WaitForSingleObject(p_cb->res.hThread, INFINITE);
        if (dwRet != WAIT_OBJECT_0) {
            DWORD err = GetLastError();
            fprintf(stderr, "\nERROR: " __FUNCTION__ ": WaitForSingleObject() failed. (err=0x%08lx)\n", err);
            goto on_error;
        }
    }

    return TRUE;

on_error:
#if USE_BARRIER
    _ReadBarrier();
#endif
    if (p_cb->res.socket != INVALID_SOCKET) {
        closesocket(p_cb->res.socket);
        p_cb->res.socket = INVALID_SOCKET;
    }
    assert(p_cb->res.hThread == NULL);
    return FALSE;
}

/// 接続します。(タイムアウト指定可能)
BOOL cli_connect_ex(
    ClientCB* p_cb,
    DWORD TCTmoIn_ms,
    unsigned (__stdcall *startAddress)(void* exinf),
    BOOL bBlocking
    )
{
    // TODO: 作成。
    return FALSE;
}

/// セッション完了を待ちます。
BOOL cli_wai_cmpl(ClientCB* p_cb)
{
    int dwRet;

    if (p_cb->res.hThread != NULL) {
        dwRet = WaitForSingleObject(p_cb->res.hThread, INFINITE);
        if (dwRet != WAIT_OBJECT_0) {
            DWORD err = GetLastError();
            fprintf(stderr, "\nERROR: " __FUNCTION__ ": WaitForSingleObject() failed. (err=0x%08lx)\n", err);
            return FALSE;
        }
        assert(p_cb->res.socket == INVALID_SOCKET);     // ソケットはスレッド終了時に閉じられているはず
        CloseHandle(p_cb->res.hThread);
        p_cb->res.hThread = NULL;
    }

    return TRUE;
}
