//
//  chncpu.h
//  chncpu
//
//  Created by 西田　耀 on 2014/06/04.
//  Copyright (c) 2014年 CHNOSProject. All rights reserved.
//

#ifndef chncpu_chncpu_h
#define chncpu_chncpu_h

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "ch4.h"

// DEBUG Flag
#define DEBUG_PRINT_APPBIN_BEFORE_EXECUTION	0
#define DEBUG_PRINT_IREG_AFTER_EXECUTION		0
#define DEBUG_PRINT_PREG_AFTER_EXECUTION    0
#define DEBUG_PRINT_OP_BINDING				0
#define DEBUG_PRINT_OP_EXECUTING			0

// VM flags
#define CHNCPU_VM_FLAG_PRINT_INFO     0x01


// Settings
#define SIZE_TMPDATA    (1024 * 1024 * 1)

#define CHNCPU_OPECODE_MAX              0xFF
#define CHNCPU_BIOS_OP_MAX				0x00
#define CHNCPU_BITS_MAX                 32
#define CHNCPU_NUMBER_OF_IREG           64
#define CHNCPU_NUMBER_OF_PREG           64
#define CHNCPU_LENGTH_OF_MAIN_MEMORY    (64 * 1024)     // per Op
#define CHNCPU_SIZE_APPBIN              (1024 * 1024 * 1)
#define CHNCPU_OPCODE_INVALID           (-1)
#define CHNCPU_MemoryIndex_INVALID      (-1)
#define CHNCPU_PREFIX_MAX               31

// Common ErrorCode
#define CHNCPU_ERR_INTERNAL                 0x0001
#define CHNCPU_ERR_INVALID_BITS             0x0002
#define CHNCPU_ERR_INVALID_OPCODE           0x0004
#define CHNCPU_ERR_INVALID_REGNUM           0x0008
#define CHNCPU_ERR_INVALID_PTYPE            0x0010
#define CHNCPU_ERR_INVALID_BINARY			0x0020
#define CHNCPU_ERR_INVALID_PREFIX           0x0040
#define CHNCPU_ERR_INVALID_EXPRESSION       0x0080
#define CHNCPU_ERR_INVALID_BIOS_FUNC_ID     0x0100
#define CHNCPU_ERR_DUPLICATED_LABEL_ID		0x0200
#define CHNCPU_ERR_NOT_FOUND_LABEL			0x0400
#define CHNCPU_ERR_NOT_MATCHED_PTYPE			0x0800
#define CHNCPU_ERR_TRUNCATED_VALUE			0x1000
#define CHNCPU_ERR_BAD_ACCESS				0x2000

// (2F)Prefix
#define CHNCPU_PREFIX_ALLOW_TRUNCATE    0x01

#define CHNCPU_PREFIX_MASK_Op_TernaryReg (CHNCPU_PREFIX_ALLOW_TRUNCATE)
#define CHNCPU_PREFIX_MASK_Op_CompareIReg (CHNCPU_PREFIX_ALLOW_TRUNCATE)

// Type of Pointer
#define CHNCPU_PType_Undefined	0x00
#define CHNCPU_PType_Label		0x01
#define CHNCPU_PType_UINT8		0x10
//	type	型名	説明
//  0x00    Undefined
//  0x01    Label
//	0x02    UINT1
//	0x03    SINT1
//  0x08    UINT4
//  0x09    SINT4
//  0x10    UINT8
//  0x11    SINT8
//  0x20    UINT16
//  0x21    SINT16
//  0x40    UINT32
//  0x41    SINT32

// Opcode
#define CHNCPU_OP_DATA	0x2E

typedef struct _CHNCPU_DATA CHNCPU_Data;
typedef struct _CHNCPU_OP_TAG CHNCPU_OpTag;
typedef struct _CHNCPU_BIOS CHNCPU_BIOS;
typedef struct _CHNCPU_OP_TABLE_SET CHNCPU_OpTableSet;
typedef struct _CHNCPU_LABEL_TAG CHNCPU_LabelTag;
typedef struct _CHNCPU_LABEL_SET CHNCPU_LabelSet;
typedef struct _CHNCPU_POINTER_TAG CHNCPU_PointerTag;
typedef struct _CHNCPU_RUN_TIME_ENVIRONMENT CHNCPU_RuntimeEnvironment;
typedef struct _CHNCPU_VM_ARGS CHNCPU_VMArgs;
typedef struct _CHNCPU_VM_ARGS_BIND_DATA_TAG CHNCPU_VMArgs_BindDataTag;

struct _CHNCPU_DATA {
    ch4_sint type;
	ch4_uint count;
    void *rawData;
    int rawDataByteSize;
    const char *filePath;
};

struct _CHNCPU_OP_TAG {
    int opCode;
    void *opCache;
};

struct _CHNCPU_BIOS {
    int (*bindFuncTable[CHNCPU_BIOS_OP_MAX + 1])(CHNCPU_RuntimeEnvironment *env, CHNCPU_OpTag *op, unsigned int prefix);
    int (*execFuncTable[CHNCPU_BIOS_OP_MAX + 1])(CHNCPU_RuntimeEnvironment *env, CHNCPU_OpTag *op);
    int (*printFuncTable[CHNCPU_BIOS_OP_MAX + 1])(CHNCPU_RuntimeEnvironment *env, CHNCPU_OpTag *op, FILE *file);
};

struct _CHNCPU_OP_TABLE_SET {
    int (*bindFuncTable[CHNCPU_OPECODE_MAX + 1])(CHNCPU_RuntimeEnvironment *env, CHNCPU_OpTag *op, unsigned int prefix);
    int (*execFuncTable[CHNCPU_OPECODE_MAX + 1])(CHNCPU_RuntimeEnvironment *env, CHNCPU_OpTag *op);
    int (*printFuncTable[CHNCPU_OPECODE_MAX + 1])(CHNCPU_RuntimeEnvironment *env, CHNCPU_OpTag *op, FILE *file);
};

struct _CHNCPU_LABEL_TAG {
	ch4_uint labelID;
	ch4_uint opt;	// 0:local(R3F Only) 1:public(can be stored to memory, pRegs)
	int mindex;
	ch4_sint pType;
	CHNCPU_LabelTag *next;
};

struct _CHNCPU_LABEL_SET {
	CHNCPU_LabelTag top;
};

struct _CHNCPU_POINTER_TAG {
	ch4_sint type;	// CHNCPU_PType_xx
	int mindex;     // for label
	int pindex;
    CHNCPU_Data *data;
};

struct _CHNCPU_RUN_TIME_ENVIRONMENT {
    CHNCPU_BIOS *bios;
	CHNCPU_OpTableSet *opSet;
	CHNCPU_LabelSet *labelSet;
    int iReg[CHNCPU_NUMBER_OF_IREG];
    int iRegBits[CHNCPU_NUMBER_OF_IREG];
	CHNCPU_PointerTag pReg[CHNCPU_NUMBER_OF_PREG];
    CHNCPU_OpTag mainmemory[CHNCPU_LENGTH_OF_MAIN_MEMORY];
    unsigned char *appbin0;
    int appbinsize;
	CH4Reader *appbinReader;
    unsigned int errFlags;
    int currentIndex;
    ch4_uint currentLabel;
};

struct _CHNCPU_VM_ARGS_BIND_DATA_TAG {
	int pReg, iReg;
    int flags;  // 0:read only, 1:write only
	CHNCPU_Data *data;
    const char *filePath;
    int pType;
};

struct _CHNCPU_VM_ARGS {
    const char *appPath;
    unsigned int VMFlags;
	CHNCPU_VMArgs_BindDataTag bindData[2];
	int bindDatas;
};



// @chncpu.c
int decodeHexString(char *src0, char *src1, unsigned char *dst0, unsigned char *dst1);
CHNCPU_RuntimeEnvironment *CHNCPU_CreateRuntimeEnvironment(CHNCPU_OpTableSet *opSet, CHNCPU_BIOS *bios);
int CHNCPU_LoadBinaryFromHexStringFilePath(CHNCPU_RuntimeEnvironment *env, const char *path);
int CHNCPU_PrepareBinaryForExecution(CHNCPU_RuntimeEnvironment *env);
int CHNCPU_Execute(CHNCPU_RuntimeEnvironment *env);
int CHNCPU_CHK_IsAvailableBits(CHNCPU_RuntimeEnvironment *env, ch4_uint bits);
int CHNCPU_AdjustValueForBit(CHNCPU_RuntimeEnvironment *env, int *value, ch4_uint bit, int prefix);

// @args.c
CHNCPU_VMArgs *bindVMArgs(int argc, const char *argv[]);
int CHNCPU_VMArgs_BindDataToRuntimeEnv(CHNCPU_RuntimeEnvironment *env, CHNCPU_VMArgs *args);
int CHNCPU_VMArgs_BindDataFromRuntimeEnv(CHNCPU_RuntimeEnvironment *env, CHNCPU_VMArgs *args);
const char *CHNCPU_VMArgs_GetNextAsString(int argc, const char *argv[], int *nextIndex);
int CHNCPU_VMArgs_GetNextAsPRegNum(int argc, const char *argv[], int *nextIndex);
int CHNCPU_VMArgs_GetNextAsIRegNum(int argc, const char *argv[], int *nextIndex);
int CHNCPU_VMArgs_GetNextAsUnsignedHexValue(int argc, const char *argv[], int *nextIndex);
int CHNCPU_VMArgs_GetNextAsSignedHexValue(int argc, const char *argv[], int *nextIndex);

// @opcode.c
CHNCPU_OpTableSet *CHNCPU_CreateOpTableSet(void);
int CHNCPU_Op_Init(CHNCPU_OpTableSet *env);
//
int CHNCPU_Op_LB_BindOperand(CHNCPU_RuntimeEnvironment *env, CHNCPU_OpTag *op, unsigned int prefix);
int CHNCPU_Op_LB_Execute(CHNCPU_RuntimeEnvironment *env, CHNCPU_OpTag *op);
int CHNCPU_Op_LB_PrintCode(CHNCPU_RuntimeEnvironment *env, CHNCPU_OpTag *op, FILE *file);
//
int CHNCPU_Op_LIMM_BindOperand(CHNCPU_RuntimeEnvironment *env, CHNCPU_OpTag *op, unsigned int prefix);
int CHNCPU_Op_LIMM_Execute(CHNCPU_RuntimeEnvironment *env, CHNCPU_OpTag *op);
int CHNCPU_Op_LIMM_PrintCode(CHNCPU_RuntimeEnvironment *env, CHNCPU_OpTag *op, FILE *file);
//
int CHNCPU_Op_PLIMM_BindOperand(CHNCPU_RuntimeEnvironment *env, CHNCPU_OpTag *op, unsigned int prefix);
int CHNCPU_Op_PLIMM_Execute(CHNCPU_RuntimeEnvironment *env, CHNCPU_OpTag *op);
int CHNCPU_Op_PLIMM_PrintCode(CHNCPU_RuntimeEnvironment *env, CHNCPU_OpTag *op, FILE *file);
//
int CHNCPU_Op_CND_BindOperand(CHNCPU_RuntimeEnvironment *env, CHNCPU_OpTag *op, unsigned int prefix);
int CHNCPU_Op_CND_Execute(CHNCPU_RuntimeEnvironment *env, CHNCPU_OpTag *op);
int CHNCPU_Op_CND_PrintCode(CHNCPU_RuntimeEnvironment *env, CHNCPU_OpTag *op, FILE *file);
//
int CHNCPU_Op_CALLBIOS_BindOperand(CHNCPU_RuntimeEnvironment *env, CHNCPU_OpTag *op, unsigned int prefix);
int CHNCPU_Op_CALLBIOS_Execute(CHNCPU_RuntimeEnvironment *env, CHNCPU_OpTag *op);
int CHNCPU_Op_CALLBIOS_PrintCode(CHNCPU_RuntimeEnvironment *env, CHNCPU_OpTag *op, FILE *file);
//
int CHNCPU_Op_LMEM_BindOperand(CHNCPU_RuntimeEnvironment *env, CHNCPU_OpTag *op, unsigned int prefix);
int CHNCPU_Op_LMEM_Execute(CHNCPU_RuntimeEnvironment *env, CHNCPU_OpTag *op);
int CHNCPU_Op_LMEM_PrintCode(CHNCPU_RuntimeEnvironment *env, CHNCPU_OpTag *op, FILE *file);
//
int CHNCPU_Op_TernaryReg_BindOperand(CHNCPU_RuntimeEnvironment *env, CHNCPU_OpTag *op, unsigned int prefix);
int CHNCPU_Op_TernaryRegBitwise_Execute(CHNCPU_RuntimeEnvironment *env, CHNCPU_OpTag *op);
int CHNCPU_Op_TernaryRegArithmetic_Execute(CHNCPU_RuntimeEnvironment *env, CHNCPU_OpTag *op);
int CHNCPU_Op_TernaryReg_PrintCode(CHNCPU_RuntimeEnvironment *env, CHNCPU_OpTag *op, FILE *file);
//
int CHNCPU_Op_CompareIReg_BindOperand(CHNCPU_RuntimeEnvironment *env, CHNCPU_OpTag *op, unsigned int prefix);
int CHNCPU_Op_CompareIReg_Execute(CHNCPU_RuntimeEnvironment *env, CHNCPU_OpTag *op);
int CHNCPU_Op_CompareIReg_PrintCode(CHNCPU_RuntimeEnvironment *env, CHNCPU_OpTag *op, FILE *file);
//
int CHNCPU_Op_DATA_BindOperand(CHNCPU_RuntimeEnvironment *env, CHNCPU_OpTag *op, unsigned int prefix);
int CHNCPU_Op_DATA_Execute(CHNCPU_RuntimeEnvironment *env, CHNCPU_OpTag *op);
int CHNCPU_Op_DATA_PrintCode(CHNCPU_RuntimeEnvironment *env, CHNCPU_OpTag *op, FILE *file);
CHNCPU_Data *CHNCPU_Op_DATA_GetData(CHNCPU_OpTag *op);
//
int CHNCPU_Op_Prefix2F_PrintWikiDoc(int opCode, FILE *file);

// @bios.c
CHNCPU_BIOS *CHNCPU_CreateBIOS(void);
int CHNCPU_BIOS_Init(CHNCPU_BIOS *bios);
int CHNCPU_BIOS_Op_putcReg_BindOperand(CHNCPU_RuntimeEnvironment *env, CHNCPU_OpTag *op, unsigned int prefix);
int CHNCPU_BIOS_Op_putcReg_Execute(CHNCPU_RuntimeEnvironment *env, CHNCPU_OpTag *op);
int CHNCPU_BIOS_Op_putcReg_PrintCode(CHNCPU_RuntimeEnvironment *env, CHNCPU_OpTag *op, FILE *file);

// @label.c
CHNCPU_LabelSet *CHNCPU_CreateLabelSet(void);
int CHNCPU_Label_Add(CHNCPU_LabelSet *lbSet, int mindex, ch4_uint labelID, ch4_uint opt, ch4_sint pType);
CHNCPU_LabelTag *CHNCPU_Label_GetTagFromLabelID(CHNCPU_LabelSet *lbSet, int labelID);

// @data.c
CHNCPU_Data *CHNCPU_DATA_CreateEmptyData(void);
void CHNCPU_DATA_FreeData(CHNCPU_Data *data);
int CHNCPU_DATA_AllocateRawDataMemory(CHNCPU_Data *data, ch4_sint dataType, ch4_uint dataCount, unsigned int *errFlags);
CHNCPU_Data *CHNCPU_DATA_CreateDataFromFileAsPType(const char dataPath[], ch4_sint pType);
int CHNCPU_DATA_AssignDataTagToPReg(CHNCPU_RuntimeEnvironment *env, ch4_uint p, CHNCPU_Data *data);
int CHNCPU_DATA_WriteRawDataToFile(CHNCPU_Data *data, const char dataPath[], int dataCount, int dataType);
int CHNCPU_DATA_ReadAtIndex(CHNCPU_Data *data, int index, unsigned int *errFlags);
int CHNCPU_DATA_CalculateByteSizeOfRawData(ch4_sint dataType, ch4_uint dataCount);
int CHNCPU_DATA_PrintData(CHNCPU_Data *data, FILE *file);

// @debug.c
void CHNCPU_DumpAppBin(CHNCPU_RuntimeEnvironment *env);
void CHNCPU_DumpIReg(CHNCPU_RuntimeEnvironment *env);
void CHNCPU_DumpPReg(CHNCPU_RuntimeEnvironment *env);
void CHNCPU_PrintErrorMessage(CHNCPU_RuntimeEnvironment *env);

#endif
